﻿using System;
using System.Collections;
using System.Collections.Generic;
using UnityEngine;

/// <summary> Base class for items that can be picked up and placed in the inventory, or held in the hand </summary>
[RequireComponent(typeof(Rigidbody))]
public abstract class Item : Interactable {
	public enum ExistenceMode { World, Inventory, PlayerHand, NPCHand }
	public ExistenceMode existenceMode { get { return _existenceMode; } }
	[SerializeField] private ExistenceMode _existenceMode;

	public override string Title { get { return displayName; } }
	public string displayName;

	public Rigidbody rigidbody { get { return _rigidbody != null ? _rigidbody : _rigidbody = GetComponent<Rigidbody>(); } }
	private Rigidbody _rigidbody;

	/// <summary> Icon in the inventory </summary>
	public Sprite icon;
	/// <summary> Position and size in inventory </summary>
	public RectInt shape;

	protected virtual void OnValidate() {
		shape.size = Vector2Int.Max(shape.size, Vector2Int.one);
	}

	public override void Interact(bool player) {
		if (Player.instance.inventory.AddItem(this)) {
			Player.instance.hotbar.AutoAddItem(this);
		} else {
			Debug.Log("No space in inventory");
		}
	}

	public abstract void PrimaryFire();

	public abstract void SecondaryFire();

	public abstract void Reload();

	public virtual void SetExistence(ExistenceMode mode) {
		switch (mode) {
			case ExistenceMode.World:
				transform.parent = null;
				gameObject.SetActive(true);
				rigidbody.useGravity = true;
				rigidbody.isKinematic = false;
				break;
			case ExistenceMode.Inventory:
				gameObject.SetActive(false);
				break;
			case ExistenceMode.PlayerHand:
				transform.parent = Player.instance.hand.transform;
				transform.localPosition = Vector3.zero;
				transform.localRotation = Quaternion.identity;
				transform.localScale = Vector3.one;
				gameObject.SetActive(true);
				rigidbody.useGravity = false;
				rigidbody.isKinematic = true;
				break;
			case ExistenceMode.NPCHand:
				gameObject.SetActive(true);
				rigidbody.useGravity = false;
				rigidbody.isKinematic = true;
				break;
		}
	}
}