﻿//Parallax occlusion mapping shader that creates a landscape in a quad
Shader "Parallax/Cutout" {
	Properties {
		
		[Header(Color)]
		_Color("Main Color", Color) = (1,1,1,1)
		_MainTex("Albedo", 2D) = "white" {}
		[Header(Metallic)][NoScaleOffset]
		_MetallicGlossMap("Metallic", 2D) = "white" {}
		_Metallic("Metallic", Range(0,1)) = 1.0
		_Glossiness("Smoothness", Range(0,1)) = 1.0
		[Header(Normal)][NoScaleOffset][Normal]
		_BumpMap("Normal", 2D) = "bump" {}
		_BumpScale("Scale", Float) = 1.0
		[Header(Height)][NoScaleOffset]
		_ParallaxMap("Height Map", 2D) = "white" {}
		_Parallax("Height", Range(0.0001,0.1)) = 0.001
		[Header(Occlusion)][NoScaleOffset]
		_OcclusionMap("Ambient Occlusion", 2D) = "white" {}
		_OcclusionStrength("AO strength", Range(0,2)) = 1.0
		[Header(Settings)]
		_Steps("Max Steps",Int) = 300
		_StepDist("StepDistance",Range(0.0001,0.01)) = 0.001
		_Cutoff("Cutoff", Range(0,1)) = 0.5
	}

		SubShader{
			Tags{ "RenderType" = "Opaque" }
			LOD 200
			CGPROGRAM

			#include "RaymarchedParallax.cginc"

	#pragma surface surf Standard vertex:vert
	#pragma target 3.0


		//Input
		float4 _Color;
		sampler2D _MainTex;
		sampler2D _MetallicGlossMap;
		half _Metallic;
		half _Glossiness;
		sampler2D _BumpMap;
		half _BumpScale;
		half _Cutoff;
		sampler2D _OcclusionMap;
		float _OcclusionStrength;


		struct Input
		{
			//What Unity can give you
			float2 uv_MainTex;

			//What you have to calculate yourself
			float3 tangentViewDir;

			fixed4 color : COLOR;

		};

		void vert(inout appdata_full i, out Input o)
		{
			UNITY_INITIALIZE_OUTPUT(Input, o);

			//Transform the view direction from world space to tangent space			
			float3 worldVertexPos = mul(unity_ObjectToWorld, i.vertex).xyz;
			float3 worldViewDir = worldVertexPos - _WorldSpaceCameraPos;

			//To convert from world space to tangent space we need the following
			//https://docs.unity3d.com/Manual/SL-VertexFragmentShaderExamples.html
			float3 worldNormal = UnityObjectToWorldNormal(i.normal);
			float3 worldTangent = UnityObjectToWorldDir(i.tangent.xyz);
			float3 worldBitangent = cross(worldNormal, worldTangent) * i.tangent.w * unity_WorldTransformParams.w;

			//Use dot products instead of building the matrix
			o.tangentViewDir = float3(
				dot(worldViewDir, worldTangent),
				dot(worldViewDir, worldNormal),
				dot(worldViewDir, worldBitangent)
				);
		}

	void surf(Input IN, inout SurfaceOutputStandard o)
	{
		float2 weightedTex = raymarchUv(IN.tangentViewDir, IN.uv_MainTex, _ParallaxMap, _Parallax);
		half finalHeight = getHeight(weightedTex, _ParallaxMap, _Parallax);

		clip((finalHeight/_Parallax)+_Cutoff);
		//Output

		//Albedo
		float3 c = tex2D(_MainTex, weightedTex);
		o.Albedo = c.rgb * _Color;

		//Metallic
		float4 m = tex2D(_MetallicGlossMap, weightedTex);
		o.Metallic = m.rgb * _Metallic;
		o.Smoothness = m.a * _Glossiness;

		//Normal
		fixed3 n = UnpackNormal(tex2D(_BumpMap, weightedTex));
		n.xy *= _BumpScale;
		n = normalize(n);
		o.Normal = n.rgb;

		//Ambient Occlusion
		half4 ao = lerp(half4(1, 1, 1, 1), tex2D(_OcclusionMap, weightedTex), _OcclusionStrength );
		o.Occlusion = ao.r;
	}
	ENDCG
	}
		FallBack "Diffuse"
}