﻿//Parallax occlusion mapping shader that creates a landscape in a quad
Shader "Parallax/Wind" {
	Properties {
		
		[Header(Color)]
		_Color("Main Color", Color) = (1,1,1,1)
		_MainTex("Albedo", 2D) = "white" {}
		[Header(Metallic)][NoScaleOffset]
		_MetallicGlossMap("Metallic", 2D) = "white" {}
		_Metallic("Metallic", Range(0,1)) = 1.0
		_Glossiness("Smoothness", Range(0,1)) = 1.0
		[Header(Normal)][NoScaleOffset][Normal]
		_BumpMap("Normal", 2D) = "bump" {}
		_BumpScale("Scale", Float) = 1.0
		[Header(Height)][NoScaleOffset]
		_ParallaxMap("Height Map", 2D) = "white" {}
		_Parallax("Height", Range(0.0001,0.1)) = 0.001
		[Header(Occlusion)][NoScaleOffset]
		_OcclusionMap("Ambient Occlusion", 2D) = "white" {}
		_OcclusionStrength("AO strength", Range(0,2)) = 1.0
		[Header(Wind)]
		_WindDisplacement("Wind Displacement", 2D) = "white" {}
		_WindDirection("Wind Direction", Vector) = (0,0,0)
		_WindStrength("Wind Strength", Float) = 0.1

		[Header(Settings)]
		_Steps("Max Steps",Int) = 300
		_StepDist("StepDistance",Range(0.0001,0.01)) = 0.001
	}

		SubShader{
			Tags{ "RenderType" = "Opaque"}
			LOD 200

			CGPROGRAM
#include "RaymarchedParallax.cginc"
	#pragma surface surf Standard vertex:vert
	#pragma target 3.0


		//Input
		float4 _Color;
		sampler2D _MainTex;
		sampler2D _MetallicGlossMap;
		half _Metallic;
		half _Glossiness;
		sampler2D _BumpMap;
		half _BumpScale;
		sampler2D _WindDisplacement;
		float2 _WindDirection;
		half _WindStrength;
		sampler2D _OcclusionMap;
		float _OcclusionStrength;


		struct Input
		{
			//What Unity can give you
			float2 uv_MainTex;
			float2 uv_WindDisplacement;

			//What you have to calculate yourself
			float3 tangentViewDir;

			fixed4 color : COLOR;

		};

		void vert(inout appdata_full i, out Input o)
		{
			UNITY_INITIALIZE_OUTPUT(Input, o);

			//Transform the view direction from world space to tangent space			
			float3 worldVertexPos = mul(unity_ObjectToWorld, i.vertex).xyz;
			float3 worldViewDir = worldVertexPos - _WorldSpaceCameraPos;

			//To convert from world space to tangent space we need the following
			//https://docs.unity3d.com/Manual/SL-VertexFragmentShaderExamples.html
			float3 worldNormal = UnityObjectToWorldNormal(i.normal);
			float3 worldTangent = UnityObjectToWorldDir(i.tangent.xyz);
			float3 worldBitangent = cross(worldNormal, worldTangent) * i.tangent.w * unity_WorldTransformParams.w;

			//Use dot products instead of building the matrix
			o.tangentViewDir = float3(
				dot(worldViewDir, worldTangent),
				dot(worldViewDir, worldNormal),
				dot(worldViewDir, worldBitangent)
				);
		}

	//Animate the uv coordinates so the landscape is moving
	float2 animateUV(float2 texturePos)
	{
		texturePos.xy += _WindDirection.xy * _Time[0];

		return texturePos;
	}

	void surf(Input IN, inout SurfaceOutputStandard o)
	{
		//uv after wind calculations
		float2 windUV = IN.uv_MainTex;

		float2 disp = tex2D(_WindDisplacement, animateUV(IN.uv_WindDisplacement));
		float2 height = tex2D(_ParallaxMap, IN.uv_MainTex);
		float2 windoffset = normalize(_WindDirection.xy);
		//windoffset *= disp[0] * _WindStrength * (getHeight(IN.uv_MainTex) - 1);
		windoffset *= disp[0] * _WindStrength;
		windoffset *= (height[0]*4)-1;
		windoffset -= _WindStrength*0.5;

		windUV += windoffset;

		float2 weightedTex = raymarchUv(IN.tangentViewDir, windUV, _ParallaxMap, _Parallax);
		//Output

		//Albedo
		float3 c = tex2D(_MainTex, weightedTex);
		o.Albedo = c.rgb * _Color;
	
		//Metallic
		float4 m = tex2D(_MetallicGlossMap, weightedTex);
		m.r *= _Metallic;
		m.a *= _Glossiness;
		o.Metallic = m.rgb;
		o.Smoothness = m.a;

		//Normal
		float3 n = UnpackNormal(tex2D(_BumpMap, weightedTex));
		n.xy *= _BumpScale;
		n = normalize(n);
		o.Normal = n.rgb;

		//Ambient Occlusion
		half4 ao = lerp(half4(1, 1, 1, 1), tex2D(_OcclusionMap, weightedTex), _OcclusionStrength );
		o.Occlusion = ao.r;
	}
	ENDCG
	}
		FallBack "Diffuse"
}