﻿using UnityEngine;

/// <summary>
/// Author: Thor Brigsted
/// -
/// Usage: Add to empty GameObject. Place Camera as child.
/// </summary>
[RequireComponent(typeof(CharacterController))]
public class PlayerController : MonoBehaviour {

	public static PlayerController instance;

	#region Inspector variables
	[Header("Settings")]
	[Tooltip("Higher values if you GOTTA GO FAST!")]
	public float movementSpeed = 6f;
	[Tooltip("Sprint speed. Set to something lower than movement speed to sneak around")]
	public float sprintSpeed = 10f;
	[Tooltip("Lower values make the floor seem slippery (0 or less disables)")]
	public float friction = 25f;
	[Tooltip("Lower values requires you to buy a larger mousepad")]
	public float mouseSens = 2f;
	[Tooltip("Lower values give a cinematic feeling (0 or less disables)")]
	public float mouseSmoothing = 0.05f;

	[Header("Air")]
	public AudioClip jumpSound;
	[Tooltip("How much force will the player jump with")]
	public float jumpForce = 10f;
	[Tooltip("Influences how fast the player falls")]
	public float playerMass = 0.05f;
	[Tooltip("If enabled, the player will automatically lose momentum if he releases movement keys while in air")]
	public bool airbrakes = true;
	[Range(0, 1)]
	[Tooltip("How much control does the player have while in the air")]
	public float aerialControl = 0.5f;
	[Tooltip("Press to toggle pause")]
	public KeyCode jumpKey = KeyCode.Space;

	[Header("Crouch")]
	[Range(0, 1)]
	[Tooltip("Height multiplier when crouching. This is a percentage of your standing height.")]
	public float crouchHeightMult = 0.5f;
	[Tooltip("Speed of crouching")]
	public float crouchSpeed = 3f;
	[Tooltip("Speed of sprinting while crouching. Or crouching while sprinting")]
	public float crouchSprintSpeed = 5f;
	[Tooltip("Press to toggle pause")]
	public KeyCode crouchKey = KeyCode.LeftControl;

	[Header("Keymapping")]
	[Tooltip("Press to toggle flying around like a ghost. Set to 'None' to disable")]
	public KeyCode noclipKey = KeyCode.V;
	[Tooltip("Press to toggle pause")]
	public KeyCode pauseKey = KeyCode.Escape;
	#endregion

	#region Public variables
	[HideInInspector]
	public bool isPaused = false, isCrouching = false;
	#endregion

	#region Private variables
	private Vector3 currentVelocity;
	private Vector2 targetMousePos;
	private CharacterController controller;
	private Camera cam;
	private bool noclip_on;
	private float playerHeight;
	private float camHeight;
	private float crouch_t = 1;
	private AudioSource audioSource { get { return _audiosource != null ? _audiosource : _audiosource = GetComponent<AudioSource>(); } }
	private AudioSource _audiosource;
	#endregion

	private void Awake() {
		instance = this;
		controller = GetComponent<CharacterController>();
		playerHeight = controller.height;
		cam = GetComponentInChildren<Camera>();
		if (cam == null) {
			Debug.LogError("You need to place a camera as child of the player GameObject");
			enabled = false;
		} else camHeight = cam.transform.localPosition.y;

		if (playerHeight * crouchHeightMult < controller.radius * 2) {
			Debug.LogWarning("Player crouch height (" + playerHeight * crouchHeightMult + ") lower than double radius (" + controller.radius * 2 + ") in CharacterController. Crouching will be capped. Consider decreasing player radius in Character Controller to fix.");
		}
		targetMousePos.x = controller.transform.rotation.eulerAngles.y;
	}

	private void Update() {

		//Toggle pause
		if (Input.GetKeyDown(pauseKey)) TogglePause();

		if (!Cursor.visible) {
			//Toggle noclip
			if (Input.GetKeyDown(noclipKey)) ToggleNoclip();

			//MouseLook
			MouseControls();

			//Crouch
			Crouch();

			//Movement
			if (noclip_on) NoclipControls();
			else MovementControls();
		}

	}

	#region Private methods
	private void Crouch() {
		if (!controller.isGrounded) return;

		float crouchHeight = playerHeight * crouchHeightMult;
		if (crouchHeight < controller.radius * 2) crouchHeight = controller.radius * 2;
		isCrouching = Input.GetKey(crouchKey);

		//Crouch
		if (isCrouching) {

			controller.height = crouchHeight;
			controller.center = new Vector3(0, -(playerHeight - crouchHeight) / 2, 0);
			//Camera smoothing
			if (crouch_t > 0) {
				crouch_t = Mathf.Clamp01(crouch_t - Time.deltaTime * 4f);
				float cam_h = AnimationCurve.EaseInOut(0f, camHeight - (playerHeight - crouchHeight), 1f, camHeight).Evaluate(crouch_t);
				cam.transform.localPosition = new Vector3(0, cam_h, 0);
			}
		}
		//Stand
		else {
			controller.height = playerHeight;
			controller.center = Vector3.zero;
			//Camera smoothing
			if (crouch_t < 1) {
				crouch_t = Mathf.Clamp01(crouch_t + Time.deltaTime * 4f);
				float cam_h = AnimationCurve.EaseInOut(0f, camHeight - (playerHeight - crouchHeight), 1f, camHeight).Evaluate(crouch_t);
				cam.transform.localPosition = new Vector3(0, cam_h, 0);
			}
		}
	}

	private void MouseControls() {
		targetMousePos.x += Input.GetAxisRaw("Mouse X") * mouseSens;
		targetMousePos.y -= Input.GetAxisRaw("Mouse Y") * mouseSens;
		targetMousePos.y = Mathf.Clamp(targetMousePos.y, -90f, 90f);

		//Keep targetMousePos.x from ever growing large enough to experience floating point inaccuracy
		if (targetMousePos.x > 360f) targetMousePos.x -= 360f;
		if (targetMousePos.x < -360f) targetMousePos.x += 360f;

		//Rotate the player on the Y-axis and the camera on the X-axis
		if (mouseSmoothing > 0) {
			transform.rotation = Quaternion.Lerp(transform.rotation, Quaternion.Euler(0, targetMousePos.x, 0), (1f / mouseSmoothing) * Time.deltaTime);
			cam.transform.localRotation = Quaternion.Lerp(cam.transform.localRotation, Quaternion.Euler(targetMousePos.y, 0, 0), (1f / mouseSmoothing) * Time.deltaTime);
		} else {
			transform.rotation = Quaternion.Euler(0, targetMousePos.x, 0);
			cam.transform.localRotation = Quaternion.Euler(targetMousePos.y, 0, 0);
		}
	}

	private void NoclipControls() {
		//Noclip moves you directly, without regard for collisions and gravity
		Vector3 mov = GetControlVector();
		mov = cam.transform.rotation * mov;
		mov *= Input.GetKey(KeyCode.LeftShift) ? sprintSpeed : movementSpeed;
		if (friction > 0) currentVelocity = Vector3.MoveTowards(currentVelocity, mov, friction * Time.deltaTime);
		else currentVelocity = mov;
		transform.position += currentVelocity * 0.1f;
	}

	private void MovementControls() {
		Vector3 mov = GetControlVector();
		mov = transform.rotation * mov;

		if (isCrouching) mov *= Input.GetKey(KeyCode.LeftShift) ? crouchSprintSpeed : crouchSpeed;
		else mov *= Input.GetKey(KeyCode.LeftShift) ? sprintSpeed : movementSpeed;

		if (controller.isGrounded) {
			//Jump
			if (Input.GetKeyDown(jumpKey)) {
				currentVelocity.y += jumpForce;
				audioSource.PlayOneShot(jumpSound);
				//Apply smoothing on x and z
				if (friction > 0) currentVelocity = Vector3.MoveTowards(currentVelocity, mov, friction * Time.deltaTime);
				else currentVelocity = mov;
				currentVelocity.y = jumpForce;
				controller.Move(currentVelocity * Time.deltaTime);
			} else {
				//Apply smoothing on x and z
				if (friction > 0) currentVelocity = Vector3.MoveTowards(currentVelocity, mov, friction * Time.deltaTime);
				else currentVelocity = mov;
				controller.Move(new Vector3(currentVelocity.x * Time.deltaTime, -0.05f, currentVelocity.z * Time.deltaTime));
			}
		} else {
			//Apply gravity
			currentVelocity += Physics.gravity * playerMass * Time.deltaTime;

			//Aerial movement
			if (airbrakes || mov != Vector3.zero) {
				float preserveY = currentVelocity.y;
				if (friction > 0) currentVelocity = Vector3.MoveTowards(currentVelocity, mov, aerialControl * friction * Time.deltaTime);
				else currentVelocity = Vector3.MoveTowards(currentVelocity, mov, aerialControl * Time.deltaTime);
				currentVelocity.y = preserveY;
			}
			controller.Move(currentVelocity * Time.deltaTime);
		}

	}

	private Vector3 GetControlVector() {
		Vector3 mov = Vector3.zero;
		if (Input.GetKey(KeyCode.W)) mov += Vector3.forward;
		if (Input.GetKey(KeyCode.A)) mov += Vector3.left;
		if (Input.GetKey(KeyCode.S)) mov += Vector3.back;
		if (Input.GetKey(KeyCode.D)) mov += Vector3.right;
		mov.Normalize();
		return mov;
	}
	#endregion

	#region Public methods
	public void ToggleNoclip() {
		ToggleNoclip(!noclip_on);
	}
	public void ToggleNoclip(bool on) {
		noclip_on = on;
		controller.enabled = !on;
	}

	public void TogglePause() {
		TogglePause(!isPaused);
	}
	public void TogglePause(bool on) {
		isPaused = on;
		if (on) CursorManager.AddUser(this);
		else CursorManager.RemoveUser(this);
	}
	#endregion
}