#include <windows.h>
#include <vector>
#include <string>

#include "fixapp.h"
#include "resource.h"
#include "Shlwapi.h"
#include "shlobj.h"
#include "fixapp_helpers.h"
#include "engine.h"
#include "unrender.h"

static std::vector<std::wstring> *renderers;

static wchar_t userini[MAX_PATH];
static wchar_t deusexini[MAX_PATH];
static wchar_t renderer[MAX_PATH];
static wchar_t tempPath[MAX_PATH];
static wchar_t deusExPath[MAX_PATH];

static const int FOV_4_3 = (int) 75;
static const int FOV_16_10 = (int) 85.2812718522;
static const int FOV_5_4 = (int) 71.4590506462;
static const int FOV_15_9 = (int) 87.61661884+1; //+1 so it rounds up to 88;
static const int FOV_16_9 = (int) 91.3154482287;

static const char BPP_16 = 16;
static const char BPP_32 = 32;

struct res
{
	unsigned short X;
	unsigned short Y;
	int fov;
};

struct res resolutions[] = {
	{640,480,FOV_4_3},
	{800,600,FOV_4_3},
	{1024,768,FOV_4_3},
	{1152,864,FOV_4_3},
	{1280,960,FOV_4_3},
	{1280,1024,FOV_5_4},
	{1400,1050,FOV_4_3},
	{1600,1200,FOV_4_3},
	{1280,800,FOV_16_10},
	{1440,900,FOV_16_10},
	{1680,1050,FOV_16_10},
	{1920,1200,FOV_16_10},
	{1280,720,FOV_16_9},
	{1920,1080,FOV_16_9},
	{1280,768,FOV_15_9},
	{1366,768,FOV_16_10}
};

void applyFOV(HWND hwndDlg, int FOV)
{
	int fovButton;
	switch (FOV)
	{
	case FOV_4_3:
		fovButton = RADIO_FOVDEFAULT;
		break;
	case FOV_5_4:
		fovButton = RADIO_FOV_5_4;
		break;
	case FOV_16_10:
		fovButton = RADIO_FOV_16_10;
		break;
	case FOV_16_9:
		fovButton = RADIO_FOV_16_9;
		break;
	case FOV_15_9:
		fovButton = RADIO_FOV_15_9;
		break;
	default:
		fovButton = RADIO_FOVCUSTOM;
		SetDlgItemInt(hwndDlg,TXT_FOV,FOV,false);
		break;

	}
	CheckRadioButton(hwndDlg,RADIO_FOVDEFAULT,RADIO_FOVCUSTOM,fovButton);
}

void readSettings(HWND hwndDlg)
{
	//Read the game's settings and apply them to the window. Also enabled the controls.


	//Read settings
	wchar_t rootWindow[1024];
	int resX = GetPrivateProfileInt(L"WinDrv.WindowsClient",L"FullscreenViewportX",1024,deusexini);
	int resY = GetPrivateProfileInt(L"WinDrv.WindowsClient",L"FullscreenViewportY",768,deusexini);
	int bitDepth = GetPrivateProfileInt(L"WinDrv.WindowsClient",L"FullscreenColorBits",BPP_32,deusexini);
	GetPrivateProfileString(L"Engine.Engine",L"root",L"DeusEx.DeusExRootWindow",rootWindow,sizeof(rootWindow),deusexini);
	int FOV = GetPrivateProfileInt(L"Engine.PlayerPawn",L"DesiredFOV",FOV_4_3,userini); //DefaultFOV is the other setting that manages the FOV
	GetPrivateProfileString(L"Engine.Engine",L"GameRenderDevice",L"SoftDrv.SoftwareRenderDevice",renderer,sizeof(renderer),deusexini);
	bool detailTextures = GetPrivateProfileBool(renderer,L"DetailTextures",deusexini);
	bool noMouseAccel = GetPrivateProfileBool(L"WinDrv.WindowsClient",L"UseDirectInput",deusexini);
	int sndLatency = GetPrivateProfileInt(L"Galaxy.GalaxyAudioSubsystem",L"Latency",40,deusexini);
	bool directSound =  GetPrivateProfileBool(L"Galaxy.GalaxyAudioSubsystem",L"UseDirectSound",deusexini);
	bool fullScreen  =  GetPrivateProfileBool(L"WinDrv.WindowsClient",L"StartupFullScreen",deusexini);
	int FPSLimit = GetPrivateProfileInt(L"WinDrv.WindowsClient",L"FPSLimit",100,deusexini);

	//Apply settings to window
	int i;
	for(i=0;i<sizeof(resolutions)/sizeof(res);i++) //If the current resolution matches one of the presets, select it
	{
		if(resX == resolutions[i].X && resY == resolutions[i].Y)
		{
			CheckRadioButton(hwndDlg,RADIO_RESCOMMON,RADIO_RESCUSTOM,RADIO_RESCOMMON);
			SendDlgItemMessage(hwndDlg,COMBO_RESOLUTION,CB_SETCURSEL,(WPARAM) i,0);
			break;
		}
	}
	if(i==sizeof(resolutions)/sizeof(res)) //No match found, show the current resolution as a custom one
	{
		CheckRadioButton(hwndDlg,RADIO_RESCOMMON,RADIO_RESCUSTOM,RADIO_RESCUSTOM);
		SetDlgItemInt(hwndDlg,TXT_RESX,resX,false);
		SetDlgItemInt(hwndDlg,TXT_RESY,resY,false);
	}

	if(bitDepth == BPP_16)
		CheckRadioButton(hwndDlg,RADIO_16BIT,RADIO_32BIT,RADIO_16BIT);
	else
		CheckRadioButton(hwndDlg,RADIO_16BIT,RADIO_32BIT,RADIO_32BIT);

	applyFOV(hwndDlg,FOV);

	strToUpper(rootWindow);
	if(wcscmp(L"OTPUIFIX.OTPROOTWINDOW",rootWindow)==0)	
		CheckRadioButton(hwndDlg,RADIO_GUIDEFAULT,RADIO_GUIFIX,RADIO_GUIFIX);
	else
		CheckRadioButton(hwndDlg,RADIO_GUIDEFAULT,RADIO_GUIFIX,RADIO_GUIDEFAULT);
	
	for(size_t i=0;i<renderers->size();i++)
	{
		if(renderers->at(i).compare(renderer)==0)	
			SendDlgItemMessage(hwndDlg,COMBO_RENDERER,CB_SETCURSEL,(WPARAM) i,0);
	}
	
	
	if(detailTextures)	
		CheckDlgButton(hwndDlg,CHK_DETAILTEX,true);

	if(noMouseAccel)
		CheckDlgButton(hwndDlg,CHK_NOMOUSEACCEL,true);
	
	if(directSound)
		CheckDlgButton(hwndDlg,CHK_DIRECTSOUND,true);

	if(fullScreen)
		CheckDlgButton(hwndDlg,CHK_FULLSCREEN,true);
	
	SetDlgItemInt(hwndDlg,TXT_LATENCY,sndLatency,false);
	SetDlgItemInt(hwndDlg,TXT_FPSLIMIT,FPSLimit,false);
	
}


void populateDiag(HWND hwndDlg)
{

	//Populate the resolution combobox
	char buffer[10];
	for(int i=0;i<sizeof(resolutions)/sizeof(res);i++)
	{
		_itoa_s(resolutions[i].X,buffer,10,10);
		strcat_s(buffer,10,"x");
		_itoa_s(resolutions[i].Y,buffer+strlen(buffer),10-strlen(buffer),10);
		SendDlgItemMessageA(hwndDlg,COMBO_RESOLUTION,CB_ADDSTRING,0,(LPARAM) buffer);
	}
	SendDlgItemMessage(hwndDlg,COMBO_RESOLUTION,CB_SETCURSEL,(WPARAM)0,0);

	//Renderers (based on UnEngineWin.h), requires appInit() to have been called
	TArray<FRegistryObjectInfo> Classes;
	Classes.Empty();

	UObject::GetRegistryObjects( Classes, UClass::StaticClass(), URenderDevice::StaticClass(), 0 );
	for( TArray<FRegistryObjectInfo>::TIterator It(Classes); It; ++It )
	{
		FString Path=It->Object, Left, Right, Temp;
		if( Path.Split(TEXT("."),&Left,&Right)  )
		{
			{
				const wchar_t* desc = Localize(*Right,TEXT("ClassCaption"),*Left);
				if(SendDlgItemMessage(hwndDlg,COMBO_RENDERER,CB_FINDSTRINGEXACT,-1,(LPARAM) desc)==CB_ERR)
				{
					SendDlgItemMessage(hwndDlg,COMBO_RENDERER,CB_ADDSTRING,0,(LPARAM) desc);
					renderers->push_back(std::wstring((wchar_t*)Path.GetCharArray().GetData()));
				}
			}
		}
	}

}

bool applySettings(HWND hwndDlg)
{
	

	//GUI scaling fix
	if(IsDlgButtonChecked(hwndDlg,RADIO_GUIFIX))
		WritePrivateProfileString(L"Engine.engine",L"root",L"otpUIfix.otpRootWindow",deusexini);
	else
		WritePrivateProfileString(L"Engine.engine",L"root",L"DeusEx.DeusExRootWindow",deusexini);	

	//Bit depth
	if(IsDlgButtonChecked(hwndDlg,RADIO_16BIT))
		WritePrivateProfileInt(L"WinDrv.WindowsClient",L"FullscreenColorBits",BPP_16,deusexini);
	else
		WritePrivateProfileInt(L"WinDrv.WindowsClient",L"FullscreenColorBits",BPP_32,deusexini);

	//FOV
	int fov;
	if(IsDlgButtonChecked(hwndDlg,RADIO_FOVDEFAULT))
		fov = FOV_4_3;
	else if(IsDlgButtonChecked(hwndDlg,RADIO_FOV_5_4))
		fov = FOV_5_4;
	else if(IsDlgButtonChecked(hwndDlg,RADIO_FOV_16_9))
		fov = FOV_16_9;
	else if(IsDlgButtonChecked(hwndDlg,RADIO_FOV_16_10))
		fov = FOV_16_10;
	else if(IsDlgButtonChecked(hwndDlg,RADIO_FOV_15_9))
		fov = FOV_15_9;
	else
		fov = GetDlgItemInt(hwndDlg,TXT_FOV,false,false);
	WritePrivateProfileInt(L"Engine.PlayerPawn",L"DesiredFOV",fov,userini);
	 WritePrivateProfileInt(L"Engine.PlayerPawn",L"DefaultFOV",fov,userini);

	//Resolution
	int resX, resY;
	if(IsDlgButtonChecked(hwndDlg,RADIO_RESCOMMON))
	{
		int i = SendDlgItemMessageA(hwndDlg,COMBO_RESOLUTION,CB_GETCURSEL,0,0);		
		resX = resolutions[i].X;
		resY = resolutions[i].Y;
	}
	else
	{
		resX = GetDlgItemInt(hwndDlg,TXT_RESX,false,false);
		resY = GetDlgItemInt(hwndDlg,TXT_RESY,false,false);
	}	
	WritePrivateProfileInt(L"WinDrv.WindowsClient",L"FullscreenViewportX",resX,deusexini);
	WritePrivateProfileInt(L"WinDrv.WindowsClient",L"FullscreenViewportY",resY,deusexini);

	//Disable mouse scaling
	WritePrivateProfileBool(L"WinDrv.WindowsClient",L"UseDirectInput",IsDlgButtonChecked(hwndDlg,CHK_NOMOUSEACCEL)!=0,deusexini);
	//DirectSound
	WritePrivateProfileBool(L"Galaxy.GalaxyAudioSubsystem",L"UseDirectSound",IsDlgButtonChecked(hwndDlg,CHK_DIRECTSOUND)!=0,deusexini);
	//Latency
	WritePrivateProfileInt(L"Galaxy.GalaxyAudioSubsystem",L"Latency",GetDlgItemInt(hwndDlg,TXT_LATENCY,false,false),deusexini);
	//Fullscreen
	WritePrivateProfileBool(L"WinDrv.WindowsClient",L"StartupFullSCreen",IsDlgButtonChecked(hwndDlg,CHK_FULLSCREEN)!=0,deusexini);
	//FPS Limit
	WritePrivateProfileInt(L"WinDrv.WindowsClient",L"FPSLimit",GetDlgItemInt(hwndDlg,TXT_FPSLIMIT,false,false),deusexini);
		

	//Renderer
	int rendererIndex = SendDlgItemMessageA(hwndDlg,COMBO_RENDERER,CB_GETCURSEL,0,0);		
	WritePrivateProfileString(L"Engine.Engine",L"GameRenderDevice",renderers->at(rendererIndex).c_str(),deusexini);
	//Detail textures
	WritePrivateProfileBool(renderers->at(rendererIndex).c_str(),L"DetailTextures",IsDlgButtonChecked(hwndDlg,CHK_DETAILTEX)!=0,deusexini);

	return true;
}

INT_PTR CALLBACK fixerDialogProc(HWND hwndDlg,UINT uMsg,WPARAM wParam,LPARAM lParam)
{
	switch (uMsg)
	{

	case WM_INITDIALOG:
	{
		renderers = new std::vector<std::wstring>();
		//Get DX system path
		GetModuleFileName(NULL,deusExPath,MAX_PATH);
		*PathFindFileName(deusExPath)=NULL;
		PathRemoveBackslash(deusExPath);
		PathQuoteSpaces(deusExPath);

		//Get settings path
		wchar_t userDataPath[MAX_PATH];
		SHGetFolderPath(NULL,CSIDL_PERSONAL,NULL,NULL,userDataPath);
		PathAppend(userDataPath,L"Deus Ex\\System");
		PathCombine(deusexini,userDataPath,L"deusex.ini");
		PathCombine(userini,userDataPath,L"user.ini");

		GetTempPath(MAX_PATH,tempPath);

		SendMessage(hwndDlg, WM_SETICON, ICON_BIG, (LPARAM) LoadIcon((HINSTANCE) GetWindowLong(hwndDlg,GWL_HINSTANCE), MAKEINTRESOURCE(IDI_ICON)));
	
		populateDiag(hwndDlg);
		readSettings(hwndDlg);
		
	}
		return false;

	
	case WM_COMMAND:
	{
		switch(HIWORD(wParam))
		{
			case CBN_SELCHANGE:
			{
		
				switch (LOWORD(wParam))
				{
				case (COMBO_RESOLUTION):
					{
						LRESULT r = SendDlgItemMessage(hwndDlg,COMBO_RESOLUTION,CB_GETCURSEL,0,0);
						applyFOV(hwndDlg,resolutions[r].fov);
					}
					return true;
				}
				break;
			}
			
			case EN_SETFOCUS:
			{
				switch (LOWORD(wParam))
				{
					case(TXT_FOV):
						CheckRadioButton(hwndDlg,RADIO_FOVDEFAULT,RADIO_FOVCUSTOM,RADIO_FOVCUSTOM);
						return true;

					case (TXT_RESX):
					case (TXT_RESY):
						CheckRadioButton(hwndDlg,RADIO_RESCOMMON,RADIO_RESCUSTOM,RADIO_RESCUSTOM);
						return true;

				}
				break;

			}
			default:
			{
				switch (LOWORD(wParam))
				{
				
					case (IDOK):				
						applySettings(hwndDlg);
					case (IDCANCEL):
						delete renderers;
						EndDialog(hwndDlg,0);
						return true;
					
					}
				break;
				}
		}
		break;
	}
	

	case WM_CLOSE:
		EndDialog(hwndDlg,0);
		return true;

	default:
		break;
	}

	return false;
}