#include <windows.h>
#include <CommCtrl.h>
#include <Shlwapi.h>
#include <cstdio>
#include <limits>

#include <cstring>
#include <vector>
#include <string>
#include <algorithm>

#include <Core.h>
#include <FConfigCacheIni.h>

#include "datadirdialog.h"
#include "fixapp.h"
#include "misc.h"

static const wchar_t* PATHS = L"Paths";

/**
Default directories the game starts out with and should be ignored.
*/
static wchar_t* standardDirs[] =
{
	L"editorres",L"goty_1",L"help",L"manual",L"maps",L"music",L"save",L"sounds",L"system",L"textures",
};

/**
Extensions that make sense to show
*/
static wchar_t* supportedExtensions[] =
{
	L".utx",L".umx",L".dx",L".u",L".unr",L".uax",
};

static HWND list;

static void moveListItem(int index, int newIndex)
{
	SendMessage(list,WM_SETREDRAW,static_cast<WPARAM>(FALSE),0);

	LVITEM item = {};	
	item.iItem = index;
	item.mask = LVIF_TEXT|LVIF_STATE;
	item.stateMask = -1;
	wchar_t buffer[MAX_PATH];
	item.pszText = buffer;
	item.cchTextMax = _countof(buffer);

	ListView_GetItem(list,&item);
	BOOL checked = ListView_GetCheckState(list,index); //Needs to be re-done when using InsertItem
	ListView_DeleteItem(list,index);
	item.iItem = newIndex;
	ListView_InsertItem(list,&item);
	ListView_SetCheckState(list,newIndex,checked);
	SendMessage(list,WM_SETREDRAW,static_cast<WPARAM>(TRUE),0);
}

static void searchDir(const wchar_t* targetDir, HWND list, bool root, wchar_t* cd)
{
	WIN32_FIND_DATA findData;
	wchar_t targetPath[MAX_PATH];
	PathCombine(targetPath,targetDir,L"*");
	HANDLE dir = FindFirstFile(targetPath,&findData);
	
	//Find subdirectories
	if(dir!=INVALID_HANDLE_VALUE)
	{
		TMap<FString,int> extensions;
		int mostCommonExtensionNum = 0;
		wchar_t mostCommonExtension[MAX_PATH];

		do
		{
			if(wcscmp(findData.cFileName,L"..") != 0 && wcscmp(findData.cFileName,L".")!=0)
			{
				if(findData.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) //Directory, search recursively
				{
					//Check if directory is one of the standard ones, if so don't process it any further
					bool isStandardDir = false;
					if(root)
					{
						wchar_t lowerCase[MAX_PATH];
						wcscpy_s(lowerCase,_countof(lowerCase),findData.cFileName);
						CharLower(lowerCase);
						auto cmp = [&lowerCase] (const wchar_t* str) {return wcscmp(str,lowerCase)==0;};
						if(std::find_if(standardDirs,standardDirs+_countof(standardDirs),cmp)<standardDirs+_countof(standardDirs))
						{
							isStandardDir = true;							
						}
						
					}
					if(!isStandardDir)
					{	
						wchar_t childPath[MAX_PATH];
						PathCombine(childPath,targetDir,findData.cFileName);
						searchDir(childPath,list,false,cd);
					}
				}
				else if(!root) //Files (and not the starting 'System' directory), process)
				{
					//Tally file extensions in directory and pick the most frequent one
					wchar_t* extensionPtr = CharLower(PathFindExtension(findData.cFileName));

					//Only count useful extensions
					auto cmp = [&] (const wchar_t* str) {return wcscmp(str,extensionPtr)==0;};
					if(std::find_if(supportedExtensions,supportedExtensions+_countof(supportedExtensions),cmp)<supportedExtensions+_countof(supportedExtensions))
					{
						int extensionNum = 0;			
						int* numPtr;
						if((numPtr=extensions.Find(extensionPtr))!=nullptr)
						{
							extensionNum = *numPtr;
						}
								
						extensions.Set(extensionPtr,++extensionNum);
						if(extensionNum>mostCommonExtensionNum) //New extension is the one with the most files
						{
							wcscpy_s(mostCommonExtension,_countof(mostCommonExtension),extensionPtr);
							mostCommonExtensionNum = extensionNum;
						}					
					}

				}
			}
		} while(FindNextFile(dir, &findData) != 0);

		FindClose(dir);

		//Done processing directory, see if we found anything
		if(mostCommonExtensionNum > 0)
		{
			//Convert to relative path
            wchar_t buf[MAX_PATH];
			PathRelativePathTo(buf,cd,FILE_ATTRIBUTE_DIRECTORY,targetPath,FILE_ATTRIBUTE_DIRECTORY);
			PathAddExtension(buf,mostCommonExtension);
			
			//Add item to list
			LVITEM item = {};
			item.iItem = std::numeric_limits<int>::max();
			item.pszText = buf;
			item.mask = LVIF_TEXT;
			LRESULT index = SendMessage(list,LVM_INSERTITEM,1,reinterpret_cast<LPARAM> (&item));
	
		}

		
	}
}

static void populateList()
{	
	//Uses USystem::Paths instead of GConfig so the order of entries is preserved
	wchar_t cd[MAX_PATH];
	GetCurrentDirectory(_countof(cd),cd);
	wchar_t cdUp[MAX_PATH];
	PathCombine(cdUp,cd,L"..");
	searchDir(cdUp, list,true,cd);

	//Give items in ini file priority and check their checkboxes
	for(int i=GSys->Paths.Num()-1;i>=0;i--)
	{
		LVFINDINFO findInfo = {};
		findInfo.flags = LVFI_STRING;
		findInfo.psz = FSTRINGCAST(GSys->Paths(i));
		int listIndex = ListView_FindItem(list,-1,&findInfo);
		if(listIndex>-1)
		{
			ListView_SetCheckState(list,listIndex,TRUE);
			moveListItem(listIndex,0);			
		}
	}
			
}

static void populateConfig()
{
	//Directly manipulating the USystem::Paths member resulted in crashes, so we use GConfig
	FString dummy;

	//Clear existing items
	TMultiMap<FString,FString>* section = GConfig->GetSectionPrivate(L"Core.System",FALSE,FALSE);
	

	//Remove path items
	for(int i=0;i<ListView_GetItemCount(list);i++)
	{
		wchar_t text[MAX_PATH];
		ListView_GetItemText(list,i,0,text,_countof(text));
		section->RemovePair(PATHS,text);
	}

	//Get the leftover items, which are the default directories
	TArray<FString> defaults;
	section->MultiFind(PATHS,defaults);
	section->Remove(PATHS);

	//Add path items to actual list
	for(int i=0;i<ListView_GetItemCount(list);i++)
	{
		wchar_t text[MAX_PATH];
		ListView_GetItemText(list,i,0,text,_countof(text));

		if(ListView_GetCheckState(list,i))
		{			
			section->Add(PATHS,text);
		}
	}

	//Re-add the defaults
	for(int i=0;i<defaults.Num();i++)
	{
		section->Add(PATHS,FSTRINGCAST(defaults(i)));
	}
	
	GSys->LoadConfig();
}

INT_PTR CALLBACK dataDirDialogProc(HWND hwndDlg,UINT uMsg,WPARAM wParam,LPARAM lParam)
{
switch (uMsg)
	{

	case WM_INITDIALOG:
		{
			SendMessage(hwndDlg, WM_SETICON, ICON_BIG,reinterpret_cast<LPARAM>(LoadIcon(reinterpret_cast<HINSTANCE>(GetWindowLong(hwndDlg,GWL_HINSTANCE)), MAKEINTRESOURCE(IDI_ICON))));

			list = GetDlgItem(hwndDlg,IDC_DDLIST);
			
			//Set up listview
			ListView_SetExtendedListViewStyle(list,LVS_EX_CHECKBOXES|LVS_EX_FULLROWSELECT);
			LONG listLong = GetWindowLongPtr(list,GWL_STYLE);
			SetWindowLongPtr(list,GWL_STYLE,listLong|LVS_NOCOLUMNHEADER);

			LVCOLUMN column = {};
			SendMessage(list,LVM_INSERTCOLUMN,0,reinterpret_cast<LPARAM>(&column));
			SendMessage(list,LVM_SETCOLUMNWIDTH,0,LVSCW_AUTOSIZE_USEHEADER);
						
			populateList();

			//Enable up/down buttons if item in list
			if(ListView_GetItemCount(list)>0)
			{
				EnableWindow(GetDlgItem(hwndDlg,IDC_UP),TRUE);
				EnableWindow(GetDlgItem(hwndDlg,IDC_DOWN),TRUE);

				ListView_SetItemState(list,0,LVIS_SELECTED,LVIS_SELECTED);
			}
		}
		return true;

	
	case WM_COMMAND:
	{
		switch(LOWORD(wParam))
		{	
		
			case (IDOK):
				EndDialog(hwndDlg,1);
				populateConfig();
				SetWindowLongPtr(hwndDlg,DWL_MSGRESULT,0);
				return true;

			case (IDCANCEL):
				EndDialog(hwndDlg,0);
				SetWindowLongPtr(hwndDlg,DWL_MSGRESULT,0);
				return true;

			case (IDC_UP):
			{
				int curSel = ListView_GetNextItem(list, -1, LVNI_SELECTED);
				if(curSel>=1)
				{
					moveListItem(curSel,curSel-1);
				}

				SetWindowLongPtr(hwndDlg,DWL_MSGRESULT,0);
				return true;
			}

			case (IDC_DOWN):
			{
				int curSel = ListView_GetNextItem(list, -1, LVNI_SELECTED);
				if(curSel<ListView_GetItemCount(list)-1)
				{
					moveListItem(curSel,curSel+1);
				}

				SetWindowLongPtr(hwndDlg,DWL_MSGRESULT,0);
				return true;
			}
			
		}
		break;
	}
	
	case WM_NOTIFY:
		{
			NMHDR* nmh = reinterpret_cast<NMHDR*>(lParam);
			if(nmh->hwndFrom == list)
			{
				switch(nmh->code)
				{
				case LVN_ITEMCHANGING:
					{
						//Detect checking/unchecking of list item checkbox
						NMLISTVIEW* itemInfo = reinterpret_cast<NMLISTVIEW*>(lParam);
						if(itemInfo->uChanged == LVIF_STATE && ((itemInfo->uNewState & LVIS_STATEIMAGEMASK) != (itemInfo->uOldState & LVIS_STATEIMAGEMASK)))
						{
							ListView_SetItemState(list,itemInfo->iItem,LVIS_SELECTED,LVIS_SELECTED);
						}
						SetWindowLongPtr(hwndDlg,DWL_MSGRESULT,0);
						return true;
					}

				default:
					break;
				}
			}
			

		}
		break;

	case WM_CLOSE:
		EndDialog(hwndDlg,0);
		SetWindowLongPtr(hwndDlg,DWL_MSGRESULT,FALSE);
		return true;

	default:
		break;
	}

	return false;
}