#include <windows.h>
#include <Shlwapi.h>

#include <vector>
#include <string>
#include <algorithm>
#include <cstdio>

#include "engine.h"
#include "unrender.h"

#include "fixapp.h"
#include "resource.h"
#include "misc.h"

static std::vector<std::wstring>* renderers; //For some reason the game crashes if this is a stack variable

static const int FOV_4_3 = 75;
static const int FOV_16_10 =  85;
static const int FOV_5_4 = 71;
static const int FOV_15_9 = 88;
static const int FOV_16_9 = 91;

static const char BPP_16 = 16;
static const char BPP_32 = 32;

struct res
{
	unsigned short X;
	unsigned short Y;
	int fov;

};

static struct res resolutions[] = {
	{640,480,FOV_4_3},
	{800,600,FOV_4_3},
	{1024,768,FOV_4_3},
	{1152,864,FOV_4_3},
	{1280,720,FOV_16_9},
	{1280,768,FOV_15_9},	
	{1280,800,FOV_16_10},
	{1280,960,FOV_4_3},
	{1280,1024,FOV_5_4},
	{1366,768,FOV_16_9},
	{1440,900,FOV_16_10},
	{1400,1050,FOV_4_3},
	{1600,1200,FOV_4_3},		
	{1680,1050,FOV_16_10},
	{1920,1200,FOV_16_10},	
	{1920,1080,FOV_16_9},		
};

static void applyFOV(HWND hwndDlg, int FOV)
{
	int fovButton;
	switch (FOV)
	{
	case FOV_4_3:
		fovButton = RADIO_FOVDEFAULT;
		break;
	case FOV_5_4:
		fovButton = RADIO_FOV_5_4;
		break;
	case FOV_16_10:
		fovButton = RADIO_FOV_16_10;
		break;
	case FOV_16_9:
		fovButton = RADIO_FOV_16_9;
		break;
	case FOV_15_9:
		fovButton = RADIO_FOV_15_9;
		break;
	default:
		fovButton = RADIO_FOVCUSTOM;
		SetDlgItemInt(hwndDlg,TXT_FOV,FOV,false);
		break;

	}
	CheckRadioButton(hwndDlg,RADIO_FOVDEFAULT,RADIO_FOVCUSTOM,fovButton);
}

static int getFOV(HWND hwndDlg)
{
	int fov;
	if(IsDlgButtonChecked(hwndDlg,RADIO_FOVDEFAULT))
	{
		fov = FOV_4_3;
	}
	else if(IsDlgButtonChecked(hwndDlg,RADIO_FOV_5_4))
	{
		fov = FOV_5_4;
	}
	else if(IsDlgButtonChecked(hwndDlg,RADIO_FOV_16_9))
	{
		fov = FOV_16_9;
	}
	else if(IsDlgButtonChecked(hwndDlg,RADIO_FOV_16_10))
	{
		fov = FOV_16_10;
	}
	else if(IsDlgButtonChecked(hwndDlg,RADIO_FOV_15_9))
	{
		fov = FOV_15_9;
	}
	else
	{
		fov = GetDlgItemInt(hwndDlg,TXT_FOV,false,false);
	}
	return fov;
}

static void readSettings(HWND hwndDlg)
{
	//Read the game's settings and apply them to the window. Also enabled the controls.


	//Read settings
	int resX = 1024;
	GConfig->GetInt(L"WinDrv.WindowsClient",L"FullscreenViewportX",resX);
	int resY = 768;
	GConfig->GetInt(L"WinDrv.WindowsClient",L"FullscreenViewportY",resY);

	int bitDepth = BPP_32;
	GConfig->GetInt(L"WinDrv.WindowsClient",L"FullscreenColorBits",bitDepth);

	const wchar_t* rootWindow = GConfig->GetStr(L"Engine.Engine",L"root");
	if(wcscmp(rootWindow,L"")==0)
	{
		rootWindow = L"DeusEx.DeusExRootWindow";
	}
	
	int FOV = FOV_4_3;
	GConfig->GetInt(L"Engine.PlayerPawn",L"DesiredFOV",FOV,L"User.ini");

	const wchar_t* renderer = GConfig->GetStr(L"Engine.Engine",L"GameRenderDevice");
	if(wcscmp(renderer,L"")==0)
	{
		renderer = L"SoftDrv.SoftwareRenderDevice";
	}
	
	BOOL detailTextures = TRUE;
	GConfig->GetBool(renderer,L"DetailTextures",detailTextures);
	
	BOOL noMouseAccel = FALSE;
	GConfig->GetBool(L"WinDrv.WindowsClient",L"UseDirectInput",noMouseAccel);
	
	int sndLatency = 40;
	GConfig->GetInt(L"Galaxy.GalaxyAudioSubsystem",L"Latency",sndLatency);

	BOOL directSound = TRUE;
	GConfig->GetBool(L"Galaxy.GalaxyAudioSubsystem",L"UseDirectSound",directSound);

	BOOL fullScreen = TRUE;
	GConfig->GetBool(L"WinDrv.WindowsClient",L"StartupFullScreen",fullScreen);
	
	int FPSLimit = 100;
	GConfig->GetInt(L"WinDrv.WindowsClient",L"FPSLimit",FPSLimit);

	//Apply settings to window
	auto cmpRes = [&resX,&resY](const res& r) {return (r.X == resX && r.Y == resY);};
	res* match = std::find_if(resolutions, resolutions+_countof(resolutions),cmpRes);
	if(match<resolutions+_countof(resolutions))
	{
		CheckRadioButton(hwndDlg,RADIO_RESCOMMON,RADIO_RESCUSTOM,RADIO_RESCOMMON);
		SendDlgItemMessage(hwndDlg,COMBO_RESOLUTION,CB_SETCURSEL,static_cast<WPARAM>(match-resolutions),0);
	}
	else
	{
		CheckRadioButton(hwndDlg,RADIO_RESCOMMON,RADIO_RESCUSTOM,RADIO_RESCUSTOM);
		SetDlgItemInt(hwndDlg,TXT_RESX,resX,false);
		SetDlgItemInt(hwndDlg,TXT_RESY,resY,false);
	}

	if(bitDepth == BPP_16)
	{
		CheckRadioButton(hwndDlg,RADIO_16BIT,RADIO_32BIT,RADIO_16BIT);
	}
	else
	{
		CheckRadioButton(hwndDlg,RADIO_16BIT,RADIO_32BIT,RADIO_32BIT);
	}

	applyFOV(hwndDlg,FOV);

	if(wcscmp(L"otpUIfix.otpRootWindow",rootWindow)==0)
	{
		CheckRadioButton(hwndDlg,RADIO_GUIDEFAULT,RADIO_GUIFIX,RADIO_GUIFIX);
	}
	else
	{
		CheckRadioButton(hwndDlg,RADIO_GUIDEFAULT,RADIO_GUIFIX,RADIO_GUIDEFAULT);
	}
	
	auto renderIt = std::find(renderers->begin(),renderers->end(),renderer);
	if(renderIt < renderers->end())
	{
			SendDlgItemMessage(hwndDlg,COMBO_RENDERER,CB_SETCURSEL,static_cast<WPARAM>(renderIt-renderers->begin()),0);
	}	
	
	if(detailTextures)
	{
		CheckDlgButton(hwndDlg,CHK_DETAILTEX,true);
	}

	if(noMouseAccel)
	{
		CheckDlgButton(hwndDlg,CHK_NOMOUSEACCEL,true);
	}
	
	if(directSound)
	{
		CheckDlgButton(hwndDlg,CHK_DIRECTSOUND,true);
	}

	if(fullScreen)
	{
		CheckDlgButton(hwndDlg,CHK_FULLSCREEN,true);
	}
	
	SetDlgItemInt(hwndDlg,TXT_LATENCY,sndLatency,false);
	SetDlgItemInt(hwndDlg,TXT_FPSLIMIT,FPSLimit,false);
	
}


static void populateDiag(HWND hwndDlg)
{

	//Populate the resolution combobox
	char buffer[10];
	for(int i=0;i<_countof(resolutions);i++)
	{
		_snprintf_s(buffer,_countof(buffer),_countof(buffer)-1,"%hdx%hd",resolutions[i].X,resolutions[i].Y);
		SendDlgItemMessageA(hwndDlg,COMBO_RESOLUTION,CB_ADDSTRING,0,reinterpret_cast<LPARAM>(buffer));
	}
	SendDlgItemMessage(hwndDlg,COMBO_RESOLUTION,CB_SETCURSEL,static_cast<WPARAM>(0),0);

	//Renderers (based on UnEngineWin.h), requires appInit() to have been called
	TArray<FRegistryObjectInfo> Classes;
	Classes.Empty();

	UObject::GetRegistryObjects( Classes, UClass::StaticClass(), URenderDevice::StaticClass(), 0 );
	for( TArray<FRegistryObjectInfo>::TIterator It(Classes); It; ++It )
	{
		FString Path=It->Object, Left, Right, Temp;
		if( Path.Split(TEXT("."),&Left,&Right)  )
		{
			{
				const wchar_t* desc = Localize(*Right,TEXT("ClassCaption"),*Left);
				if(SendDlgItemMessage(hwndDlg,COMBO_RENDERER,CB_FINDSTRINGEXACT,-1,(LPARAM) desc)==CB_ERR)
				{
					SendDlgItemMessage(hwndDlg,COMBO_RENDERER,CB_ADDSTRING,0,(LPARAM) desc);
					renderers->push_back(std::wstring((wchar_t*)Path.GetCharArray().GetData()));
				}
			}
		}
	}

}

static bool applySettings(HWND hwndDlg)
{
	
	//GUI scaling fix
	wchar_t* rootWindow = L"DeusEx.DeusExRootWindow";
	if(IsDlgButtonChecked(hwndDlg,RADIO_GUIFIX))
	{
		rootWindow = L"otpUIfix.otpRootWindow";
	}
	GConfig->SetString(L"Engine.engine",L"root",rootWindow);

	//Bit depth
	char bitDepth = BPP_32;
	if(IsDlgButtonChecked(hwndDlg,RADIO_16BIT))
	{
		bitDepth = BPP_16;		
	}
	GConfig->SetInt(L"WinDrv.WindowsClient",L"FullscreenColorBits",bitDepth);

	//FOV
	int fov = getFOV(hwndDlg);
	GConfig->SetInt(L"Engine.PlayerPawn",L"DesiredFOV",fov,L"User.ini");
	GConfig->SetInt(L"Engine.PlayerPawn",L"DefaultFOV",fov,L"User.ini");

	//Resolution
	unsigned short resX, resY;
	if(IsDlgButtonChecked(hwndDlg,RADIO_RESCOMMON))
	{
		int i = SendDlgItemMessageA(hwndDlg,COMBO_RESOLUTION,CB_GETCURSEL,0,0);		
		resX = resolutions[i].X;
		resY = resolutions[i].Y;
	}
	else
	{
		resX = GetDlgItemInt(hwndDlg,TXT_RESX,false,false);
		resY = GetDlgItemInt(hwndDlg,TXT_RESY,false,false);
	}
	GConfig->SetInt(L"WinDrv.WindowsClient",L"FullscreenViewportX",resX);
	GConfig->SetInt(L"WinDrv.WindowsClient",L"FullscreenViewportY",resY);

	//Disable mouse scaling
	GConfig->SetBool(L"WinDrv.WindowsClient",L"UseDirectInput",IsDlgButtonChecked(hwndDlg,CHK_NOMOUSEACCEL)!=0);
	//DirectSound
	GConfig->SetBool(L"Galaxy.GalaxyAudioSubsystem",L"UseDirectSound",IsDlgButtonChecked(hwndDlg,CHK_DIRECTSOUND)!=0);
	//Latency
	GConfig->SetInt(L"Galaxy.GalaxyAudioSubsystem",L"Latency",GetDlgItemInt(hwndDlg,TXT_LATENCY,false,false));
	//Fullscreen
	GConfig->SetBool(L"WinDrv.WindowsClient",L"StartupFullSCreen",IsDlgButtonChecked(hwndDlg,CHK_FULLSCREEN)!=0);
	//FPS Limit
	GConfig->SetInt(L"WinDrv.WindowsClient",L"FPSLimit",GetDlgItemInt(hwndDlg,TXT_FPSLIMIT,false,false));
		

	//Renderer
	int rendererIndex = SendDlgItemMessageA(hwndDlg,COMBO_RENDERER,CB_GETCURSEL,0,0);
	GConfig->SetString(L"Engine.Engine",L"GameRenderDevice",renderers->at(rendererIndex).c_str());
	//Detail textures
	GConfig->SetBool(renderers->at(rendererIndex).c_str(),L"DetailTextures",IsDlgButtonChecked(hwndDlg,CHK_DETAILTEX)!=0);

	return true;
}

INT_PTR CALLBACK fixerDialogProc(HWND hwndDlg,UINT uMsg,WPARAM wParam,LPARAM lParam)
{
	switch (uMsg)
	{

	case WM_INITDIALOG:
	{
		renderers = new std::vector<std::wstring>();
		
		SendMessage(hwndDlg, WM_SETICON, ICON_BIG, (LPARAM) LoadIcon((HINSTANCE) GetWindowLong(hwndDlg,GWL_HINSTANCE), MAKEINTRESOURCE(IDI_ICON)));
	
		populateDiag(hwndDlg);
		readSettings(hwndDlg);
		
	}
	return false;

	
	case WM_COMMAND:
	{
		switch(HIWORD(wParam))
		{
			case CBN_SELCHANGE:
			{	
				switch (LOWORD(wParam))
				{
				case (COMBO_RESOLUTION):
					{
						LRESULT r = SendDlgItemMessage(hwndDlg,COMBO_RESOLUTION,CB_GETCURSEL,0,0);
						applyFOV(hwndDlg,resolutions[r].fov);
					}
					SetWindowLongPtr(hwndDlg,DWL_MSGRESULT,0);
					return true;
				}
				break;
			}
			
			case EN_SETFOCUS:
			{
				switch (LOWORD(wParam))
				{
					case(TXT_FOV):
						CheckRadioButton(hwndDlg,RADIO_FOVDEFAULT,RADIO_FOVCUSTOM,RADIO_FOVCUSTOM);
						SetWindowLongPtr(hwndDlg,DWL_MSGRESULT,0);
						return true;

					case (TXT_RESX):
					case (TXT_RESY):
						CheckRadioButton(hwndDlg,RADIO_RESCOMMON,RADIO_RESCUSTOM,RADIO_RESCUSTOM);
						SetWindowLongPtr(hwndDlg,DWL_MSGRESULT,0);
						return true;

				}
				break;

			}
			default:
			{
				switch (LOWORD(wParam))
				{
				
					case (IDOK):				
						applySettings(hwndDlg);
						SetWindowLongPtr(hwndDlg,DWL_MSGRESULT,0);
					case (IDCANCEL):
						delete renderers;
						EndDialog(hwndDlg,0);
						SetWindowLongPtr(hwndDlg,DWL_MSGRESULT,0);
						return true;
					
					}
				break;
				}
		}
		break;
	}
	

	case WM_CLOSE:
		EndDialog(hwndDlg,0);
		return true;

	default:
		break;
	}

	return false;
}