#include "stdafx.h"
#include "DataDirDialog.h"
#include "Misc.h"
#include "resource.h"

const wchar_t* CDataDirDialog::sm_pszPaths = L"Paths";

const CDataDirDialog::StandardDirsArray CDataDirDialog::sm_StandardDirs =
{
	L"editorres",L"goty_1",L"help",L"manual",L"maps",L"music",L"save",L"sounds",L"system",L"textures",
};

const CDataDirDialog::SupportedExtensionsArray CDataDirDialog::sm_SupportedExtensions =
{
	L".utx",L".umx",L".dx",L".u",L".unr",L".uax",
};

CDataDirDialog::CDataDirDialog()
{

}

CDataDirDialog::~CDataDirDialog()
{

};

bool CDataDirDialog::Show(HWND hWndParent) const
{
	return DialogBoxParam(GetModuleHandle(0),MAKEINTRESOURCE(IDD_DATADIRS),hWndParent,DataDirDialogProc,reinterpret_cast<LPARAM>(this)) == 1;
}

void CDataDirDialog::MoveListItem(int iIndex, int iNewIndex) const
{
	SendMessage(m_hWndList,WM_SETREDRAW,static_cast<WPARAM>(FALSE),0);

	wchar_t szBuffer[MAX_PATH];
	LVITEM Item = {};	
	Item.iItem = iIndex;
	Item.mask = LVIF_TEXT|LVIF_STATE;
	Item.stateMask = static_cast<UINT>(-1);	
	Item.pszText = szBuffer;
	Item.cchTextMax = _countof(szBuffer);

	ListView_GetItem(m_hWndList,&Item);
	BOOL bChecked = ListView_GetCheckState(m_hWndList,iIndex); //Needs to be re-done when using InsertItem
	ListView_DeleteItem(m_hWndList,iIndex);
	Item.iItem = iNewIndex;
	ListView_InsertItem(m_hWndList,&Item);
	ListView_SetCheckState(m_hWndList,iNewIndex,bChecked);
	SendMessage(m_hWndList,WM_SETREDRAW,static_cast<WPARAM>(TRUE),0);
}

void CDataDirDialog::SearchDir(const wchar_t* pszTargetDir, bool bRoot, const wchar_t* pszCurrentDir) const
{
	assert(pszTargetDir);
	assert(pszCurrentDir);

	WIN32_FIND_DATA FindData;
	wchar_t szTargetPath[MAX_PATH];
	PathCombine(szTargetPath,pszTargetDir,L"*");
	HANDLE hDir = FindFirstFile(szTargetPath,&FindData);

	//Find subdirectories
	if(hDir!=INVALID_HANDLE_VALUE)
	{
		TMap<FString,int> Extensions;
		int iMostCommonExtensionCount = 0;
		wchar_t szMostCommonExtension[MAX_PATH];

		do
		{
			if(wcscmp(FindData.cFileName,L"..")!=0 && wcscmp(FindData.cFileName,L".")!=0)
			{
				if(FindData.dwFileAttributes & FILE_ATTRIBUTE_DIRECTORY) //Directory, search recursively
				{
					//Check if directory is one of the standard ones, if so don't process it any further
					bool bIsStandardDir = false;
					if(bRoot)
					{
						auto cmp = [&FindData] (const wchar_t* str) {return _wcsicmp(str,FindData.cFileName)==0;};
						if(std::find_if(sm_StandardDirs.cbegin(),sm_StandardDirs.cend(),cmp)!=sm_StandardDirs.cend())
						{
							bIsStandardDir = true;
						}

					}
					if(!bIsStandardDir)
					{	
						wchar_t szChildPath[MAX_PATH];
						PathCombine(szChildPath,pszTargetDir,FindData.cFileName);
						SearchDir(szChildPath,false,pszCurrentDir);
					}
				}
				else if(!bRoot) //Files (and not the starting 'System' directory), process)
				{
					//Tally file extensions in directory and pick the most frequent one
					wchar_t* pszExtensionPtr = PathFindExtension(FindData.cFileName);

					//Only count useful extensions
					auto cmp = [&pszExtensionPtr] (const wchar_t* str) {return _wcsicmp(str,pszExtensionPtr)==0;};
					if(std::find_if(sm_SupportedExtensions.begin(),sm_SupportedExtensions.end(),cmp)!=sm_SupportedExtensions.end())
					{
						int iExtensionCount = 0;
						int* pNum;
						if((pNum=Extensions.Find(pszExtensionPtr))!=nullptr)
						{
							iExtensionCount = *pNum;
						}

						Extensions.Set(pszExtensionPtr,++iExtensionCount);
						if(iExtensionCount>iMostCommonExtensionCount) //New extension is the one with the most files
						{
							wcscpy_s(szMostCommonExtension,pszExtensionPtr);
							iMostCommonExtensionCount = iExtensionCount;
						}
					}

				}
			}
		} while(FindNextFile(hDir, &FindData) != 0);

		FindClose(hDir);

		//Done processing directory, see if we found anything
		if(iMostCommonExtensionCount > 0)
		{	
			//Convert to relative path
			wchar_t szBuf[MAX_PATH];
			PathRelativePathTo(szBuf,pszCurrentDir,FILE_ATTRIBUTE_DIRECTORY,szTargetPath,FILE_ATTRIBUTE_DIRECTORY);
			PathAddExtension(szBuf,szMostCommonExtension);

			//Add item to list
			LVITEM Item = {};
			Item.iItem = std::numeric_limits<int>::max();
			Item.pszText = szBuf;
			Item.mask = LVIF_TEXT;
			SendMessage(m_hWndList,LVM_INSERTITEM,1,reinterpret_cast<LPARAM>(&Item));

		}
	}
}

void CDataDirDialog::PopulateList() const
{	
	assert(GSys);

	//Uses USystem::Paths instead of GConfig so the order of entries is preserved
	wchar_t szCurrentDir[MAX_PATH];
	GetCurrentDirectory(_countof(szCurrentDir),szCurrentDir);
	wchar_t szDirUp[MAX_PATH];
	PathCombine(szDirUp,szCurrentDir,L"..");
	SearchDir(szDirUp,true,szCurrentDir);

	//Give items in ini file priority and check their checkboxes
	for(int i=GSys->Paths.Num()-1;i>=0;i--)
	{
		LVFINDINFO FindInfo = {};
		FindInfo.flags = LVFI_STRING;
		FindInfo.psz = *GSys->Paths(i);
		int listIndex = ListView_FindItem(m_hWndList,-1,&FindInfo);
		if(listIndex>-1)
		{
			ListView_SetCheckState(m_hWndList,listIndex,TRUE);
			MoveListItem(listIndex,0);
		}
	}
}

void CDataDirDialog::PopulateConfig() const
{
	assert(GSys);

	//Directly manipulating the USystem::Paths member resulted in crashes, so we use GConfig
	FString Dummy;

	//Clear existing items
	TMultiMap<FString,FString>* Section = GConfig->GetSectionPrivate(L"Core.System",FALSE,FALSE);
	assert(Section);

	//Remove path items
	for(int i=0;i<ListView_GetItemCount(m_hWndList);i++)
	{
		wchar_t szText[MAX_PATH];
		ListView_GetItemText(m_hWndList,i,0,szText,_countof(szText));
		Section->RemovePair(sm_pszPaths,szText);
	}

	//Get the leftover items, which are the default directories
	TArray<FString> Defaults;
	Section->MultiFind(sm_pszPaths,Defaults);
	Section->Remove(sm_pszPaths);

	//Add path items to actual list
	for(int i=0;i<ListView_GetItemCount(m_hWndList);i++)
	{
		wchar_t szText[MAX_PATH];
		ListView_GetItemText(m_hWndList,i,0,szText,_countof(szText));

		if(ListView_GetCheckState(m_hWndList,i))
		{			
			Section->Add(sm_pszPaths,szText);
		}
	}

	//Re-add the defaults
	for(int i=0;i<Defaults.Num();i++)
	{
		Section->Add(sm_pszPaths,*Defaults(i));
	}

	GSys->LoadConfig();
}

INT_PTR CALLBACK CDataDirDialog::DataDirDialogProc(HWND hwndDlg,UINT uMsg,WPARAM wParam,LPARAM lParam)
{
	CDataDirDialog* pThis = reinterpret_cast<CDataDirDialog*>(GetProp(hwndDlg,L"this"));

	switch (uMsg)
	{

	case WM_INITDIALOG:
		{
			SetProp(hwndDlg,L"this",reinterpret_cast<HANDLE>(lParam));
			pThis =  reinterpret_cast<CDataDirDialog*>(lParam);
			pThis->m_hWnd = hwndDlg;
			pThis->m_hWndList = GetDlgItem(hwndDlg,IDC_DDLIST);

			SendMessage(hwndDlg, WM_SETICON, ICON_BIG,reinterpret_cast<LPARAM>(LoadIcon(reinterpret_cast<HINSTANCE>(GetWindowLong(hwndDlg,GWL_HINSTANCE)), MAKEINTRESOURCE(IDI_ICON))));

			//Set up listview
			ListView_SetExtendedListViewStyle(pThis->m_hWndList,LVS_EX_CHECKBOXES|LVS_EX_FULLROWSELECT);
			LONG lList = GetWindowLongPtr(pThis->m_hWndList,GWL_STYLE);
			SetWindowLongPtr(pThis->m_hWndList,GWL_STYLE,lList|LVS_NOCOLUMNHEADER);

			LVCOLUMN Column = {};
			SendMessage(pThis->m_hWndList,LVM_INSERTCOLUMN,0,reinterpret_cast<LPARAM>(&Column));
			SendMessage(pThis->m_hWndList,LVM_SETCOLUMNWIDTH,0,LVSCW_AUTOSIZE_USEHEADER);

			pThis->PopulateList();

			//Enable up/down buttons if item in list
			if(ListView_GetItemCount(pThis->m_hWndList)>0)
			{
				EnableWindow(GetDlgItem(hwndDlg,IDC_UP),TRUE);
				EnableWindow(GetDlgItem(hwndDlg,IDC_DOWN),TRUE);

				ListView_SetItemState(pThis->m_hWndList,0,LVIS_SELECTED,LVIS_SELECTED);
			}
		}
		return TRUE;

	case WM_COMMAND:
		switch(LOWORD(wParam))
		{	

		case IDOK:
			EndDialog(hwndDlg,1);
			pThis->PopulateConfig();			
			return TRUE;

		case IDCANCEL:
			EndDialog(hwndDlg,0);				
			return TRUE;

		case IDC_UP:
			{
				int iCurSel = ListView_GetNextItem(pThis->m_hWndList, -1, LVNI_SELECTED);
				if(iCurSel>=1)
				{
					pThis->MoveListItem(iCurSel,iCurSel-1);
				}				
			}
			return TRUE;

		case IDC_DOWN:
			{
				int iCurSel = ListView_GetNextItem(pThis->m_hWndList, -1, LVNI_SELECTED);
				if(iCurSel>-1 && iCurSel<ListView_GetItemCount(pThis->m_hWndList)-1)
				{
					pThis->MoveListItem(iCurSel,iCurSel+1);
				}
			}
			return TRUE;

		}
		break;

	case WM_NOTIFY:
		{
			NMHDR* pNMH = reinterpret_cast<NMHDR*>(lParam);
			assert(pNMH);
			if(pNMH->hwndFrom == pThis->m_hWndList)
			{
				switch(pNMH->code)
				{
				case LVN_ITEMCHANGING:
					{
						//Detect checking/unchecking of list item checkbox
						NMLISTVIEW* pItemInfo = reinterpret_cast<NMLISTVIEW*>(lParam);
						assert(pItemInfo);
						if(pItemInfo->uChanged == LVIF_STATE && ((pItemInfo->uNewState & LVIS_STATEIMAGEMASK) != (pItemInfo->uOldState & LVIS_STATEIMAGEMASK)))
						{
							ListView_SetItemState(pThis->m_hWndList,pItemInfo->iItem,LVIS_SELECTED,LVIS_SELECTED);
						}
					}
					return TRUE;
				}
			}
		}
		break;

	case WM_CLOSE:
		EndDialog(hwndDlg,0);
		return TRUE;
	}

	return FALSE;
}