#include "stdafx.h"
#include "FixApp.h"
#include "Misc.h"
#include "resource.h"

const wchar_t* CFixApp::sm_pszOTPRootWindow = L"otpUIfix.otpRootWindow";
const wchar_t* CFixApp::sm_pszDefaultRootWindow = L"DeusEx.DeusExRootWindow";

/**
Resolution <-> FOV table
*/
const CFixApp::ResolutionArray CFixApp::sm_Resolutions = {{
	{640,480,sm_iFov_4_3},
	{800,600,sm_iFov_4_3},
	{1024,768,sm_iFov_4_3},
	{1152,864,sm_iFov_4_3},
	{1280,720,sm_iFov_16_9},
	{1280,768,sm_iFov_15_9},	
	{1280,800,sm_iFov_16_10},
	{1280,960,sm_iFov_4_3},
	{1280,1024,sm_iFov_5_4},
	{1366,768,sm_iFov_16_9},
	{1440,900,sm_iFov_16_10},
	{1400,1050,sm_iFov_4_3},
	{1600,1200,sm_iFov_4_3},		
	{1680,1050,sm_iFov_16_10},
	{1920,1200,sm_iFov_16_10},	
	{1920,1080,sm_iFov_16_9},
	{2560,1440,sm_iFov_16_9},
	{2560,1600,sm_iFov_16_10},
	{2880,1800,sm_iFov_16_10},
}};

CFixApp::CFixApp()
{

}

CFixApp::~CFixApp()
{

}

bool CFixApp::Show(HWND hWndParent) const
{
	return DialogBoxParam(GetModuleHandle(0),MAKEINTRESOURCE(IDD_DIALOG2),hWndParent,FixAppDialogProc,reinterpret_cast<LPARAM>(this)) == 1;
}

void CFixApp::SetFOV(int iFOV) const
{
	int fovButton;
	switch (iFOV)
	{
	case sm_iFov_4_3:
		fovButton = RADIO_FOVDEFAULT;
		break;
	case sm_iFov_5_4:
		fovButton = RADIO_FOV_5_4;
		break;
	case sm_iFov_16_10:
		fovButton = RADIO_FOV_16_10;
		break;
	case sm_iFov_16_9:
		fovButton = RADIO_FOV_16_9;
		break;
	case sm_iFov_15_9:
		fovButton = RADIO_FOV_15_9;
		break;
	default:
		fovButton = RADIO_FOVCUSTOM;
		SetDlgItemInt(m_hWnd,TXT_FOV,iFOV,false);
		break;

	}
	CheckRadioButton(m_hWnd,RADIO_FOVDEFAULT,RADIO_FOVCUSTOM,fovButton);
}

int CFixApp::GetFOV() const
{
	int iFov;
	if(IsDlgButtonChecked(m_hWnd,RADIO_FOVDEFAULT))
	{
		iFov = sm_iFov_4_3;
	}
	else if(IsDlgButtonChecked(m_hWnd,RADIO_FOV_5_4))
	{
		iFov = sm_iFov_5_4;
	}
	else if(IsDlgButtonChecked(m_hWnd,RADIO_FOV_16_9))
	{
		iFov = sm_iFov_16_9;
	}
	else if(IsDlgButtonChecked(m_hWnd,RADIO_FOV_16_10))
	{
		iFov = sm_iFov_16_10;
	}
	else if(IsDlgButtonChecked(m_hWnd,RADIO_FOV_15_9))
	{
		iFov = sm_iFov_15_9;
	}
	else
	{
		iFov = static_cast<decltype(iFov)>(GetDlgItemInt(m_hWnd,TXT_FOV,false,false));
	}
	return iFov;
}

void CFixApp::ReadSettings() const
{
	//Read the game's settings and apply them to the window. Also enables the controls.
	int iResX = 1024;
	GConfig->GetInt(L"WinDrv.WindowsClient",L"FullscreenViewportX",iResX);
	int iResY = 768;
	GConfig->GetInt(L"WinDrv.WindowsClient",L"FullscreenViewportY",iResY);

	int iBitDepth = sm_iBPP_32;
	GConfig->GetInt(L"WinDrv.WindowsClient",L"FullscreenColorBits",iBitDepth);

	const wchar_t* pszRootWindow = GConfig->GetStr(L"Engine.Engine",L"root");
	if(wcscmp(pszRootWindow,L"")==0)
	{
		pszRootWindow = sm_pszDefaultRootWindow;
	}
	
	int iFOV = sm_iFov_4_3;
	GConfig->GetInt(L"Engine.PlayerPawn",L"DesiredFOV",iFOV,L"User.ini");

	const wchar_t* pszRenderer = GConfig->GetStr(L"Engine.Engine",L"GameRenderDevice");
	if(wcscmp(pszRenderer,L"")==0)
	{
		pszRenderer = L"SoftDrv.SoftwareRenderDevice";
	}
	
	BOOL bDetailTextures = TRUE;
	GConfig->GetBool(pszRenderer,L"DetailTextures",bDetailTextures);
	
	BOOL bNoMouseAccel = FALSE;
	GConfig->GetBool(L"WinDrv.WindowsClient",L"UseDirectInput",bNoMouseAccel);
	
	int iSndLatency = 40;
	GConfig->GetInt(L"Galaxy.GalaxyAudioSubsystem",L"Latency",iSndLatency);

	BOOL bDirectSound = TRUE;
	GConfig->GetBool(L"Galaxy.GalaxyAudioSubsystem",L"UseDirectSound",bDirectSound);

	BOOL bFullScreen = TRUE;
	GConfig->GetBool(L"WinDrv.WindowsClient",L"StartupFullScreen",bFullScreen);
	
	int iFPSLimit = 100;
	GConfig->GetInt(L"WinDrv.WindowsClient",L"FPSLimit",iFPSLimit);

	//Apply settings to window
	auto cmpRes = [&iResX,&iResY](const Resolution& r) {return (r.iX == iResX && r.iY == iResY);};
	auto itRes = std::find_if(sm_Resolutions.begin(), sm_Resolutions.end(),cmpRes);
	if(itRes!=sm_Resolutions.end())
	{
		CheckRadioButton(m_hWnd,RADIO_RESCOMMON,RADIO_RESCUSTOM,RADIO_RESCOMMON);
		SendDlgItemMessage(m_hWnd,COMBO_RESOLUTION,CB_SETCURSEL,static_cast<WPARAM>(itRes-sm_Resolutions.begin()),0);
	}
	else
	{
		CheckRadioButton(m_hWnd,RADIO_RESCOMMON,RADIO_RESCUSTOM,RADIO_RESCUSTOM);
		SetDlgItemInt(m_hWnd,TXT_RESX,iResX,false);
		SetDlgItemInt(m_hWnd,TXT_RESY,iResY,false);
	}

	if(iBitDepth == sm_iBPP_16)
	{
		CheckRadioButton(m_hWnd,RADIO_16BIT,RADIO_32BIT,RADIO_16BIT);
	}
	else
	{
		CheckRadioButton(m_hWnd,RADIO_16BIT,RADIO_32BIT,RADIO_32BIT);
	}

	SetFOV(iFOV);

	if(_wcsicmp(sm_pszOTPRootWindow,pszRootWindow)==0)
	{
		CheckRadioButton(m_hWnd,RADIO_GUIDEFAULT,RADIO_GUIFIX,RADIO_GUIFIX);
	}
	else if(_wcsicmp(sm_pszDefaultRootWindow,pszRootWindow)==0)
	{
		CheckRadioButton(m_hWnd,RADIO_GUIDEFAULT,RADIO_GUIFIX,RADIO_GUIDEFAULT);
	}
	else
	{		
		EnableWindow(GetDlgItem(m_hWnd,RADIO_GUIDEFAULT),FALSE);
		EnableWindow(GetDlgItem(m_hWnd,RADIO_GUIFIX),FALSE);
		SetWindowText(GetDlgItem(m_hWnd,STATIC_GUI),L"GUI (Custom GUI found)");
		EnableWindow(GetDlgItem(m_hWnd,STATIC_GUI),FALSE);
	}
	
	auto renderIt = std::find(m_Renderers.begin(),m_Renderers.end(),pszRenderer);
	if(renderIt != m_Renderers.end())
	{
			SendDlgItemMessage(m_hWnd,COMBO_RENDERER,CB_SETCURSEL,static_cast<WPARAM>(renderIt-m_Renderers.begin()),0);
	}	
	
	if(bDetailTextures)
	{
		CheckDlgButton(m_hWnd,CHK_DETAILTEX,true);
	}

	if(bNoMouseAccel)
	{
		CheckDlgButton(m_hWnd,CHK_NOMOUSEACCEL,true);
	}
	
	if(bDirectSound)
	{
		CheckDlgButton(m_hWnd,CHK_DIRECTSOUND,true);
	}

	if(bFullScreen)
	{
		CheckDlgButton(m_hWnd,CHK_FULLSCREEN,true);
	}
	
	SetDlgItemInt(m_hWnd,TXT_LATENCY,iSndLatency,false);
	SetDlgItemInt(m_hWnd,TXT_FPSLIMIT,iFPSLimit,false);
	
}


void CFixApp::PopulateDialog()
{

	//Populate the resolution combobox
	char szBuffer[10];
	for(auto it=sm_Resolutions.begin(); it!=sm_Resolutions.end();++it)
	{
		_snprintf_s(szBuffer,_countof(szBuffer)-1,"%dx%d",it->iX,it->iY);
		SendDlgItemMessageA(m_hWnd,COMBO_RESOLUTION,CB_ADDSTRING,0,reinterpret_cast<LPARAM>(szBuffer));
	}
	SendDlgItemMessage(m_hWnd,COMBO_RESOLUTION,CB_SETCURSEL,0,0);

	//Renderers (based on UnEngineWin.h), requires appInit() to have been called
	TArray<FRegistryObjectInfo> Classes;
	Classes.Empty();

	UObject::GetRegistryObjects( Classes, UClass::StaticClass(), URenderDevice::StaticClass(), 0 );
	for( TArray<FRegistryObjectInfo>::TIterator It(Classes); It; ++It )
	{
		FString Path=It->Object, Left, Right;
		if( Path.Split(TEXT("."),&Left,&Right)  )
		{
			const wchar_t* pszDesc = Localize(*Right,TEXT("ClassCaption"),*Left);
			assert(pszDesc);
			if(SendDlgItemMessage(m_hWnd,COMBO_RENDERER,CB_FINDSTRINGEXACT,static_cast<WPARAM>(-1),reinterpret_cast<LPARAM>(pszDesc))==CB_ERR)
			{
				SendDlgItemMessage(m_hWnd,COMBO_RENDERER,CB_ADDSTRING,0,reinterpret_cast<LPARAM>(pszDesc));
				m_Renderers.push_back(std::wstring(static_cast<wchar_t*>(Path.GetCharArray().GetData())));
			}
		}
	}

}

void CFixApp::ApplySettings() const
{
	
	//GUI scaling fix
	if(IsDlgButtonChecked(m_hWnd,RADIO_GUIFIX))
	{
		GConfig->SetString(L"Engine.engine",L"root",sm_pszOTPRootWindow);
	}
	else if(IsDlgButtonChecked(m_hWnd,RADIO_GUIDEFAULT))
	{
		GConfig->SetString(L"Engine.engine",L"root",sm_pszDefaultRootWindow);
	}	

	//Bit depth
	unsigned char iBitDepth = IsDlgButtonChecked(m_hWnd,RADIO_16BIT) ? sm_iBPP_16 : sm_iBPP_32;	
	GConfig->SetInt(L"WinDrv.WindowsClient",L"FullscreenColorBits",iBitDepth);

	//FOV
	int fov = GetFOV();
	GConfig->SetInt(L"Engine.PlayerPawn",L"DesiredFOV",fov,L"User.ini");
	GConfig->SetInt(L"Engine.PlayerPawn",L"DefaultFOV",fov,L"User.ini");

	//Resolution
	size_t iResX, iResY;
	if(IsDlgButtonChecked(m_hWnd,RADIO_RESCOMMON))
	{
		int i = SendDlgItemMessageA(m_hWnd,COMBO_RESOLUTION,CB_GETCURSEL,0,0);		
		iResX = sm_Resolutions[i].iX;
		iResY = sm_Resolutions[i].iY;
	}
	else
	{
		iResX = GetDlgItemInt(m_hWnd,TXT_RESX,false,false);
		iResY = GetDlgItemInt(m_hWnd,TXT_RESY,false,false);
	}
	GConfig->SetInt(L"WinDrv.WindowsClient",L"FullscreenViewportX",iResX);
	GConfig->SetInt(L"WinDrv.WindowsClient",L"FullscreenViewportY",iResY);

	//Disable mouse scaling
	GConfig->SetBool(L"WinDrv.WindowsClient",L"UseDirectInput",IsDlgButtonChecked(m_hWnd,CHK_NOMOUSEACCEL)!=0);
	//DirectSound
	GConfig->SetBool(L"Galaxy.GalaxyAudioSubsystem",L"UseDirectSound",IsDlgButtonChecked(m_hWnd,CHK_DIRECTSOUND)!=0);
	//Latency
	GConfig->SetInt(L"Galaxy.GalaxyAudioSubsystem",L"Latency",GetDlgItemInt(m_hWnd,TXT_LATENCY,false,false));
	//Fullscreen
	GConfig->SetBool(L"WinDrv.WindowsClient",L"StartupFullSCreen",IsDlgButtonChecked(m_hWnd,CHK_FULLSCREEN)!=0);
	//FPS Limit
	GConfig->SetInt(L"WinDrv.WindowsClient",L"FPSLimit",GetDlgItemInt(m_hWnd,TXT_FPSLIMIT,false,false));
		
	//Renderer
	int iRendererIndex = SendDlgItemMessageA(m_hWnd,COMBO_RENDERER,CB_GETCURSEL,0,0);
	GConfig->SetString(L"Engine.Engine",L"GameRenderDevice",m_Renderers.at(iRendererIndex).c_str());
	//Detail textures
	GConfig->SetBool(m_Renderers.at(iRendererIndex).c_str(),L"DetailTextures",IsDlgButtonChecked(m_hWnd,CHK_DETAILTEX)!=0);
}

INT_PTR CALLBACK CFixApp::FixAppDialogProc(HWND hwndDlg,UINT uMsg,WPARAM wParam,LPARAM lParam)
{	
	CFixApp* pThis = reinterpret_cast<CFixApp*>(GetProp(hwndDlg,L"this"));

	switch (uMsg)
	{
	case WM_INITDIALOG:
		{
			//Get all object oriented like
			SetProp(hwndDlg,L"this",reinterpret_cast<HANDLE>(lParam));
			pThis =  reinterpret_cast<CFixApp*>(lParam);
			pThis->m_hWnd = hwndDlg;

			SendMessage(hwndDlg, WM_SETICON, ICON_BIG, (LPARAM) LoadIcon((HINSTANCE) GetWindowLong(hwndDlg,GWL_HINSTANCE), MAKEINTRESOURCE(IDI_ICON)));
			pThis->PopulateDialog();
			pThis->ReadSettings();

		}
	return TRUE;
	
	case WM_COMMAND:
		switch(HIWORD(wParam))
		{
		case CBN_SELCHANGE:
			{	
				switch (LOWORD(wParam))
				{
				case COMBO_RESOLUTION:
					{
						int iS = SendDlgItemMessage(hwndDlg,COMBO_RESOLUTION,CB_GETCURSEL,0,0);
						pThis->SetFOV(sm_Resolutions[iS].iFov);
						CheckRadioButton(hwndDlg,RADIO_RESCOMMON,RADIO_RESCUSTOM,RADIO_RESCOMMON);
					}
					return TRUE;
				}
				break;
			}

		case EN_SETFOCUS:
			{
				switch (LOWORD(wParam))
				{
				case TXT_FOV:
					CheckRadioButton(hwndDlg,RADIO_FOVDEFAULT,RADIO_FOVCUSTOM,RADIO_FOVCUSTOM);
					return TRUE;

				case TXT_RESX:
				case TXT_RESY:
					CheckRadioButton(hwndDlg,RADIO_RESCOMMON,RADIO_RESCUSTOM,RADIO_RESCUSTOM);
					return TRUE;

				}
				break;

			}
		default:
			{
				switch (LOWORD(wParam))
				{

				case IDOK:				
					pThis->ApplySettings();
				case IDCANCEL:
					EndDialog(hwndDlg,0);
					return TRUE;

				}
				break;
			}
		}
		break;

	case WM_CLOSE:
		EndDialog(hwndDlg,0);
		return TRUE;

	}

	return FALSE;
}