#pragma comment(linker,"/manifestdependency:\"type='win32' name='Microsoft.Windows.Common-Controls' version='6.0.0.0' processorArchitecture='*' publicKeyToken='6595b64144ccf1df' language='*'\"")

#include "stdafx.h"
#include "DeusExeFileManager.h"
#include "Misc.h"
#include "RawInput.h"
#include "LauncherDialog.h"
#include "Fixapp.h"
#include "Detoureds.h"
#include "ExecHook.h"

extern "C" {TCHAR GPackage[64]=TEXT("Launch");} //Will be set to exe name later

//Unreal engine framework classes
static LARGE_INTEGER s_iPerfCounterFreq;
static int s_iFPSLimit;
static UBOOL s_bDirectInput;
static HWND s_hWnd;

static void MainLoop( UEngine* pEngine )
{
	assert(pEngine);
	float fDeltaTime;
	LARGE_INTEGER iOldTime;
	LARGE_INTEGER iTime;

	if(!QueryPerformanceCounter(&iOldTime)) //Initial time
	{
		return;
	}
	
	MSG Msg;
	while(GIsRunning && !GIsRequestingExit)
	{
		QueryPerformanceCounter(&iTime);
		fDeltaTime = (iTime.QuadPart-iOldTime.QuadPart) / static_cast<float>(s_iPerfCounterFreq.QuadPart);

		//Lock game update speed to fps limit
		if((s_iFPSLimit == 0 || fDeltaTime>=1.0f/s_iFPSLimit) && (!pEngine->GetMaxTickRate() || fDeltaTime>=1.0f/pEngine->GetMaxTickRate()))
		{
			pEngine->Tick(fDeltaTime);
			if( GWindowManager )
			{
				GWindowManager->Tick(fDeltaTime);
			}
			iOldTime = iTime;
		}

		//Get rid of double mouse cursors in DirectInput mode as they go out of sync (making you click outside the window)
		if(s_bDirectInput)
		{
			assert(s_hWnd);
			if(GetForegroundWindow()==s_hWnd)
			{
				RECT Rect;
				GetWindowRect(s_hWnd,&Rect);
				while(ShowCursor(FALSE)>0); //Hide cursor
				Rect.left = Rect.right = (Rect.left+Rect.right)/2;
				Rect.top = Rect.bottom = (Rect.top+Rect.bottom)/2;				
				ClipCursor(&Rect);
			}
			else
			{
				ClipCursor(nullptr);
				ShowCursor(TRUE);
			}
		}

		while(PeekMessage(&Msg,NULL,0,0,PM_REMOVE))
		{
			bool bSkipMessage=false;

			switch(Msg.message)
			{
			case WM_QUIT:
				GIsRequestingExit = 1;
				break;

			case WM_MOUSEMOVE: //The issue is that DirectInput means these are ignored, as we're fixing that make sure they won't get caught anyway
				if(s_bDirectInput)
				{
					bSkipMessage=true;
				}
				break;
			case WM_INPUT: //Fix menu mouse cursor in DirectInput mode.
				{
					if(s_bDirectInput && pEngine->Client->Viewports.Num()>0) //If user closes window, viewport disappears before we get WM_QUIT
					{
						RAWINPUT raw;
						unsigned int rawSize = sizeof(raw);
						GetRawInputData(reinterpret_cast<HRAWINPUT>(Msg.lParam), RID_INPUT, &raw, &rawSize, sizeof(RAWINPUTHEADER));
						pEngine->MouseDelta(pEngine->Client->Viewports(0),0,raw.data.mouse.lLastX/1.5f,raw.data.mouse.lLastY/1.5f);
						bSkipMessage=true;
					}
				}
				break;

			}

			if(!bSkipMessage)
			{
				TranslateMessage( &Msg );
				DispatchMessage( &Msg );
			}
		}
	}
	
}

bool DoLauncherDialog()
{
	CLauncherDialog LD;
	return LD.Show(NULL);
}

INT WINAPI WinMain( HINSTANCE /*hInInstance*/, HINSTANCE /*hPrevInstance*/, LPSTR /*lpCmdLine*/, INT /*nCmdShow*/ )
{
	INITCOMMONCONTROLSEX CommonControlsInfo;
	CommonControlsInfo.dwSize = sizeof(INITCOMMONCONTROLSEX);
	CommonControlsInfo.dwICC = ICC_LISTVIEW_CLASSES;
	if(InitCommonControlsEx(&CommonControlsInfo)!=TRUE)
	{
		return EXIT_FAILURE;
	}

	//Overwrite default settings. I'd rather not change default.ini but the game copies it over real system.ini before I'm able to patch up that
	wchar_t szIniLoc[MAX_PATH];
	GetGameSystemDir(szIniLoc);
	PathAppend(szIniLoc,L"Default.ini");
	WritePrivateProfileString(L"Engine.Engine",L"GameRenderDevice",L"D3DDrv.D3DRenderDevice",szIniLoc);
	WritePrivateProfileString(L"WinDrv.WindowsClient",L"FullscreenColorBits",L"32",szIniLoc);

	//If -localdata command line option present, don't use user documents for data
	SetUseUserDocs(wcswcs(GetCommandLine(),L"-localdata")==nullptr);
	
	appStrcpy(GPackage,appPackage());

	//Init core	
	FMallocWindows Malloc;
	FOutputDeviceFile Log;
	FOutputDeviceWindowsError Error;
	FFeedbackContextWindows Warn;
	FFileManagerDeusExe FileManager;
	appInit(GPackage,GetCommandLine(), &Malloc, &Log, &Error, &Warn, &FileManager, FConfigCacheIni::Factory, 1);	

	GIsStarted = 1;
	GIsServer = 1;
	GIsClient = !ParseParam(appCmdLine(), TEXT("SERVER"));
	GIsEditor = 0;
	GIsScriptable = 1;
	GLazyLoad = !GIsClient;

	assert(GConfig);
	GConfig->Flush(false);
	bool bSkipLauncherDialog = (!GIsClient || ParseParam(appCmdLine(), TEXT("skipdialog")));

	if(InitDetours()==false)
	{
		Log.Log(L"Failed to initialize detours.");
	}
	if(SetProcessAffinityMask(GetCurrentProcess(),0x1)==FALSE) //Force on single CPU
	{
		Log.Log(L"Failed to set process affinity.");
	}
	if(!SetDEP(0)) //Disable DEP for process (also need NXCOMPAT=NO)
	{
		Log.Log(L"Failed to set process DEP flags.");
	}
	if(QueryPerformanceFrequency(&s_iPerfCounterFreq)==FALSE)
	{
		Error.Log(L"Failed to query performance counter.");
	}
	
	//Show launcher dialog
	if( ParseParam(appCmdLine(),TEXT("changevideo")))
	{
		CFixApp FixApp;
		FixApp.Show(NULL);
	}
	else if(bSkipLauncherDialog || DoLauncherDialog())
	{
		GConfig->Flush(true); //Reload config file to pick up any changes made by fixer app.
		
		//Init windowing
		InitWindowing();
	
		//Create log window
		std::unique_ptr<WLog> LogWindowPtr(new WLog( Log.Filename, Log.LogAr, TEXT("GameLog")));
		GLogWindow = LogWindowPtr.get(); //Yup...
		GLogWindow->OpenWindow(!GIsClient, 0);				
		GLogWindow->Log( NAME_Title, LocalizeGeneral("Start") );

		//Create command processor, fixes dedicated server preferences
		FExecHook ExecHook;
		GExec = &ExecHook;

		//Init engine
		UClass* pEngineClass = UObject::StaticLoadClass( UGameEngine::StaticClass(), NULL, TEXT("ini:Engine.Engine.GameEngine"), NULL, LOAD_NoFail, NULL );
		assert(pEngineClass);
		UEngine* pEngine = ConstructObject<UEngine>(pEngineClass);
		assert(pEngine);
		if(!pEngine)
		{
			Error.Log(L"Engine initialization failed.");
		}
		pEngine->Init();
		
		GLogWindow->SetExec(pEngine); //If we directly set GExec, only our custom commands work
		GLogWindow->Log( NAME_Title, LocalizeGeneral("Run") );
		
		//Initialize raw input
		if(pEngine->Client && pEngine->Client->Viewports.Num()>0)
		{
			s_hWnd = static_cast<HWND>(pEngine->Client->Viewports(0)->GetWindow());
			GConfig->GetBool(L"WinDrv.WindowsClient",L"UseDirectInput",s_bDirectInput);
			if(s_bDirectInput && !RegisterRawInput(s_hWnd))
			{
				Error.Log(L"Raw input: Failed to register raw input device.");
			}
		}
		else
		{
			Log.Log(L"Raw input: Unable to get window handle.");
		}		
	
		GConfig->GetInt(L"WinDrv.WindowsClient",L"FPSLimit",s_iFPSLimit);

		//Main loop
		GIsRunning = 1;
		if(!GIsRequestingExit)
		{
			MainLoop(pEngine);
		}
		GIsRunning = 0;
		
		GLogWindow->Log(NAME_Title, LocalizeGeneral("Exit"));
	}

	//Uninit
	appPreExit();
	appExit();
	GIsStarted = 0;

	UninitDetours();
	return EXIT_SUCCESS;
}