#include "stdafx.h"
#include "FixApp.h"
#include "Misc.h"
#include "resource.h"

const wchar_t* const CFixApp::sm_pszOTPRootWindow = L"otpUIfix.otpRootWindow";
const wchar_t* const CFixApp::sm_pszDefaultRootWindow = L"DeusEx.DeusExRootWindow";

CFixApp::CFixApp()
{

}

CFixApp::~CFixApp()
{

}

bool CFixApp::Show(const HWND hWndParent) const
{
	return DialogBoxParam(GetModuleHandle(0),MAKEINTRESOURCE(IDD_DIALOG2),hWndParent,FixAppDialogProc,reinterpret_cast<LPARAM>(this)) == 1;
}

void CFixApp::ReadSettings()
{
	assert(GConfig);

	//Resolution
	int iResX = 1024;
	GConfig->GetInt(L"WinDrv.WindowsClient", L"FullscreenViewportX", iResX);
	int iResY = 768;
	GConfig->GetInt(L"WinDrv.WindowsClient", L"FullscreenViewportY", iResY);

	wchar_t szBuffer[20];
	_snwprintf_s(szBuffer, _TRUNCATE, L"%dx%d", iResX, iResY);
	const int iComboRes = ComboBox_FindStringExact(m_hWndCBResolutions, -1, szBuffer);

	if(iComboRes != CB_ERR) //Can fail if res not supported
	{
		ComboBox_SetCurSel(m_hWndCBResolutions, iComboRes);
		CheckRadioButton(m_hWnd, RADIO_RESCOMMON, RADIO_RESCUSTOM, RADIO_RESCOMMON);
	}
	else
	{
		CheckRadioButton(m_hWnd,RADIO_RESCOMMON,RADIO_RESCUSTOM,RADIO_RESCUSTOM);
		SetDlgItemInt(m_hWnd,TXT_RESX,iResX,FALSE);
		SetDlgItemInt(m_hWnd,TXT_RESY,iResY,FALSE);
	}

	//Bit depth
	int iBitDepth = sm_iBPP_32;
	GConfig->GetInt(L"WinDrv.WindowsClient", L"FullscreenColorBits", iBitDepth);

	if(iBitDepth == sm_iBPP_16)
	{
		CheckRadioButton(m_hWnd,RADIO_16BIT,RADIO_32BIT,RADIO_16BIT);
	}
	else
	{
		CheckRadioButton(m_hWnd,RADIO_16BIT,RADIO_32BIT,RADIO_32BIT);
	}

	//FOV
	const float fDefaultFOV = Misc::GetDefaultFOV();
	float fFOV = fDefaultFOV;
	GConfig->GetFloat(L"Engine.PlayerPawn", L"DefaultFOV", fFOV, *static_cast<FConfigCacheIni*>(GConfig)->UserIni);
	UBOOL bUseAutoFOV = TRUE;
	GConfig->GetBool(PROJECTNAME, L"UseAutoFOV", bUseAutoFOV);

	if(bUseAutoFOV)
	{
		CheckRadioButton(m_hWnd, RADIO_FOVDEFAULT, RADIO_FOVCUSTOM, RADIO_FOVAUTO);
	}
	else if (fFOV == fDefaultFOV)
	{
		CheckRadioButton(m_hWnd, RADIO_FOVDEFAULT, RADIO_FOVCUSTOM, RADIO_FOVDEFAULT);
	}
	else
	{
		CheckRadioButton(m_hWnd, RADIO_FOVDEFAULT, RADIO_FOVCUSTOM, RADIO_FOVCUSTOM);		
	}
	SetDlgItemInt(m_hWnd, TXT_FOV, static_cast<UINT>(fFOV), FALSE);

	//GUI scaling fix
	const wchar_t* pszRootWindow = GConfig->GetStr(L"Engine.Engine", L"root");
	assert(pszRootWindow);
	if(_wcsicmp(sm_pszOTPRootWindow,pszRootWindow)==0)
	{
		CheckDlgButton(m_hWnd, CHK_GUIFIX,1);
	}
	else if(wcscmp(pszRootWindow, L"") == 0 || _wcsicmp(sm_pszDefaultRootWindow, pszRootWindow) == 0)
	{
		CheckRadioButton(m_hWnd, RADIO_GUIDEFAULT, RADIO_GUIFIX, RADIO_GUIDEFAULT);
	}
	else
	{
		const HWND h = GetDlgItem(m_hWnd, CHK_GUIFIX);
		EnableWindow(h, FALSE);
		SetWindowText(h, L"GUI (Custom GUI found)");
	}
	
	//Renderer
	const wchar_t* pszRenderer = GConfig->GetStr(L"Engine.Engine", L"GameRenderDevice");
	if(pszRenderer[0]=='\0')
	{
		pszRenderer = L"SoftDrv.SoftwareRenderDevice";
	}

	const auto renderIt = std::find(m_Renderers.cbegin(), m_Renderers.cend(), pszRenderer);
	if(renderIt != m_Renderers.cend())
	{
		ComboBox_SetCurSel(m_hWndCBRenderers, renderIt - m_Renderers.begin());
	}	
	
	//Detail textures
	BOOL bDetailTextures = TRUE;
	GConfig->GetBool(pszRenderer, L"DetailTextures", bDetailTextures);
	if(bDetailTextures)
	{
		CheckDlgButton(m_hWnd,CHK_DETAILTEX,1);
	}

	//Mouse acceleration
	BOOL bNoMouseAccel = FALSE;
	GConfig->GetBool(PROJECTNAME, L"RawInput", bNoMouseAccel);
	if(bNoMouseAccel)
	{
		CheckDlgButton(m_hWnd,CHK_NOMOUSEACCEL,1);
	}
	
	//DirectSound
	BOOL bDirectSound = TRUE;
	GConfig->GetBool(L"Galaxy.GalaxyAudioSubsystem", L"UseDirectSound", bDirectSound);
	if(bDirectSound)
	{
		CheckDlgButton(m_hWnd,CHK_DIRECTSOUND,1);
	}

	//Full-screen
	BOOL bFullScreen = TRUE;
	GConfig->GetBool(L"WinDrv.WindowsClient", L"StartupFullScreen", bFullScreen);
	if(bFullScreen)
	{
		CheckDlgButton(m_hWnd,CHK_FULLSCREEN,1);
	}
	
	//Audio latency
	int iSndLatency = 40;
	GConfig->GetInt(L"Galaxy.GalaxyAudioSubsystem", L"Latency", iSndLatency);
	SetDlgItemInt(m_hWnd,TXT_LATENCY,iSndLatency,FALSE);

	//FPS limit
	int iFPSLimit = 0;
	GConfig->GetInt(L"WinDrv.WindowsClient", L"FPSLimit", iFPSLimit);
	SetDlgItemInt(m_hWnd, TXT_FPSLIMIT, iFPSLimit, FALSE);
}

void CFixApp::PopulateDialog()
{
	//Populate the resolution combobox
	DEVMODE dm = {};
	dm.dmSize = sizeof(dm);
	Resolution r= {};
	for(DWORD iModeNum = 0; EnumDisplaySettings(NULL, iModeNum, &dm) != FALSE; iModeNum++)
	{
		if(dm.dmPelsWidth != r.iX || dm.dmPelsHeight != r.iY) //Only add each res once, but don't actually check if it matches the current color depth/refresh rate
		{
			r.iX = dm.dmPelsWidth;
			r.iY = dm.dmPelsHeight;
			m_Resolutions.push_back(r);

			wchar_t szBuffer[20];
			_snwprintf_s(szBuffer, _TRUNCATE, L"%dx%d", r.iX, r.iY);
			int iIndex = ComboBox_AddString(m_hWndCBResolutions, szBuffer);
			ComboBox_SetItemData(m_hWndCBResolutions, iIndex, &(m_Resolutions.back()));
		}		
	}
	ComboBox_SetCurSel(m_hWndCBResolutions, 0);


	//Renderers (based on UnEngineWin.h), requires appInit() to have been called
	TArray<FRegistryObjectInfo> Classes;
	Classes.Empty();

	UObject::GetRegistryObjects( Classes, UClass::StaticClass(), URenderDevice::StaticClass(), 0 );
	for( TArray<FRegistryObjectInfo>::TIterator It(Classes); It; ++It )
	{
		FString Path = It->Object, Left, Right;
		if( Path.Split(L".",&Left,&Right)  )
		{
			const wchar_t* pszDesc = Localize(*Right,L"ClassCaption",*Left);
			assert(pszDesc);
			if(ComboBox_FindStringExact(m_hWndCBRenderers, -1, pszDesc) == CB_ERR)
			{
				ComboBox_AddString(m_hWndCBRenderers, pszDesc);
				m_Renderers.emplace_back(static_cast<wchar_t*>(Path.GetCharArray().GetData()));
			}
		}
	}

}

void CFixApp::ApplySettings() const
{
	assert(GConfig);

	//GUI scaling fix
	if(IsWindowEnabled(GetDlgItem(m_hWnd, CHK_GUIFIX)))
	{
		if(IsDlgButtonChecked(m_hWnd, CHK_GUIFIX))
		{
			GConfig->SetString(L"Engine.engine", L"root", sm_pszOTPRootWindow);
		}
		else
		{
			GConfig->SetString(L"Engine.engine", L"root", sm_pszDefaultRootWindow);
		}
	}

	//Bit depth
	const unsigned char iBitDepth = IsDlgButtonChecked(m_hWnd,RADIO_16BIT) ? sm_iBPP_16 : sm_iBPP_32;	
	GConfig->SetInt(L"WinDrv.WindowsClient",L"FullscreenColorBits",iBitDepth);

	//Resolution
	size_t iResX, iResY;
	if(IsDlgButtonChecked(m_hWnd,RADIO_RESCOMMON))
	{
		const int i = ComboBox_GetCurSel(m_hWndCBResolutions);
		const Resolution* pRes = reinterpret_cast<Resolution*>(ComboBox_GetItemData(m_hWndCBResolutions,i));
		iResX = pRes->iX;
		iResY = pRes->iY;
	}
	else
	{
		iResX = GetDlgItemInt(m_hWnd, TXT_RESX, nullptr, FALSE);
		iResY = GetDlgItemInt(m_hWnd, TXT_RESY, nullptr, FALSE);
	}
	GConfig->SetInt(L"WinDrv.WindowsClient",L"FullscreenViewportX",iResX);
	GConfig->SetInt(L"WinDrv.WindowsClient", L"WindowedViewportX", iResX);
	GConfig->SetInt(L"WinDrv.WindowsClient",L"FullscreenViewportY",iResY);
	GConfig->SetInt(L"WinDrv.WindowsClient", L"WindowedViewportY", iResY);

	//FOV
	float fFOV;
	if(IsDlgButtonChecked(m_hWnd, RADIO_FOVAUTO))
	{
		fFOV = Misc::CalcFOV(iResX, iResY);
		GConfig->SetBool(PROJECTNAME, L"UseAutoFOV", TRUE);
	}
	else
	{
		if(IsDlgButtonChecked(m_hWnd, RADIO_FOVDEFAULT))
		{
			fFOV = Misc::GetDefaultFOV();
			GConfig->SetBool(PROJECTNAME, L"UseAutoFOV", FALSE);
		}
		else
		{
			fFOV = static_cast<float>(GetDlgItemInt(m_hWnd, TXT_FOV, nullptr, FALSE));
			GConfig->SetBool(PROJECTNAME, L"UseAutoFOV", FALSE);
		}

		const wchar_t* pszUserIni = *static_cast<FConfigCacheIni*>(GConfig)->UserIni;
		GConfig->SetFloat(L"Engine.PlayerPawn", L"DesiredFOV", fFOV, pszUserIni);
		GConfig->SetFloat(L"Engine.PlayerPawn", L"DefaultFOV", fFOV, pszUserIni);
	}
	
	//Disable mouse scaling
	GConfig->SetBool(PROJECTNAME,L"RawInput",IsDlgButtonChecked(m_hWnd,CHK_NOMOUSEACCEL)!=0);
	//DirectSound
	GConfig->SetBool(L"Galaxy.GalaxyAudioSubsystem",L"UseDirectSound",IsDlgButtonChecked(m_hWnd,CHK_DIRECTSOUND)!=0);
	//Audio latency
	GConfig->SetInt(L"Galaxy.GalaxyAudioSubsystem", L"Latency", GetDlgItemInt(m_hWnd, TXT_LATENCY, nullptr, FALSE));
	//Full-screen
	GConfig->SetBool(L"WinDrv.WindowsClient",L"StartupFullSCreen",IsDlgButtonChecked(m_hWnd,CHK_FULLSCREEN)!=0);
	//FPS Limit
	GConfig->SetInt(L"WinDrv.WindowsClient", L"FPSLimit", GetDlgItemInt(m_hWnd, TXT_FPSLIMIT, nullptr, FALSE));
		
	//Renderer
	const int iRendererIndex = ComboBox_GetCurSel(m_hWndCBRenderers);
	GConfig->SetString(L"Engine.Engine",L"GameRenderDevice",m_Renderers[iRendererIndex].c_str());
	//Detail textures
	GConfig->SetBool(m_Renderers[iRendererIndex].c_str(),L"DetailTextures",IsDlgButtonChecked(m_hWnd,CHK_DETAILTEX)!=0);
}

INT_PTR CALLBACK CFixApp::FixAppDialogProc(HWND hwndDlg,UINT uMsg,WPARAM wParam,LPARAM lParam)
{	
	CFixApp* pThis = reinterpret_cast<CFixApp*>(GetProp(hwndDlg,L"this"));

	switch (uMsg)
	{
	case WM_INITDIALOG:
		{
			//Get all object oriented like
			SetProp(hwndDlg,L"this",reinterpret_cast<HANDLE>(lParam));
			pThis =  reinterpret_cast<CFixApp*>(lParam);
			pThis->m_hWnd = hwndDlg;
			pThis->m_hWndCBRenderers = GetDlgItem(hwndDlg, COMBO_RENDERER);
			pThis->m_hWndCBResolutions = GetDlgItem(hwndDlg, COMBO_RESOLUTION);
			SendMessage(hwndDlg, WM_SETICON, ICON_BIG, reinterpret_cast<LPARAM>(LoadIcon(reinterpret_cast<HINSTANCE>(GetWindowLong(hwndDlg, GWL_HINSTANCE)), MAKEINTRESOURCE(IDI_ICON))));

			pThis->PopulateDialog();
			pThis->ReadSettings();
		}
	return TRUE;
	
	case WM_COMMAND:
		switch(HIWORD(wParam))
		{
		case CBN_SELCHANGE:
			switch(LOWORD(wParam))
			{
			case COMBO_RESOLUTION:					 
				CheckRadioButton(hwndDlg, RADIO_RESCOMMON, RADIO_RESCUSTOM, RADIO_RESCOMMON);
				return TRUE;				
			}
			break;
		case EN_SETFOCUS:
			switch(LOWORD(wParam))
			{
			case TXT_FOV:
				CheckRadioButton(hwndDlg, RADIO_FOVDEFAULT, RADIO_FOVCUSTOM, RADIO_FOVCUSTOM);
				return TRUE;

			case TXT_RESX:
			case TXT_RESY:
				CheckRadioButton(hwndDlg, RADIO_RESCOMMON, RADIO_RESCUSTOM, RADIO_RESCUSTOM);
				return TRUE;

			}
			break;
		default:
			switch(LOWORD(wParam))
			{

			case IDOK:
				pThis->ApplySettings();
			case IDCANCEL:
				EndDialog(hwndDlg, 0);
				return TRUE;

			}
			break;
		}
		break;

	case WM_CLOSE:
		EndDialog(hwndDlg,0);
		return TRUE;

	}

	return FALSE;
}