#include "stdafx.h"
#include "FileManagerDeusExe.h"
#include "Misc.h"
#include "RawInput.h"
#include "LauncherDialog.h"
#include "Fixapp.h"
#include "Detoureds.h"
#include "ExecHook.h"

//Do not put before stdafx.h
#pragma comment(linker,"/manifestdependency:\"type='win32' name='Microsoft.Windows.Common-Controls' version='6.0.0.0' processorArchitecture='*' publicKeyToken='6595b64144ccf1df' language='*'\"")

extern "C" {wchar_t GPackage[64] = L"Launch"; } //Will be set to exe name later

//Unreal engine framework classes
static LARGE_INTEGER s_iPerfCounterFreq;
static float s_fFPSLimit; //Because GetMaxTickRate() is float
static UBOOL s_bRawInput;
static HWND s_hWnd;
static UBOOL s_bAutoFov;
static int s_iSizeX;
static int s_iSizeY;
static UViewport* s_pViewPort; //If user closes window, viewport disappears before we get WM_QUIT
static bool s_bPrevInMenu;

static void ApplyAutoFOV(const int iSizeX, const int iSizeY)
{
    assert(s_pViewPort);
    const float fFOV = Misc::CalcFOV(iSizeX, iSizeY);
    wchar_t szCmd[12];
    swprintf_s(szCmd, L"fov %6.3f", fFOV);

    s_pViewPort->Exec(szCmd);
    s_iSizeX = iSizeX;
    s_iSizeY = iSizeY;
}

static void MainLoop(UEngine* const pEngine)
{
    assert(pEngine);

    LARGE_INTEGER iOldTime;
    if(!QueryPerformanceCounter(&iOldTime)) //Initial time
    {
        return;
    }

    while (GIsRunning && !GIsRequestingExit)
    {
        LARGE_INTEGER iTime;
        QueryPerformanceCounter(&iTime);
        const float fDeltaTime = (iTime.QuadPart - iOldTime.QuadPart) / static_cast<float>(s_iPerfCounterFreq.QuadPart);

        //Lock game update speed to fps limit
        if((s_fFPSLimit == 0.0f || fDeltaTime >= 1.0f / s_fFPSLimit) && (pEngine->GetMaxTickRate() == 0.0f || fDeltaTime >= 1.0f / pEngine->GetMaxTickRate()))
        {
            pEngine->Tick(fDeltaTime);
            if(GWindowManager)
            {
                GWindowManager->Tick(fDeltaTime);
            }
            iOldTime = iTime;
        }
        
        if(pEngine->Client->Viewports.Num() == 0) //If user closes window, viewport disappears before we get WM_QUIT
        {
            s_pViewPort = nullptr;
        }

        POINT p;
        GetCursorPos(&p);
        const bool bMouseOverWindow = WindowFromPoint(p) == s_hWnd;
        const bool bHasFocus = GetFocus() == s_hWnd;
        if(s_pViewPort)
        {
            assert(s_hWnd);

            RECT rClientArea;
            GetClientRect(s_hWnd, &rClientArea);

            //PeekMessage() doesn't get WM_SIZE
            //Default/desired FOV check is so we don't change FOV while zoomed in
            if(s_bAutoFov && s_pViewPort->Actor->DefaultFOV == s_pViewPort->Actor->DesiredFOV)
            {
                int iSizeX = rClientArea.right - rClientArea.left;
                int iSizeY = rClientArea.bottom - rClientArea.top;

                //Handle auto FOV
                if(s_iSizeX != iSizeX || s_iSizeY != iSizeY)
                {
                    ApplyAutoFOV(iSizeX, iSizeY);
                }
            }
            
            //pEngine->Client->Viewports(0)->SetMouseCapture()'s cursor centering doesn't work with raw input.
            //Why doesn't it work? Because we block WM_MOUSEMOVE messages, which the game apparently uses to center the cursor.
            //SetCursorPos() still works, though, which I'd assume the game uses; ClipCursor() didn't exist until Win2000.
            //Also, if you force the game to turn off mouse centering, the camera doesn't work; does it use the WM_MOUSEMOVE messages generated by SetCursorPos() to actually move the camera?

            //Issue: using raw input, in full-screen mode you can move the cursor around while controlling the camera, if you then open the menu and slightly move the mouse
            //The game's cursor will snap to the Windows mouse cursor position.
            //Theory as to why: SetMouseCapture() without clipping resets the mouse position to previous (looking at headers / UT X driver code).
            //In full-screen mode this is not done when going to the menu (observed in Windows Input mode, cursor keeps being centered).
            //Because the game uses relative messages for menu mouse input (MouseDelta(), not MousePosition()) this doesn't matter.

            //Other observed behavior in Windows Input mode, running windowed: mouse is clipped to window dimensions + centered in menu mode (like in camera mode)
            //Until alt+tab or mission start, at which point it's not clipped and window can be resized

            //Forcing mouse to be centered in menu mode makes it feel weird, doesn't match Windows mouse cursor movements

            /* Tests
            1. Does menu cursor track Windows cursor nicely
            2. Can cursor immediately leave window when menu first pops up (who cares)
            3. Does resize cursor pop up on window edges
            4. When alt+tabbing while not in a menu, make sure mouse isn't clipped to game window area
            5. Both windowed and full screen: when having controlled the camera and then entering a menu, the mouse should either be centered or in the position where it last was.
               When touching the mouse, it should not teleport due to having been moved in camera mode.
            5a. Still happens in raw input + windowed mode when entering menu without having first moved mouse, acceptable.
            6. When alt+tabbing and not in a menu, make sure camera isn't controlled by mouse movements until the window is clicked
            7. Make sure Windows mouse cursor is not visible (other than during testing)
            8. Make sure preferences window is usable (no hidden cursor) and that it doesn't pop up a phantom cursor in menu mode
            9. When looking around with preferences window on top, cursor doesn't appear
            10. In fullscreen mode, when rapidly clicking, window isn't minimized
            11. In two-monitor fullscreen make sure mouse can't move outside of monitor
            */

            const APlayerPawnExt* const pPlayer = static_cast<APlayerPawnExt*>(s_pViewPort->Actor);
            assert(pPlayer);
            XRootWindow* const pRoot = static_cast<XRootWindow*>(pPlayer->rootWindow);
            assert(pRoot);

            const bool bInMenu = pRoot->IsMouseGrabbed()!=0;

            if (s_bRawInput && s_pViewPort && s_pViewPort->IsFullscreen())
            {
                if (bInMenu && !s_bPrevInMenu) //Fixes that in fullscreen mode, windows mouse cursor pos isn't matched to DX menu cursor
                {
                    float fX, fY;
                    pRoot->GetRootCursorPos(&fX, &fY);
                    POINT p{static_cast<int>(fX), static_cast<int>(fY)};
                    ClientToScreen(s_hWnd, &p);
                    SetCursorPos(p.x, p.y);
                }
                ClipCursor(&rClientArea); //Fixed being able to move cursor outside of fullscreen game on dual monitor systems
            }
            s_bPrevInMenu = bInMenu;

            const bool bMouseInClientRect = ScreenToClient(s_hWnd, &p) && PtInRect(&rClientArea, p); //This makes sure resize cursor isn't hidden
            const bool bCaptured = GetCapture() == s_hWnd;
            if (bMouseInClientRect && (bMouseOverWindow || bCaptured)) //Want to show cursor when over preferences window when we don't have focus, but not when it's under the window if we do
            {
                while(ShowCursor(FALSE) > 0); //Get rid of double mouse cursors when game doesn't clip it
            }
            else
            {
                while(ShowCursor(TRUE) <= 0);
            }
        }

        MSG Msg;
        while (PeekMessage(&Msg, NULL, 0, 0, PM_REMOVE))
        {
            bool bSkipMessage = false;

            switch (Msg.message)
            {
            case WM_QUIT:
                GIsRequestingExit = 1;
                break;

            case WM_MOUSEMOVE:
                if (s_pViewPort && s_bRawInput)
                {
                    if (bMouseOverWindow) //Because preferences window defers mousemove calls to us, somehow
                    {
                        //Use WM_MOUSEMOVE to control menu cursor
                        const int iXPos = GET_X_LPARAM(Msg.lParam);
                        const int iYPos = GET_Y_LPARAM(Msg.lParam);
                        pEngine->MousePosition(s_pViewPort, 0, static_cast<float>(iXPos), static_cast<float>(iYPos));
                    }
                    bSkipMessage = true;
                }
                break;

            case WM_INPUT:
            {
                //Use raw input to control camera
                if (s_pViewPort && bHasFocus)
                {
                    RAWINPUT raw;
                    unsigned int rawSize = sizeof(raw);
                    GetRawInputData(reinterpret_cast<HRAWINPUT>(Msg.lParam), RID_INPUT, &raw, &rawSize, sizeof(RAWINPUTHEADER));

                    const float fDeltaX = static_cast<float>(raw.data.mouse.lLastX);
                    const float fDeltaY = static_cast<float>(raw.data.mouse.lLastY);
                    if(fDeltaX != 0.0f)
                    {
                        pEngine->InputEvent(s_pViewPort, EInputKey::IK_MouseX, EInputAction::IST_Axis, fDeltaX);
                    }
                    if(fDeltaY != 0.0f)
                    {
                        pEngine->InputEvent(s_pViewPort, EInputKey::IK_MouseY, EInputAction::IST_Axis, -fDeltaY);
                    }
                    bSkipMessage = true;
                }
            }
                break;
            }

            if(!bSkipMessage)
            {
                TranslateMessage(&Msg);
                DispatchMessage(&Msg);
            }
        }
    }

}

static bool DoLauncherDialog()
{
    CLauncherDialog LD;
    return LD.Show(NULL);
}

INT WINAPI WinMain(HINSTANCE /*hInInstance*/, HINSTANCE /*hPrevInstance*/, LPSTR /*lpCmdLine*/, INT /*nCmdShow*/)
{
    INITCOMMONCONTROLSEX CommonControlsInfo;
    CommonControlsInfo.dwSize = sizeof(INITCOMMONCONTROLSEX);
    CommonControlsInfo.dwICC = ICC_TREEVIEW_CLASSES | ICC_LINK_CLASS;
    if(InitCommonControlsEx(&CommonControlsInfo) != TRUE)
    {
        return EXIT_FAILURE;
    }

    appStrcpy(GPackage, appPackage());

    //Init core
    FMallocWindows Malloc;
    FOutputDeviceFile Log;
    FOutputDeviceWindowsError Error;
    FFeedbackContextWindows Warn;

    //If -localdata command line option present, don't use user documents for data; can't use appCmdLine() yet.
    std::unique_ptr<FFileManagerDeusExe> pFileManager(wcswcs(GetCommandLine(), L" -localdata") == nullptr ? new FFileManagerDeusExeUserDocs : new FFileManagerDeusExe);

    appInit(GPackage, GetCommandLine(), &Malloc, &Log, &Error, &Warn, pFileManager.get(), FConfigCacheIni::Factory, 1);
    GLog->Logf(L"Deus Exe: version %s.", Misc::GetVersion());

    pFileManager->AfterCoreInit();

    GIsStarted = 1;
    GIsServer = 1;
    GIsClient = !ParseParam(appCmdLine(), L"SERVER");
    GIsEditor = 0;
    GIsScriptable = 1;
    GLazyLoad = !GIsClient;

    assert(GConfig);

    if(!InitDetours())
    {
        Log.Log(L"Failed to initialize detours.");
    }
    if(!Misc::SetDEP(0)) //Disable DEP for process (also need NXCOMPAT=NO); needed for Galaxy.dll
    {
        Log.Log(L"Failed to set process DEP flags.");
    }
    if(QueryPerformanceFrequency(&s_iPerfCounterFreq) == FALSE)
    {
        Error.Log(L"Failed to query performance counter.");
    }

    int iFirstRun = 0;
    GConfig->GetInt(L"FirstRun", L"FirstRun", iFirstRun);
    const bool bFirstRun = iFirstRun < ENGINE_VERSION;
    if(bFirstRun) //Select better default options
    {
        GConfig->SetString(L"Engine.Engine", L"GameRenderDevice", L"D3DDrv.D3DRenderDevice");
        GConfig->SetString(L"WinDrv.WindowsClient", L"FullscreenColorBits", L"32");
        wchar_t szTemp[1024];
        _itow_s(GetSystemMetrics(SM_CXSCREEN), szTemp, 10);
        GConfig->SetString(L"WinDrv.WindowsClient", L"FullscreenViewportX", szTemp);
        _itow_s(GetSystemMetrics(SM_CYSCREEN), szTemp, 10);
        GConfig->SetString(L"WinDrv.WindowsClient", L"FullscreenViewportY", szTemp);
        GConfig->SetBool(PROJECTNAME, L"RawInput", TRUE);
        GConfig->SetBool(PROJECTNAME, L"UseAutoFOV", TRUE);
        GConfig->SetInt(PROJECTNAME, L"FPSLimit", 200);
    }

    //Show launcher dialog
    if(ParseParam(appCmdLine(), L"changevideo") || bFirstRun)
    {
        CFixApp FixApp;
        FixApp.Show(NULL);
        if(bFirstRun)
        {
            GConfig->SetInt(L"FirstRun", L"FirstRun", ENGINE_VERSION);
        }
    }
    if(!GIsClient || ParseParam(appCmdLine(), TEXT("skipdialog")) || DoLauncherDialog()) //Here the game actually starts
    {
        pFileManager->OnGameStart();

        BOOL bUseSingleCPU = FALSE;
        GConfig->GetBool(PROJECTNAME, L"UseSingleCPU", bUseSingleCPU);
        if (bUseSingleCPU)
        {
            if (SetProcessAffinityMask(GetCurrentProcess(), 0x1) == FALSE) //Force on single CPU
            {
                Log.Log(L"Failed to set process affinity.");
            }
        }

        GConfig->GetBool(PROJECTNAME, L"RawInput", s_bRawInput);
        if(s_bRawInput) //If raw input is enabled, disable DirectInput
        {
            GConfig->SetBool(L"WinDrv.WindowsClient", L"UseDirectInput", FALSE);
        }

        //Init windowing
        InitWindowing();

        //Create log window
        const std::unique_ptr<WLog> LogWindowPtr = std::make_unique<WLog>(Log.Filename, Log.LogAr, L"GameLog");
        GLogWindow = LogWindowPtr.get(); //Yup...
        GLogWindow->OpenWindow(!GIsClient, 0);
        GLogWindow->Log(NAME_Title, LocalizeGeneral("Start"));

        //Create command processor, fixes dedicated server preferences
        FExecHook ExecHook;
        GExec = &ExecHook;

        //Init engine
        UClass* const pEngineClass = UObject::StaticLoadClass(UGameEngine::StaticClass(), nullptr, L"ini:Engine.Engine.GameEngine", nullptr, LOAD_NoFail, nullptr);
        assert(pEngineClass);
        UEngine* const pEngine = ConstructObject<UEngine>(pEngineClass);
        assert(pEngine);
        if(!pEngine)
        {
            Error.Log(L"Engine initialization failed.");
        }

        pEngine->Init();

        GLogWindow->SetExec(pEngine); //If we directly set GExec, only our custom commands work
        GLogWindow->Log(NAME_Title, LocalizeGeneral("Run"));

        //Find window handle
        if(GIsClient)
        {
            if(pEngine->Client && pEngine->Client->Viewports.Num() > 0)
            {
                s_pViewPort = pEngine->Client->Viewports(0);
                s_hWnd = static_cast<const HWND>(s_pViewPort->GetWindow());
            }
            else
            {
                Log.Log(L"Unable to get viewport.");
            }
        }

        //Initialize raw input
        if(s_bRawInput && s_hWnd)
        {
            if(!RegisterRawInput(s_hWnd))
            {
                Error.Log(L"Raw input: Failed to register raw input device.");
            }
        }

        int iFPSLimit;
        GConfig->GetInt(L"WinDrv.WindowsClient", L"FPSLimit", iFPSLimit);
        s_fFPSLimit = static_cast<float>(iFPSLimit);

        GConfig->GetBool(PROJECTNAME, L"UseAutoFOV", s_bAutoFov);
        if(GIsClient && s_bAutoFov)
        {
            RECT r;
            GetClientRect(s_hWnd, &r);
            int iSizeX = r.right - r.left;
            int iSizeY = r.bottom - r.top;
            ApplyAutoFOV(iSizeX, iSizeY);
        }

        //Main loop
        GIsRunning = 1;
        if(!GIsRequestingExit)
        {
            MainLoop(pEngine);
        }
        GIsRunning = 0;

        GLogWindow->Log(NAME_Title, LocalizeGeneral("Exit"));
    }

    //Uninit
    appPreExit();
    appExit();
    GIsStarted = 0;

    UninitDetours();
    return EXIT_SUCCESS;
}