#include "tcpclient.h"

void TcpClient::onRead()
{
    // clear buffer from previous, complete reads.
    if ( _rxBuffer.contains("\\final\\") )
    {
        // if buffer contains "\final\", a previous message was completed. buffer is safe to clear now.
        _rxBuffer = "";
    }

    // read from tcp connection
    QByteArray receiveBuffer = _tcpSocket->readAll();
    _log->logEvent("tcp", QStringLiteral("%1 sent '%2'").arg(_clientLabel, receiveBuffer.data()) );

    // final received?
    _rxBuffer += receiveBuffer;
    if ( ! receiveBuffer.contains("\\final\\") )
    {
        // wait for data to be complete
        return;
    }

    // reset timeout
    _timeOut.start();

    // process data according to \key\value formatting
    QMultiHash<QString, QString> receiveData = parseGameSpy0Buffer(_rxBuffer);

    // part 1: validation challenge.
    if ( receiveData.contains("validate") )
    {
        // combined queries provide two gamenames: one for authentication and one for list request.
        QString gamename = ( receiveData.values("gamename").size() >= 2
                             ? receiveData.values("gamename").takeLast() // removes duplicate gamename
                             : receiveData.value("gamename","") );

        // sanity check
        if ( _gameInfoDetails->supportedGames.contains(gamename) )
        {
            // get response
            AuthResult authResult = validateGamename(CLIENT,
                                                     gamename,
                                                     receiveData.value("validate",""),
                                                     _gameInfoDetails->supportedGames.value(gamename).cipher,
                                                     _secure,
                                                     receiveData.value("enctype", "0").toInt() );

            // authenticate
            if ( authResult.auth )
            {
                _hasValidated = true;
                _log->logEvent("auth", QStringLiteral("%1 passed validation for %2").arg(_clientLabel, gamename));
            }
            else
            {
                _log->logEvent("auth", QStringLiteral("%1 failed validation for %2").arg(_clientLabel, gamename));

                // log detailed information on failure
                _log->logEvent("secure", QStringLiteral("secure: '%1', gamename: '%2', validate: '%3', expected: '%4'")
                                            .arg(_secure, gamename, receiveData.value("validate", "null"), authResult.validate ));

                // kick client after unsuccessful validation
                this->disconnect();
            }
        }
        else
        {
            _log->logEvent("support", QStringLiteral("%1 tried to validate for unknown game %2").arg(_clientLabel, gamename));

            // log detailed information on failure
            _log->logEvent("secure", QStringLiteral("secure: '%1', gamename: '%2', validate: '%3'")
                                        .arg(_secure, gamename, receiveData.value("validate", "null") ));
            // kick client after unsuccessful validation
            this->disconnect();
        }

    } // hasValidated ?

    // part 2: after validation, send serverlist
    if ( receiveData.contains("list") )
    {
        // list request is only granted after validation
        if ( ! _hasValidated )
        {
            // kick client after unsuccessful validation
            _log->logEvent("auth", QStringLiteral("%1 requested a list for %2 without validating").arg(_clientLabel, receiveData.value("gamename", "")) );
            this->disconnect();
            return;
        }

        // string format or compressed format?
        bool cmp = (receiveData.value("list","").compare("cmp", Qt::CaseInsensitive) == 0 );

        // verify that cmp is properly implemented (FIXME: remove me later)
        if (cmp) _log->logEvent("debug", QStringLiteral("%1 requested a compressed list").arg(_clientLabel) );

        // get list from db and send it (if no gamename provided, result will be an empty list)
        QByteArray writeBuffer = _databaseHandle->compileServerlist(receiveData.value("gamename", ""), _serverttl_s, cmp, false);
        _tcpSocket->write(writeBuffer);

        _log->logEvent("list", QStringLiteral("%1 received the list for %2").arg(_clientLabel, receiveData.value("gamename", "")));

        // all done
        this->disconnect();

    } // list

    // part 2b: 333networks synchronisation
    if ( receiveData.contains("sync") )
    {
        // list request is only granted after validation
        if ( ! _hasValidated )
        {
            // kick client after unsuccessful validation
            _log->logEvent("auth", QStringLiteral("%1 requested to sync for %2 without validating").arg(_clientLabel, receiveData.value("gamename", "")) );
            this->disconnect();
            return;
        }

        // do not sync with ourself
        if ( _masterserverIdentity.compare( receiveData.value("msid", "")) == 0 )
        {
            // msid match -- skip syncing
            _tcpSocket->write("\\final\\");
            _log->logEvent("list", QStringLiteral("skipping sync for self at %1").arg(_clientLabel) );
            this->disconnect();
            return;
        }

        // sync request has options "all" or "gamename_1 gamename_2 gamename_n"
        QStringList gamenameList;
        if ( receiveData.value("sync","").toLower().compare("all") == 0 )
        {
            // get list of gamenames from database
            gamenameList = _databaseHandle->getGamenames(_serverttl_s);
        }
        else
        {
            // get requested gamenames
            QStringList rawList = receiveData.value("sync","").toLower().split(" ");

            // only load gamenames that are in our list of available games
            QStringListIterator rawListNames(rawList);
            while ( rawListNames.hasNext() )
            {
                QString rawGamename = overrideGamename( rawListNames.next() );
                if ( _gameInfoDetails->supportedGames.contains(rawGamename) )
                {
                    gamenameList.append(rawGamename);
                }
            }
        }

        // get compiled sync list for all requested gamenames
        QByteArray writeBuffer = _databaseHandle->compileSyncList(gamenameList, _serverttl_s);
        _tcpSocket->write(writeBuffer);

        // log (displays all the gamenames that were synced or empty when no gamenames/servers were available)
        _log->logEvent("list", QStringLiteral("%1 received the sync list for %2").arg(_clientLabel, gamenameList.join(",")) );

        // all done
        this->disconnect();

    } // sync

    // optional: echo
    if ( receiveData.contains("echo") )
    {
        _log->logEvent("echo", QStringLiteral("%1 echoed %2").arg(_clientLabel, receiveData.value("echo", "")) );
        _tcpSocket->write("\\echo_reply\\" + receiveData.value("echo", "").toUtf8() );
    }

    // else

    // unknown, log as error
    if ( ! receiveData.contains("validate") and
         ! receiveData.contains("list") and
         ! receiveData.contains("sync") and
         ! receiveData.contains("echo") )
    {
        _log->logEvent("unknown", QStringLiteral("%1 with unknown request %2").arg(_clientLabel, _rxBuffer.data() ) );
        this->disconnect();
    }

}
