#ifndef BEACONSERVER_H
#define BEACONSERVER_H

#include <QTimer>
#include <QUdpSocket>
#include <QNetworkDatagram>
#include <QHostInfo>

#include "Core/version.h"
#include "Settings/structsettings.h"
#include "Games/gameinfo.h"
#include "Logger/logger.h"
#include "Database/databasehandle.h"

#include "Protocols/GameSpy0/gamespy0.h"
#include "Protocols/GameSpy0/securevalidate.h"

// helper for replyQuery()
extern int queryId;

// udp beacon server
struct HeartBeat {
    qint64          time = QDateTime::currentSecsSinceEpoch();
    QHostAddress    address;
    unsigned short  port;
    QString         gamename;
    QString         secure;
    QString         protocol = "unknown";
    bool            firstBeacon = true;
};

class BeaconServer : public QObject
{
    Q_OBJECT
public:
    BeaconServer();
    bool init(const sSettings                      &settings,
              const QSharedPointer<GameInfo>       &gameInfoDetails,
              const QSharedPointer<DatabaseHandle> &databaseHandle,
              const QSharedPointer<Logger>         &log);

    // activate listener and broadcast
    bool listen();
    bool uplink();

private: // general udp task handles
    bool _init = false;

    sSettings                       _settings;
    QSharedPointer<GameInfo>        _gameInfoDetails;
    QSharedPointer<DatabaseHandle>  _databaseHandle;
    QSharedPointer<Logger>          _log;

    // udp socket
    QUdpSocket _udpSocket;

    // shared functions
    QStringList replyQuery(const QMultiHash<QString, QString> &query);


private: // udp beacon server

    // udp beacon server functions
    bool isLocal(const QHostAddress &address);

    // timer to sweep up abandoned beacons
    QTimer     _sweepTimer;

    // local settings
    int  _timeOutTime_ms = 5000;
    bool _allowLocal;

    // store information about unverified beacons
    QHash<QString, HeartBeat> _beaconList;

private slots: // udp beacon server event slots
    void onUdpRead();
    void onUdpTimedOut();


private: // broadcast heartbeat

    // broadcast heartbeat functions
    void sendUplink(QHostAddress ip, unsigned short port);

    // outbound heartbeat socket, timer and content
    QTimer _heartbeatTimer;
    QString _heartbeatData;

    // resolved addresses
    QTimer _resolveTimer;
    QHash<QString, QHostAddress> _uplinkAddress;

    // settings
    int _broadcastInterval  = 60000;      // 1 minute
    int _resolveInterval    = 60000 * 30; // every half hour

private slots: // broadcast heartbeat events
    void onResolveTimer();
    void onHeartbeatTimer();
};

#endif // BEACONSERVER_H
