#include "servercheckerscheduler.h"
#include <QDebug>

ServerCheckerScheduler::ServerCheckerScheduler()
{
}

bool ServerCheckerScheduler::init(const sSettings &settings,
                                  const QSharedPointer<GameInfo> &gameInfoDetails,
                                  const QSharedPointer<DatabaseHandle> &databaseHandle,
                                  const QSharedPointer<Logger> &log)
{
    this->_settings          = settings;
    this->_gameInfoDetails   = gameInfoDetails;
    this->_databaseHandle    = databaseHandle;
    this->_log               = log;

    // checker timers
    connect(&_cycleTimer, &QTimer::timeout, this, &ServerCheckerScheduler::onCheckerCycle);
    connect(&_nextTimer,  &QTimer::timeout, this, &ServerCheckerScheduler::onCheckNextServer);
    connect(&_udpSocket,  &QUdpSocket::readyRead, this, &ServerCheckerScheduler::onCheckerResponse);

    // scheduler ready for use
    _init = true;
    return _init;
}

bool ServerCheckerScheduler::scheduleChecker()
{
    // require that .init() was completed first
    if (! _init)
    {
        return false;
    }

    // start the reset timer
    _cycleTimer.start(_cycleTime_ms);

    // complete startup
    _log->logEvent("info", QStringLiteral("checking servers every %1 minutes")
                            .arg(QString::number(_cycleTime_ms/60000)));

    if (DEBUG)
    {
        //-----------------------------------------------
        // dummy data to test serverchecker
        QHash<QString, QString> server = {
            {"ip" ,     "84.83.176.234"},
            {"port",    "7738"},
            {"gamename", "ut"},
        };
        _pendingList.append(server);
        //-----------------------------------------------
    }

    return true;
}

void ServerCheckerScheduler::onCheckerCycle()
{
    // announce
    _log->logEvent("check","restarting server checker cycle");
    _pendingIndex = 0;
    _serverIndex  = -1;

    // (re)set timers
    _nextTimer.setInterval(_nextTime_ms);
    _nextTimer.start();
}

void ServerCheckerScheduler::onCheckNextServer()
{
    // case by case: pending list cleared?
    if ( _pendingIndex < _pendingList.size() )
    {
        // query server
        QHash<QString, QString> server = _pendingList.value( _pendingIndex++ );

        // send datagram (function)
        QHostAddress ip( server.value("ip") );
        unsigned short port = server.value("port").toUShort();
        // TODO: determine by gameInfo.list.protocolType whether secure challenge is needed (and other info)
        this->sendUdpStatus(ip, port, server.value("gamename"), true);

        // log
        _log->logEvent("checker",QStringLiteral("check pending %1:%2").arg(ip.toString(), QString::number(port)));

        // done for this cycle.
        return;
    }

    // case by case: serverlist cleared?
    QHash<QString, QString> nextServer = _databaseHandle->getNextServer(_serverIndex, _settings.Network.serverttl_s);
    if ( nextServer.value("id","-1").toInt() >= 0 )
    {
        // send datagram (function)
        QHostAddress ip( nextServer.value("ip") );
        unsigned short port = nextServer.value("port").toUShort();

        // query server
        QHash<QString, QString> server = _pendingList.value( _pendingIndex++ );

        // send datagram (function)
        this->sendUdpStatus(ip, port, server.value("gamename"), false);

        // log
        _log->logEvent("checker",QStringLiteral("check serverlist %1:%2").arg(ip.toString(), QString::number(port)));

        // index for next cycle
        _serverIndex = nextServer.value("id").toInt();
    }
}

void ServerCheckerScheduler::sendUdpStatus(const QHostAddress   &ip,
                                           const unsigned short &port,
                                           const QString        &gamename,
                                           const bool           &doChallenge)
{
    // add to _beaconList;
    HeartBeat newServer;
    newServer.address   = ip;
    newServer.port      = port;
    newServer.gamename  = gamename;

    // shorthand label
    QString serverLabel = QStringLiteral("%1:%2").arg(ip.toString(), QString::number(port));
    _beaconList[serverLabel] = newServer; // overwrites (timed out / existing) entry

    // do secure challenge?
    if (doChallenge)
    {
        // create datagram
        QString secure = genChallengeString(6, false);
        newServer.secure    = secure;

        QString challenge = QStringLiteral("\\secure\\%1").arg(secure);
        QNetworkDatagram udpDatagram (challenge.toUtf8(), ip, port);

        // send check
        _udpSocket.writeDatagram(udpDatagram);
    }

    // send status request (receives all server information)
    QString checkRequest = "\\status\\";
    QNetworkDatagram udpDatagram (checkRequest.toUtf8(), ip, port);

    // send check
    _udpSocket.writeDatagram(udpDatagram);
}


void ServerCheckerScheduler::onCheckerResponse()
{
    // process incoming udp data
    while ( _udpSocket.hasPendingDatagrams() )
    {
        // get sender and payload
        QNetworkDatagram    datagram        = _udpSocket.receiveDatagram();
        QHostAddress        senderAddress   = QHostAddress( datagram.senderAddress().toIPv6Address() );
        int                 senderPort      = datagram.senderPort();
        QString             receiveBuffer   = datagram.data();

        _log->logEvent("udp", QStringLiteral("%1:%2 sent '%3'")
                           .arg(senderAddress.toString(), QString::number(senderPort), receiveBuffer) );

        // shorthand label
        QString serverLabel = QStringLiteral("%1:%2").arg(senderAddress.toString(), QString::number(senderPort));

        // continue receiving data until message is complete
        HeartBeat newServer = _beaconList[serverLabel];
        newServer.data += receiveBuffer;

        // all data received?
        if ( ! newServer.data.contains("\\final\\") )
        {
            continue;
        }

        // parse data and update (or insert) database
        QHash<QString,QString> receiveData = parseGameSpy0Buffer(newServer.data);

        // validate challenge
        if ( receiveData.contains("validate") and
             ! newServer.secure.isEmpty() and
             ! newServer.gamename.isEmpty() )
        {
            // get response
            AuthResult authResult = validateGamename(CLIENT,
                                                     newServer.gamename,
                                                     receiveData.value("validate",""),
                                                     _gameInfoDetails->supportedGames.value(newServer.gamename).cipher,
                                                     newServer.secure,
                                                     receiveData.value("enctype", "0").toInt() );

            // compare with received response
            if ( authResult.auth )
            {
                // server authenticated - log and add to database
                _log->logEvent("validate", QStringLiteral("successful validate from %1:%2 for %3")
                               .arg(newServer.address.toString(), QString::number(newServer.port),newServer.gamename) );

                // prevent duplicate entries (server possibly was added between secure and validate due to duplicate uplink entry in server config)
                if ( ! _databaseHandle->updateServer(newServer.address, newServer.port, newServer.gamename, true) )
                {
                    // add to database
                    _databaseHandle->insertServer(newServer.address, newServer.port, newServer.gamename, true);

                    // log type "uplink" for first heartbeat. from now on, this server is logged with type "heartbeat"
                    _log->logEvent("uplink", QStringLiteral("%1:%2 for %3")
                                   .arg(newServer.address.toString(), QString::number(newServer.port),newServer.gamename) );

                } // else: updated existing entry, ignore

                // remove from temporary/pending list
                _beaconList.remove(serverLabel);
            }
            else // log failed validate
            {
                _log->logEvent("validate", QStringLiteral("failed validate from %1:%2 for %3")
                               .arg(newServer.address.toString(), QString::number(newServer.port),newServer.gamename) );
                _log->logEvent("secure", QStringLiteral("secure: '%1', gamename: '%2', validate: '%3', expected: '%4'")
                                            .arg(newServer.secure, newServer.gamename, receiveData.value("validate", "null"), authResult.validate ));

                // do not proceed with server
                continue;
            }

            // end validate -- server authenticated
        }

        // server was inserted in database.
        // now update/insert additional server information
        // TODO


    }

}
