#include "settings.h"

bool Settings::loadSettings(const QString &applicationPath)
{
    this->_applicationPath = applicationPath;

    // determine if file exists
    if ( ! QFile(applicationPath+"/"+_settingsPath).exists() )
    {
        // no settings file exists. ask to generate clean settings file.
        logPrimitive() << "No valid settings file was found at " << _settingsPath << endl
                       << "Do you want to generate a clean settings file? [y/N]" << endl;

        // if opted for new config, create a new config file at the provided location
        if ( QTextStream(stdin).readLine().startsWith("y") )
        {
            this->writeSettings();

            // inform that a file was written
            logPrimitive() << "Please update your settings in " << _settingsPath << endl
                           << "and restart the application." << endl;
        }
        return false;
    }

    // open settings
    QSettings settingsFile(applicationPath+"/"+_settingsPath, QSettings::IniFormat);
    if ( settingsFile.status() != QSettings::NoError )
    {
        // error occurred. report and quit.
        logPrimitive() << "An error occurred while loading the configuration (" << QString::number( settingsFile.status() ) << ").";
        return false;
    }

    // public details
    settings.PublicInformation.hostname  = settingsFile.value("PublicDetails/Hostname",  "").toString();
    settings.PublicInformation.adminName = settingsFile.value("PublicDetails/AdminName", "").toString();
    settings.PublicInformation.contact   = settingsFile.value("PublicDetails/Contact",   "").toString();

    // network settings
    settings.Network.beaconPort  = static_cast<unsigned short>( settingsFile.value("Network/BeaconPort", settings.Network.beaconPort).toInt() );
    settings.Network.listenPort  = static_cast<unsigned short>( settingsFile.value("Network/ListenPort", settings.Network.listenPort).toInt() );
    settings.Network.serverttl_s = settingsFile.value("Network/ServerLifeTimeSeconds",                   settings.Network.serverttl_s).toInt();
    settings.Network.allowLocal  = settingsFile.value("Network/AllowLocal",                              settings.Network.allowLocal).toBool();

    // syncer settings
    settings.Syncer.doUplink  = settingsFile.value("Syncer/DoUplink",  settings.Syncer.doUplink).toBool();
    settings.Syncer.doSync    = settingsFile.value("Syncer/DoSync",    settings.Syncer.doSync).toBool();
    settings.Syncer.syncGames = settingsFile.value("Syncer/SyncGames", settings.Syncer.syncGames).toString().toLower();

    // error and number of items
    int parseError = -1;
    int len = settingsFile.beginReadArray("Syncer");

    // read list of syncer items
    for (int i = 0; i < len; i++)
    {
        settingsFile.setArrayIndex(i);
        sSyncServer syncServer;
        QStringList strServer = settingsFile.value("SyncServer").toString().split(",");

        if ( strServer.length() >= 3 )
        {
            // parse
            syncServer.remoteAddress = strServer.value(0).trimmed();
            syncServer.beaconPort    = strServer.value(1).toUShort();
            syncServer.listenPort    = strServer.value(2).toUShort();

            // sanity checks
            if (syncServer.remoteAddress.length() > 0 and syncServer.beaconPort > 0 and syncServer.listenPort > 0)
            {
                // add
                settings.Syncer.syncServers.append(syncServer);
                continue;
            }
        }

        // else input error, do not continue parsing
        parseError = i;
        break;
    }
    settingsFile.endArray();

    // database settings
    settings.Database.databaseType = settingsFile.value("Database/Type", "").toString().toLower();
    settings.Database.dbname       = settingsFile.value("Database/DatabaseName", "").toString();
    settings.Database.user         = settingsFile.value("Database/Username", "").toString();
    settings.Database.pass         = settingsFile.value("Database/Password", "").toString();

    // logging settings
    settings.Logging.cycle           = settingsFile.value("Logging/CycleLogs", "never").toString();
    settings.Logging.suppressLog     = settingsFile.value("Logging/SuppressLog", "debug").toString();
    settings.Logging.suppressDisplay = settingsFile.value("Logging/SuppressDisplay", "none").toString();

    // cleaner than a reportError function
    try {
        // sanity checks
        if (settings.Network.beaconPort == 0)   throw "Network/BeaconPort";
        if (settings.Network.listenPort == 0)   throw "Network/ListenPort";
        if (settings.Network.serverttl_s <= 0)  throw "Network/ServerLifeTimeSeconds";

        // syncer
        if (parseError >= 0) throw QStringLiteral("Syncer/SyncServer[%1]").arg(QString::number(parseError+1));

        // database type
        if (settings.Database.databaseType.isEmpty()) throw "Database/Type";

        // double check that details are filled in
        if (settings.PublicInformation.hostname.length() <= 0)  throw "PublicDetails/Hostname";
        if (settings.PublicInformation.adminName.length() <= 0) throw "PublicDetails/AdminName";
        if (settings.PublicInformation.contact.length() <= 0)   throw "PublicDetails/Contact";
    }
    catch (QString error)
    {
        logPrimitive() << "One or more settings are incorrect: setting " << error << " has an incorrect value!" << endl
                       << "Please correct the value and restart the application." << endl;
        return false;
    }

    // write settings after loading. newly introduced settings are saved in existing
    // settings, existing settings remain. con: overwrites settings in alphabetical order
    this->writeSettings();

    // so far, everything correct
    return true;

}
