#include "syncclient.h"

SyncClient::SyncClient(const sSettings settings,
                       const QSharedPointer<GameInfo>        &gameInfoDetails,
                       const QSharedPointer<DatabaseHandle>  &databaseHandle,
                       const QSharedPointer<Logger>          &log,
                       const QHostAddress remoteAddress,
                       unsigned short remotePort)
{
    // pass interfaces
    this->_settings          = settings;
    this->_gameInfoDetails   = gameInfoDetails;
    this->_databaseHandle    = databaseHandle;
    this->_log               = log;

    // connect events and functions
    connect(&_tcpSocket, &QTcpSocket::connected,    this, &SyncClient::onConnect);
    connect(&_tcpSocket, &QTcpSocket::readyRead,    this, &SyncClient::onRead);
    connect(&_tcpSocket, &QTcpSocket::disconnected, this, &SyncClient::onDisconnect);
    connect(&_timeOut,   &QTimer::timeout,          this, &SyncClient::onTimeOut);

    // timeout
    _timeOut.setInterval( _timeOutTime_ms );
    _timeOut.start();

    // connect to remote masterserver
    _log->logEvent("tcp", QStringLiteral("connecting to %1:%2").arg(remoteAddress.toString(), QString::number(remotePort)) );
    _tcpSocket.connectToHost(remoteAddress, remotePort);


}

// accept both domain string and qhostaddress as input
SyncClient::SyncClient(const sSettings settings,
                       const QSharedPointer<GameInfo>        &gameInfoDetails,
                       const QSharedPointer<DatabaseHandle>  &databaseHandle,
                       const QSharedPointer<Logger>          &log,
                       const QString remoteAddress,
                       unsigned short remotePort)
{
    // pass interfaces
    this->_settings          = settings;
    this->_gameInfoDetails   = gameInfoDetails;
    this->_databaseHandle    = databaseHandle;
    this->_log               = log;

    // connect events and functions
    connect(&_tcpSocket, &QTcpSocket::connected,    this, &SyncClient::onConnect);
    connect(&_tcpSocket, &QTcpSocket::readyRead,    this, &SyncClient::onRead);
    connect(&_tcpSocket, &QTcpSocket::disconnected, this, &SyncClient::onDisconnect);
    connect(&_timeOut,   &QTimer::timeout,          this, &SyncClient::onTimeOut);

    // timeout
    _timeOut.setInterval( _timeOutTime_ms );
    _timeOut.start();

    // connect to remote masterserver
    _log->logEvent("tcp", QStringLiteral("connecting to %1:%2").arg(remoteAddress, QString::number(remotePort)) );
    _tcpSocket.connectToHost(remoteAddress, remotePort);
}

void SyncClient::onConnect()
{
    // reset timeout
    _timeOut.start();

    // convenience label to found
    _clientLabel = QStringLiteral("%1:%2").arg(_tcpSocket.peerAddress().toString(), QString::number(_tcpSocket.peerPort()));

    // log
    _log->logEvent("tcp", QStringLiteral("%1 connected").arg(_clientLabel) );
}

void SyncClient::onRead()
{
    // reset timeout after receiving (any) data
    _timeOut.start();

    // read from tcp connection and append to buffer
    QByteArray receiveBuffer = _tcpSocket.readAll();
    _rxBuffer.append( receiveBuffer );
    _log->logEvent("tcp", QStringLiteral("%1 sent %2 characters").arg(_clientLabel, QString::number(receiveBuffer.length()) ) );

    // remote masterserver opens with secure challenge
    if ( _rxBuffer.contains("secure") )
    {
        // parse to hash
        QMultiHash<QString, QString> receiveData = parseGameSpy0Buffer(receiveBuffer);

        // generate response
        QStringList response = replyQuery(receiveData);

        // return response
        _tcpSocket.write(response.join("").toLatin1());

        // sync request
        QString request = QStringLiteral("\\sync\\%1\\msid\\%2")
                .arg(_settings.Syncer.syncGames, _settings.masterserverIdentity);
        _tcpSocket.write(request.toLatin1());

        // all relevant information received. clear buffer for next interaction
        _rxBuffer = "";
        return;
    }

    if ( _rxBuffer.contains("final") )
    {
        // parse to hash: receivedData format is {gamename} => {string of addresses}
        QMultiHash<QString, QString> receiveData = parseGameSpy0Buffer(_rxBuffer);
        receiveData.remove("final"); // prevent "final" from registering as gamename

        // count number of addresses for logging
        int totalServerCount = 0;

        // parse to list of {gamename} = [list of ip+port]
        QHash<QString, QList<sServer>> gameServerList;
        QHashIterator<QString,QString> receiveDataIterator(receiveData);
        while ( receiveDataIterator.hasNext() )
        {
            // {gamename} => {string of addresses}
            receiveDataIterator.next();
            QString gamename      = receiveDataIterator.key();
            QString addressBuffer = receiveDataIterator.value();

            // split address list in ip:port
            QList<sServer> serverList;
            QStringList addressList = addressBuffer.split(" ", QString::SkipEmptyParts);
            QStringListIterator listIterator(addressList);
            while ( listIterator.hasNext() )
            {
                // split address into ip and port
                QStringList address = listIterator.next().split(":");
                sServer server;
                server.ip   = QHostAddress(address.value(0));
                server.port = address.value(1).toUShort();

                // append to list of servers
                serverList.append(server);
            }

            // append to list of gamename => list of servers
            int serverCount = serverList.count();
            if ( serverCount > 0 )
            {
                gameServerList.insert(gamename, serverList);
                totalServerCount += serverCount;
            }
        }

        // report in log
        _log->logEvent("sync", QStringLiteral("received %1 servers in %2 games").arg( QString::number(totalServerCount), QString::number(gameServerList.count()) ));
    }

    // else keep appending data until \\final\\ is received
}

void SyncClient::onDisconnect()
{
    _timeOut.stop();
    _log->logEvent("tcp", QStringLiteral("%1 disconnected").arg(_clientLabel) );
    this->deleteLater();
}

void SyncClient::onTimeOut()
{
    _log->logEvent("tcp", QStringLiteral("%1 timed out").arg(_clientLabel) );
    this->disconnect();
}
