#include "beaconserver.h"

void BeaconServer::onUdpRead()
{
    while ( _udpSocket.hasPendingDatagrams() )
    {
        // get sender and payload
        QNetworkDatagram    datagram        = _udpSocket.receiveDatagram();
        QHostAddress        senderAddress   = QHostAddress( datagram.senderAddress().toIPv6Address() );
        int                 senderPort      = datagram.senderPort();
        QString             receiveBuffer   = datagram.data();

        _log->logEvent("udp", QStringLiteral("%1:%2 sent '%3'")
                           .arg(senderAddress.toString(), QString::number(senderPort), receiveBuffer) );

        // determine protocol and response based on the first character (backslash, byte value, ... )
        unsigned short protocol_chooser = receiveBuffer.at(0).unicode();
        switch(protocol_chooser)
        {
        /*
         * GameSpy v0 protocol.
         * Used by Unreal/UT, Postal2, Rune, DeusEx among others
         */
        case 92:
            {
                // parse key/value pairs and QHash label
                QMultiHash<QString, QString> receiveData = parseGameSpy0Buffer(receiveBuffer);
                QString senderAddressLabel = QStringLiteral("%1:%2").arg(senderAddress.toString(), QString::number(senderPort));

                // received a heartbeat?
                if ( receiveData.contains("heartbeat") )
                {
                    // check if local addres (if enabled/disabled)
                    if (! _allowLocal and isLocal( senderAddress ) )
                    {
                            // log for admin's attention -- change config on either gameserver or masterserver
                            _log->logEvent("local", QStringLiteral("received heartbeat %2 from local address: %1")
                                           .arg(senderAddressLabel, receiveBuffer) );

                            // ignore disabled local beacon
                            continue;
                    }

                    // store heartbeat and request authentication
                    HeartBeat newBeacon;
                    newBeacon.secure    = genChallengeString(6, false);
                    newBeacon.address   = senderAddress;
                    newBeacon.port      = receiveData.value("heartbeat", "0").toUShort();
                    newBeacon.gamename  = receiveData.value("gamename", "unknown");
                    newBeacon.protocol  = "gamespy0";

                    //
                    // TODO: remove address and port from HeartBeat struct. these are directly
                    // retrieved from the datagram. ip/port are duplicate.
                    //


                    // sanity check: known game(name)
                    if ( _gameInfoDetails->supportedGames.contains( newBeacon.gamename ) )
                    {
                        // valid port and/or default port?
                        if (newBeacon.port == 0)
                        {
                            // override with default port if possible
                            if ( _gameInfoDetails->supportedGames.value(newBeacon.gamename).port > 0 )
                            {
                                newBeacon.port = _gameInfoDetails->supportedGames.value( newBeacon.gamename ).port;
                            }
                            else // no valid port available. log and abort.
                            {
                                _log->logEvent("heartbeat", QStringLiteral("%1 invalid port for %2")
                                               .arg(senderAddress.toString(), newBeacon.gamename) );
                                continue;
                            }
                        }
                    }
                    else // unknown game. log and abort.
                    {
                        _log->logEvent("unsupported", QStringLiteral("%1:%2 for unsupported game %3")
                                       .arg(senderAddress.toString(), QString::number(newBeacon.port), newBeacon.gamename) );
                        continue;
                    }

                    // if this server already exists, update it
                    if ( _databaseHandle->updateServer(newBeacon.address, newBeacon.port, newBeacon.gamename, true) > 0)
                    {
                        // log update
                        _log->logEvent("heartbeat", QStringLiteral("%1:%2 for %3")
                                .arg(newBeacon.address.toString(), QString::number(newBeacon.port), newBeacon.gamename) );
                        continue;
                    }
                    else
                    {
                        // TODO: remove me when it works decently and doesn't need logging anymore.
                        _log->logEvent("heartbeat", QStringLiteral("new beacon %1:%2 for %3, now challenging")
                                .arg(newBeacon.address.toString(), QString::number(newBeacon.port), newBeacon.gamename) );
                    }

                    // store heartbeat in temporary list
                    _beaconList.remove(senderAddressLabel);
                    _beaconList.insert(senderAddressLabel, newBeacon);

                    // request authentication from remote server
                    _udpSocket.writeDatagram( datagram.makeReply( QStringLiteral("\\secure\\%1").arg(newBeacon.secure).toUtf8() ) );

                    // nothing left to do for this beacon.
                    continue;
                }

                // received response to authentication request
                if ( receiveData.contains("validate") )
                {
                    // load existing information
                    HeartBeat valBeacon = _beaconList.value(senderAddressLabel);

                    // empty heartbeat? then received validate is timed out or a spoof
                    if (valBeacon.address == QHostAddress::Null || valBeacon.gamename.length() <= 0)
                    {
                        _log->logEvent("validate", QStringLiteral("unexpected validate from %1").arg(senderAddress.toString()));
                        continue;
                    }

                    // get response
                    AuthResult authResult = validateGamename(CLIENT,
                                                             valBeacon.gamename,
                                                             receiveData.value("validate",""),
                                                             _gameInfoDetails->supportedGames.value(valBeacon.gamename).cipher,
                                                             valBeacon.secure,
                                                             receiveData.value("enctype", "0").toInt() );

                    // compare with received response
                    if ( authResult.auth )
                    {
                        // server authenticated - log and add to database
                        _log->logEvent("validate", QStringLiteral("successful validate from %1:%2 for %3")
                                       .arg(valBeacon.address.toString(), QString::number(valBeacon.port),valBeacon.gamename) );

                        // prevent duplicate entries (server possibly was added between secure and validate due to duplicate uplink entry in server config)
                        if ( ! _databaseHandle->updateServer(valBeacon.address, valBeacon.port, valBeacon.gamename, true) )
                        {
                            // add to database
                            _databaseHandle->insertServer(valBeacon.address, valBeacon.port, valBeacon.gamename, true);

                            // log type "uplink" for first heartbeat. from now on, this server is logged with type "heartbeat"
                            _log->logEvent("uplink", QStringLiteral("%1:%2 for %3")
                                           .arg(valBeacon.address.toString(), QString::number(valBeacon.port),valBeacon.gamename) );

                        } // else: updated existing entry, ignore

                        // remove from temporary/pending list
                        _beaconList.remove(senderAddressLabel);
                    }
                    else // log failed validate
                    {
                        _log->logEvent("validate", QStringLiteral("failed validate from %1:%2 for %3")
                                       .arg(valBeacon.address.toString(), QString::number(valBeacon.port),valBeacon.gamename) );
                        _log->logEvent("secure", QStringLiteral("secure: '%1', gamename: '%2', validate: '%3', expected: '%4'")
                                                    .arg(valBeacon.secure, valBeacon.gamename, receiveData.value("validate", "null"), authResult.validate ));
                    }

                    // end validate -- no further tasks
                    continue;
                }

                // status queries directed at masterserver
                if (receiveData.contains("secure") or
                    receiveData.contains("basic") or
                    receiveData.contains("info") or
                    receiveData.contains("rules") or
                    receiveData.contains("status") or
                    receiveData.contains("echo") )
                {
                    // parse response query
                    QStringList response = replyQuery(receiveData);

                    // return response
                    _udpSocket.writeDatagram( datagram.makeReply( response.join("").toLatin1() ) );

                    // log incoming query
                    _log->logEvent("query", QStringLiteral("%1 queried us with %2")
                                   .arg(senderAddress.toString(), receiveData.keys().join(", ") ) );

                    // log echo separately
                    if ( receiveData.contains("echo") )
                        _log->logEvent("echo", QStringLiteral("%1: '%2'")
                                       .arg(senderAddress.toString(), receiveData.value("echo") ) );

                    continue;
                }

                // ignore trailing queryid+final
                if (receiveData.contains("queryid") and receiveData.contains("final"))
                {
                    // receive queryid and final from the query
                    receiveData.remove("queryid");
                    receiveData.remove("final");

                    // nothing remains? then ignore. otherwise, proceed to "unknown query"
                    if ( receiveData.size() <= 0)
                        continue;
                }

                // received another type of query?
                _log->logEvent("unknown", QStringLiteral("received unknown udp uplink %1 from %2")
                               .arg(receiveBuffer, senderAddress.toString()) );

                // no other tasks
                break;
            }

        default:
        {
            // received another type of query?
            _log->logEvent("unknown", QStringLiteral("received unknown message %1 from %2")
                           .arg(receiveBuffer, senderAddress.toString()) );
            break;
        }
        }
    }
}
