
package MasterServer::MasterQuery;

use strict;
use warnings;
use Data::Dumper 'Dumper';
use AnyEvent;
use AnyEvent::Handle;
use Exporter 'import';

our @EXPORT = qw| MasterQueryScheduler QueryMaster QueryGameName |;

################################################################################
##
##   MasterServer Query Scheduler
##
##   Commands the "Query MasterServer" function to query a MasterServer every
##   /interval/ seconds. Stores the received addresses in the "ip_list" table 
##   of the database, so they may be processed further.
##   Adds new addresses, updates the time on excisting ones.
##   Args: @masterserver list must be set above.
################################################################################
sub MasterQueryScheduler {
  my $self = shift;
  
  my $i = 0;
  return AnyEvent->timer (
    after     => $self->{MasterQueryTime}[0],
    interval  => $self->{MasterQueryTime}[1],
    cb        => sub {
      # check if there's a master server entry to be queried. If not, return
      # to zero and go all over again.
      $i = 0 unless $self->{master_server}[$i];
      
      my $n = 1; # n * 30 = delay between different game queries
      $self->{heap} = ();
      for (@{$self->{master_games}}) {
        $self->{heap}->{$_} = $self->QueryGameName($i, $_, $self->{MasterQueryTime}[2]*$n++);
      }

      #increment counter
      $i++;
    }
  );
}

sub QueryGameName {
  my ($self, $i, $gamename, $delay) = @_;

  return AnyEvent->timer (
    after     => $delay,
    cb        => sub {
      return unless $self->{master_server}[$i];
      
      # Query the master for $gamename games      
      $self->log("[OK] > Querying $self->{master_server}[$i]{label} for $gamename servers.");
      $self->QueryMaster($self->{master_server}[$i], $gamename);

      #increment counter
      $i++;
    }
  );
}

################################################################################
##
##   Query MasterServer function
##
##   Communicate with other MasterServers to obtain more server addresses.
##   Args: hash with {label, ip, port, gamename}
##   Function is called by AnyEvent->timer() further below
################################################################################
sub QueryMaster {
  my ($self, $ms, $gamename) = @_;
  
  #DEBUG
  #print "Querying for more servers.\n";

  # list to store all IPs in.
  my $master_list = "";
  
  # connection handle
  my $handle; 
  $handle = new AnyEvent::Handle(
    connect  => [$ms->{ip} => $ms->{port}],
    timeout  => 5,
    poll     => 'r',
    on_error => sub {$self->log("[E] $! on $ms->{label}."); $handle->destroy;},
    on_eof   => sub {
      # address list
      my @list;

      # remove last \final      
      $master_list =~ s/\\final//g;

      # process received data
      # l(ocations, \label\ip:port\) split up in a(ddress) and p(ort)
      foreach my $l (split(/\\/, $master_list)){
        if ($l =~ /:/){
          my ($a,$q) = $l =~ /(.*):(.*)/;
          push(@list, ($a.':'.$q));
        }
      }
      
      # print findings
      $self->log("[TCP] > Found ". scalar @list ." servers at $ms->{label} for $gamename.");
      
      # Debug only list if one or more servers were found
      #if (scalar @list > 0) {$self->log("[TCP] > \"$gamename\"\t\t - ". scalar @list ." servers at $ms->{label}.");}
      

      # SQLite is slow, therefore use transactions.
      $self->{dbh}->begin_work;      
      #my $t = time;      
     
      # add newly found entries to db
      foreach my $l (@list) {
      
        # break ip:port into $a (ip) and $q (query port)
        if ($l =~ /:/) {
          my ($a,$q) = $self->valid_address($l);
          
          # if valid IP and valid port
          if ($a && $q){
          
            # add server
            $self->addServer($a, $q, $gamename);
            
          }
          else{
            # log as error, leaves a trace
            $self->log("[E] Invalid IP found at $ms->{label}: $l!"); 
          }
        }
      }
      
      # end transaction, commit        
      #$self->log("[DB] > Masterlist for $gamename updated in ".(time-$t). " seconds.");
      $self->{dbh}->commit;
      
      # destroy socket and let's all go home.
      $handle->destroy;
    },
  );

  # and finally handle any remaining data as body
  $handle->on_read(sub {
     
    # receive and clear buffer
    my $m = $_[0]->rbuf;
    $_[0]->rbuf = "";

    # remove string terminator: sometimes trailing slashes are missing from
    # sender, so secure is actually secure\0. Causes invalid responses, 
    # is not pretty, that kind of stuff.
    chop $m if $m =~ m/secure/;
        
    # part 1: receive \basic\\secure\$key
    if ($m =~ m/basic\\\\secure/) {

      # hash $m into %r
      my %r;
      $m =~ s/\\([^\\]+)\\([^\\]+)/$r{$1}=$2/eg;
      
      # respond to the validate challenge
      my $validate = $self->validate($gamename, $r{secure});

      # part 2: send \gamename\ut\location\0\validate\$validate\final\
      $handle->push_write("\\gamename\\$gamename\\location\\0\\validate\\$validate\\final\\");
      
      # part 3: also request the list \list\gamename\ut\
      $handle->push_write("\\list\\\\gamename\\$gamename\\");
    } # end secure
    
    # part 3b: receive the entire list in multiple steps.
    if ($m =~ m/\\ip\\/) {
      # add buffer to the list
      $master_list .= $m;
    } # end ip

  });
}

1;
