
package MasterServer::UBrowserServer;

use strict;
use warnings;
use AnyEvent::Socket;
use AnyEvent::Handle;
use Exporter 'import';

our @EXPORT = qw| UBrowserServer|;

# ubrowser handle manager
my %ubrowser_clients;

# Convenience function to clean out the $ubrowser tcp_server handles after EOF
sub clean_handle {
  my ($c) = @_;

  # clean and close the connection
  delete ($ubrowser_clients{$c});  
  $c->destroy();
}

################################################################################
##
##   Master Server UBrowser TCP Listener
##
##   Waits for incoming TCP connections from UT clients and other MasterServers.
##   Returns (after validation) a list of IP:port addresses of UT servers.
##   Args: $listen_port must be set above.
##
##   Main loop starts/continues here.
################################################################################
sub UBrowserServer {
  my $self = shift;

  my $ubrowser = tcp_server undef, $self->{listen_port}, sub {
    my ($fh, $a, $p) = @_;
    
    # validated? yes = 1 no = 0
    my $val = 0; 
    
    #  received variables, processed
    my %r;
    
    #DEBUG
    #print "New connection from $a $p\n";
    
    # prep a challenge
    my $secure = $self->secure();
    
    # handle received data
    my $h; $h = AnyEvent::Handle->new(
      fh        => $fh,
      poll      => 'r',
      on_eof    => \&clean_handle,
      on_error  => \&clean_handle,
      timeout  => 5,
      on_read   => sub {
        my ($c) = @_;
        my $m = $c->rbuf;
        
        # clear the buffer
        $c->rbuf = "";

        # utgslive uses this type of query. Is not supported yet. Report error.      
        if ($m =~ /final\\\\list/) {
          
          # log this disgression
          $self->log("[E] > $a:$p - Not Supported. Are you using UTGS?\n");
          
          # report error to client if supported
          # TODO: support UTGS(live) and other clients
          $c->push_write("\\echo\\333networks does not support this function. Are you using UTGS?\\");
        }

        if ($m =~ m/validate/) {
          # part 2: receive \gamename\ut\location\0\validate\$validate\final\
          $m =~ s/\\([^\\]+)\\([^\\]+)/$r{$1}=$2/eg;
          
          # pass or fail the secure challenge        
          if (exists $r{gamename}) {
            $val = $self->hasValidKey($r{gamename}, $secure, $r{validate});
          }

          # DEBUG print
          $self->log("[SEC] > $a:$p validated with $val for $r{gamename}, $secure, $r{validate}");
        } # end if validate
        
        # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #
        # At this point, it either passes the test and sends the \list\ command,
        # or it fails and disconnects.
        # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # # #

        if ($m =~ m/list/) {
          # part 3: wait for the requested action: \list\gamename\ut\

          # extract and splice \list\ command from buf
          my $i = index($m,'\\',1);
          my $j = substr($m, 1, $i-1);
          $m = substr($m, $i);
          $m =~ s/\\([^\\]+)\\([^\\]+)/$r{$1}=$2/eg;
          
          # confirm validation and game supported (ut)
          # TODO now only support for ut99, expand to more gamespy v0 games
          if ($val) {
          
            # Query database for servers that have been seen within the last 3 hours
            my $list = $self->{dbh}->selectall_arrayref(
                         "SELECT ip, port 
                          FROM serverlist
                          WHERE updated > datetime(CURRENT_TIMESTAMP, '-10600 seconds')
                          AND gamename = ?", undef, lc $r{gamename});
            
            # Format the DB list into response query string
            my $response_list = "\\basic\\";
            
            foreach $_ (@{$list}) {
              $response_list .= "\\ip\\$_->[0]:$_->[1]";
            }
            
            # end response query
            $response_list .= "\\final\\";
            
            # send to UT client
            $c->push_write($response_list);
            
            # log successful (debug)
            $self->log("[TCP] > $a:$p successfully retrieved the list for $r{gamename}.");
            
            # clean and close the connection
            delete ($ubrowser_clients{$c});  
            $c->destroy();
          }
          else {
            # proper syntax/protocol, but incorrect game type. Therefore respond with
            # an 'empty' list, returning only \final\.
            $c->push_write("\\echo\\333networks failed to validate your request.\\final\\");
            
            # log it too
            $self->log("[E] > UBrowser $a:$p failed validation for $r{gamename}");

            # clean and close the connection
            delete ($ubrowser_clients{$c});  
            $c->destroy();          
          }
        } # end if list
        
        # improper syntax/protocol
        if ($m =~ m/!(validate|list)/) {
          
          # error message to client
          $c->push_write("\\echo\\333networks did not understand your request. ".
                         "Contact us via 333networks.com\\final\\");

          # and log it
          $self->log("[E] > Invalid request from UBrowser $a:$p with unknown message \"$m\"");

          # clean and close the connection
          delete ($ubrowser_clients{$c});  
          $c->destroy();                
          
        } # end if weird query
      },
    );
    
    # part 1: send \basic\\secure\$key\
    $h->push_write("\\basic\\\\secure\\$secure\\");
    
    # keep handle alive longer.
    $ubrowser_clients{$h} = $h;
    return;
  };
  
  # startup of TCP server complete 
  $self->log("[OK] > Listening for UBrowser connections on port $self->{listen_port}.");
  return $ubrowser;
}

1;
