class OPIO extends Object DLLBind(OPIOFile);

/** IMPORTED FUNCTIONS, DO NOT USE DIRECTLY */
dllimport private final function string _GetGameDirectory();
dllimport private final function string _GetAppDataDirectory();
dllimport private final function _CurrentDirectorySet(string s);
dllimport private final function string _CurrentDirectoryGet();
dllimport private final function string _InitFileList(out int err, out int folder);
dllimport private final function string _FileListGetNext(out int fin, out int folder);
dllimport private final function bool _CreateDirectory();
dllimport private final function bool _RemoveDirectory();
dllimport private final function bool _WriteStringToFile(string file, string str, int append);
dllimport private final function string _ReadStringLineFromFile(out int fhandle, string file, out int liner);

/**
 * Initialize IOFile class.
 *
 * @param dir	    Current directory. If not provided, %appdata%\UDK is used.
 */
final function Initialize(optional string dir)
{
	if (dir != "")
	{
		_CurrentDirectorySet(dir);
	}
	else
	{
		_CurrentDirectorySet(_GetAppDataDirectory() $ "\\UDK");
		CreateDirectory();
	}
}

/**
 * Set current directory.
 *
 * @param dir	    Current directory.
 */
final function SetDirectory(string dir)
{
	_CurrentDirectorySet(dir);
}

/**
 * Get current directory.
 *
 * @return Current directory.
 */
final function string GetDirectory()
{
	return _CurrentDirectoryGet();
}

/**
 * Obtain UDK.exe working directory.
 *
 * @return UDK working directory.
 */
final function string GetGameDirectory()
{
	return _GetGameDirectory();
}

/**
 * Create current directory.
 *
 * @return Returns true if directory was created or already exists, false if failed.
 */
final function bool CreateDirectory()
{
	return _CreateDirectory();
}

/**
 * Remove current directory. Current directory is set to be parent of removed directory.
 *
 * @return Returns true if directory was removed, false if failed. If failed, current directory is not changed.
 */
final function bool RemoveDirectory()
{
	return _RemoveDirectory();
}

/**
 * Append new folder to current directory and create one if it doesn't exist. Current directory is changed.
 *
 * @param folder       Folder name.
 * @return Returns true if directory was successfuly appended, false otherwise. If failed, current directory is not changed.
 */
final function bool AppendDirectory(string folder)
{
	local string cur, old;

	cur = _CurrentDirectoryGet();
	old = cur;
	cur = cur $ "\\" $ folder;
	_CurrentDirectorySet(cur);
	if (_CreateDirectory()) return true;
	else
	{
		_CurrentDirectorySet(old);
		return false;
	}
}

/**
 * Set current directory to parent.
 *
 * @return Returns false if already at root.
 */
final function bool ParentDirectory()
{
	local string cur;
	local int i;

	cur = _CurrentDirectoryGet();
	i = InStr(cur, "\\", true);
	if (i == -1) return false;
	cur = Left(cur, i);

	_CurrentDirectorySet(cur);
}

/**
 * Get all files in current directory.
 *
 * @return Array of files in current directory.
 */
final function array<string> GetAllFiles()
{
	return GetAll(0);
}

/**
 * Get all folders in current directory.
 *
 * @return Array of folders in current directory.
 */
final function array<string> GetAllFolders()
{
	return GetAll(1);
}

/**
 * Write text to file. If file does not exists, new is created. If file exists, it's content is erased first.
 *
 * @param filen     Name of the file to write to.
 * @param text      Text to write to file.
 * @return Returns true if write was successful, otherwise false.
 */
final function bool WriteText(string filen, string text)
{
	return _WriteStringToFile(filen, text, 0);
}

/**
 * Append text to file. If file does not exists, new is created. If file exists, it's content is not erased, provided text is appended to the end of the file.
 *
 * @param filen     Name of the file to write to.
 * @param text      Text to write to file.
 * @return Returns true if write was successful, otherwise false.
 */
final function bool AppendText(string filen, string text)
{
	return _WriteStringToFile(filen, text, 1);
}

/**
 * Append endline characters to the end of the file.
 *
 * @param filen     Name of the file to write to.
 * @return Returns true if write was successful, otherwise false.
 */
final function bool AppendEndLine(string filen)
{
	return _WriteStringToFile(filen, "\n", 1);
}

/**
 * Append text lines to the file.
 *
 * @param filen     Name of the file to write to.
 * @param lines     Array of text lines.
 * @return Returns true if write was successful, otherwise false.
 */
final function bool AppendLines(string filen, array<string> lines)
{
	return AWLines(filen, lines, 1);
}

/**
 * Write text lines to the file.
 *
 * @param filen     Name of the file to write to.
 * @param lines     Array of text lines.
 * @return Returns true if write was successful, otherwise false.
 */
final function bool WriteLines(string filen, array<string> lines)
{
	return AWLines(filen, lines, 0);
}

/**
 * Read all text from file.
 *
 * @param filen     Name of the file to read from.
 * @param text      Out string parameter where to save read text to.
 * @return Returns true if read was successful, otherwise false.
 */
final function bool ReadText(string filen, out string text)
{
	local int h, lr;
	local string strline;

	h = 0;
	text = _ReadStringLineFromFile(h, filen, lr);
	if (lr != 1) return false;

	while (h != 0)
	{
		strline = _ReadStringLineFromFile(h, "", lr);
		if (lr == 1) text = text $ "\n" $ strline;
	}

	return true;
}

/**
 * Read all text lines from file.
 *
 * @param filen     Name of the file to read from.
 * @param lines      Out array of strings where read lines are saved to.
 * @return Returns true if read was successful, otherwise false.
 */
final function bool ReadLines(string filen, out array<string> lines)
{
	local int h, lr;
	local string strline;

	lines.Length = 0;

	h = 0;
	strline = _ReadStringLineFromFile(h, filen, lr);
	if (lr != 1) return false;

	lines.AddItem(strline);

	while (h != 0)
	{
		strline = _ReadStringLineFromFile(h, "", lr);
		if (lr == 1) lines.AddItem(strline);
	}

	return true;
}

/**
 * Open or create file, which can be then used to write to or read from.
 *
 * @param filen     Name of the file.
 * @return Returns OPIOFile object, which can be used to write and read from file.
 */
final function OPIOFile OpenFile(string filen)
{
	local OPIOFile f;

	f = new class'OPIOFile';
	f.IOInstance = self;
	f.FileName = filen;

	return f;
}


/** PRIVATE FUNCTIONS */

private final function bool AWLines(string filen, array<string> lines, int append)
{
	local string str;
	local int i;

	for (i = 0; i < lines.Length; i++)
		str = str $ lines[i] $ "\n";

	return _WriteStringToFile(filen, str, append);
}

private final function array<string> GetAll(int f)
{
	local string s;
	local array<string> files;
	local int err, folder;

	s = _InitFileList(err, folder);
	if (err == 0)
	{
		do
		{
			if (s != "." && s != ".." && folder == folder)
				files.AddItem(s);
			s = _FileListGetNext(err, folder);
		} until (err != 0);
	}

	return files;
}



DefaultProperties
{
}
