//=============================================================================
// HXAugmentationManager
//=============================================================================
class HXAugmentationManager extends AugmentationManager;

/*
struct S_AugInfo
{
	var int NumSlots;
	var int AugCount;
	var int KeyBase;
};

var travel S_AugInfo AugLocs[7];
var DeusExPlayer Player;				// which player am I attached to?
var travel Augmentation FirstAug;		// Pointer to first Augmentation

// All the available augmentations 
var Class<Augmentation> augClasses[25];
var Class<Augmentation> defaultAugs[3];

var localized string AugLocationFull;
var localized String NoAugInSlot;
*/

var localized string MsgAugAlreadyInstalled;
var localized String MsgInvalidAugSlot;

// ----------------------------------------------------------------------------
// Network Replication.
// ----------------------------------------------------------------------------

replication
{
/*
	//variables server to client
	reliable if ((Role == ROLE_Authority) && (bNetOwner))
		AugLocs, FirstAug, Player;

	//functions client to server
	reliable if (Role < ROLE_Authority)
		ActivateAugByKey, AddAllAugs, SetAllAugsToMaxLevel, ActivateAll, DeactivateAll, GivePlayerAugmentation;
*/

	// Client to server.
	reliable if ( Role<ROLE_Authority )
		MPGivePlayerAugmentation, ForceDeactivateAll; // Maybe ForceDeactivateAll isn't needed to be RPC'ed.
}

// ----------------------------------------------------------------------------
// CreateAugmentations()
// ----------------------------------------------------------------------------

function CreateAugmentations(DeusExPlayer newPlayer)
{
	local int augIndex;
	local Augmentation anAug;
	local Augmentation lastAug;

	FirstAug = None;
	LastAug  = None;

	player = newPlayer;

	for(augIndex=0; augIndex<arrayCount(augClasses); augIndex++)
	{
		if (augClasses[augIndex] != None)
		{
			anAug = Spawn(augClasses[augIndex], Self);
			anAug.Player = player;

			// Manage our linked list
			if (anAug != None)
			{
				if (FirstAug == None)
				{
					FirstAug = anAug;
				}
				else
				{
					LastAug.next = anAug;
				}

				LastAug  = anAug;
			}
		}
	}
}

// ----------------------------------------------------------------------------
// AddDefaultAugmentations()
// ----------------------------------------------------------------------------

function AddDefaultAugmentations()
{
	local int augIndex;

	for(augIndex=0; augIndex<arrayCount(defaultAugs); augIndex++)
	{
		if (defaultAugs[augIndex] != None)
			GivePlayerAugmentation(defaultAugs[augIndex]);
	}
}

// ----------------------------------------------------------------------------
// RefreshAugDisplay()
//
// Refreshes the Augmentation display with all the augs that are 
// currently active.
// ----------------------------------------------------------------------------

simulated function RefreshAugDisplay()
{
	local Augmentation anAug;

	if (player == None)
		return;

	// First make sure there are no augs visible in the display
	player.ClearAugmentationDisplay();

	anAug = FirstAug;
	while(anAug != None)
	{
		// First make sure the aug is active if need be
		if (anAug.bHasIt)
		{
			if (anAug.bIsActive)
			{
				anAug.GotoState('Active');

				// Now, if this is an aug that isn't *always* active, then 
				// make sure it's in the augmentation display

				if (!anAug.bAlwaysActive)
					player.AddAugmentationDisplay(anAug);
			}
			else if ((player.bHUDShowAllAugs) && (!anAug.bAlwaysActive))
			{
				player.AddAugmentationDisplay(anAug);		
			}
		}

		anAug = anAug.next;
	}
}

// ----------------------------------------------------------------------------
// NumAugsActive()
//
// How many augs are currently active?
// ----------------------------------------------------------------------------

simulated function int NumAugsActive()
{
	local Augmentation anAug;
	local int count;

	if (player == None)
		return 0;

	count = 0;
	anAug = FirstAug;
	while(anAug != None)
	{
		if (anAug.bHasIt && anAug.bIsActive && !anAug.bAlwaysActive)
			count++;

		anAug = anAug.next;
	}

	return count;
}

// ----------------------------------------------------------------------------
// SetPlayer()
// ---------------------------------------------------------------------

function SetPlayer(DeusExPlayer newPlayer)
{
	local Augmentation anAug;

	player = newPlayer;

	anAug = FirstAug;
	while(anAug != None)
	{
		anAug.player = player;
		anAug = anAug.next;
	}
}

// ----------------------------------------------------------------------------
// BoostAugs()
// ----------------------------------------------------------------------------

function BoostAugs(bool bBoostEnabled, Augmentation augBoosting)
{
	local Augmentation anAug;

	anAug = FirstAug;
	while(anAug != None)
	{
		// Don't boost the augmentation causing the boosting!
		if (anAug != augBoosting)
		{
			if (bBoostEnabled)
			{
				if (anAug.bIsActive && !anAug.bBoosted && (anAug.CurrentLevel < anAug.MaxLevel))
				{
					anAug.Deactivate();
					anAug.CurrentLevel++;
					anAug.bBoosted = True;
					anAug.Activate();
				}
			}
			else if (anAug.bBoosted)
			{
				anAug.CurrentLevel--;
				anAug.bBoosted = False;
			}
		}
		anAug = anAug.next;
	}
}

// ----------------------------------------------------------------------------
// GetClassLevel()
// this returns the level, but only if the augmentation is
// currently turned on
// ----------------------------------------------------------------------------

simulated function int GetClassLevel(class<Augmentation> augClass)
{
	local Augmentation anAug;

	anAug = FirstAug;
	while(anAug != None)
	{
		if (anAug.Class == augClass)
		{
			if (anAug.bHasIt && anAug.bIsActive)
				return anAug.CurrentLevel;
			else
				return -1;
		}

		anAug = anAug.next;
	}

	return -1;
}

// ----------------------------------------------------------------------------
// GetAugLevelValue()
//
// takes a class instead of being called by actual augmentation
// ----------------------------------------------------------------------------

simulated function float GetAugLevelValue(class<Augmentation> AugClass)
{
	local Augmentation anAug;
	local float retval;

	retval = 0;

	anAug = FirstAug;
	while(anAug != None)
	{
		if (anAug.Class == augClass)
		{
			if (anAug.bHasIt && anAug.bIsActive)
				return anAug.LevelValues[anAug.CurrentLevel];
			else
				return -1.0;
		}

		anAug = anAug.next;
	}

	return -1.0;
}

// ----------------------------------------------------------------------------
// ActivateAll()
//
// Loops through all the Augmentations, deactivating any that are active.
// ----------------------------------------------------------------------------

function ActivateAll()
{
	local Augmentation anAug;

	// Only allow this if the player still has 
	// Bioleectric Energy(tm)

	if ((player != None) && (player.Energy > 0))
	{
		anAug = FirstAug;
		while(anAug != None)
		{
			if (!anAug.bIsActive && !anAug.bAlwaysActive && HXAugmentation(anAug).bActivateWithAll )
				anAug.Activate();
			anAug = anAug.next;
		}
	}
}

// ----------------------------------------------------------------------------
// DeactivateAll()
//
// Loops through all the Augmentations, deactivating any that are active.
// Intended to be used for a player keybinding or UI interaction.
//
// THIS IS NOT INTENDED TO BE USED BY CODE TO FORCE ALL AUGMENTATIONS OFF.
// ----------------------------------------------------------------------------

function DeactivateAll()
{
	local Augmentation anAug;

	anAug = FirstAug;
	while(anAug != None)
	{
		if (anAug.bIsActive && !anAug.bAlwaysActive && HXAugmentation(anAug).bDeactivateWithAll )
			anAug.Deactivate();
		anAug = anAug.next;
	}
}

// ----------------------------------------------------------------------------
// ForceDeactivateAll()
//
// Loops through all the Augmentations, deactivating any that are active.
// Intended to be used for a player keybinding or UI interaction.
//
// This is intended to be used by code which forces augs to be off.
// ----------------------------------------------------------------------------

function ForceDeactivateAll()
{
	local Augmentation anAug;

	anAug = FirstAug;
	while(anAug != None)
	{
		if (anAug.bIsActive && !anAug.bAlwaysActive)
			anAug.Deactivate();
		anAug = anAug.next;
	}
}

// ----------------------------------------------------------------------------
// FindAugmentation()
//
// Returns the augmentation based on the class name
// ----------------------------------------------------------------------------

simulated function Augmentation FindAugmentation(Class<Augmentation> findClass)
{
	local Augmentation anAug;

	anAug = FirstAug;
	while(anAug != None)
	{
		if (anAug.Class == findClass)
			break;

		anAug = anAug.next;
	}

	return anAug;
}

// ----------------------------------------------------------------------------
// MPGivePlayerAugmentation()
// ----------------------------------------------------------------------------

function MPGivePlayerAugmentation( class<Augmentation> NewAugClass, HXAugmentationCannister AugCan, HXMedicalBot MedBot )
{
	local Augmentation NewAug;

	if ( AugCan==None || AugCan.bDeleteMe )
		return;

	NewAug = FindAugmentation( NewAugClass );

	// shouldn't happen, but you never know!
	if ( NewAug==None )
		return; 

	if ( NewAug.bHasIt )
	{
		//NewAug.IncLevel();
		Player.ClientMessage( MsgAugAlreadyInstalled );
	}
	else if ( AreSlotsFull( NewAug ) )
	{
		Player.ClientMessage( AugLocationFull );
	}
	else
	{
		NewAug.bHasIt = true;

		if ( NewAug.bAlwaysActive )
		{
			NewAug.bIsActive = true;
			NewAug.GotoState('Active');
		}
		else
		{
			NewAug.bIsActive = false;
		}

		if ( Player.Level.Netmode==NM_Standalone )
			Player.ClientMessage( Sprintf(NewAug.AugNowHaveAtLevel,NewAug.AugmentationName,NewAug.CurrentLevel+1) );

		// Manage our AugLocs[] array
		AugLocs[NewAug.AugmentationLocation].AugCount++;
		
		// Assign hot key to new aug (must be after before AugCount is incremented!).
		NewAug.HotKeyNum = AugLocs[NewAug.AugmentationLocation].AugCount + AugLocs[NewAug.AugmentationLocation].KeyBase;

		if ( !NewAug.bAlwaysActive && Player.bHUDShowAllAugs )
			Player.AddAugmentationDisplay( NewAug );

		// Play a cool animation.
		MedBot.PlayAnim( 'Scan' );

		// Now Destroy the Augmentation cannister.
		Player.DeleteInventory( AugCan );
		AugCan.Destroy();
	}
}

// ----------------------------------------------------------------------------
// GivePlayerAugmentation()
// ----------------------------------------------------------------------------

function Augmentation GivePlayerAugmentation( class<Augmentation> NewAugClass )
{
	local Augmentation NewAug;

	NewAug = FindAugmentation( NewAugClass );

	// shouldn't happen, but you never know!
	if ( NewAug==None )
		return None;

	if ( NewAug.bHasIt )
	{
		//NewAug.IncLevel();
		Player.ClientMessage( MsgAugAlreadyInstalled, 'StatusLine' );
	}
	else if ( AreSlotsFull( NewAug ) )
	{
		Player.ClientMessage( AugLocationFull, 'StatusLine' );
	}
	else
	{
		NewAug.bHasIt = true;

		if ( NewAug.bAlwaysActive )
		{
			NewAug.bIsActive = true;
			NewAug.GotoState('Active');
		}
		else
		{
			NewAug.bIsActive = false;
		}

		if ( Player.Level.Netmode==NM_Standalone )
			Player.ClientMessage( Sprintf(NewAug.AugNowHaveAtLevel,NewAug.AugmentationName,NewAug.CurrentLevel+1), 'StatusLine' );

		// Manage our AugLocs[] array
		AugLocs[NewAug.AugmentationLocation].AugCount++;
		
		// Assign hot key to new aug (must be after before AugCount is incremented!).
		NewAug.HotKeyNum = AugLocs[NewAug.AugmentationLocation].AugCount + AugLocs[NewAug.AugmentationLocation].KeyBase;

		if ( !NewAug.bAlwaysActive && Player.bHUDShowAllAugs )
			Player.AddAugmentationDisplay( NewAug );
	}

	return NewAug;
}

// ----------------------------------------------------------------------------
// AreSlotsFull()
//
// For the given Augmentation passed in, checks to see if the slots
// for this aug are already filled up.  This is used to prevent to 
// prevent the player from adding more augmentations than the slots
// can accomodate.
// ----------------------------------------------------------------------------

simulated function Bool AreSlotsFull(Augmentation augToCheck)
{
	local int num;
   //local bool bHasMPConflict;
	local Augmentation anAug;

	// You can only have a limited number augmentations in each location, 
	// so here we check to see if you already have the maximum allowed.

	num = 0;
   //bHasMPConflict = false;
	anAug = FirstAug;
	while(anAug != None)
	{
		if (anAug.AugmentationName != "")
			if (augToCheck != anAug)
            //if (Level.Netmode == NM_Standalone)
            //{
               if (augToCheck.AugmentationLocation == anAug.AugmentationLocation)
                  if (anAug.bHasIt)
                     num++;
            //}
            //else
            //{
               //if ((AnAug.MPConflictSlot == AugToCheck.MPConflictSlot) && (AugToCheck.MPConflictSlot != 0) && (AnAug.bHasIt))
               //{
                  //bHasMPConflict = true;
               //}
            //}		
		anAug = anAug.next;
	}
	//if (Level.NetMode == NM_Standalone)
      return (num >= AugLocs[augToCheck.AugmentationLocation].NumSlots);
   //else
      //return bHasMPConflict;
}

// ----------------------------------------------------------------------------
// CalcEnergyUse()
//
// Calculates energy use for all active augmentations
// ----------------------------------------------------------------------------

// Sth. like ConsumeEnergy( float DeltaTime ) which directly sucks player dry and deactives augs may be nice.
simulated function Float CalcEnergyUse(float deltaTime)
{
	local float energyUse, energyMult;
	local Augmentation anAug;
   local Augmentation PowerAug;

	energyUse = 0;
	energyMult = 1.0;

	anAug = FirstAug;
	while(anAug != None)
	{
    if (anAug.IsA('HXAugPower'))
      PowerAug = anAug;

		if (anAug.bHasIt && anAug.bIsActive)
		{
			energyUse += ((anAug.GetEnergyRate()/60) * deltaTime);
			if (anAug.IsA('HXAugPower'))
      {
				energyMult = anAug.LevelValues[anAug.CurrentLevel];
      }
		}
		anAug = anAug.next;
	}

	// check for the power augmentation
	energyUse *= energyMult;

	return energyUse;
}

// ----------------------------------------------------------------------------
// AddAllAugs()
// ----------------------------------------------------------------------------

function AddAllAugs()
{
	local int augIndex;

	// Loop through all the augmentation classes and create
	// any augs that don't exist.  Then set them all to the 
	// maximum level.

	for(augIndex=0; augIndex<arrayCount(augClasses); augIndex++)
	{
		if (augClasses[augIndex] != None)
			GivePlayerAugmentation(augClasses[augIndex]);
	}
}

// ----------------------------------------------------------------------------
// SetAllAugsToMaxLevel()
// ----------------------------------------------------------------------------

function SetAllAugsToMaxLevel()
{
	local Augmentation anAug;

	anAug = FirstAug;
	while(anAug != None)
	{
		if (anAug.bHasIt)
			anAug.CurrentLevel = anAug.MaxLevel;

		anAug = anAug.next;
	}
}

// ----------------------------------------------------------------------------
// IncreaseAllAugs()
// ----------------------------------------------------------------------------

function IncreaseAllAugs(int Amount)
{
   local Augmentation anAug;
   
   anAug = FirstAug;
   while(anAug != None)
   {
      if (anAug.bHasIt)
         anAug.CurrentLevel = Min(anAug.CurrentLevel + Amount, anAug.MaxLevel);

      anAug = anAug.next;
   }
}

// ----------------------------------------------------------------------------
// ResetAugmentations()
// ----------------------------------------------------------------------------

function ResetAugmentations()
{
	local Augmentation anAug;
	local Augmentation nextAug;
    local int LocIndex;

	anAug = FirstAug;
	while(anAug != None)
	{
		nextAug = anAug.next;
		anAug.Destroy();
		anAug = nextAug;
	}

	FirstAug = None;

    //Must also clear auglocs.
    for (LocIndex = 0; LocIndex < 7; LocIndex++)
    {
        AugLocs[LocIndex].AugCount = 0;
    }
   
}

// ----------------------------------------------------------------------------
// FindAugmentationByKey()
// ----------------------------------------------------------------------------

simulated function HXAugmentation FindAugmentationByKey( int AugKey )
{	
	local Augmentation Aug;
	local int AugHotKey;
	local bool bActivated;

	if ( AugKey<0 || AugKey>9 )
		return None;

	AugHotKey = AugKey+3;

	bActivated = False;

	for ( Aug=FirstAug; Aug!=None; Aug=Aug.Next )
		if ( Aug.HotKeyNum==AugHotKey && Aug.bHasIt )
			return HXAugmentation(Aug);

	return None;
}

// ----------------------------------------------------------------------------
// ActivateAugmentationByKey()
// ----------------------------------------------------------------------------

function ActivateAugmentationByKey( int AugKey )
{
	local HXAugmentation Augmentation;

	// Warn here?
	if ( AugKey<0 || AugKey>9 )
		return;

	Augmentation = FindAugmentationByKey( AugKey );
	if ( Augmentation==None )
	{
		// I'm unsure about the Tag though. Do hotkeys work inside the persona screens? --han
		Player.ClientMessage( NoAugInSlot, 'StatusLine' );
		return;
	}

	// Avoid complaining about not being able to activate an already activate Augmentation.
	if ( Player.Energy<=0.0 && !Augmentation.bAlwaysActive && !Augmentation.bIsActive )
	{
		// !! Needs visual feedback !! Maybe sth. like FailActivate()
		Player.ClientMessage( Player.EnergyDepleted, 'StatusLine' );

		// Audio feedback.
		if ( Augmentation.DeactivateSound!=None )
			PlaySound( Augmentation.DeactivateSound, SLOT_None );
		return;
	}

	// No try to activate.
	Augmentation.Activate();
}

// ----------------------------------------------------------------------------
// DeactivateAugmentationByKey()
// ----------------------------------------------------------------------------

function DeactivateAugmentationByKey( int AugKey )
{
	local HXAugmentation Augmentation;

	// Warn here?
	if ( AugKey<0 || AugKey>9 )
		return;

	Augmentation = FindAugmentationByKey( AugKey );
	if ( Augmentation==None )
	{
		// I'm unsure about the Tag though. Do hotkeys work inside the persona screens? --han
		Player.ClientMessage( NoAugInSlot, 'StatusLine' );
		return;
	}

	// No try to activate.
	Augmentation.Deactivate();
}

// ----------------------------------------------------------------------------
// ToggleAugmentationByKey()
// ----------------------------------------------------------------------------

function ToggleAugmentationByKey( int AugKey )
{
	local HXAugmentation Augmentation;

	// Warn here?
	if ( AugKey<0 || AugKey>9 )
		return;

	Augmentation = FindAugmentationByKey( AugKey );

	// Just pass aug not found case forward.
	if ( Augmentation==None || !Augmentation.bIsActive )
		ActivateAugmentationByKey( AugKey );
	else 
		DeactivateAugmentationByKey( AugKey );
}

// ----------------------------------------------------------------------------
// Deprecated.
// ----------------------------------------------------------------------------

function bool ActivateAugByKey( int AugKey )
{	
	Warn( "Called." );
}

// ----------------------------------------------------------------------------
// ----------------------------------------------------------------------------

defaultproperties
{
	AugLocationFull="You can't add any more augmentations to that location!"
	NoAugInSlot="There is no augmentation in that slot"
	MsgAugAlreadyInstalled="You already have this augmentation!"
	MsgInvalidAugSlot="This is not a valid augmentation slot"
	AugLocs(0)=(NumSlots=1,KeyBase=4)
	AugLocs(1)=(NumSlots=1,KeyBase=7)
	AugLocs(2)=(NumSlots=3,KeyBase=8)
	AugLocs(3)=(NumSlots=1,KeyBase=5)
	AugLocs(4)=(NumSlots=1,KeyBase=6)
	AugLocs(5)=(NumSlots=2,KeyBase=2)
	AugLocs(6)=(NumSlots=3,KeyBase=11)
	AugClasses(0)=HXAugSpeed
	AugClasses(1)=HXAugTarget
	AugClasses(2)=HXAugCloak
	AugClasses(3)=HXAugBallistic
	AugClasses(4)=HXAugRadarTrans
	AugClasses(5)=HXAugShield
	AugClasses(6)=HXAugEnviro
	AugClasses(7)=HXAugEMP
	AugClasses(8)=HXAugCombat
	AugClasses(9)=HXAugHealing
	AugClasses(10)=HXAugStealth
	AugClasses(11)=HXAugIFF
	AugClasses(12)=HXAugLight
	AugClasses(13)=HXAugMuscle
	AugClasses(14)=HXAugVision
	AugClasses(15)=HXAugDrone
	AugClasses(16)=HXAugDefense
	AugClasses(17)=HXAugAqualung
	AugClasses(18)=HXAugDatalink
	AugClasses(19)=HXAugHeartLung
	AugClasses(20)=HXAugPower
	DefaultAugs(0)=HXAugLight
	DefaultAugs(1)=HXAugIFF
	DefaultAugs(2)=HXAugDatalink
	bGameRelevant=True
	bHidden=True
	bTravel=True
}
