//=============================================================================
// HXAutoTurret.
//=============================================================================
class HXAutoTurret extends HXDecoration;

var HXAutoTurretGun gun;
var Class<HXAutoTurretGun> GunClass;

var() localized String titleString;		// So we can name specific turrets in multiplayer
var() bool bTrackPawnsOnly;
var() bool bTrackPlayersOnly;
var() bool bActive;
var() int maxRange;
var() float fireRate;
var() float gunAccuracy;
var() int gunDamage;
var() int ammoAmount;
var Actor curTarget;
var Actor prevTarget;         // target we had last tick.
var Pawn safeTarget;          // in multiplayer, this actor is strictly off-limits
                               // Usually for the player who activated the turret.
var float fireTimer;
var bool bConfused;				// used when hit by EMP
var float confusionTimer;		// how long until turret resumes normal operation
var float confusionDuration;	// how long does an EMP hit last?
var Actor LastTarget;			// what was our last target?
var float pitchLimit;			// what's the maximum pitch?
var Rotator origRot;			// original rotation
var bool bPreAlarmActiveState;	// was I previously awake or not?
var bool bDisabled;				// have I been hacked or shut down by computers?
//var float TargetRefreshTime;      // used for multiplayer to reduce rate of checking for targets.

var int team;						// Keep track of team the turrets on

//var int mpTurretDamage;			// Settings for multiplayer
//var int mpTurretRange;

var bool bComputerReset;			// Keep track of if computer has been reset so we avoid all actors checks

var bool bSwitching;
var float SwitchTime, beepTime;
var Pawn savedTarget;

// ----------------------------------------------------------------------------
// Networking replication.
// ----------------------------------------------------------------------------

replication
{
	// Server to client.
	reliable if ( Role==ROLE_Authority )
		SafeTarget, bDisabled, bActive, Team, TitleString, bTrackPawnsOnly, bTrackPlayersOnly;
}


// ----------------------------------------------------------------------------
// PreBeginPlay()
// ----------------------------------------------------------------------------

simulated function PreBeginPlay()
{
	local Vector V1, V2;
	local Rotator Rot;

	Super.PreBeginPlay();

	if ( Role==ROLE_Authority )
	{
		Rot = Rotation;
		Rot.Pitch = 0;
		Rot.Roll = 0;
		OrigRot = Rot;

		if ( Gun==None )
		{
			Gun = Spawn( GunClass, Self,, Location, Rot );
		}
		//else
			//Log( "Already found GUN!!!!!!!!!!!!!!!!!" );

		if ( Gun!=None )
		{
			V1.X = 0;
			V1.Y = 0;
			V1.Z = CollisionHeight + Gun.Default.CollisionHeight;
			V2 = V1 >> Rotation;
			V2 += Location;
			Gun.SetLocation( V2 );
			Gun.SetBase( Self );
		}

		// Set up the alarm listeners.
		// HX_HAN: Moved to PostPostBeginPlay().
		//AISetEventCallback( 'Alarm', 'AlarmHeard' );
	}
}

// ----------------------------------------------------------------------------
// PostBeginPlay()
// ----------------------------------------------------------------------------

simulated function PostBeginPlay()
{
	if ( Role==ROLE_Authority )
	{
		// Note: Move this to PostPostBeginPlay() too?
		SafeTarget = None;
		PrevTarget = None;
		//TargetRefreshTime = 0;
	}

	Super.PostBeginPlay();
}

// ----------------------------------------------------------------------------
// PostPostBeginPlay()
// ----------------------------------------------------------------------------

simulated function PostPostBeginPlay()
{
	if ( Role==ROLE_Authority )
	{
		// Set up the alarm listeners. Do this in PostPostBeginPlay(),
		// so it is done each time the map is (re)loaded and i can 
		// clear EventManager which hopefully solves remaining listen server
		// crash issue.
		AISetEventCallback( 'Alarm', 'AlarmHeard' );
	}

	Super.PostPostBeginPlay();
}

// ----------------------------------------------------------------------------
// Succeeds()
//
// Intended to be called by Mutator code if Self was spawned as a successor
// to Other. Super call recommended.
// ----------------------------------------------------------------------------

function Succeeds( Actor Other )
{
	local AutoTurret OtherAutoTurret;

	// Call Super.
	Super.Succeeds( Other );

	// Special AutoTurret Init.
	OtherAutoTurret = AutoTurret(Other);
	if ( OtherAutoTurret!=None )
	{
		// Is everything needed or are they more like identical?
		TitleString				= OtherAutoTurret.TitleString;
		bTrackPawnsOnly		= OtherAutoTurret.bTrackPawnsOnly;
		bTrackPlayersOnly	= OtherAutoTurret.bTrackPlayersOnly;
		bActive						= OtherAutoTurret.bActive;
		MaxRange					= OtherAutoTurret.MaxRange;
		FireRate					= OtherAutoTurret.FireRate;
		GunAccuracy				= OtherAutoTurret.GunAccuracy;
		GunDamage					= OtherAutoTurret.GunDamage;
		AmmoAmount				= OtherAutoTurret.AmmoAmount;

		// Is this still needed?
		OtherAutoTurret.bActive = False;
		OtherAutoTurret.bHidden = True;
		OtherAutoTurret.DrawType = DT_None;
		OtherAutoTurret.Style = STY_None;
	}
}

// ----------------------------------------------------------------------------
// Trigger()
// ----------------------------------------------------------------------------

function Trigger( Actor Other, Pawn Instigator )
{
	if ( bConfused || bDisabled )
		return;

	// If we are triggered, turn us on.
	if ( !bActive )
	{
		bActive = True;
		AmbientSound = Default.AmbientSound;
	}

	Super.Trigger( Other, Instigator );
}

// ----------------------------------------------------------------------------
// UnTrigger()
// ----------------------------------------------------------------------------

function UnTrigger( Actor Other, Pawn Instigator )
{
	if ( bConfused || bDisabled )
		return;

	// If we are untriggered, turn us off.
	if ( bActive )
	{
		bActive = False;
		AmbientSound = None;
	}

	Super.UnTrigger( Other, Instigator );
}

// ----------------------------------------------------------------------------
// Destroyed()
// ----------------------------------------------------------------------------

simulated function Destroyed()
{
	if ( Role==ROLE_Authority )
	{
		if ( Gun!=None )
		{
			Gun.Destroy();
			Gun = None;
		}
	}

	Super.Destroyed();		
}

// ----------------------------------------------------------------------------
// UpdateSwitch()
// ----------------------------------------------------------------------------

function UpdateSwitch()
{
	if ( Level.Timeseconds > SwitchTime )
	{
		bSwitching = False;
		//safeTarget = savedTarget;
		SwitchTime = 0;
		beepTime = 0;
	}
	else
	{
		if ( Level.Timeseconds > beepTime )
		{
			PlaySound(Sound'TurretSwitch', SLOT_Interact, 1.0,, maxRange );
			beepTime = Level.Timeseconds + 0.75;
		}
	}
}

// ----------------------------------------------------------------------------
// SetSafeTarget()
// ----------------------------------------------------------------------------

function SetSafeTarget( Pawn NewSafeTarget )
{
	local DeusExPlayer aplayer;

	bSwitching = True;
	SwitchTime = Level.Timeseconds + 2.5;
	beepTime = 0.0;
   safeTarget = newSafeTarget;
   //savedTarget = newSafeTarget;
}

// ----------------------------------------------------------------------------
// Tick()
// ----------------------------------------------------------------------------

function Tick( float DeltaTime )
{
	local Pawn pawn;
	local ScriptedPawn sp;
	local DeusExDecoration deco;
	local float near;
	local Rotator destRot;
	local bool bSwitched;

	Super.Tick(deltaTime);

	bSwitched = False;

	if ( bSwitching )
	{
		UpdateSwitch();
		return;
	}

	if (bConfused)
	{
		confusionTimer += deltaTime;

		// pick a random facing
		if (confusionTimer % 0.25 > 0.2)
		{
			gun.DesiredRotation.Pitch = origRot.Pitch + (pitchLimit / 2 - Rand(pitchLimit));
			gun.DesiredRotation.Yaw = 65535*FRand();
		}
		if (confusionTimer > confusionDuration)
		{
			bConfused = False;
			confusionTimer = 0;
			confusionDuration = Default.confusionDuration;
		}
	}

	if (bActive && !bDisabled)
	{
		curTarget = None;

		if ( !bConfused )
		{
			if (Role == ROLE_Authority)
			{
				//
				// Logic table for turrets
				//
				// bTrackPlayersOnly		bTrackPawnsOnly		Should Attack
				// 			T						X				Allies
				//			F						T				Enemies
				//			F						F				Everything
				//
         
				// Attack allies and neutrals
				if (bTrackPlayersOnly || (!bTrackPlayersOnly && !bTrackPawnsOnly))
				{
					foreach gun.VisibleActors(class'Pawn', pawn, maxRange, gun.Location)
					{
						if (pawn.bDetectable && !pawn.bIgnore)
						{
							if (pawn.IsA('DeusExPlayer'))
							{
								// If the player's RadarTrans aug is off, the turret can see him
								if (DeusExPlayer(pawn).AugmentationSystem.GetAugLevelValue(class'HXAugRadarTrans') == -1.0)
								{
									curTarget = pawn;
									break;
								}
							}
							//
							//else if (pawn.IsA('ScriptedPawn') && (ScriptedPawn(pawn).GetPawnAllianceType(GetPlayerPawn()) != ALLIANCE_Hostile))
							else if (pawn.IsA('ScriptedPawn') && (ScriptedPawn(pawn).GetAllianceType('Player') != ALLIANCE_Hostile))
							{
								curTarget = pawn;
								break;
							}
						}
					}
				}
         
				if (!bTrackPlayersOnly)
				{
					// Attack everything
					if (!bTrackPawnsOnly)
					{
						foreach gun.VisibleActors(class'DeusExDecoration', deco, maxRange, gun.Location)
						{
							if (!deco.IsA('HXElectronicDevices') && !deco.IsA('HXAutoTurret') &&
								!deco.bInvincible && deco.bDetectable && !deco.bIgnore)
							{
								curTarget = deco;
								break;
							}
						}
					}
            
					// Attack enemies
					foreach gun.VisibleActors(class'ScriptedPawn', sp, maxRange, gun.Location)
					{
						//if (sp.bDetectable && !sp.bIgnore && (sp.GetPawnAllianceType(GetPlayerPawn()) == ALLIANCE_Hostile))
						if (sp.bDetectable && !sp.bIgnore && (sp.GetAllianceType('Player') == ALLIANCE_Hostile))
						{
							curTarget = sp;
							break;
						}
					}
				}
			}

			// if we have a target, rotate to face it
			if (curTarget != None)
			{
				destRot = Rotator(curTarget.Location - gun.Location);
				gun.DesiredRotation = destRot;
				near = pitchLimit / 2;
				gun.DesiredRotation.Pitch = FClamp(gun.DesiredRotation.Pitch, origRot.Pitch - near, origRot.Pitch + near);
			}
			else
				gun.DesiredRotation = origRot;
		}
	}
	else
	{
		if ( !bConfused )
			gun.DesiredRotation = origRot;
	}

	near = (Abs(gun.Rotation.Pitch - gun.DesiredRotation.Pitch)) % 65536;
	near += (Abs(gun.Rotation.Yaw - gun.DesiredRotation.Yaw)) % 65536;

	if (bActive && !bDisabled)
	{
		// play an alert sound and light up
		if ((curTarget != None) && (curTarget != LastTarget))
			PlaySound(Sound'Beep6',,,, 1280);

		// if we're aiming close enough to our target
		if (curTarget != None)
		{
			gun.MultiSkins[1] = Texture'RedLightTex';
			if ((near < 4096) && (((Abs(gun.Rotation.Pitch - destRot.Pitch)) % 65536) < 8192))
			{
				if (fireTimer > fireRate)
				{
					Fire();
					fireTimer = 0;
				}
			}
		}
		else
		{
			if (gun.IsAnimating())
				gun.PlayAnim('Still', 10.0, 0.001);

			if (bConfused)
				gun.MultiSkins[1] = Texture'YellowLightTex';
			else
				gun.MultiSkins[1] = Texture'GreenLightTex';
		}

		fireTimer += deltaTime;
		LastTarget = curTarget;
	}
	else
	{
		if (gun.IsAnimating())
			gun.PlayAnim('Still', 10.0, 0.001);
		gun.MultiSkins[1] = None;
	}

	// make noise if we're still moving
	if (near > 64)
	{
		gun.AmbientSound = Sound'AutoTurretMove';
		if (bConfused)
			gun.SoundPitch = 128;
		else
			gun.SoundPitch = 64;
	}
	else
		gun.AmbientSound = None;
}

// ----------------------------------------------------------------------------
// Fire()
// ----------------------------------------------------------------------------

function Fire()
{
	local Vector HitLocation, HitNormal, StartTrace, EndTrace, X, Y, Z;
	local Rotator rot;
	local Actor hit;
	local ShellCasing shell;
	local Spark spark;
	local Pawn attacker;

	if (!gun.IsAnimating())
		gun.LoopAnim('Fire');

	// CNN - give turrets infinite ammo
//	if (ammoAmount > 0)
//	{
//		ammoAmount--;
		GetAxes(gun.Rotation, X, Y, Z);
		StartTrace = gun.Location;
		EndTrace = StartTrace + gunAccuracy * (FRand()-0.5)*Y*1000 + gunAccuracy * (FRand()-0.5)*Z*1000 ;
		EndTrace += 10000 * X;
		hit = Trace(HitLocation, HitNormal, EndTrace, StartTrace + gun.CollisionRadius * 1.2 * X, True);

		// spawn some effects
    if ((HXGameInfo(Level.Game) != None) && (!HXGameInfo(Level.Game).bSpawnEffects))
    {
       shell = None;
    }
    else
    {
       shell = Spawn(class'ShellCasing',,, gun.Location);
    }
		if (shell != None)
			shell.Velocity = Vector(gun.Rotation - rot(0,16384,0)) * 100 + VRand() * 30;

		MakeNoise(1.0);
		PlaySound(sound'PistolFire', SLOT_None);
		AISendEvent('LoudNoise', EAITYPE_Audio);

		// muzzle flash
		gun.LightType = LT_Steady;
		gun.MultiSkins[2] = Texture'FlatFXTex34';
		SetTimer(0.1, False);

		// HX_HAN do it every time.
		// randomly draw a tracer
		//if (FRand() < 0.5)
		//{
			if (VSize(HitLocation - StartTrace) > 250)
			{
				rot = Rotator(EndTrace - StartTrace);
				//Spawn(class'HXTracer',,, StartTrace + 96 * Vector(rot), rot);
				Spawn(class'HXNetTemporaryTracer',,, StartTrace + 96 * Vector(rot), rot);
			}
		//}

		if (hit != None)
		{
         if ((HXGameInfo(Level.Game) != None) && (!HXGameInfo(Level.Game).bSpawnEffects))
         {
            spark = None;
         }
         else
         {
            // spawn a little spark and make a ricochet sound if we hit something
            spark = spawn(class'Spark',,,HitLocation+HitNormal, Rotator(HitNormal));
         }

			if (spark != None)
			{
				spark.DrawScale = 0.05;
				PlayHitSound(spark, hit);
			}

			attacker = None;
			// HX_HAN: Took that crap out.
			/*if ((curTarget == hit) && !curTarget.IsA('PlayerPawn'))
				attacker = GetPlayerPawn();*/
			if (Level.NetMode != NM_Standalone)
				attacker = safetarget;
			if ( hit.IsA('DeusExPlayer') && ( Level.NetMode != NM_Standalone ))
				DeusExPlayer(hit).myTurretKiller = Self;

			//Log( hit $ ".TakeDamage( " $ gunDamage $ ", " $ attacker $ ", " $ HitLocation $ ", " $ (1000.0*X) $ ", 'AutoShot')" );
			hit.TakeDamage(gunDamage, attacker, HitLocation, 1000.0*X, 'AutoShot');

			if (hit.bIsPawn && !(hit.IsA('HXScriptedPawn') && HXScriptedPawn(hit).bIsRobot) )
				SpawnBlood(HitLocation, HitNormal);
			else if ((hit == Level) || hit.IsA('Mover'))
				SpawnEffects(HitLocation, HitNormal, hit);
			// HX_HAN, Uhm, what about robots?
		}
//	}
//	else
//	{
//		PlaySound(sound'DryFire', SLOT_None);
//	}
}

// ----------------------------------------------------------------------------
// SpawnBlood()
// ----------------------------------------------------------------------------

function SpawnBlood(Vector HitLocation, Vector HitNormal)
{
	local rotator rot;

	rot = Rotator(Location - HitLocation);
	rot.Pitch = 0;
	rot.Roll = 0;

   if ((HXGameInfo(Level.Game) != None) && (!HXGameInfo(Level.Game).bSpawnEffects))
      return;

	spawn(class'BloodSpurt',,,HitLocation+HitNormal, rot);
	spawn(class'BloodDrop',,,HitLocation+HitNormal);
	if (FRand() < 0.5)
		spawn(class'BloodDrop',,,HitLocation+HitNormal);
}

// ----------------------------------------------------------------------------
// SpawnEffects()
// ----------------------------------------------------------------------------

simulated function SpawnEffects( Vector HitLocation, Vector HitNormal, Actor Other )
{
	local SmokeTrail puff;
	local int i;
	local BulletHole hole;
	local Rotator rot;

   if ((HXGameInfo(Level.Game) != None) && (!HXGameInfo(Level.Game).bSpawnEffects))
      return;

   if (FRand() < 0.5)
	{
		puff = spawn(class'SmokeTrail',,,HitLocation+HitNormal, Rotator(HitNormal));
		if (puff != None)
		{
			puff.DrawScale *= 0.3;
			puff.OrigScale = puff.DrawScale;
			puff.LifeSpan = 0.25;
			puff.OrigLifeSpan = puff.LifeSpan;
		}
	}

	if ( Other != None && !Other.IsA('HXBreakableGlass'))
		for (i=0; i<2; i++)
			if (FRand() < 0.8)
				spawn(class'Rockchip',,,HitLocation+HitNormal);

	hole = spawn(class'BulletHole', Other,, HitLocation, Rotator(HitNormal));

	// should we crack glass?
	if (GetWallMaterial(HitLocation, HitNormal) == 'Glass')
	{
		if (FRand() < 0.5)
			hole.Texture = Texture'FlatFXTex29';
		else
			hole.Texture = Texture'FlatFXTex30';

		hole.DrawScale = 0.1;
		hole.ReattachDecal();
	}
}

// ----------------------------------------------------------------------------
// PlayHitSound()
// ----------------------------------------------------------------------------

function Name GetWallMaterial( Vector HitLocation, Vector HitNormal )
{
	local vector EndTrace, StartTrace;
	local actor newtarget;
	local int texFlags;
	local name texName, texGroup;
	local Texture Tex;

	StartTrace = HitLocation + HitNormal*16;		// make sure we start far enough out
	EndTrace = HitLocation - HitNormal;

	foreach TraceTextures(class'Actor', newtarget, Tex, texName, texGroup, texFlags, StartTrace, HitNormal, EndTrace)
		if ((newtarget == Level) || newtarget.IsA('Mover'))
			break;

	return texGroup;
}

// ----------------------------------------------------------------------------
// PlayHitSound()
// ----------------------------------------------------------------------------

function PlayHitSound( Actor DestActor, Actor HitActor )
{
	local float rnd;
	local sound snd;

	rnd = FRand();

	if (rnd < 0.25)
		snd = sound'Ricochet1';
	else if (rnd < 0.5)
		snd = sound'Ricochet2';
	else if (rnd < 0.75)
		snd = sound'Ricochet3';
	else
		snd = sound'Ricochet4';

	// play a different ricochet sound if the object isn't damaged by normal bullets
	if (hitActor != None) 
	{
		if (hitActor.IsA('DeusExDecoration') && (DeusExDecoration(hitActor).minDamageThreshold > 10))
			snd = sound'ArmorRicochet';
		else if ( hitActor.bIsPawn && hitActor.IsA('HXScriptedPawn') && HXScriptedPawn(hitActor).bIsRobot )
			snd = sound'ArmorRicochet';
	}

	if (destActor != None)
		destActor.PlaySound(snd, SLOT_None,,, 1024, 1.1 - 0.2*FRand());
}

// ----------------------------------------------------------------------------
// Timer()
// ----------------------------------------------------------------------------

simulated function Timer()
{
	// Turn off the muzzle flash.

	Gun.LightType = LT_None;
	Gun.MultiSkins[2] = None;
}

// ----------------------------------------------------------------------------
// AlarmHeard()
// ----------------------------------------------------------------------------

function AlarmHeard( Name Event, EAIEventState State, XAIParams Params )
{
	switch ( State )
	{
		case EAISTATE_Begin:
			if ( !bActive )
			{
				bPreAlarmActiveState = bActive;
				bActive = True;
			}
			break;
	
		case EAISTATE_End:
			if ( bActive )
			{
				bActive = bPreAlarmActiveState;
			}
			break;

		default:
			break;
	}
}

// ----------------------------------------------------------------------------
// state Active
// ----------------------------------------------------------------------------

auto state Active
{
	function TakeDamage(int Damage, Pawn EventInstigator, vector HitLocation, vector Momentum, name DamageType)
	{
		local float mindmg;

		if (DamageType == 'EMP')
		{
			// duration is based on daamge
			// 10 seconds min to 30 seconds max
			mindmg = Max(Damage - 15.0, 0.0);
			confusionDuration += mindmg / 5.0;
         confusionDuration = FClamp(confusionDuration,10.0,30.0);
			confusionTimer = 0;
			if (!bConfused)
			{
				bConfused = True;
				PlaySound(sound'EMPZap', SLOT_None,,, 1280);
			}
			return;
		}
		//if (( Level.NetMode != NM_Standalone ) && (EventInstigator.IsA('DeusExPlayer')))
			//DeusExPlayer(EventInstigator).ServerConditionalNotifyMsg( DeusExPlayer(EventInstigator).MPMSG_TurretInv );

		Super.TakeDamage(Damage, EventInstigator, HitLocation, Momentum, DamageType);
	}
}

// ----------------------------------------------------------------------------
// ----------------------------------------------------------------------------

defaultproperties
{
	GunClass=HXAutoTurretGun
	bAlwaysRelevant=True 
	TitleString="AutoTurret"
	bTrackPlayersOnly=True
	bActive=True
	MaxRange=512
	FireRate=0.25
	GunAccuracy=0.5
	GunDamage=5
	AmmoAmount=1000
	ConfusionDuration=10.0
	PitchLimit=11000.0
	Team=-1
	HitPoints=50
	MinDamageThreshold=50
	bHighlight=False
	bClientHighlight=False
	ItemName="Turret Base"
	bPushable=False
	Physics=PHYS_None
	Mesh=AutoTurretBase
	SoundRadius=48
	SoundVolume=192
	AmbientSound=AutoTurretHum
	CollisionRadius=14.00
	//CollisionHeight=20.20
	CollisionHeight=19.45
	Mass=50.0
	Buoyancy=10.0
	bVisionImportant=True
}
