//=============================================================================
// HXBarkManager.
//=============================================================================
class HXBarkManager extends HXInfo
	transient;

// A globally unique identifier.
struct BarkInfo
{
	var HXScriptedPawn BarkPawn;
	var Name           ConName;
	var float          BarkDuration;
	var float          BarkTimer;
	var EBarkModes     BarkMode;
	var int            BarkPriority;
	var int            PlayingSoundId;
};

var BarkInfo CurrentBarks[8];
var BarkInfo RecentBarks[32];

var float PerCharDelay;					
var float MinimumTextPause;
var float BarkModeSpacer;
var int   MaxCurrentBarks;
var float MaxBarkExpirationTimer;
var float MaxAllowableRadius;
var float MaxHiddenHeightDifference;
var float MaxVisibleHeightDifference;

var HXGameInfo Game;

// Localized Strings.
var localized string BarkTagString;

// ----------------------------------------------------------------------
// Init()
//
// Called by HXGameInfo during InitGame.
// ----------------------------------------------------------------------

function Init( HXGameInfo InGame )
{
	Game = InGame;
}

// ----------------------------------------------------------------------
// Tick()
// ----------------------------------------------------------------------

function Tick(float deltaTime)
{
	//Log( "I'm"@Self, 'DevBark' );

	Super(Actor).Tick(deltaTime);

	UpdateCurrentBarks(deltaTime);
	UpdateRecentBarks(deltaTime);
}

// ----------------------------------------------------------------------
// HasBark()
// ----------------------------------------------------------------------

function bool HasBark( HXScriptedPawn BarkPawn, EBarkModes NewBarkMode )
{
	local Name ConName;

	if ( BarkPawn==None || BarkPawn.ConListItems==None )
		return False;

	// First attempt to find this Conversation
	ConName = BuildBarkName( BarkPawn, NewBarkMode );

	// Okay, we have the name of the bark, now attempt to find a Conversation based on this name.
	return ConListItem(BarkPawn.ConListItems).FindConversationByName(ConName)!=None;
}

// ----------------------------------------------------------------------
// StartBark()
// ----------------------------------------------------------------------

function bool StartBark( HXScriptedPawn BarkPawn, EBarkModes NewBarkMode )
{
	local Name         ConName;
	local Conversation Con;
	local int          barkIndex;
	local float        barkDuration, TestDistance, DeltaZ;
	local bool         bBarkStarted;
	local String       ConSpeechString;
	local ConSpeech    ConSpeech;
	local Sound        speechAudio;
	local bool         bHaveSpeechAudio;
	local int          playingSoundID;

	local HXPlayerPawn RelevantPlayers[32], /*HXPP, */TestPlayer;
	local int					 RelevantPlayerCount;

	local Pawn P;
	local int i;

	if ( BarkPawn==None )
		return False;

	// HX: walk through pawn list instead. Maybe one should do the other checks first.
	/*for ( P=Level.PawnList; P!=None; P=P.NextPawn )
	{
		if ( !P.bIsPlayer )
			continue;
		HXPP = HXPlayerPawn(P);
		if ( HXPP==None )
			continue;
		if ( CheckRadius(HXPP,BarkPawn) && CheckHeightDifference(HXPP,BarkPawn) )
			RelevantPlayers[RelevantPlayerCount++] = HXPP;
	}*/

	// Now try precalculated PlayerDistances.
	for ( i=0; i<32/*ArrayCount(RelevantPlayers) && i<ArrayCount(BarkPawn.PlayerDistances)*/; i++ )
	{
		// 'Error, Context expression: Variable is too large (256 bytes, 255 max)' my ass.
		BarkPawn.GetPlayerDistance( i, TestPlayer, TestDistance );

		// Check whether entry is valid.
		if ( TestPlayer==None )
			break;

		// CheckRadius.
		if ( TestDistance>MaxAllowableRadius )
			break;

		// CheckHeightDifference.
		// If the ScriptedPawn can see the player, than allow a taller height difference.
		// Otherwise make it pretty small (this is done especially to prevent the bark from playing through ceilings).
		/*if ( (TestPlayer.FastTrace(BarkPawn.Location) && TestPlayer.CheckConversationHeightDifference(BarkPawn,MaxVisibleHeightDifference)) || TestPlayer.CheckConversationHeightDifference(BarkPawn,MaxHiddenHeightDifference) )
			RelevantPlayers[RelevantPlayerCount++] = TestPlayer;*/

		// Imho this code is weired as hell. --han
		DeltaZ = Abs(Abs(TestPlayer.Location.Z-BarkPawn.Location.Z)-Abs(TestPlayer.Default.CollisionHeight-TestPlayer.CollisionHeight)); // HX_NOTE: added another Abs

		// MaxHiddenHeightDifference<MaxVisibleHeightDifference, so there is no need to trace in this case, otherwise check MaxVisibleHeightDifference before trace.
		if (  DeltaZ<=(Abs(TestPlayer.CollisionHeight-BarkPawn.CollisionHeight)+MaxHiddenHeightDifference)
			|| (DeltaZ<=(Abs(TestPlayer.CollisionHeight-BarkPawn.CollisionHeight)+MaxVisibleHeightDifference) && TestPlayer.FastTrace(BarkPawn.Location)) )
			RelevantPlayers[RelevantPlayerCount++] = TestPlayer;
	}
	if ( RelevantPlayerCount==0 )
		return False;

	// Now attempt to find this Conversation, should usually work as available barks are cached inside ScriptedPawn.
	ConName = BuildBarkName( BarkPawn, NewBarkMode );

	// Okay, we have the name of the bark, now attempt to find a Conversation based on this name.
	Con = ConListItem(BarkPawn.ConListItems).FindConversationByName(ConName);

	// Return if there is no conversation.
	if ( Con==None )
		return False;

	// Check whether this con is masked by flags.
	if ( !Game.CheckFlagRefs(Con.FlagRefList) )
	{
		Log( "Skipping flag masked bark conversation: "$Con$".", 'DevBark' );
		return False;
	}

	// Abort if we don't get a valid barkIndex back.
	BarkIndex = GetAvailableCurrentBarkSlot( BarkPawn, NewBarkMode );
	if ( BarkIndex==-1 )
		return False;

	// Make sure that another NPC isn't already playing this particular bark.
	if ( IsBarkPlaying(ConName) )
		return False;

	// Now check to see if the same kind of bark has already been played by this NPC within a certain range of time. 
	if ( HasBarkTypePlayedRecently(BarkPawn,NewBarkMode) )
		return False;

	// Process conversation and grab Speech.
	ConSpeech = GetBarkSpeech( BarkPawn, Con );

	bHaveSpeechAudio = False;

	// Play the audio (if we have audio)
	if ((ConSpeech != None) && (ConSpeech.soundID != -1))
	{
		speechAudio = Con.GetSpeechAudio(ConSpeech.soundID);

		if (speechAudio != None)
		{
			bHaveSpeechAudio = True;
			//playingSoundID = BarkPawn.PlaySound(speechAudio, SLOT_Talk,,,1024.0);
			playingSoundID = Game.BroadcastActorSpeech( speechAudio $ "", BarkPawn, 1024.0 );
			barkDuration = Con.GetSpeechLength(ConSpeech.soundID);
		}
	}

	// If we don't have any audio, then calculate the timer based on the 
	// length of the speech text.

	if ((ConSpeech != None) && (!bHaveSpeechAudio))
		barkDuration = FMax(Len(ConSpeech.speech) * perCharDelay, minimumTextPause);

	// Show the speech if Subtitles are on
	//if ((DeusExPlayer(owner) != None) && (DeusExPlayer(owner).bSubtitles) && (ConSpeech != None) && (ConSpeech.speech != ""))
	//{
		//rootWindow.hud.barkDisplay.AddBark(ConSpeech.speech, barkDuration, BarkPawn);
	//}

	if ( (ConSpeech != None) && (ConSpeech.speech != "") )
		for ( i = 0; i < RelevantPlayerCount; i ++ )
			RelevantPlayers[i].ClientAddBarkDisplay( ConSpeech.speech, barkDuration, BarkPawn.UnfamiliarName );

	// Keep track fo the bark
	SetBarkInfo(barkIndex, ConName, BarkPawn, newBarkMode, barkDuration, playingSoundID);

	bBarkStarted = True;
}

// ----------------------------------------------------------------------
// IsBarkPlaying()
//
// Loops through all the active barks and checks to see if a bark
// with the same name is already playing.
// ----------------------------------------------------------------------

function bool IsBarkPlaying(Name ConName)
{
	local int barkIndex;

	for (barkIndex=0; barkIndex < maxCurrentBarks; barkIndex++)
	{
		if (currentBarks[barkIndex].conName == conName)
		{
			return True;
		}
	}

	return False;
}

// ----------------------------------------------------------------------
// HasBarkTypePlayedRecently()
//
// Checks to see if a bark type has been played recently for 
// the given ScriptedPawn passed in.
// ----------------------------------------------------------------------

function bool HasBarkTypePlayedRecently(HXScriptedPawn newBarkPawn, EBarkModes newBarkMode)
{
	local int barkIndex;

	for (barkIndex=0; barkIndex < arrayCount(recentBarks); barkIndex++)
	{
		if ((recentBarks[barkIndex].barkPawn == newBarkPawn) && (recentBarks[barkIndex].barkMode == newBarkMode))
		{
			return True;
		}
	}

	return False;
}

// ----------------------------------------------------------------------
// CheckRadius()
//
// Returns True if this conversation can be invoked given the 
// invoking actor and the distance between the actor and the player
// ----------------------------------------------------------------------

/*function bool CheckRadius(HXPlayerPawn Player, HXScriptedPawn invokePawn)
{
	local Int  invokeRadius;
	local Int  dist;

	dist = VSize(Player.Location - invokePawn.Location);

	return (dist <= maxAllowableRadius);
}*/

// ----------------------------------------------------------------------
// CheckHeightDifference()
// ----------------------------------------------------------------------

/*function bool CheckHeightDifference(HXPlayerPawn Player, HXScriptedPawn invokePawn)
{
	// If the ScriptedPawn can see the player, than allow a taller height difference.
	// Otherwise make it pretty small (this is done especially to prevent the 
	// bark from playing through ceilings).

	if (Player.FastTrace(invokePawn.Location))
		return Player.CheckConversationHeightDifference(invokePawn, maxVisibleHeightDifference);
	else
		return Player.CheckConversationHeightDifference(invokePawn, maxHiddenHeightDifference);
}*/

// ----------------------------------------------------------------------
// SetBarkInfo()
// ----------------------------------------------------------------------

function SetBarkInfo(
	int barkIndex, 
	Name conName,
	HXScriptedPawn newBarkPawn, 
	EBarkModes newBarkMode, 
	Float barkDuration, 
	int playingSoundID)
{
	RemoveCurrentBark(barkIndex);

	currentBarks[barkIndex].conName        = conName;
	currentBarks[barkIndex].barkPawn       = newBarkPawn;
	currentBarks[barkIndex].barkDuration   = barkDuration;
	currentBarks[barkIndex].barkTimer      = 0.0;
	currentBarks[barkIndex].barkMode       = newBarkMode;
	currentBarks[barkIndex].playingSoundID = playingSoundID;

	// Determine the bark priority based on the mode
	currentBarks[barkIndex].barkPriority  = GetBarkPriority(newBarkMode);
}

// ----------------------------------------------------------------------
// GetAvailableCurrentBarkSlot()
//
// Loops through the bark slots and checks to see if any are avaialble
// or if any are in use by this NPC.  
// ----------------------------------------------------------------------

function int GetAvailableCurrentBarkSlot(
	HXScriptedPawn newBarkPawn, 
	EBarkModes newBarkMode)
{
	local int barkIndex;
	local int emptyBarkIndex;
	local int barkPriority;

	barkPriority = GetBarkPriority(newBarkMode);

	emptyBarkIndex = -1;

	for (barkIndex=0; barkIndex < maxCurrentBarks; barkIndex++)
	{
		if (currentBarks[barkIndex].barkPawn == None)
		{
			emptyBarkIndex = barkIndex;
		}
		else
		{
			if (currentBarks[barkIndex].barkPawn == newBarkPawn)
			{
				// Only override if the new bark is a higher priority than
				// the previous bark.
				if (currentBarks[barkindex].barkPriority < barkPriority)
				{
					emptyBarkIndex = barkIndex;
					break;
				}
			}
			else if (currentBarks[barkIndex].barkMode == newBarkMode)
			{
				// Check to see if a similar bark has been started very close to
				// this one, in which case we want to not start this bark (so they
				// don't clobber one another)

				if (currentBarks[barkIndex].barkTimer < barkModeSpacer)
				{
					// abort
					break;
				}
			}
		}
	}

	return emptyBarkIndex;
}

// ----------------------------------------------------------------------
// GetAvailableRecentBarkSlot()
// ----------------------------------------------------------------------

function int GetAvailableRecentBarkSlot()
{
	local int barkIndex;
	local int emptyBarkIndex;
	local int oldestBarkIndex;
	local float expireTimer;

	emptyBarkIndex  = -1;
	oldestBarkIndex = -1;
	expireTimer     = maxBarkExpirationTimer;

	for (barkIndex=0; barkIndex < arrayCount(recentBarks); barkIndex++)
	{
		if (recentBarks[barkIndex].barkPawn == None)
		{
			emptyBarkIndex = barkIndex;
			break;
		}
		else if ((recentBarks[barkIndex].barkDuration - recentBarks[barkIndex].barkTimer) < expireTimer)
		{
			expireTimer     = recentBarks[barkIndex].barkDuration - recentBarks[barkIndex].barkTimer;
			oldestBarkIndex = barkIndex;
		}
	}

	// If we found an empty slot, use it.  Otherwise use the bark that will 
	// expire first

	if (emptyBarkIndex == -1)
		emptyBarkIndex = oldestBarkIndex;

	return emptyBarkIndex;
}

// ----------------------------------------------------------------------
// UpdateCurrentBarks()
// ----------------------------------------------------------------------

function UpdateCurrentBarks(Float deltaTime)
{
	local int barkIndex;

   // DEUS_EX AMSD In multiplayer, for now, kill barks.
   //if (Level.NetMode != NM_Standalone)
      //return;

	for (barkIndex=0; barkIndex < maxCurrentBarks; barkIndex++)
	{
		if (currentBarks[barkIndex].barkPawn != None)
		{
			currentBarks[barkIndex].barkTimer += deltaTime;

			if (currentBarks[barkIndex].barkTimer >= currentBarks[barkIndex].barkDuration)
				RemoveCurrentBark(barkIndex);
		}
	}
}

// ----------------------------------------------------------------------
// UpdateRecentBarks()
// ----------------------------------------------------------------------

function UpdateRecentBarks(Float deltaTime)
{
	local int barkIndex;

   // DEUS_EX AMSD In multiplayer, for now, kill barks.
   //if (Level.NetMode != NM_Standalone)
      //return;

   for (barkIndex=0; barkIndex < arrayCount(recentBarks); barkIndex++)
	{
		if (recentBarks[barkIndex].barkPawn != None)
		{
			recentBarks[barkIndex].barkTimer += deltaTime;

			if (recentBarks[barkIndex].barkTimer >= recentBarks[barkIndex].barkDuration)
				RemoveRecentBark(barkIndex);
		}
	}
}

// ----------------------------------------------------------------------
// RemoveCurrentBark()
// ----------------------------------------------------------------------

function RemoveCurrentBark(int barkIndex)
{
	// First add to the RecentBarkList
	MoveCurrentBarkToRecent(barkIndex);

	currentBarks[barkIndex].barkPawn       = None;
	currentBarks[barkIndex].conName        = '';
	currentBarks[barkIndex].barkDuration   = 0.0;
	currentBarks[barkIndex].barkTimer      = 0.0;
	currentBarks[barkIndex].barkMode       = BM_Idle;
	currentBarks[barkIndex].barkPriority   = 0;
	currentBarks[barkIndex].playingSoundID = 0;
}

// ----------------------------------------------------------------------
// MoveCurrentBarkToRecent()
// ----------------------------------------------------------------------

function MoveCurrentBarkToRecent(int currentBarkIndex)
{
	local int recentBarkIndex;

	recentBarkIndex = GetAvailableRecentBarkSlot();

	if (recentBarkIndex != -1)
	{
		recentBarks[recentBarkIndex].conName      = currentBarks[currentBarkIndex].conName;
		recentBarks[recentBarkIndex].barkPawn     = currentBarks[currentBarkIndex].barkPawn;
		recentBarks[recentBarkIndex].barkDuration = GetBarkTimeout(currentBarks[currentBarkIndex].barkMode);
		recentBarks[recentBarkIndex].barkTimer    = 0.0;
		recentBarks[recentBarkIndex].barkMode     = currentBarks[currentBarkIndex].barkMode;
		recentBarks[recentBarkIndex].barkPriority = currentBarks[currentBarkIndex].barkPriority;
	}
}

// ----------------------------------------------------------------------
// RemoveRecentBark()
// ----------------------------------------------------------------------

function RemoveRecentBark(int barkIndex)
{
	recentBarks[barkIndex].conName       = '';
	recentBarks[barkIndex].barkPawn      = None;
	recentBarks[barkIndex].barkDuration  = 0.0;
	recentBarks[barkIndex].barkTimer     = 0.0;
	recentBarks[barkIndex].barkMode      = BM_Idle;
	recentBarks[barkIndex].barkPriority  = 0;
}

// ----------------------------------------------------------------------
// GetBarkPriority()
// ----------------------------------------------------------------------

function int GetBarkPriority(EBarkModes barkMode)
{
	local int barkPriority;

	switch(barkMode)
	{
		case BM_Idle:
			barkPriority = 1;
			break;

		case BM_CriticalDamage:
			barkPriority = 5;
			break;

		case BM_AreaSecure:
			barkPriority = 2;
			break;

		case BM_TargetAcquired:
			barkPriority = 3;
			break;

		case BM_TargetLost:
			barkPriority = 2;
			break;

		case BM_GoingForAlarm:
			barkPriority = 4;
			break;

		case BM_OutOfAmmo:
			barkPriority = 2;
			break;

		case BM_Scanning:
			barkPriority = 2;
			break;

		case BM_Futz:
			barkPriority = 3;
			break;

		case BM_OnFire:
			barkPriority = 5;
			break;

		case BM_TearGas:
			barkPriority = 4;
			break;

		case BM_Gore:
			barkPriority = 3;
			break;

		case BM_Surprise:
			barkPriority = 5;
			break;

		case BM_PreAttackSearching:
			barkPriority = 4;
			break;

		case BM_PreAttackSighting:
			barkPriority = 5;
			break;

		case BM_PostAttackSearching:
			barkPriority = 4;
			break;

		case BM_SearchGiveUp:
			barkPriority = 2;
			break;

		case BM_AllianceHostile:
			barkPriority = 3;
			break;

		case BM_AllianceFriendly:
			barkPriority = 2;
			break;
	}

	return barkPriority;
}

// ----------------------------------------------------------------------
// GetBarkTimeout()
// ----------------------------------------------------------------------

function Float GetBarkTimeout(EBarkModes barkMode)
{
	local Float barkTimeout;

	switch(barkMode)
	{
		case BM_Futz:
			barkTimeout = 1.0;
			break;

		default:
			barkTimeout = 10.0;
			break;
	}

	return barkTimeout;
}

// ----------------------------------------------------------------------
// BuildBarkName()
// ----------------------------------------------------------------------

function Name BuildBarkName(HXScriptedPawn newBarkPawn, EBarkModes newBarkMode)
{
	local String conStringName;

	//SetRootWindow();

	// Use the "BarkBindName" unless it's blank, in which case
	// we'll fall back to the "BindName" used for normal 
	// conversations

	if (newBarkPawn.BarkBindName == "") 
		conStringName = newBarkPawn.BindName $ "_Bark";
	else
		conStringName = newBarkPawn.BarkBindName $ "_Bark";

	switch(newBarkMode)
	{
		case BM_Idle:
			conStringName = conStringName $ "Idle";
			break;

		case BM_CriticalDamage:
			conStringName = conStringName $ "CriticalDamage";
			break;

		case BM_AreaSecure:
			conStringName = conStringName $ "AreaSecure";
			break;

		case BM_TargetAcquired:
			conStringName = conStringName $ "TargetAcquired";
			break;

		case BM_TargetLost:
			conStringName = conStringName $ "TargetLost";
			break;

		case BM_GoingForAlarm:
			conStringName = conStringName $ "GoingForAlarm";
			break;

		case BM_OutOfAmmo:
			conStringName = conStringName $ "OutOfAmmo";
			break;

		case BM_Scanning:
			conStringName = conStringName $ "Scanning";
			break;

		case BM_Futz:
			conStringName = conStringName $ "Futz";
			break;

		case BM_OnFire:
			conStringName = conStringName $ "OnFire";
			break;

		case BM_TearGas:
			conStringName = conStringName $ "TearGas";
			break;

		case BM_Gore:
			conStringName = conStringName $ "Gore";
			break;

		case BM_Surprise:
			conStringName = conStringName $ "Surprise";
			break;

		case BM_PreAttackSearching:
			conStringName = conStringName $ "PreAttackSearching";
			break;

		case BM_PreAttackSighting:
			conStringName = conStringName $ "PreAttackSighting";
			break;

		case BM_PostAttackSearching:
			conStringName = conStringName $ "PostAttackSearching";
			break;

		case BM_SearchGiveUp:
			conStringName = conStringName $ "SearchGiveUp";
			break;

		case BM_AllianceHostile:
			conStringName = conStringName $ "AllianceHostile";
			break;

		case BM_AllianceFriendly:
			conStringName = conStringName $ "AllianceFriendly";
			break;
	}

	// Take the string name and convert it to a name
	return StringToName(conStringName);
}

// ----------------------------------------------------------------------
// ScriptedPawnDied()
// ----------------------------------------------------------------------

function ScriptedPawnDied(HXScriptedPawn deadPawn)
{
	local int barkIndex;
	local DeusExPlayer player;

	// Loop through our active barks and see if one of them is 
	// owned by the dead dude.

	for (barkIndex=0; barkIndex < maxCurrentBarks; barkIndex++)
	{
		if (currentBarks[barkIndex].barkPawn == deadPawn)
		{
			//player = DeusExPlayer(GetPlayerPawn());

			// Stop the sound and remove the bark
			//if (player != None)
			//{
				//player.StopSound(currentBarks[barkIndex].playingSoundID);
				Game.BroadcastActorStopSpeech( currentBarks[barkIndex].playingSoundID );
				RemoveCurrentBark(barkIndex);
			//}
		}
	}
}

// ----------------------------------------------------------------------
// CheckFlagRefs()
//
// Loops through the flagrefs passed in and sees if the current flag
// settings in the game match this set of flags.  Returns True if so,
// otherwise False.
// ----------------------------------------------------------------------

function bool CheckFlagRefs( ConFlagRef FlagRef )
{
	local ConFlagRef Ref;

	// Loop through our list of FlagRef's, checking the value of each.
	// If we hit a bad match, then we'll stop right away since there's
	// no point of continuing.
	Ref = FlagRef;
	while( Ref!=None )
	{
		if ( Game.Steve.FlagBase.GetBool(Ref.FlagName)!=Ref.Value )
			return False;
		Ref = Ref.NextFlagRef;
	}
	
	// If we made it this far, then the flags check out.
	return True;
}

// ----------------------------------------------------------------------
// GetBarkSpeech()
// ----------------------------------------------------------------------

function ConSpeech GetBarkSpeech( HXScriptedPawn BarkPawn, Conversation Con )
{
	local ConEvent Event;

	// Abort if we don't have a valid conversation
	if ( Con==None )
		return None;

	// Loop through the events until we hit some speech.
	Event = Con.EventList;
	while ( Event!=None )
	{
		switch( Event.EventType )
		{
			// Original.
			case ET_Speech:
				return ConEventSpeech(Event).ConSpeech;
				break;
			case ET_Jump:
				Event = ProcessBarkEventJump( BarkPawn, Con, ConEventJump(Event) );
				break;
			case ET_Random:
				Event = ProcessBarkEventRandomLabel( BarkPawn, Con, ConEventRandomLabel(Event) );
				break;
			case ET_End:
				return None;
				break;

			// Additional.
			case ET_CheckFlag:
				Event = ProcessBarkEventCheckFlag( BarkPawn, Con, ConEventCheckFlag(Event) );
				break;
			case ET_SetFlag:
				Event = ProcessBarkEventSetFlag( BarkPawn, Con, ConEventSetFlag(Event) );
				break;
			case ET_Trigger:
				Event = ProcessBarkEventTrigger( BarkPawn, Con, ConEventTrigger(Event) );
				break;
			case ET_AddGoal:
				Event = ProcessBarkEventAddGoal( BarkPawn, Con, ConEventAddGoal(Event) );
				break;
			case ET_AddNote:
				Event = ProcessBarkEventAddNote( BarkPawn, Con, ConEventAddNote(Event) );
				break;
			case ET_AddSkillPoints:
				Event = ProcessBarkEventAddSkillPoints( BarkPawn, Con, ConEventAddSkillPoints(Event) );
				break;

			default:
				Event = Event.NextEvent;
				break;
		}
	}
	return None;
}

// ----------------------------------------------------------------------
// ProcessBarkEventJump()
// ----------------------------------------------------------------------

function ConEvent ProcessBarkEventJump( HXScriptedPawn BarkPawn, Conversation Con, ConEventJump Event )
{
	local ConEvent NextEvent;

	// Check to see if the jump label is empty.  If so, then we just want
	// to fall through to the next event.  This can happen when jump
	// events get inserted during the import process.  ConEdit will not
	// allow the user to create events like this. 
	if ( Event.JumpLabel=="" )
		return Event.NextEvent;

	// Jumping to events in another Bark(Conversation) is not supported.
	if ( Event.JumpCon!=None && Event.JumpCon!=Con )
		return None;

	// Ordinary jump.
	return Con.GetEventFromLabel(Event.jumpLabel);
}

// ----------------------------------------------------------------------
// ProcessBarkEventRandomLabel()
// ----------------------------------------------------------------------

function ConEvent ProcessBarkEventRandomLabel( HXScriptedPawn BarkPawn, Conversation Con, ConEventRandomLabel Event )
{
	// Pick a random label.
	return Con.GetEventFromLabel( Event.GetRandomLabel() );
}

// ----------------------------------------------------------------------
// ProcessBarkEvent()
// ----------------------------------------------------------------------

function ConEvent ProcessBarkEventCheckFlag( HXScriptedPawn BarkPawn, Conversation Con, ConEventCheckFlag Event )
{
	// Failed.
	if ( !CheckFlagRefs(Event.FlagRef) )
		return Event.NextEvent;
	
	// Passed.
	return Con.GetEventFromLabel( Event.SetLabel );
}

// ----------------------------------------------------------------------
// ProcessBarkEventSetFlag()
// ----------------------------------------------------------------------

function ConEvent ProcessBarkEventSetFlag( HXScriptedPawn BarkPawn, Conversation Con, ConEventSetFlag Event )
{
	local ConFlagRef Ref;

	// Just follow the chain of flag references and set the flags to the proper value!
	Ref = Event.FlagRef;

	while ( Ref!=None )
	{
		// This is so awful nasty, need to get rid of flag replication anyway. --han.
		Game.Steve.Flagbase.SetBool( Ref.FlagName, Ref.Value );
		Game.Steve.Flagbase.SetExpiration( Ref.FlagName, FLAG_Bool, Ref.Expiration ); 
		Game.FlagReplicationInfo.SetBoolFlag( Ref.FlagName, Ref.Value );

		Ref = Ref.NextFlagRef;
	}
	return Event.NextEvent;
}

// ----------------------------------------------------------------------
// ProcessBarkEventTrigger()
// ----------------------------------------------------------------------

function ConEvent ProcessBarkEventTrigger( HXScriptedPawn BarkPawn, Conversation Con, ConEventTrigger Event )
{
	local Actor A;

	// Loop through all the actors, firing a trigger for each using the BarkPawn as Other and Instigator for triggering.
	foreach AllActors( Class'Actor', A, Event.TriggerTag )
		A.Trigger( BarkPawn, BarkPawn );

	return Event.NextEvent;
}

// ----------------------------------------------------------------------
// ProcessBarkEventAddGoal()
// ----------------------------------------------------------------------

function ConEvent ProcessBarkEventAddGoal( HXScriptedPawn BarkPawn, Conversation Con, ConEventAddGoal Event )
{
	// Add goal.
	if ( !Event.bGoalCompleted )
		Game.AddGoal( Event.GoalName, Event.bPrimaryGoal, Event.GoalText );
	// Or mark as completed.
	else
		Game.GoalCompleted( Event.GoalName ); 

	return Event.NextEvent;
}

// ----------------------------------------------------------------------
// ProcessBarkEventAddNote()
// ----------------------------------------------------------------------

function ConEvent ProcessBarkEventAddNote( HXScriptedPawn BarkPawn, Conversation Con, ConEventAddNote Event )
{
	// Only add the note if it hasn't been added already (in case the PC has the same conversation more than once)
	// This bNoteAdded probably won't persist across level channels. --han
	if ( !Event.bNoteAdded )
	{
		// Add the note to the list of notes
		Game.AddNote( Event.NoteText, False, True );
		Event.bNoteAdded = True;
	}
	return Event.NextEvent;
}

// ----------------------------------------------------------------------
// ProcessBarkEventAddSkillPoints()
// ----------------------------------------------------------------------

function ConEvent ProcessBarkEventAddSkillPoints( HXScriptedPawn BarkPawn, Conversation Con, ConEventAddSkillPoints Event )
{
	Game.SkillPointsAdd( Event.PointsToAdd, BarkTagString );

	return Event.NextEvent;
}

// ----------------------------------------------------------------------------
// PruneCurrent()
//
// Called before current map progress is saved just before switching level.
// Out parameter can be used to flag or unflag actor as transient.
//
// Be aware that this can break linked lists such as Level.PawnList.
// ----------------------------------------------------------------------------

simulated event PruneCurrent( out byte bTransient )
{
	bTransient = 1;
}

// ----------------------------------------------------------------------
// ----------------------------------------------------------------------

defaultproperties
{
	bGameRelevant=True
	BarkTagString="(Bark)"
	RemoteRole=ROLE_None
	PerCharDelay=0.1
	MinimumTextPause=3.0
	BarkModeSpacer=1.0
	MaxCurrentBarks=4
	MaxBarkExpirationTimer=30.0
	MaxAllowableRadius=1000
	MaxHiddenHeightDifference=100
	MaxVisibleHeightDifference=500
	bHidden=True
	bTravel=False
}