//=============================================================================
// HXClientCameraPoint.
//=============================================================================
class HXClientCameraPoint extends HXInfo
	transient;

//
// DEUS_EX CNN
// CameraPoint.uc - Cinematic camera control script
// These points are to be placed in the editor to control
// the camera during a cinematic cut-scene.
//

enum EClientCameraCommand
{
	CAMCMD_MOVE,						// cut (or smoothly move if timeSmooth > 0) to XYZPYR
	CAMCMD_PUSH,						// move forward along view axis
	CAMCMD_FOV,							// change FOV
	CAMCMD_TILT,						// change Pitch
	CAMCMD_PAN,							// change Yaw
	CAMCMD_ROLL,						// change Roll
	CAMCMD_WAIT,						// wait time or for a specific event
	CAMCMD_TRIGGER						// trigger an event
};

var() EClientCameraCommand	cmd;			// what command to execute
var() float				value;			// parameter for command
var() name				eventName;		// used by CAMCMD_WAIT_EVENT
var() float				timeSmooth;		// execute command smoothly over time (0 = instant)
var() float				timeWaitPost;	// delay after executing command
var() bool				bParallel;		// execute the next command simultaneously
var() bool				bRandom;		// the next point is chosen at random from other bRandoms
var() int				randomCount;	// number of times to choose randomly
var() int				postRandomNum;	// sequence to start at after random sequence

// internal vars not editable by the user
var() int			sequenceNum;	// assigned in editor automatically
var HXClientCameraPoint			nextPoint;		// next point to be executed - points linked at runtime
var HXClientCameraPoint			prevPoint;		// point that was just executed - used by bParallel
var float				curTime;		// timer used by movement routines
var vector				startLoc;		// init location at beginning of command
var rotator				startRot;		// init rotation at beginning of command
var float				startFOV;		// init FOV at beginning of command
var vector				endLoc;			// dest location
var rotator				endRot;			// dest rotation
var float				endFOV;			// dest FOV
var PlayerPawn			Player;			// which player is being controlled
var bool				bTickReady;		// can we tick now?
var bool				bFirstRandom;	// are we the first random point in this sequence?
var int					randomRemain;	// how many random choices are left?
var HXClientCameraPoint			continuePoint;	// where we continue after a random sequence

enum EFFNextStateCode
{
	NSC_None,
	NSC_RunningBegin,
	NSC_RunningNext
};

// FastForward
var EFFNextStateCode FFNextStateCode;
var float FFTimerRemaining;
var float FFSleepRemaining;
var bool bHasFFTimerRunning;
var bool bHasFFWaitSleep;

// FastForwardCode maybe i should add a check if state code is called twice or if it wasnt called at all
function FastForward( float TimeOffset )
{
	local HXClientCameraPoint Point;

	// Idle.Begin:
	Player = GetPlayerPawn();

	if (Player == None)
	{
		log("** WARNING - HXClientCameraPoint.Idle.Begin() - "$sequenceNum$" - can't find the player!");
		Destroy();
	}

	// grab the player, make it not collide, and freeze it's controls
	Player.GotoState('Paralyzed', 'Letterbox');
	Player.bHidden = True;
	Player.bCollideWorld = False;
	Player.SetCollision(False, False, False);
	Player.BaseEyeHeight = 0.0;

	if (Player.Shadow != None)
	{
		Player.Shadow.Destroy();
		Player.Shadow = None;
	}

	InitAllPoints();
	FFGotoState( 'Running', 'Begin' );

	// FF at 20 fps
	while ( TimeOffset >= 0.05 )
	{
		FFStep( 0.05 );

		TimeOffset -= 0.05;
	}

	//FFStep( TimeOffset );

	// Cleanup timer and sleep
	for ( Point = Self; Point != None; Point = Point.nextPoint )
	{
		// Check remaining timer
		if ( Point.bHasFFTimerRunning )
		{
			Point.bHasFFTimerRunning = False;
			Point.SetTimer( Point.FFTimerRemaining, False );
		}

		// Check remaining sleep
		if ( Point.bHasFFWaitSleep )
		{
			Point.bHasFFWaitSleep = False;
			Point.GotoState( 'Running', 'FFNextSleepRemaining' );
		}
	}
}

function FFStep( float DeltaTime )
{
	local HXClientCameraPoint Point;

	if ( sequenceNum != 0 )
		Log( "FFStep(): invalid sequenceNum", 'CameraPoint' );

	//foreach AllActors( class'HXClientCameraPoint', Point )
	for ( Point = Self; Point != None; Point = Point.nextPoint )
	{
		// Check timer
		if ( Point.bHasFFTimerRunning )
		{
			Point.FFTimerRemaining -= DeltaTime;

			if ( Point.FFTimerRemaining < 0 )
			{
				Point.bHasFFTimerRunning = False;
				Point.FFTimer();
			}
		}

		// Check Sleep
		if ( Point.bHasFFWaitSleep )
		{
			Point.FFSleepRemaining -= DeltaTime;

			if ( Point.FFSleepRemaining < 0 )
			{
				Point.bHasFFWaitSleep = False;
				Point.FFRunningNext2();
			}
		}

		// Tick point
		Point.FFTick( DeltaTime );
		Point.FFStateCode();
	}
}

function FFStateCode()
{
	switch ( FFNextStateCode )
	{
		case NSC_None:
			break;

		case NSC_RunningBegin:
			FFNextStateCode = NSC_None;
			FFRunningBegin();
			break;

		case NSC_RunningNext:
			FFNextStateCode = NSC_None;
			FFRunningNext();
			break;
	
		default:
			FFNextStateCode = NSC_None;
			break;
	}
}

function FFGotoState( name NewState, name NewLabel )
{
	bHasFFWaitSleep = False;

	if ( NewState == 'Running' )
	{
		if ( NewLabel == 'Begin' )
		{
			FFNextStateCode = NSC_RunningBegin;
			GotoState( 'Running', 'FFRunningDummy' );
			return;
		}
		else if ( NewLabel == 'Next' )
		{
			FFNextStateCode = NSC_RunningNext;
			GotoState( 'Running', 'FFRunningDummy' );
			return;
		}
	}
	else if ( NewState == 'Idle' )
	{
		if ( NewLabel == 'Begin' )
		{
			 // Idle.Wait does nothing.
			FFNextStateCode = NSC_None;
			GotoState('Idle', 'Wait');
			return;
		}
	}

	GotoState( NewState, NewLabel );
}

// wait for the timer to expire
function FFTimer()
{
	FFNext();
}

function FFSetTimer( float timeSmooth )
{
	FFTimerRemaining = timeSmooth;
	bHasFFTimerRunning = True;
}

// called when command is complete to move on to next command
function FFNext()
{
	FFGotoState( 'Running', 'Next' );
}

function FFRunningNext()
{
	if (!bParallel)
		bTickReady = False;

	// if there's a postdelay, wait for it
	if ( timeWaitPost > 0.0 )
	{
		FFSleep( timeWaitPost );
		return;
	}

	FFRunningNext2();
}

function FFSleep( float timeWaitPost )
{
	FFSleepRemaining = timeWaitPost;
	bHasFFWaitSleep = True;
}

// The part after sleep
function FFRunningNext2()
{
	if (bRandom && !bParallel)
	{
		bTickReady = False;

		// if this is the first random encounter, set some values
		if (randomRemain == -1)
		{
			bFirstRandom = True;
			randomRemain = randomCount;
			SetContinuePoint(); // Ok
		}

		// if we're done being random, continue where we left off
		if (randomRemain == 0)
			continuePoint.FFWakeUp();
		else
			FFWakeUpRandomPoint();
	}
	else if (nextPoint != None)
		nextPoint.FFWakeUp();
//	else
		Log("** HXClientCameraPoint.FFRunningNext2() - "$sequenceNum$" - finished with path!", 'CameraPoint' );

	//Wait:
	FFGoToSleep();
}

// called to wake this camerapoint up
function FFWakeUp()
{
	FFGotoState( 'Running', 'Begin' );
}

// called to put this camerapoint into it's idle state
function FFGoToSleep()
{
	if ( IsInState('Running') )
		FFGotoState('Idle', 'Wait'); // Idle, Wait does nothing.
	else
		Log("** WARNING - HXClientCameraPoint.FFGoToSleep() - "$sequenceNum$" - already sleeping!");
}

// called to find another random camerapoint
function FFWakeUpRandomPoint()
{
	local HXClientCameraPoint point;
	local int count, rnd;

	// count how many there are
	count = 0;
	foreach AllActors(class'HXClientCameraPoint', point)
		if (point.bRandom)
			count++;

	// pick one at random
	rnd = Rand(count);
	count = 0;
	foreach AllActors(class'HXClientCameraPoint', point)
		if (point.bRandom && (count++ == rnd))
		{
			point.randomRemain = randomRemain - 1;
			point.continuePoint = continuePoint;
			point.FFWakeUp();
			break;
		}
}

function FFRunningBegin()
{
	Log("FFRunningBegin ** starting "$sequenceNum, 'CameraPoint' );
	// save the starting location/rotation
	startLoc = Player.Location;
	startRot = Player.ViewRotation;
	startFOV = Player.FOVAngle;

	// init the destination
	endLoc = startLoc;
	endRot = startRot;
	endFOV = startFOV;

	// evaluate the command
	FFEvaluateCommand();
	bTickReady = True;

	// should we execute the next command in parallel?
	if (bParallel)
	{
		nextPoint.prevPoint = Self;
		FFNext();
	}
}

// evaluate the current command
function bool FFEvaluateCommand()
{
	local vector loc;
	local HXClientCameraPoint HXCCP;

	loc = vect(0,0,0);

	// init the movement timer
	curTime = 0.0;

	log("** HXClientCameraPoint.FFEvaluateCommand - "$sequenceNum$" - cmd = "$cmd$" value = "$value$" eventName = "$eventName);

	switch (cmd)
	{
		case CAMCMD_MOVE:		endLoc = Location;
								endRot = Rotation;
								break;

		case CAMCMD_PUSH:		loc.X = value;
								endLoc += (loc >> Player.ViewRotation);
								break;

		case CAMCMD_FOV:		endFOV += value;
								break;

		case CAMCMD_TILT:		endRot.Pitch += value;
								break;

		case CAMCMD_PAN:		endRot.Yaw += value;
								break;

		case CAMCMD_ROLL:		endRot.Roll += value;
								break;

		case CAMCMD_WAIT:		if (eventName != '')
								{
									Tag = eventName;
									FFGoToSleep();
								}
								else
									FFSetTimer(timeSmooth);
								break;

		case CAMCMD_TRIGGER:	if (eventName != '')
								{
									foreach AllActors(class'HXClientCameraPoint', HXCCP, eventName)
										HXCCP.FFTrigger();
								}
								break;
	}

	// if we are running in parallel with the previous point,
	// add its parameters to ours
	if (prevPoint != None)
	{
		endLoc += prevPoint.endLoc - prevPoint.startLoc;
		endRot += prevPoint.endRot - prevPoint.startRot;
		endFOV += prevPoint.endFOV - prevPoint.startFOV;
	}
}

function FFTrigger() {}
function FFTick( float DeltaTime ) {}

// link all CameraPoints together
function InitAllPoints()
{
	local HXClientCameraPoint cur, next;
	local int num;
	local rotator rot;

	cur = Self;
	num = 0;
	while (num != 999999)
	{
		num = 999999;
		foreach AllActors(class'HXClientCameraPoint', next)
		{
			// clamp the rotation
			rot.Pitch = Rotation.Pitch % 65536;
			rot.Yaw = Rotation.Yaw % 65536;
			rot.Roll = Rotation.Roll % 65536;
			SetRotation(rot);

			// find the next highest number in the sequence
			if ((next.sequenceNum < num) && (next.sequenceNum > cur.sequenceNum))
			{
				num = next.sequenceNum;
				cur.nextPoint = next;
			}
		}

		if (num != 999999)
		{
			Log("** HXClientCameraPoint.InitAllPoints - linked "$cur.sequenceNum$" to "$num, 'CameraPoint');
			cur = cur.nextPoint;
		}
	}
}

//
// state control code
//
state Idle
{
	// wait to be triggered
	function Trigger(Actor Other, Pawn Instigator)
	{
		Log( "Idle.Trigger() sequenceNum = " $ sequenceNum, 'CameraPoint' );
		Super.Trigger(Other, Instigator);
		Tag = '';
		Next();
	}

	function FFTrigger()
	{
		Log( "Idle.FFTrigger() sequenceNum = " $ sequenceNum, 'CameraPoint' );
		Tag = '';
		FFNext();
	}

Begin:
	// find the player in this level
	// if there's not a player, destroy ourself
	Player = GetPlayerPawn();
	if (Player == None)
	{
		log("** WARNING - HXClientCameraPoint.Idle.Begin() - "$sequenceNum$" - can't find the player!", 'CameraPoint');
		Destroy();
		Stop;
	}

	// if we are the first one, link up the points and begin execution
	if (sequenceNum == 0)
	{
		// grab the player, make it not collide, and freeze it's controls
		Player.GotoState('Paralyzed', 'Letterbox');
		Player.bHidden = True;
		Player.bCollideWorld = False;
		Player.SetCollision(False, False, False);
		Player.BaseEyeHeight = 0.0;

		if (Player.Shadow != None)
		{
			Player.Shadow.Destroy();
			Player.Shadow = None;
		}
		InitAllPoints();
		GotoState('Running');
	}

Wait:
	// do nothing until somebody wakes us up
	Stop;
}

state Running
{
	// used for smooth moves/rotations
	function Tick(float deltaTime)
	{
		local float alpha, beta;
		local vector loc;
		local rotator rot, diffrot;
		local float fov;

		if (!bTickReady)
			return;

		Super.Tick(deltaTime);

		// update the timer and see if we're done
		curTime += deltaTime;

		if (curTime >= timeSmooth)
		{
			Player.ViewRotation = endRot;
			Player.SetLocation(endLoc);
			Player.SetFOVAngle(endFOV);
			Player.DesiredFOV = endFOV;
			Next();
		}
		else
		{
			alpha = curTime / timeSmooth;

			// make sure we rotate the shortest direction
			diffrot = endRot - startRot;

			if (diffrot.Pitch >= 32768)
				diffrot.Pitch = diffrot.Pitch - 65536;
			else if (diffrot.Pitch <= -32768)
				diffrot.Pitch = diffrot.Pitch + 65536;

			if (diffrot.Yaw >= 32768)
				diffrot.Yaw = diffrot.Yaw - 65536;
			else if (diffrot.Yaw <= -32768)
				diffrot.Yaw = diffrot.Yaw + 65536;

			if (diffrot.Roll >= 32768)
				diffrot.Roll = diffrot.Roll - 65536;
			else if (diffrot.Roll <= -32768)
				diffrot.Roll = diffrot.Roll + 65536;

			// calculate our smoothing coefficient
			beta = 3.0*alpha*alpha - 2.0*alpha*alpha*alpha;

			// smooth the camera rotation, movement, and FOV
			rot = startRot + beta * diffrot;
			loc = startLoc + beta * (endLoc - startLoc);
			fov = startFOV + beta * (endFOV - startFOV);

			Player.SetLocation(loc);
			Player.ViewRotation = rot;
			Player.SetFOVAngle(fov);
			Player.DesiredFOV = fov;
		}
	}

	function FFTick( float DeltaTime )
	{
		local float alpha, beta;
		local vector loc;
		local rotator rot, diffrot;
		local float fov;

		if (!bTickReady)
			return;

		//Global.FFTick( DeltaTime );

		//Super.Tick(deltaTime);

		// update the timer and see if we're done
		curTime += deltaTime;

		if (curTime >= timeSmooth)
		{
			Player.ViewRotation = endRot;
			Player.SetLocation(endLoc);
			Player.SetFOVAngle(endFOV);
			Player.DesiredFOV = endFOV;
			FFNext();
		}
		else
		{
			alpha = curTime / timeSmooth;

			// make sure we rotate the shortest direction
			diffrot = endRot - startRot;

			if (diffrot.Pitch >= 32768)
				diffrot.Pitch = diffrot.Pitch - 65536;
			else if (diffrot.Pitch <= -32768)
				diffrot.Pitch = diffrot.Pitch + 65536;

			if (diffrot.Yaw >= 32768)
				diffrot.Yaw = diffrot.Yaw - 65536;
			else if (diffrot.Yaw <= -32768)
				diffrot.Yaw = diffrot.Yaw + 65536;

			if (diffrot.Roll >= 32768)
				diffrot.Roll = diffrot.Roll - 65536;
			else if (diffrot.Roll <= -32768)
				diffrot.Roll = diffrot.Roll + 65536;

			// calculate our smoothing coefficient
			beta = 3.0*alpha*alpha - 2.0*alpha*alpha*alpha;

			// smooth the camera rotation, movement, and FOV
			rot = startRot + beta * diffrot;
			loc = startLoc + beta * (endLoc - startLoc);
			fov = startFOV + beta * (endFOV - startFOV);

			Player.SetLocation(loc);
			Player.ViewRotation = rot;
			Player.SetFOVAngle(fov);
			Player.DesiredFOV = fov;
		}	
	}

Begin:
//	log("** starting "$sequenceNum);
	// save the starting location/rotation
	startLoc = Player.Location;
	startRot = Player.ViewRotation;
	startFOV = Player.FOVAngle;

	// init the destination
	endLoc = startLoc;
	endRot = startRot;
	endFOV = startFOV;

	// evaluate the command
	EvaluateCommand();
	bTickReady = True;

	// should we execute the next command in parallel?
	if (bParallel)
	{
		nextPoint.prevPoint = Self;
		Next();
	}

	Stop;

FFRunningDummy:
	Stop;

FFNextSleepRemaining:
	// Sleep remaining time
	Sleep( FFSleepRemaining );
	Goto( 'NextPostTimer' );

Next:
	if (!bParallel)
		bTickReady = False;

	// if there's a postdelay, wait for it
	if (timeWaitPost > 0.0)
		Sleep(timeWaitPost);

NextPostTimer:

	if (bRandom && !bParallel)
	{
		bTickReady = False;

		// if this is the first random encounter, set some values
		if (randomRemain == -1)
		{
			bFirstRandom = True;
			randomRemain = randomCount;
			SetContinuePoint();
		}

		// if we're done being random, continue where we left off
		if (randomRemain == 0)
			continuePoint.WakeUp();
		else
			WakeUpRandomPoint();
	}
	else if (nextPoint != None)
		nextPoint.WakeUp();
//	else
	Log("** HXClientCameraPoint.Running.Next - "$sequenceNum$" - finished with path!", 'CameraPoint' );

Wait:
	GoToSleep();
}


//
// state control functions
//

// called to find another random camerapoint
function WakeUpRandomPoint()
{
	local HXClientCameraPoint point;
	local int count, rnd;

	// count how many there are
	count = 0;
	foreach AllActors(class'HXClientCameraPoint', point)
		if (point.bRandom)
			count++;

	// pick one at random
	rnd = Rand(count);
	count = 0;
	foreach AllActors(class'HXClientCameraPoint', point)
		if (point.bRandom && (count++ == rnd))
		{
			point.randomRemain = randomRemain - 1;
			point.continuePoint = continuePoint;
			point.WakeUp();
			break;
		}
}

// called to set the point to continue at after the random sequence
function SetContinuePoint()
{
	local HXClientCameraPoint point;
	
	foreach AllActors(class'HXClientCameraPoint', point)
		if (point.sequenceNum == postRandomNum)
		{
			continuePoint = point;
			break;
		}
}

// called when command is complete to move on to next command
function Next()
{
	GotoState('Running', 'Next');
}

// called to wake this camerapoint up
function WakeUp()
{
	GotoState('Running', 'Begin');
}

// called to put this camerapoint into it's idle state
function GoToSleep()
{
	if (IsInState('Running'))
		GotoState('Idle', 'Wait');
	else
		log("** WARNING - HXClientCameraPoint.GoToSleep() - "$sequenceNum$" - already sleeping!");
}

// wait for the timer to expire
function Timer()
{
	Next();
}

// evaluate the current command
function bool EvaluateCommand()
{
	local vector loc;
	local Actor A;

	loc = vect(0,0,0);

	// init the movement timer
	curTime = 0.0;

	log("** HXClientCameraPoint.EvaluateCommand - "$sequenceNum$" - cmd = "$cmd$" value = "$value$" eventName = "$eventName);

	switch (cmd)
	{
		case CAMCMD_MOVE:		endLoc = Location;
								endRot = Rotation;
								break;

		case CAMCMD_PUSH:		loc.X = value;
								endLoc += (loc >> Player.ViewRotation);
								break;

		case CAMCMD_FOV:		endFOV += value;
								break;

		case CAMCMD_TILT:		endRot.Pitch += value;
								break;

		case CAMCMD_PAN:		endRot.Yaw += value;
								break;

		case CAMCMD_ROLL:		endRot.Roll += value;
								break;

		case CAMCMD_WAIT:		if (eventName != '')
								{
									Tag = eventName;
									GoToSleep();
								}
								else
									SetTimer(timeSmooth, False);
								break;

		case CAMCMD_TRIGGER:	if (eventName != '')
								{
									foreach AllActors(class'Actor', A, eventName)
										A.Trigger(None, None);
								}
								break;
	}

	// if we are running in parallel with the previous point,
	// add its parameters to ours
	if (prevPoint != None)
	{
		endLoc += prevPoint.endLoc - prevPoint.startLoc;
		endRot += prevPoint.endRot - prevPoint.startRot;
		endFOV += prevPoint.endFOV - prevPoint.startFOV;
	}
}

defaultproperties
{
	SoundVolume=0
	CollisionRadius=10.000000
	CollisionHeight=10.000000
	randomRemain=-1
	Texture=Texture'Engine.S_Camera'
	RemoteRole=ROLE_None
}
