//=============================================================================
// HXComputers.
//=============================================================================
class HXComputers extends HXElectronicDevices
	abstract;

struct sSpecialOptions
{
	var() localized string	Text;
	var() localized string	TriggerText;
	var() string			userName;
	var() name				TriggerEvent;
	var() name				UnTriggerEvent;
	var() bool				bTriggerOnceOnly;
	var bool				bAlreadyTriggered;
};

var() localized sSpecialOptions specialOptions[4];

var class<HXNetworkTerminal> TerminalType;
var HXNetworkTerminal Termwindow;
var bool bOn;
var bool bAnimating;
var bool bLockedOut;				// true if this terminal is locked out
var() float lockoutDelay;			// delay until locked out terminal can be used
var float lockoutTime;				// time when terminal was locked out
var float lastHackTime;				// last time the terminal was hacked
var HXPlayerPawn curFrobber;     // player currently frobbing.
var localized String msgLockedOut;

enum EAccessLevel
{
	AL_Untrained,
	AL_Trained,
	AL_Advanced,
	AL_Master
};

// userlist information
struct sUserInfo
{
	var() string		UserName;
	var() string		Password;
	var() EAccessLevel	AccessLevel;
};

var() sUserInfo UserList[8];

// Specific location information.
var() String  NodeName;
var() String  TitleString;
var() Texture TitleTexture;

var() string TextPackage; // Package from which text for emails/bulletins for this computer should be loaded.

//
// 15 DeusEx + 15 Custom ComputerNodes (for Mods).
//
// NodeInfo is stored in HXNetworkTerminal. To add you own nodes subclass the appropreciate HXNetworkTerminal
// subclass, add your own node descriptions to it and set it as TerminalType for your HXComputers subclass.
//
enum EComputerNodes
{
	CN_UNATCO, 
	CN_VersaLife,
	CN_QueensTower,
	CN_USNavy,
	CN_MJ12Net,
	CN_PageIndustries,
	CN_Area51,
	CN_Everett,
	CN_NSF,
	CN_NYC,
	CN_China,
	CN_HKNet,
	CN_QuickStop,
	CN_LuckyMoney,
	CN_Illuminati,
	CN_Custom_00,
	CN_Custom_01,
	CN_Custom_02,
	CN_Custom_03,
	CN_Custom_04,
	CN_Custom_05,
	CN_Custom_06,
	CN_Custom_07,
	CN_Custom_08,
	CN_Custom_09,
	CN_Custom_10,
	CN_Custom_11,
	CN_Custom_12,
	CN_Custom_13,
	CN_Custom_14
};

var() EComputerNodes ComputerNode;

// alarm vars
var float LastAlarmTime;		// last time the alarm was sounded
var int AlarmTimeout;			// how long before the alarm silences itself

var localized string CompInUseMsg;

var string MissionNumber; // DeusExLevelInfo is not on client!

var bool bPlayActivationAnimations;

var Pawn alarmInstigator;

// ----------------------------------------------------------------------------
// Network replication.
// ----------------------------------------------------------------------------

replication
{
	// server to client
	reliable if (Role == ROLE_Authority)
		ComputerNode, specialOptions, TextPackage, lockoutDelay, userList, missionNumber;
}


// ----------------------------------------------------------------------------
// PreBeginPlay
// ----------------------------------------------------------------------------

function PreBeginPlay()
{
	local DeusExLevelInfo info;

	Super.PreBeginPlay();

	foreach AllActors(class'DeusExLevelInfo', info)
		break;

	// hack for the DX.DX splash level
	if (info != None) 
	{
		if (info.MissionNumber < 10)
			MissionNumber = "0" $ String(info.MissionNumber);
		else
			MissionNumber = String(info.MissionNumber);
	}
}

// ----------------------------------------------------------------------------
// PostBeginPlay
// ----------------------------------------------------------------------------

function PostBeginPlay()
{
   Super.PostBeginPlay();
   curFrobber = None;
}

// ----------------------------------------------------------------------------
// Succeeds()
//
// Intended to be called by Mutator code if Self was spawned as a successor
// to Other. Super call recommended.
// ----------------------------------------------------------------------------

function Succeeds( Actor Other )
{
	local Computers OtherComputers;
	local int i;

	// Call Super.
	Super.Succeeds( Other );

	// Special  Init.
	OtherComputers = Computers(Other);
	if ( OtherComputers!=None )
	{
		SetPropertyText( "ComputerNode", OtherComputers.GetPropertyText("ComputerNode") );

		LockoutDelay = OtherComputers.LockoutDelay;
		NodeName     = OtherComputers.NodeName;
		TitleString  = OtherComputers.TitleString;
		TitleTexture = OtherComputers.TitleTexture;
		TextPackage	 = OtherComputers.TextPackage;

		for ( i=0; i<4; i++ )
		{
			SpecialOptions[i].Text              = OtherComputers.SpecialOptions[i].Text;
			SpecialOptions[i].TriggerText       = OtherComputers.SpecialOptions[i].TriggerText;
			SpecialOptions[i].UserName          = OtherComputers.SpecialOptions[i].UserName;
			SpecialOptions[i].TriggerEvent      = OtherComputers.SpecialOptions[i].TriggerEvent;
			SpecialOptions[i].UnTriggerEvent    = OtherComputers.SpecialOptions[i].UnTriggerEvent;
			SpecialOptions[i].bTriggerOnceOnly  = OtherComputers.SpecialOptions[i].bTriggerOnceOnly;
		}
		for ( i=0; i<8; i++ )
		{
			UserList[i].UserName	= OtherComputers.UserList[i].UserName;
			UserList[i].Password	= OtherComputers.UserList[i].Password;

			switch ( OtherComputers.UserList[i].AccessLevel )
			{
				case AL_Untrained: UserList[i].AccessLevel = AL_Untrained; break;
				case AL_Trained:   UserList[i].AccessLevel = AL_Trained;   break;
				case AL_Advanced:  UserList[i].AccessLevel = AL_Advanced;  break;
				case AL_Master:    UserList[i].AccessLevel = AL_Master;    break;
			}
		}
	}
}

//
// Alarm functions for when you get caught hacking
//
function BeginAlarm()
{
	AmbientSound = Sound'Klaxon2';
	SoundVolume = 128;
	SoundRadius = 64;
	SoundPitch = 64;
	lastAlarmTime = Level.TimeSeconds;

	alarmInstigator = curFrobber;
	AIStartEvent('Alarm', EAITYPE_Audio, SoundVolume/255.0, 25*(SoundRadius+1));

	// make sure we can't go into stasis while we're alarming
	bStasis = False;
}

function EndAlarm()
{
	AmbientSound = Default.AmbientSound;
	SoundVolume = Default.SoundVolume;
	SoundRadius = Default.SoundRadius;
	SoundPitch = Default.SoundPitch;
	lastAlarmTime = 0;

	alarmInstigator = None;
	AIEndEvent('Alarm', EAITYPE_Audio);

	// reset our stasis info
	bStasis = Default.bStasis;
}

function Tick(float deltaTime)
{
	Super.Tick(deltaTime);

   // DEUS_EX AMSD IN multiplayer, set lockout to 0
   //if (Level.NetMode != NM_Standalone)
      //bLockedOut = False;

	// shut off the alarm if the timeout has expired
	if (lastAlarmTime != 0)
	{
		if (Level.TimeSeconds - lastAlarmTime >= alarmTimeout)
			EndAlarm();
	}
}
// ----------------------------------------------------------------------------
// ChangePlayerVisibility()
// ----------------------------------------------------------------------------

function ChangePlayerVisibility(bool bInviso)
{
	local DeusExPlayer player;

   if (Level.NetMode != NM_Standalone)
      return;
	player = DeusExPlayer(GetPlayerPawn());
	if (player != None)
		player.MakePlayerIgnored(!bInviso);
}

// ----------------------------------------------------------------------------
// state On
// ----------------------------------------------------------------------------

state On
{
	function Tick(float deltaTime)
	{
		Global.Tick(deltaTime);

		if (bOn)
		{
			if ((termwindow == None) && (Level.NetMode == NM_Standalone))
         {
				GotoState('Off');
         }            
         if (curFrobber == None)
         {
            GotoState('Off');
         }
         else if (VSize(curFrobber.Location - Location) > 1500)
         {
            log("Disabling computer "$Self$" because user "$curFrobber$" was too far away");
			//Probably should be "GotoState('Off')" instead, but no good way to test, so I'll leave it alone.
            curFrobber = None;
         }
		}
	}

Begin:
	if (!bOn)
	{
    AdditionalActivation(curFrobber);
		bAnimating = True;

		if (bPlayActivationAnimations)
		{
			PlayAnim('Activate');
			FinishAnim();
		}

		bOn = True;
		bAnimating = False;
		ChangePlayerVisibility(False);
    TryInvoke();
	}
}

// ----------------------------------------------------------------------------
// state Off
// ----------------------------------------------------------------------------

auto state Off
{
Begin:
	if (bOn)
	{
    AdditionalDeactivation(curFrobber);
		ChangePlayerVisibility(True);
		bAnimating = True;

		if (bPlayActivationAnimations)
		{
			PlayAnim('Deactivate');
			FinishAnim();
		}

		bOn = False;
		bAnimating = False;
		if (bLockedOut)
			BeginAlarm();

		// Resume any datalinks that may have started while we were 
		// in the computers (don't want them to start until we pop back out)
		ResumeDataLinks();
    curFrobber = None;
	}
}

// ----------------------------------------------------------------------------
// ResumeDataLinks()
// ----------------------------------------------------------------------------

function ResumeDataLinks()
{
	if (curFrobber != None)
	{
		curFrobber.ClientResumeDataLinks();
	}
}

// ----------------------------------------------------------------------------
// TryInvoke()
// ----------------------------------------------------------------------------

function TryInvoke()
{
   if (IsInState('Off'))
      return;
   
   if (!Invoke())
   {
      GotoState('Off');
   }

   return;
}

// ----------------------------------------------------------------------------
// Invoke()
// ----------------------------------------------------------------------------

function bool Invoke()
{
	local HXPlayerPawn player;

	if (termwindow != None)
		return False;

	player = curFrobber;
	if (player != None)
	{
      //pass timing info so the player can keep the time uptodate on his end.
      player.HXInvokeComputerScreen(self, lastHackTime, Level.TimeSeconds);
      // set owner for relevancy fer sure;
      SetOwner(Player);
	}

	return True;
}

// ----------------------------------------------------------------------------
// CloseOut()
// ----------------------------------------------------------------------------

function CloseOut()
{
   if (curFrobber != None)
   {
      //curFrobber = None;
      GotoState('Off');
   }
}

// ----------------------------------------------------------------------------
// Frob()
// ----------------------------------------------------------------------------

function Frob(Actor Frobber, Inventory frobWith)
{
	local HXPlayerPawn player;
	local float elapsed, delay;

	// Don't allow someone else to use the computer when already in use.
	if (curFrobber != None)
	{
		if ( Frobber != curFrobber && HXPlayerPawn(Frobber) != None )
			HXPlayerPawn(Frobber).ClientMessage(Sprintf(CompInUseMsg,curFrobber.PlayerReplicationInfo.PlayerName));
		return;
	}

	Super.Frob(Frobber, frobWith);

	// DEUS_EX AMSD get player from frobber, not from getplayerpawn
	player = HXPlayerPawn(Frobber);
	if (player != None)
	{
		if (bLockedOut)
		{
			// computer skill shortens the lockout duration
			delay = lockoutDelay / player.SkillSystem.GetSkillLevelValue(class'HXSkillComputer');

			elapsed = Level.TimeSeconds - lockoutTime;
			if (elapsed < delay)
				player.ClientMessage(Sprintf(msgLockedOut, Int(delay - elapsed)));
			else
				bLockedOut = False;
		}
		if (!bAnimating && !bLockedOut)
		{
			curFrobber = player;
			GotoState('On');
		}
	}
}

// ----------------------------------------------------------------------------
// AdditionalActivation()
// Called for subclasses to do any additional activation steps.
// ----------------------------------------------------------------------------

function AdditionalActivation(DeusExPlayer ActivatingPlayer)
{
}

// ----------------------------------------------------------------------------
// AdditionalDeactivation()
// ----------------------------------------------------------------------------

function AdditionalDeactivation(DeusExPlayer DeactivatingPlayer)
{
}

// ----------------------------------------------------------------------------
// ActivateSpecialOption()
// ----------------------------------------------------------------------------

function ActivateSpecialOption( int specialIndex )
{
	local Actor A;

	if (specialIndex != -1)
	{
		// Make sure this option wasn't already triggered
		if ( !specialOptions[specialIndex].bAlreadyTriggered )
		{
			if ( specialOptions[specialIndex].TriggerEvent != '' )
				foreach AllActors(class'Actor', A, specialOptions[specialIndex].TriggerEvent)
					A.Trigger(None, curFrobber);
		
			if ( specialOptions[specialIndex].UnTriggerEvent != '' )
				foreach AllActors(class'Actor', A, specialOptions[specialIndex].UnTriggerEvent)
					A.UnTrigger(None, curFrobber);
		
			if ( specialOptions[specialIndex].bTriggerOnceOnly )
				specialOptions[specialIndex].bAlreadyTriggered = True;

			// Display a message			
			//winSpecialInfo.SetText( specialOptions[specialIndex].TriggerText );
			BroadcastMessage( specialOptions[specialIndex].TriggerText );
		}
	}
}

// ----------------------------------------------------------------------------
// ----------------------------------------------------------------------------

defaultproperties
{
	bOn=True
	bPlayActivationAnimations=True
	LockoutDelay=30.0
	LastHackTime=-9999.0
	MsgLockedOut="Terminal is locked out for %d more seconds"
	NodeName="UNATCO"
	TitleString="United Nations Anti-Terrorist Coalition (UNATCO)"
	TitleTexture=ComputerLogonLogoUNATCO
	TextPackage="DeusExText"
	AlarmTimeout=30
	CompInUseMsg="The computer is already in use by %s"
	Mass=20.0
	Buoyancy=5.0
}
