//=============================================================================
// HXConsole.
//=============================================================================
class HXConsole extends Console;

#exec Texture Import NAME=HXConsoleBack File=Textures\HXConsole2.bmp

var(PhonemeDemo) float PhonemeTimeout;
var(PhonemeDemo) float PhonemeUpdateTime;
var(PhonemeDemo) int   PhonemesPerLine;
var bool bPhonemeDemo;

struct PhonemeInfo
{
	var Actor  Actor;
	var Name   ActorName;
	var String CurrentPhoneme;
	var String AccPhonemes;
	var float  NextUpdateTime;
};

var PhonemeInfo PhonemeInfos[16];

// Size the view up.
exec function ViewUp()
{
	BorderSize = Clamp( BorderSize-1, 0, MaxBorder );
}

// Size the view down.
exec function ViewDown()
{
	BorderSize = Clamp( BorderSize+1, 0, MaxBorder );
}

// Variant of Message, which just adds it to the dedicated console view.
event MessageHistory( PlayerReplicationInfo PRI, coerce string Msg, name N )
{
	if( Msg!="" )
	{
		TopLine		     = (TopLine+1) % MaxLines;
		NumLines	     = Min(NumLines+1,MaxLines-1);
		MsgType[TopLine] = N;
		//MsgTime		     = 6.0;
		MsgTime		     = -1.0;
		TextLines++;
		MsgText[TopLine] = Msg;
		MsgPlayer[TopLine] = PRI;
		MsgTick[TopLine] = MsgTickTime + MsgTime;
	}
}

// Forces all visible foreground messages into an existence as console log.
function ArchiveMessages()
{
	local int i;

	for (i=0; i<MaxLines; i++)
	{
		MsgTick[i] = 0.0; // MsgTickTime-1.0; ?
	}
	MsgTime = 0.0; // Is this needed?
}

// Fetches the oldest still visible message and sets it to be timed out.
function bool SnagMessage( out string OutMsgText, out name OutMsgType )
{
	local int iLine, iBackTrack;

	for ( iLine=0; iLine<MaxLines; iLine++ )
	{
		// Needs starting off TopLine;
		iBackTrack = (TopLine-iLine+MaxLines) % MaxLines;

		if ( MsgTick[iBackTrack]>MsgTickTime )
		{
			OutMsgText = MsgText[iBackTrack];
			OutMsgType = MsgType[iBackTrack];

			// Now hide.
			MsgTick[iBackTrack] = MsgTickTime-1.0;
			return true;
		}
	}

	return false;
}

// Called by the engine when a key, mouse, or joystick button is pressed
// or released, or any analog axis movement is processed.
event bool KeyEvent( EInputKey Key, EInputAction Action, FLOAT Delta )
{
	//Log( Self $ ".KeyEvent( "$GetEnum(enum'EInputKey',Key)$", "$GetEnum(enum'EInputAction',Action)$", "$Delta$" )", 'DevInput' );

	if( Action!=IST_Press )
	{
		return false;
	}
	// DEUS_EX CNN - Shift-Tilde is console now
	else if( Key==IK_Tilde )
	{
		if( ConsoleDest==0.0 )
		{
			ConsoleDest=0.6;
			GotoState('Typing');
		}
		else GotoState('');
		return true;
	}
	else return false;
}

// Debug output for FastTextureTrace.
/*function FastTextureTraceRender( Canvas C )
{
	local int Slot;
	local string Text;
	local PlayerPawn Player;
	local Vector  TraceEnd;
	local Texture Texture;
	local Name    TextureName;
	local Name    TextureGroup;
	local int     TextureFlags;

	Player = Viewport.Actor;

	// Trace out in front of us.
	TraceEnd = Player.Location + (Vector(Player.ViewRotation) * 256.0);

	Class'HXActor'.Static.StaticFastTextureTrace( Player.Level, TraceEnd, Texture, TextureName, TextureGroup, TextureFlags, Player.Location );

	C.Font = TimeDemoFont;
	C.DrawColor.R = 255;
	C.DrawColor.G = 255;
	C.DrawColor.B = 255;
	C.SetPos( 0, 0.2*C.ClipY );
	C.DrawText( "FastTextureTrace:" @ Texture );
}*/

// Called after rendering the world view.
event PostRender( Canvas C )
{
	local int YStart, YEnd, Y, I, J, Line, iLine;

	if(bNoDrawWorld)
	{
		C.SetPos(0,0);
		C.DrawPattern( Texture'Border', C.ClipX, C.ClipY, 1.0 );
	}

	if( bTimeDemo )
	{
		TimeDemoCalc();
		TimeDemoRender( C );
	}

	PhonemeDemoRender( C );

	// Temporary. Remove me.
	/*FastTextureTraceRender( C );*/

	// call overridable "level action" rendering code to draw the "big message"
	DrawLevelAction( C );

	// If the console has changed since the previous frame, draw it.
	if ( ConsoleLines > 0 )
	{
		// DEUS_EX AJY
		C.Style = 4;
		C.DrawColor.R = 128;
		C.DrawColor.G = 128;
		C.DrawColor.B = 128;
		C.SetPos(0.0, ConsoleLines - FrameY);
		C.SetOrigin(0.0, 0.0);
		C.DrawTile(ConBackground, FrameX, FrameY, 0.0, 0.0, FrameX, FrameY);
		C.Style = 1;
	}

	// Draw border.
	if ( BorderLines > 0 || BorderPixels > 0 )
	{
		YStart 	= BorderLines + ConsoleLines;
		YEnd 	= FrameY - BorderLines;
		if ( BorderLines > 0 )
		{
			C.SetOrigin(0.0, 0.0);
			C.SetPos(0.0, 0.0);
			C.DrawPattern( Border, FrameX, BorderLines, 1.0 );
			C.SetPos(0.0, YEnd);
			C.DrawPattern( Border, FrameX, BorderLines, 1.0 );
		}
		if ( BorderPixels > 0 )
		{
			C.SetOrigin(0.0, 0.0);
			C.SetPos(0.0, YStart);
			C.DrawPattern( Border, BorderPixels, YEnd - YStart, 1.0 );
			C.SetPos( FrameX - BorderPixels, YStart );
			C.DrawPattern( Border, BorderPixels, YEnd - YStart, 1.0 );
		}
	}

	// Draw console text.
	C.SetOrigin(0.0, 0.0);
	if ( ConsoleLines > 0 )
		DrawConsoleView( C );
	else
		DrawSingleView( C );
}

//-----------------------------------------------------------------------------
// state Typing
//
// State used while typing a command on the console.
// ----------------------------------------------------------------------------

state Typing
{
	function bool KeyEvent( EInputKey Key, EInputAction Action, FLOAT Delta )
	{
		local string Temp;

		if( Key==IK_Enter )
		{
			bNoStuff = false;

			if( Scrollback!=0 )
			{
				Scrollback=0;
			}
			else
			{
				if( TypedStr!="" )
				{
					// Print to console.
					if( ConsoleLines!=0 )
						Message( None, "(>" @ TypedStr, 'Console' );

					// Update history buffer.
					History[HistoryCur++ % MaxHistory] = TypedStr;
					if( HistoryCur > HistoryBot )
						HistoryBot++;
					if( HistoryCur - HistoryTop >= MaxHistory )
						HistoryTop = HistoryCur - MaxHistory + 1;

					// Make a local copy of the string.
					Temp=TypedStr;
					TypedStr="";

					//
					// Workaround for say/teamsay messages starting with ; not showing up.
					//
					while ( InStr(Temp," ")==0 ) // Remove leading spaces, this is always safe to do.
						Temp = Mid(Temp,1);

					if ( Left(Temp,4)~="say " )
					{
						Temp = Mid(Temp,4);
						while ( InStr(Temp," ")==0 )
							Temp = Mid(Temp,1);

						if ( Temp!="" )
							Viewport.Actor.Say( Temp );
					}
					else if ( Left(Temp,8)~="teamsay " )
					{
						Temp = Mid(Temp,8);
						while ( InStr(Temp," ")==0 )
							Temp = Mid(Temp,1);

						if ( Temp!="" )
							HXPlayerPawn(Viewport.Actor).TeamSay( Temp );
					}
					else if ( !ConsoleCommand(Temp) )
					{
						Message( None, Localize("Errors","Exec","Core"), 'Console' );
					}
					Message( None, "", 'Console' );
				}
				if( ConsoleDest==0.0 )
					GotoState('');
				Scrollback=0;
			}
			return true;
		}

		return Super.KeyEvent( Key, Action, Delta );
	}
}

// Phoneme debugging functions.
exec function PhonemeDemo( bool bEnable )
{
	local int Slot;
	bPhonemeDemo = bEnable;

	if ( bPhonemeDemo )
	{
		// Reset info.
		for ( Slot=0; Slot<ArrayCount(PhonemeInfos); Slot++ )
			ResetPhonemeSlot( Slot );
		Log( "PhonemeDemo is now on.", 'DevPhoneme' );
	}
	else
		Log( "PhonemeDemo is now off.", 'DevPhoneme' );
}

function PhonemeDemoRender( Canvas C )
{
	local int Slot;
	local float W, H, Y;
	local string Text;

	C.Font = TimeDemoFont;
	C.DrawColor.R = 255;
	C.DrawColor.G = 255;
	C.DrawColor.B = 255;

	Y = 0.3*C.ClipY;

	for ( Slot=0; Slot<ArrayCount(PhonemeInfos); Slot++ )
	{
		if ( PhonemeInfos[Slot].ActorName != '' )
		{
			if ( (PhonemeInfos[Slot].NextUpdateTime-PhonemeUpdateTime+PhonemeTimeout) > Viewport.Actor.Level.TimeSeconds )
			{
				// Draw Info.
				Text = PhonemeInfos[Slot].ActorName $ ": " $ PhonemeInfos[Slot].AccPhonemes;

				C.SetPos( 0, Y );
				C.TextSize( Text, W, H );
				C.DrawText( Text );
				Y += H;
			}
			else
			{
				// Timed out.
				LogPhonemeLine( Slot );
				ResetPhonemeSlot( Slot );
			}
		}
	}
}

function LogPhonemeLine( int Slot )
{
	Log( PhonemeInfos[Slot].ActorName $ ": " $ PhonemeInfos[Slot].AccPhonemes, 'DevPhoneme' );
}

function ResetPhonemeSlot( int Slot )
{
	PhonemeInfos[Slot].Actor          = None;
	PhonemeInfos[Slot].ActorName      = '';
	PhonemeInfos[Slot].CurrentPhoneme = "";
	PhonemeInfos[Slot].AccPhonemes    = "";
	PhonemeInfos[Slot].NextUpdateTime = Viewport.Actor.Level.TimeSeconds - 1.0;
	
}

function UpdatePhonemeSlot( int Slot, Actor Speaker, String Phoneme )
{
	PhonemeInfos[Slot].Actor     = Speaker;
	PhonemeInfos[Slot].ActorName = Speaker.Name;

	if ( PhonemeInfos[Slot].NextUpdateTime < Viewport.Actor.Level.TimeSeconds )
	{
		// Flush.
		if ( Len(PhonemeInfos[Slot].AccPhonemes) == PhonemesPerLine )
		{
			LogPhonemeLine( Slot );
			PhonemeInfos[Slot].AccPhonemes = "";
		}
		PhonemeInfos[Slot].CurrentPhoneme = Phoneme;
		PhonemeInfos[Slot].AccPhonemes    = PhonemeInfos[Slot].AccPhonemes $ Phoneme;
		PhonemeInfos[Slot].NextUpdateTime = Viewport.Actor.Level.TimeSeconds + PhonemeUpdateTime;
	}
}

function PhonemeNotify( Actor Speaker, String Phoneme )
{
	local int i, Slot, EmptySlot;

	if ( !bPhonemeDemo )
		return;

	//Log( "PhonemeNotify( " $ Speaker $ ", " $ Phoneme $ " )", 'DevPhoneme' );

	EmptySlot = -1;
	Slot      = -1;

	for ( i=0; i<ArrayCount(PhonemeInfos); i++ )
	{
		if ( PhonemeInfos[i].Actor == Speaker )
		{
			Slot = i;
			break;
		}
		else if ( PhonemeInfos[i].Actor == None && EmptySlot == -1 )
			EmptySlot = i;
	}

	// New Actor.
	if ( Slot == -1 )
	{
		if ( EmptySlot == -1 )
		{
			Log( "Warning: PhonemeInfo is full, rejecting " $ Speaker $ ".", 'DevPhoneme' );
			return;
		}
		Slot = EmptySlot;
		ResetPhonemeSlot( Slot );
	}

	// Now we have a valid slot, so update it.
	UpdatePhonemeSlot( Slot, Speaker, Phoneme );
}

// ----------------------------------------------------------------------------
// ----------------------------------------------------------------------------

defaultproperties
{
	ConBackground=Texture'HX.HXConsoleBack'
	PhonemesPerLine=64
	PhonemeTimeout=4.0
	PhonemeUpdateTime=0.1
	bPhonemeDemo=False
}
