//=============================================================================
// HXConversationTrigger.
//
// Triggers a conversation when touched.
//=============================================================================
class HXConversationTrigger extends HXTrigger;

var() Name   ConversationTag; // Matched to the conversation file which has all of the conversation events in it.
var() String BindName;        // !! Shadows Actor.BindName !!
var() Name   CheckFlag;
var() bool   bCheckFalse;

// ----------------------------------------------------------------------------
// Succeeds()
//
// Intended to be called by Mutator code if Self was spawned as a successor
// to Other. Super call recommended.
// ----------------------------------------------------------------------------

function Succeeds( Actor Other )
{
	local ConversationTrigger OtherConversationTrigger;

	// Call Super.
	Super.Succeeds( Other );

	// Special ConversationTrigger Init.
	OtherConversationTrigger = ConversationTrigger(Other);
	if ( OtherConversationTrigger!=None )
	{
		ConversationTag = OtherConversationTrigger.ConversationTag;
		BindName        = OtherConversationTrigger.BindName;
		CheckFlag       = OtherConversationTrigger.CheckFlag;
		bCheckFalse     = OtherConversationTrigger.bCheckFalse;
	}
}

// ----------------------------------------------------------------------------
// Touch()
// ----------------------------------------------------------------------------

singular function Touch( Actor Other )
{
	// Is relevant now contains all condition handling.
	if ( !IsRelevant(Other) )
		return;

	if ( StartConversation(HXPlayerPawn(Other)) )
		Super.Touch( Other );
}

// ----------------------------------------------------------------------------
// Trigger()
// ----------------------------------------------------------------------------

singular function Trigger( Actor Other, Pawn Instigator )
{
	local Actor Actor;

	// Is relevant now contains all condition handling.
	if ( !IsRelevant(Instigator) )
		return;

	if ( StartConversation(HXPlayerPawn(Instigator)) )
		Super.Trigger( Other, Instigator );
}

// ----------------------------------------------------------------------------
// IsRelevant()
//
// See whether the other actor is relevant to this trigger.
// ----------------------------------------------------------------------------

function bool IsRelevant( Actor Other )
{
	// BindName and ConversationTag need to be present to do anything.
	if ( BindName=="" || ConversationTag=='' )
		return false;

	// Only works for PlayerPawns.
	if ( !Other.bIsPawn || HXPlayerPawn(Other)==None )
		return false;

	// Trivial accept for no CheckFlag.
	if ( CheckFlag=='' )
		return true;
	
	// Otherwise see that Flag matches bCheckFalse setting.
	return Game.Steve.FlagBase.GetBool(CheckFlag)!=bCheckFalse;
}

// ----------------------------------------------------------------------------
// StartConversation()
// ----------------------------------------------------------------------------

function bool StartConversation( HXPlayerPawn PlayerPawn )
{
	local Actor Actor;

	foreach AllActors( class'Actor', Actor )
		if ( Actor.BindName==BindName )
			return Game.StartConversationByName( PlayerPawn, ConversationTag, Actor );

	return false;
}

// ----------------------------------------------------------------------------
// ----------------------------------------------------------------------------

defaultproperties
{
	bTriggerOnceOnly=True
	CollisionRadius=96.0
}
