//=============================================================================
// HXDataLinkTrigger.
//
// Triggers a DataLink event when touched
//
// * DatalinkTag is matched to the conversation file which has all of
//   the DataLink events in it.
//
// We might possibly need to monitor a flag and trigger a DataLink event
// when that flag changes during the mission.  This could be done in the
// individual mission's script file update loop.
//=============================================================================
class HXDataLinkTrigger extends HXTrigger;

var() Name DatalinkTag;
var() Name CheckFlag;
var() bool bCheckFalse;

var bool  bStartedViaTrigger;
var bool  bStartedViaTouch;
var bool  bCheckFlagTimer;
var float CheckFlagTimer;

var HXPlayerPawn Player;
var Actor TriggerOther;
var Pawn  TriggerInstigator;

// ----------------------------------------------------------------------------
// Succeeds()
//
// Intended to be called by Mutator code if Self was spawned as a successor
// to Other. Super call recommended.
// ----------------------------------------------------------------------------

function Succeeds( Actor Other )
{
	local DataLinkTrigger OtherDataLinkTrigger;

	// Call Super.
	Super.Succeeds( Other );

	// Special DataLinkTrigger Init.
	OtherDataLinkTrigger = DataLinkTrigger(Other);
	if ( OtherDataLinkTrigger!=None )
	{
		DatalinkTag = OtherDataLinkTrigger.DatalinkTag;
		CheckFlag   = OtherDataLinkTrigger.CheckFlag;
		bCheckFalse = OtherDataLinkTrigger.bCheckFalse;
	}
}

// ----------------------------------------------------------------------------
// Timer()
// ----------------------------------------------------------------------------

function Timer()
{
	if ( bCheckFlagTimer && EvaluateFlag() && Player!=None )
		Game.StartDataLinkTransmission( String(DatalinkTag), Self );
}

// ----------------------------------------------------------------------------
// Trigger()
// ----------------------------------------------------------------------------

singular function Trigger( Actor Other, Pawn Instigator )
{
	// Only set the Player if the Player isn't already set and the "bCheckFlagTimer" variable is false.
	if ( Player==None || (Player!=None && !bCheckFlagTimer) )
		Player = HXPlayerPawn(Instigator);

	// Only works for HXPlayerPawns.
	if ( Player==None )
		return;

	if ( EvaluateFlag() )
	{
		if ( Game.StartDataLinkTransmission(String(DatalinkTag),Self) )
		{
			bStartedViaTrigger = true;
			TriggerOther       = Other;
			TriggerInstigator  = Instigator;
		}
	}
	else if ( CheckFlag!='' )
	{
		bStartedViaTrigger = true;
		TriggerOther       = Other;
		TriggerInstigator  = Instigator;
	}
}

// ----------------------------------------------------------------------------
// Touch()
// ----------------------------------------------------------------------------

singular function Touch( Actor Other )
{
	// Only set the Player if the Player isn't already set and the "bCheckFlagTimer" variable is false.
	if ( Player==None || (Player!=None && !bCheckFlagTimer) )
		Player = HXPlayerPawn(Other);

	// Only works for HXPlayerPawns.
	if ( Player==None )
		return;

	if ( EvaluateFlag() )
	{
		if ( Game.StartDataLinkTransmission(String(DatalinkTag),Self) )
		{
			bStartedViaTouch = true;
			TriggerOther     = Other;
		}
	}
	else if ( CheckFlag!='' )
	{
		bStartedViaTouch   = True;
		TriggerOther       = Other;
	}
}

// ----------------------------------------------------------------------------
// UnTouch()
// 
// Used to monitor the state of the "CheckFlag" variable if it's set.
// This is so a Player can be sitting inside the radius of a trigger
// and if the "CheckFlag" suddenly is valid, the trigger will play.
// ----------------------------------------------------------------------------

function UnTouch( Actor Other )
{
	bCheckFlagTimer = false;

	Super.UnTouch( Other );
}

// ----------------------------------------------------------------------------
// EvaluateFlag()
// ----------------------------------------------------------------------------

function bool EvaluateFlag()
{
	local bool bSuccess;

	if ( CheckFlag!='' )
	{
		if ( Game!=None && Game.Steve.FlagBase!=None )
		{
			if ( !Game.Steve.FlagBase.GetBool(CheckFlag) )
				bSuccess = bCheckFalse;
			else
				bSuccess = !bCheckFalse;

			// Don't run to early.
			if ( !Player.bPostLoginRun ) 
				bSuccess = false;

			// If the flag check fails, then make sure the Tick() event 
			// is active so we can continue to check the flag while 
			// the Player is inside the radius of the trigger.
			if ( !bSuccess )
			{
				bCheckFlagTimer = true;
				SetTimer( CheckFlagTimer, false );
			}
		}
		else
		{
			bSuccess = false;
		}
	}
	else
	{
		// Don't run to early.
		if ( Player.bPostLoginRun )
		{
			bSuccess = true;
		}
		else
		{
			bSuccess = false;

			bCheckFlagTimer = true;
			SetTimer( CheckFlagTimer, false );
		}
	}

	return bSuccess;
}

// ----------------------------------------------------------------------------
// DatalinkFinished()
// ----------------------------------------------------------------------------

function DatalinkFinished()
{
	if ( bStartedViaTrigger )
		Super.Trigger( TriggerOther, TriggerInstigator );
	else if ( bStartedViaTouch )
		Super.Touch( TriggerOther );
}

// ----------------------------------------------------------------------------
// ----------------------------------------------------------------------------

defaultproperties
{
	CheckFlagTimer=1.0
	bTriggerOnceOnly=True
	CollisionRadius=96.0
}
