//=============================================================================
// HXDecoration.
//
// TODO:
//  * Work on Fly code.
//  * Predict just Rotation of Bump() code clientside and and use this for 
//    rendering when delta to Rotation is smaller then a threshold
//    (about 2x net resoluton for Rotation).
//  * bFloating should just get set for decorations which can actually float.
//    and not for all things (imagine valves placed unterwater).
//  * Figure out whether you can interact with flies in any form. If not
//    make them clientside only.
//=============================================================================
class HXDecoration extends DeusExDecoration
	native
	nativereplication
	abstract;

var transient HXGameInfo Game;
var Name PrecessorName;
var bool bSpawning;

//
// Additional Timers.
//
var float BurnTime; // Used by BurnTimer().
var float PushTime; // User by PushTimer().

//
// Intention is to just create these proxies on demand with a
// limited LifeSpan (about 100s for now) if needed and to reset
// the LifeSpan each time they are used. This avoid having plenty
// of actors which are not needed (e.g. RenderProxy is currently
// just needed if another Pawn carries the decoration to avoid it
// beeing translucent, just a few decorations smoke, etc.).
//
var transient Actor RenderProxy; // Used by render iterator to render decorations itsself if needed.
var transient Actor SmokeProxy;  // Intended for later use.

/*
// for destroying them
var() travel int HitPoints;
var() int minDamageThreshold;
var() bool bInvincible;
var() class<Fragment> fragType;

// information for floating/bobbing decorations
var() bool bFloating;
//var rotator origRot;

// lets us attach a decoration to a mover
var() name moverTag;

// object properties
var() bool bFlammable;				// can this object catch on fire?
var() float Flammability;			// how long does the object burn?
var() bool bExplosive;				// does this object explode when destroyed?
var() int explosionDamage;			// how much damage does the explosion cause?
var() float explosionRadius;		// how big is the explosion?

var() bool bHighlight;				// should this object not highlight when focused?

var() bool bCanBeBase;				// can an actor stand on this decoration?

var() bool bGenerateFlies;			// does this actor generate flies?

//var int pushSoundId;				// used to stop playing the push Sound

var int gradualHurtSteps;			// how many separate explosions for the staggered HurtRadius
var int gradualHurtCounter;			// which one are we currently doing

var name NextState;					// for queueing states
var name NextLabel;					// for queueing states
*/

var HXPawnGenerator FlyGenerator;

/*
var localized string itemArticle;	
var localized string itemName;		// human readable name !! Replicate this.

native(2101) final function ConBindEvents();
*/

var bool bIsContainer; // Whether to use Containers style behaviour in Destroyed().
var bool bIsHangingDecoration; // Whether this Actor is a HangingDecoration.
var bool bIsInformationDevice; // Whether this Actor is a InformationDevices.
var bool bIsHackableDevice;    // Whether this Actor is a HackableDevices.
var bool bIsSeat; // Whether this Actor is a Seat.
var bool bIsCart; // Whether this Actor is a Cart.

//
// Variables for Enter/LeaveWorld.
//
var() bool bInWorld;
//var Vector WorldPosition;
var Sound  WorldAmbientSound;

//
// Seat related variables. Moved here out of Seat.
//
var() Vector SitPoint[4];
var Actor SittingActor[4];
var int NumSitPoints;
var Vector InitialPosition;

var bool bClientHighlight;
var bool bContentsSpawned;
var bool bVanished;

//
// Enables special DesiredRotation replication (used for SatelliteDishes).
// 
// if ( Role==ROLE_Authority && (!bSpecialDesired || (bSpecialDesired && bNetInitial)) && !bCarriedItem && (DrawType==DT_Mesh || DrawType==DT_Brush) && (bNetInitial || bSimulatedPawn || RemoteRole<ROLE_SimulatedProxy) )
//   Rotation;
// if ( Role==ROLE_Authority && RemoteRole==ROLE_SimulatedProxy && (bNetInitial || bSpecialDesired) && Physics==PHYS_Rotating) )
//   bFixedRotationDir, bRotateToDesired, RotationRate, DesiredRotation;
//
var bool bSpecialDesired;

// Swimming related variables used by HXDecorationRenderIterator.
var float SwimPhi;   // Phase offset for floating to avoid syncronized swimming (Random value).
var float SwimAlpha; // Used to blend in swimming to remove stuttering on ZoneChanges (e.g. when deco is floating on surface).
var float SwimOmega; // Angular frequency for floating effect (Default: 2PI * 0.25Hz).
var Rotator SwimAmp; // Amplitude for swimming. (Default: (512,256,512)).

// String to Name conversion.
native static final function Name StringToName( coerce String Str );

// Matches Extension.Window() which actually just returns a line feed.
static final function String CarriageReturn()
{
	return Chr(10);
}

// Short form of CarriageReturn().
static final function String CR()
{
	return Chr(10);
}

//
// Fixed and extended version of TraceTexture() which also returns
// the Texture hit (Hint: There are Sound properties on Texture).
//
native(3340) final iterator function TraceTextures
(
	Class<Actor> BaseClass,
	out actor Actor,
	out Texture Tex,
	out name TexName,
	out name TexGroup,
	out int Flags,
	out vector HitLoc,
	out vector HitNorm,
	vector End,
	optional vector Start,
	optional vector Extent
);

//
// First Hit variant of TraceTextures.
// (The same as Trace is to TraceActors)
//
native(3341) final function Actor TraceSingleTexture
(
	Class<Actor> BaseClass,
	out Texture  Texture,
	out Name     TextureName,
	out Name     TextureGroup,
	out int      TextureFlags,
	out Vector   HitLocation,
	out Vector   HitNormal,
	Vector       TraceEnd,
	optional Vector TraceStart,
	optional bool bTraceActors,
	optional Vector TraceExtent
);

//
// Behaves the same FastTrace but returns the
// surface texture if a hit occured.
//
/*native(3342) final function bool FastTextureTrace
(
	Vector          TraceEnd,
	out Texture     Texture,
	out Name        TextureName,
	out Name        TextureGroup,
	out int         TextureFlags,
	optional Vector TraceStart
);*/

//
// Returns true if did not hit non transparent world geometry.
// This might not be exact as this might check a coplanar bsp surface.
// However most windows are on a different plane, so this works
// fine nearly all the time (expect for maybe ugly maps) so it would
// not be worth the additional performance hit.
//
// Code was kindly donated by Smirftsch (oldunreal.com).
//
native(3343) final function bool FastRepTrace
(
	vector          TraceEnd,
	optional vector TraceStart
);

//
// Stops the specified or all Sound slots for an Actor.
//
// These functions do work in network. However, ClientHearSound() is NOT
// reliable, so it could happen that the PlaySound() call reaches the client
// after the StopSoundId/StopSoundSlot/StopAllSoundSlots call. So try to avoid 
// starting the Sound and stopping it at the same or next few ticks and you
// are safe (the same applies to stop the Sound to just play another).
//
native(3244) final function StopSoundSlot( ESoundSlot Slot );
native(3245) final function StopAllSoundSlots();
native simulated event DemoStopSoundSlot( ESoundSlot Slot );
native simulated event DemoStopAllSoundSlots();

event BurnTimer(); // Used by state Burning.

// ----------------------------------------------------------------------------
// Network replication.
// ----------------------------------------------------------------------------

replication
{
	reliable if ( Role==ROLE_Authority )
		bClientHighlight;

	reliable if( bDemoRecording )
		DemoStopSoundSlot, DemoStopAllSoundSlots;
}

// ----------------------------------------------------------------------------
// Succeeds()
//
// Intended to be called by Mutator code if Self was spawned as a successor
// to Other. Super call recommended.
// ----------------------------------------------------------------------------

function Succeeds( Actor Other )
{
	local DeusExDecoration OtherDeusExDecoration;
	local Decoration OtherDecoration;

	// Save Precessors name aside for debugging purposes.
	PrecessorName        = Other.Name;

	// Collision and Blocking.
	bCollideWhenPlacing = Other.bCollideWhenPlacing; // Still used for far moving after spawn.
	bCollideWorld       = Other.bCollideWorld;

	SetCollisionSize( Other.CollisionRadius, Other.CollisionHeight );
	SetCollision( Other.bCollideActors, Other.bBlockActors, Other.bBlockPlayers );

	// Movement.
	Mass			= Other.Mass;
	Buoyancy	= Other.Buoyancy;
	AttachTag = Other.AttachTag;

	// Advanced (partial).
	bOwned  = Other.bOwned;
	bHidden = Other.bHidden;

	// Object.
	Group = Other.Group;

	// Events.
	Event = Other.Event;

	// Conversation -- Hopefully someday moved to some StationaryPawn class.
	BindName         = Other.BindName;
	BarkBindName     = Other.BarkBindName;
	FamiliarName     = Other.FamiliarName;
	UnfamiliarName   = Other.UnfamiliarName;
	ConStartInterval = Other.ConStartInterval;

	// Animation variables.
	//AnimSequence = Other.AnimSequence; // Animation sequence we're playing.
	//AnimFrame    = Other.AnimFrame;    // Current animation frame, 0.0 to 1.0.
	//AnimRate     = Other.AnimRate;     // Animation rate in frames per second, 0=none, negative=velocity scaled.
	//LODBias      = Other.LODBias;

	// Drawing effect.
	DrawType     = Other.DrawType;

	// Style for rendering sprites, meshes.
	Style				= Other.Style;

	// Other display properties.
	//Sprite			= Other.Sprite;
	//Texture			= Other.Texture;
	//Skin				= Other.Skin;
	//Mesh				= Other.Mesh;
	DrawScale		= Other.DrawScale;
	//ScaleGlow		= Other.ScaleGlow;   // There are some crappy ScaleGlow and
	//AmbientGlow	= Other.AmbientGlow; // AmbientGlow settings, which now look even more odd.
	Fatness			= Other.Fatness;

	// Display.
	bUnlit           = Other.bUnlit;
	bNoSmooth        = Other.bNoSmooth;
	bParticles       = Other.bParticles;
	bRandomFrame     = Other.bRandomFrame;
	bMeshEnviroMap   = Other.bMeshEnviroMap;
	//bMeshCurvy       = Other.bMeshCurvy;
	//VisibilityRadius = Other.VisibilityRadius
	//VisibilityHeight = Other.VisibilityHeight;

	// Not yet implemented.
	//bShadowCast = Other.bShadowCast;     // Casts shadows.

	// Multiple skin support.
	//for ( i = 0; i < ArrayCount(Other.MultiSkins); i++ )
		//MultiSkins[i] = Other.MultiSkins[i];

	// Lighting info.
	LightBrightness = Other.LightBrightness;
	LightHue        = Other.LightHue;
	LightSaturation = Other.LightSaturation;

	// Light modulation.
	LightType        = Other.LightType;

	// Spatial light effect to use.
	LightEffect      = Other.LightEffect;

	// Light properties.
	LightRadius      = Other.LightRadius;
	LightPeriod      = Other.LightPeriod;
	LightPhase       = Other.LightPhase;
	LightCone        = Other.LightCone;
	VolumeBrightness = Other.VolumeBrightness;
	VolumeRadius     = Other.VolumeRadius;
	VolumeFog        = Other.VolumeFog;

	// Lighting.
	bSpecialLit			 = Other.bSpecialLit;
	//bActorShadows		 = Other.bActorShadows;
	//bCorona					 = Other.bCorona;
	//bLensFlare       = Other.bLensFlare;

	// Ambient sound.
	SoundRadius	 = Other.SoundRadius;
	SoundVolume  = Other.SoundVolume;
	SoundPitch   = Other.SoundPitch;
	AmbientSound = Other.AmbientSound;

	// Regular sounds.
	//TransientSoundVolume = Other.TransientSoundVolume; // !! Not replicated.
	//TransientSoundRadius = Other.TransientSoundRadius; // !! Not replicated.

	// Set BindName to empty string.
	Other.BindName = "";

	// Prevent Destoyed() fire up events!
	Other.Event = '';
	Other.Group = '';

	// Maybe this helps a bit
	Other.AttachTag = '';

	// Special Decoration Init.
	OtherDecoration = Decoration(Other);
	if ( OtherDecoration!=None )
	{
		EffectWhenDestroyed	= OtherDecoration.EffectWhenDestroyed;
		bPushable						= OtherDecoration.bPushable;
		bOnlyTriggerable		= OtherDecoration.bOnlyTriggerable;
		PushSound						= OtherDecoration.PushSound;
		Contents						= OtherDecoration.Contents;
		Content2						= OtherDecoration.Content2;
		Content3						= OtherDecoration.Content3;
		EndPushSound				= OtherDecoration.EndPushSound;
	}

	// Special DeusExDecoration Init.
	OtherDeusExDecoration = DeusExDecoration(Other);
	if ( OtherDeusExDecoration!=None )
	{
		HitPoints						= OtherDeusExDecoration.HitPoints;
		MinDamageThreshold	= OtherDeusExDecoration.MinDamageThreshold;
		bInvincible					= OtherDeusExDecoration.bInvincible;

		// Just copy if non default.
		//if ( OtherDeusExDecoration.FragType!=OtherDeusExDecoration.Default.FragType )
			FragType = OtherDeusExDecoration.FragType;

		bFloating						= OtherDeusExDecoration.bFloating;
		MoverTag						= OtherDeusExDecoration.MoverTag;
		bFlammable					= OtherDeusExDecoration.bFlammable;
		Flammability				= OtherDeusExDecoration.Flammability;
		bExplosive					= OtherDeusExDecoration.bExplosive;
		ExplosionDamage			= OtherDeusExDecoration.ExplosionDamage;
		ExplosionRadius			= OtherDeusExDecoration.ExplosionRadius;
		bHighlight					= OtherDeusExDecoration.bHighlight;
		bCanBeBase					= OtherDeusExDecoration.bCanBeBase;
		bGenerateFlies			= OtherDeusExDecoration.bGenerateFlies;

		// Only copy non default ItemName. ItemName isn't replicated (yet). Copy ItemArticle along.
		if ( OtherDeusExDecoration.ItemName!=OtherDeusExDecoration.default.ItemName )
		{
			ItemName    = OtherDeusExDecoration.ItemName;
			ItemArticle = OtherDeusExDecoration.ItemArticle;
		}

		// quirks for simulated decorations
		//bClientGenerateFlies	= OtherDeusExDecoration.bGenerateFlies;
		bClientHighlight			= OtherDeusExDecoration.bHighlight;

		// Remove old MoverTag.
		OtherDeusExDecoration.MoverTag = ''; 

		// Prevent the old decorations from spawning flies.
		OtherDeusExDecoration.bGenerateFlies = false;
	}
}

// ----------------------------------------------------------------------------
// IsInRelevant() -- Whether code is currently running inside during
//                   spawn inside a mutator in which case the properties
//                   are not yet set properly.
// ----------------------------------------------------------------------------

simulated function bool IsInRelevant()
{
	if ( Level.Game==None ) // Will happen on client too.
		return False;
	return Level.Game.IsInState( 'InIsRelevant' );
}

// ----------------------------------------------------------------------------
// Spawned()
// ----------------------------------------------------------------------------

simulated event Spawned()
{
	Super.Spawned();

	if ( Level.bStartup )
		bGameRelevant = true;

	Game = HXGameInfo(Level.Game);
}

// ----------------------------------------------------------------------------
// PreBeginPlay()
// ----------------------------------------------------------------------------

function PreBeginPlay()
{
	if ( bDeleteMe || IsInRelevant() )
		return;

	// Handle autodestruction if desired. Moved this here because the
	// config properties won't have their correct value set when spawned inside
	// the mutator, so you can't replace them at first pass there.
	if ( !bGameRelevant && Level.NetMode!=NM_Client && !Level.Game.IsRelevant(Self) )
	{
		Destroy();
		return;
	}

	if ( Role==ROLE_Authority )
	{
		// Enforce HX classes?
		if ( EffectWhenDestroyed!=None )
			Game.ModifyActorClass( EffectWhenDestroyed );
		if ( Contents!=None )
			Game.ModifyInventoryClass( Contents );
		if ( Content2!=None )
			Game.ModifyInventoryClass( Content2 );
		if ( Content3!=None )
			Game.ModifyInventoryClass( Content3 );
	}

	// Report default item names as bug
	if ( ItemName==Class'HXDecoration'.Default.ItemName )
		Warn( Self $ " has default ItemName." );
}

// ----------------------------------------------------------------------------
// BeginPlay()
//
// if we are already floating, then set our ref points
// ----------------------------------------------------------------------------

function BeginPlay()
{
	local class<Actor> FlyGeneratorClass;
	local Mover Mover;

	// Bail out if alreadys destroyed (e.g. replaced by a mutator).
	if ( bDeleteMe )
		return;

	// Randomize swimming phase offset to avoid synchronized swimming.
	SwimPhi = SwimOmega * FRand();

	// Init SwimAlpha based on bFloating.
	if ( Role==ROLE_Authority )
	{
		if ( bFloating )
			SwimAlpha = 1.0;
		else
			SwimAlpha = 0.0;
	}

	// Code running inside mutator. Hence config properties are not yet valid.
	if ( IsInRelevant() )
	{
	}
	else
	{
		// Attach us to the mover that was tagged
		AttachToMover( MoverTag );
		ChoosePushSound();

		if ( Level.NetMode!=NM_Client && bGenerateFlies && FlyGenerator==None && FRand()<0.1 )
		{
			FlyGeneratorClass = class'FlyGenerator';
			Game.ModifyActorClass( FlyGeneratorClass );

			FlyGenerator = Spawn(class<HXPawnGenerator>(FlyGeneratorClass),,, Location, Rotation);
		}
	}
}

// ----------------------------------------------------------------------------
// PostBeginPlay()
// ----------------------------------------------------------------------------

function PostBeginPlay()
{
	// Bail out if alreadys destroyed (e.g. replaced by a mutator).
	if ( bDeleteMe )
		return;

	// Code running inside mutator. Hence config properties are not yet valid.
	if ( IsInRelevant() )
	{
	}
	else
	{
		if ( !bInWorld )
		{
			// tricky
			bInWorld = true;
			LeaveWorld();
		}
	}
}

// ----------------------------------------------------------------------------
// SetInitialState()
//
// Called after PostBeginPlay.
// ----------------------------------------------------------------------------

simulated event SetInitialState()
{
	if ( bDeleteMe || IsInRelevant() )
		return;

	if ( InitialState!='' )
		GotoState( InitialState );
	else
		GotoState( 'Auto' );

	bSpawning = false;
}

// ----------------------------------------------------------------------------
// PostPostBeginPlay()
// ----------------------------------------------------------------------------

simulated event PostPostBeginPlay()
{
	//local ConListItem ConListItem;

	// This may need to be reset, as a new GameInfo is spawned each time.
	Game = HXGameInfo(Level.Game);

	Super(Decoration).PostPostBeginPlay();

	// Bail out if alreadys destroyed (e.g. replaced by a mutator).
	if ( bDeleteMe )
		return;

	// Decoration can be client side, so be careful to not invoke ConBindEvents.
	if ( Level.NetMode!=NM_Client )
	{
		// Bind any conversation events to this Decoration.
		ConBindEvents();

		// Dump bound conversations.
		/*if ( ConListItem(ConListItems)!=None )
			Log( "ConListItems for" @ Self, 'DevConversation' );
		for ( ConListItem=ConListItem(ConListItems); ConListItem!=None; ConListItem=ConListItem.Next )
			Log( "  " $ ConListItem, 'DevConversation' );*/
	}
}

// ----------------------------------------------------------------------------
// PostNetBeginPlay()
// ----------------------------------------------------------------------------

simulated function PostNetBeginPlay()
{
	// Randomize swimming phase offset to avoid synchronized swimming.
	SwimPhi = SwimOmega * FRand();

	// Init SwimAlpha based on bFloating.
	if ( bFloating )
		SwimAlpha = 1.0;
	else
		SwimAlpha = 0.0;

	// Attach us to the mover that was tagged
	AttachToMover( MoverTag );
	//ChoosePushSound();
}

// ----------------------------------------------------------------------------
// FellOutOfWorld()
// ----------------------------------------------------------------------------

event FellOutOfWorld()
{
	if ( PrecessorName!='' )
	{
		Log( Sprintf("Precessor of falling out of world %s was %s.",Name,PrecessorName), 'FellOutOfWorld' );
	}

	SetPhysics( PHYS_None );
	Destroy();
}	

// ----------------------------------------------------------------------------
// ChoosePushSound()
// ----------------------------------------------------------------------------

simulated function ChoosePushSound()
{
	// HX_HAN: This should be set along with the fragtype.
	switch ( FragType )
	{
		case Class'GlassFragment':
		case Class'PaperFragment':
		case Class'PlasticFragment':
		case Class'Rockchip' :
			PushSound = Sound'PushPlastic';
			break;

		case Class'MetalFragment':
			PushSound = Sound'PushMetal';
			break;

		case Class'WoodFragment':
			PushSound = Sound'PushWood';
			break;

		default:
			break;
	}
}

// ----------------------------------------------------------------------------
// AttachToMover()
// ----------------------------------------------------------------------------

simulated function AttachToMover( Name AttachMoverTag )
{
	local Mover Mover;

	// Attach us to the mover that was tagged
	if ( AttachMoverTag!='' )
	{
		foreach AllActors( Class'Mover', Mover, AttachMoverTag )
		{
			SetBase( Mover );
			SetPhysics( PHYS_None );
			bInvincible = True;
			bCollideWorld = False;
		}
	}
}

// ----------------------------------------------------------------------------
// TravelPostAccept()
// ----------------------------------------------------------------------------

function TravelPostAccept()
{
}

// ----------------------------------------------------------------------------
// Landed()
// 
// Called when we hit the ground
// ----------------------------------------------------------------------------

function Landed( Vector HitNormal )
{
	local Rotator rot;
	local Sound HitSound;

	// make it lay flat on the ground
	bFixedRotationDir = False;
	Rot = Rotation;
	Rot.Pitch = 0;
	Rot.Roll  = 0;
	SetRotation( Rot );

	// Play a Sound effect if it's falling fast enough.
	if ( Velocity.Z<=-200.0  )
	{
		switch ( FragType )
		{
			case Class'WoodFragment':
				if ( Mass<=20.0 )
					HitSound = Sound'WoodHit1';
				else
					HitSound = Sound'WoodHit2';
				break;

			case Class'MetalFragment':
				if ( Mass<=20.0 )
					HitSound = Sound'MetalHit1';
				else
					HitSound = Sound'MetalHit2';
				break;

			case Class'PlasticFragment':
				if ( Mass<=20.0 )
					HitSound = Sound'PlasticHit1';
				else
					HitSound = Sound'PlasticHit2';
				break;

			case Class'GlassFragment':
				if ( Mass<=20.0 )
					HitSound = Sound'GlassHit1';
				else
					HitSound = Sound'GlassHit2';
				break;

			default:
				if ( Mass<=20.0 )
					HitSound = Sound'PaperHit1';
				else
					HitSound = Sound'PaperHit2';
				break;
		}

		if ( HitSound!=None )
			PlaySound( HitSound, SLOT_None );

		// Alert NPCs that I've landed.
		AISendEvent( 'LoudNoise', EAITYPE_Audio ); // Does this get cleared?
	}

	bWasCarried = False;
	bBobbing    = False;

	if ( /*Role==ROLE_Authority &&*/ ((bExplosive && VSize(Velocity)>425.0) || (!bExplosive && Velocity.Z<-500.0)) )
		TakeDamage( (1-Velocity.Z/30), Instigator, Location, Vect(0,0,0), 'Fell' );
}

// ----------------------------------------------------------------------------
// SupportActor()
//
// Called when somebody lands on us
// ----------------------------------------------------------------------------

singular function SupportActor( Actor StandingActor )
{
	local Vector NewVelocity;
	local float Angle, ZVelocity;
	local float BaseMass, StandingMass;
	local float Damage;

	ZVelocity = StandingActor.Velocity.Z;

	// We've been stomped!
	if ( ZVelocity<-500.0 )
	{
		StandingMass = FMax( 1.0, StandingActor.Mass );
		BaseMass     = FMax( 1.0, Mass);
		Damage       = 1.0 - StandingMass/BaseMass * ZVelocity/30.0;

		TakeDamage( Damage, StandingActor.Instigator, StandingActor.Location, 0.2*StandingActor.Velocity, 'Stomped' );
	}

	// Toss StandingActor around.
	if ( !bCanBeBase )
	{
		Angle = FRand()*Pi*2;
		NewVelocity.X = Cos( Angle );
		NewVelocity.Y = Sin( Angle );
		NewVelocity.Z = 0;
		NewVelocity  *= FRand()*25.0 + 25.0;
		NewVelocity  += StandingActor.Velocity;
		NewVelocity.Z = 50.0;
		StandingActor.Velocity = NewVelocity;
		StandingActor.SetPhysics( PHYS_Falling );
	}
	else
	{
		StandingActor.SetBase( Self );
	}
}

// ----------------------------------------------------------------------------
// ResetScaleGlow()
//
// Reset the ScaleGlow to the default
// Decorations modify ScaleGlow using damage
// ----------------------------------------------------------------------------

function ResetScaleGlow()
{
	if ( !bInvincible )
		ScaleGlow = float(HitPoints) / float(Default.HitPoints) * 0.9 + 0.1;
}

// ----------------------------------------------------------------------------
// BaseChange()
//
// Ripped out most of the code from the original BaseChange; the equivalent
// functionality has been moved to Landed() and SupportActor()
// ----------------------------------------------------------------------------

singular function BaseChange()
{
	//Log( "BaseChange(), bStartup="$Level.bStartup, Name );

	bBobbing = False;

	if( Base==None && (bPushable/* || IsA('Carcass')*/) && Physics==PHYS_None )
		SetPhysics( PHYS_Falling );

	// Make sure if a decoration is accidentally dropped we reset it's parameters correctly.
	// HX_HAN: Removed this hack, fucks up replacement CreateExplosiveSmall in 04_NYC_Hotel
	//SetCollision( Default.bCollideActors, Default.bBlockActors, Default.bBlockPlayers );
	//Style  = Default.Style;
	//bUnlit = Default.bUnlit;
	//ResetScaleGlow();
}

// ----------------------------------------------------------------------------
// Tick()
// ----------------------------------------------------------------------------

simulated function Tick( float DeltaTime )
{
	local float VelocitySize;
	local Pawn Pawn;

	// Authority.
	if ( Role==ROLE_Authority )
	{
		// Update scale glow.
		ResetScaleGlow();

		// Scale NetPriority with SwimAlpha.
		NetPriority = Default.NetPriority * (1.0 + Max(0.0,SwimAlpha));

		// BOOGER!  This is a hack!
		// Ideally, we'd set the base of the fly generator to this decoration,
		// but unfortunately this prevents the player from picking up the
		// decoration... need to fix!
		// HX_HAN: Work on it.
		if ( FlyGenerator!=None )
		{
			if ( FlyGenerator.Location!=Location )
				FlyGenerator.SetLocation( Location );
			if ( FlyGenerator.Rotation!=Rotation )
				FlyGenerator.SetRotation( Rotation );
		}

		// If we have any conversations, check to see if we're close enough
		// to the player to start one (and all the other checks that take place
		// when a valid conversation can be started);
		//
		// Notes:
		//  * Assume ScriptedPawns.bDoLowPriority distance as near enough
		//  * Maybe do store distances on decorations, but not much decorations
		//    have conversations, so the additional IsA() would likely be slower.
		if ( bInWorld && ConListItems!=None && Game!=None )
			for( Pawn=Level.PawnList; Pawn!=None; Pawn=Pawn.NextPawn )
				if ( Pawn.bIsPlayer && VSize(Pawn.Location-Location)<1200 )
					Game.StartConversation( HXPlayerPawn(Pawn), Self, IM_Radius );

		// This seems to be a hackfix for PHYS_Rolling (gets set when you bump them) to prevent
		// them from moving further as vector components get rounded to integers and it looks like
		// they make some random movement on occasion on clients. A better way might be to reset
		// Physics to PHYS_Falling or sth, but I'll keep it this way for now, as resetting physics
		// might cause other problems instead.
		if ( Level.Netmode!=NM_Standalone )
		{
			VelocitySize = VSize(Velocity);
			if ( VelocitySize>0 && VelocitySize<5 )
				Velocity *= 0;
		}
	}
	// Not Authority.
	//else
	//{
	//}
}

// ----------------------------------------------------------------------------
// ZoneChange()
// this decoration will now float with cool bobbing if it is
// buoyant enough
// ----------------------------------------------------------------------------

singular function ZoneChange( ZoneInfo NewZone )
{
	local float SplashSize;
	local Actor Splash;

	if ( NewZone.bWaterZone )
	{
		if( bSplash && !Region.Zone.bWaterZone && Mass<=Buoyancy 
			&& ((Abs(Velocity.Z) < 100) || (Mass == 0)) && (FRand() < 0.05) && !PlayerCanSeeMe() ) // !! PlayerCanSeeMe() checks all listed Pawns.
		{
			bSplash = false;
			SetPhysics(PHYS_None);
		}
		else if( !Region.Zone.bWaterZone && (Velocity.Z < -200) )
		{
			// Else play a splash.
			SplashSize = FClamp(0.0001 * Mass * (250 - 0.5 * FMax(-600,Velocity.Z)), 1.0, 3.0 );
			if( NewZone.EntrySound != None )
				PlaySound( NewZone.EntrySound, SLOT_Interact, SplashSize );
			if( NewZone.EntryActor!=None )
			{
				Splash = Spawn( NewZone.EntryActor ); 
				if ( Splash!=None )
					Splash.DrawScale = SplashSize;
			}
		}
		bSplash = True;
	}
	else if ( Region.Zone.bWaterZone && (Buoyancy > Mass) )
	{
		//bBobbing = true;      // removed (looked stupid) - DEUS_EX CNN
		if( Buoyancy > 1.1 * Mass )
			Buoyancy = 0.95 * Buoyancy; // waterlog
		else if( Buoyancy > 1.03 * Mass )
			Buoyancy = 0.99 * Buoyancy;
	}

	if( NewZone.bPainZone && NewZone.DamagePerSec>0 )
		TakeDamage( 100, None, Location, Vect(0,0,0), NewZone.DamageType );

	if ( bFloating && !NewZone.bWaterZone )
	{
		bFloating = False;
		//Velocity = Vect(0,0,0); // Try this.
		return;
	}

	if ( NewZone.bWaterZone )
		ExtinguishFire();

	if ( NewZone.bWaterZone && !bFloating && Buoyancy>Mass )
	{
		bFloating = True;
	}
}

// ----------------------------------------------------------------------------
// Bump()
// copied from Engine\Classes\Decoration.uc
// modified so we can have strength modify what you can push
// ----------------------------------------------------------------------------

simulated function Bump( Actor Other )
{
	local HXScriptedPawn OtherScriptedPawn;
	local HXDecoration OtherDecoration;
	local HXPlayerPawn OtherPlayer;
	local Rotator Rot;
	local float MaxPush, VelScale;
	local int AugLevel, AugMult;

	/*if ( Role<ROLE_Authority )
		Log( Self $ "Bump()" );*/

	if ( Other!=None )
	{
		OtherPlayer = HXPlayerPawn(Other);
		if ( OtherPlayer==None )
		{
			OtherScriptedPawn = HXScriptedPawn(Other);
			if ( OtherScriptedPawn==None )
				OtherDecoration = HXDecoration(Other);
		}
	}

	// HX_NOTE: only for own player bOnFire is replicated
	if ( ROLE==ROLE_Authority  && bFlammable && !Region.Zone.bWaterZone )
	{
		// If we are bumped by a burning player, then set us on fire.
		if ( OtherPlayer!=None )
		{
			if ( OtherPlayer.bOnFire )
				GotoState('Burning');
		}
		// If we are bumped by a burning scriptedpawn, then set us on fire.
		else if ( OtherScriptedPawn!=None )
		{
			if ( OtherScriptedPawn.bOnFire && !OtherScriptedPawn.bIsRobot )
				GotoState('Burning');
		}
		// If we are bumped by a burning decoration, then set us on fire.
		else if ( OtherDecoration!=None )
		{
			// HX_HAN: Really check for OtherDecoration.bFlammable?
			if ( OtherDecoration.bFlammable && OtherDecoration.IsInState('Burning') )
				GotoState('Burning');
		}
	}

	// Check to see if the actor touched is the Player Character
	if ( OtherPlayer!=None )
	{
		// If we are being carried, ignore Bump().
		// HX_HAN: CarriedDecoration of other players gets replicated now.
		if ( OtherPlayer.CarriedDecoration==Self )
			return;
	}

	// Why does this run on client?
	if ( bPushable && OtherPlayer!=None && OtherPlayer.Mass>40 /*&& Physics!=PHYS_Falling*/ )
	{
		// A little bit of a hack...
		// Make sure this decoration isn't being bumped from above or below
		if (abs(Location.Z-Other.Location.Z) < (CollisionHeight+Other.CollisionHeight-1))
		{
			//Log( Self $ ".Bump() OtherPlayer.AugMuscleClassLevel = " $ OtherPlayer.AugMuscleClassLevel );

			MaxPush  = 100;
			AugMult  = 1;
			AugLevel = OtherPlayer.AugMuscleClassLevel;
			
			if ( AugLevel>=0 )
				AugMult = AugLevel+2;
			MaxPush *= AugMult;

			if ( Mass<=MaxPush )
			{
				// slow it down based on how heavy it is and what level my augmentation is
				velscale = FClamp((50.0 * augMult) / Mass, 0.0, 1.0);
				if (velscale < 0.25)
					velscale = 0;

				// Apply more velocity than normal if we're floating.
				// HX_HAN: This might be too much.
				if ( bFloating )
					Velocity = Other.Velocity;
				else
					Velocity = Other.Velocity * velscale;

				if ( Physics!=PHYS_Falling )
					Velocity.Z = 0;

				if ( ROLE==ROLE_Authority )
					if (!bFloating && !bPushSoundPlaying && (Mass > 15))
						StartPushSound( VelScale );

				if ( !bFloating && Physics!=PHYS_Falling )
					SetPhysics( PHYS_Rolling );

				PushTime = 0.2;
				Instigator = Pawn(Other);

				// Impart angular velocity (yaw only) based on where we are bumped from
				// NOTE: This is faked, but it looks cool
				// HX_HAN: This is annoying.
				rot = Rotator((Other.Location - Location) << Rotation);
				rot.Pitch = 0;
				rot.Roll = 0;

				// ignore which side we're pushing from
				if (rot.Yaw >= 24576)
					rot.Yaw -= 32768;
				else if (rot.Yaw >= 8192)
					rot.Yaw -= 16384;
				else if (rot.Yaw <= -24576)
					rot.Yaw += 32768;
				else if (rot.Yaw <= -8192)
					rot.Yaw += 16384;

				// scale it down based on mass and apply the new "rotational force"
				rot.Yaw *= velscale * 0.025;
				SetRotation(Rotation+rot);
			}
		}
	}
}

// ----------------------------------------------------------------------------
// PushTimer()
//
// Shuts off the looping push Sound
// ----------------------------------------------------------------------------

event PushTimer()
{
	StopPushSound();
}

// ----------------------------------------------------------------------------
// DropThings()
// 
// drops everything that is based on this decoration
// ----------------------------------------------------------------------------

function DropThings()
{
	local Actor Actor;

	// HX_NOTE: maybe check for other decoration on top of us!

	// drop everything that is on us
	foreach BasedActors( Class'Actor', Actor )
		if ( !Actor.IsA('ParticleGenerator') )
			Actor.SetPhysics( PHYS_Falling );
}

// ----------------------------------------------------------------------------
// EnterConversationState()
// ----------------------------------------------------------------------------

function EnterConversationState( bool bFirstPerson, optional bool bAvoidState )
{
	// First check to see if we're already in a conversation state, 
	// in which case we'll abort immediately
	if ( GetStateName()=='Conversation' || GetStateName()=='FirstPersonConversation' )
		return;

	NextState = GetStateName();

	// If bAvoidState is set, then we don't want to jump into the conversaton state because bad things might happen otherwise.
	if ( !bAvoidState )
	{
		if ( bFirstPerson )
			GotoState( 'FirstPersonConversation' );
		else
			GotoState( 'Conversation' );
	}
}

// ----------------------------------------------------------------------------
// EndConversation()
// ----------------------------------------------------------------------------

function EndConversation()
{
	LastConEndTime = Level.TimeSeconds;

	Enable( 'Bump' );

	GotoState( NextState );
}

// ----------------------------------------------------------------------------
// state Conversation 
//
// Just sit here until the conversation is over
// ----------------------------------------------------------------------------

state Conversation
{
	ignores Bump, Frob;

Idle:
	Sleep(1.0);
	goto('Idle');

Begin:

	// Make sure we're not on fire!
	ExtinguishFire();

	goto('Idle');
}

// ----------------------------------------------------------------------------
// state FirstPersonConversation 
//
// Just sit here until the conversation is over
// ----------------------------------------------------------------------------

state FirstPersonConversation
{
	ignores Bump, Frob;

Idle:
	Sleep(1.0);
	goto('Idle');

Begin:
	goto('Idle');
}

// ----------------------------------------------------------------------------
// Explode()
// Blow it up real good!
// ----------------------------------------------------------------------------

function Explode(vector HitLocation)
{
	local ShockRing ring;
	local ScorchMark s;
	local ExplosionLight light;
	local int i;

	// make sure we wake up when taking damage
	bStasis = False;

	// Just in case anyone think it makes sense that an explosive
	// crate or sth. generates flies.
	if ( FlyGenerator!=None )
	{
		FlyGenerator.StopGenerator();
		FlyGenerator = None;
	}

	// alert NPCs that I'm exploding
	AISendEvent('LoudNoise', EAITYPE_Audio, , explosionRadius * 16);

	if (explosionRadius <= 128)
		PlaySound(Sound'SmallExplosion1', SLOT_None,,, explosionRadius*16);
	else
		PlaySound(Sound'LargeExplosion1', SLOT_None,,, explosionRadius*16);

	// draw a pretty explosion
	light = Spawn(class'ExplosionLight',,, HitLocation);
	if (explosionRadius < 128)
	{
		Spawn(class'ExplosionSmall',,, HitLocation);
		light.size = 2;
	}
	else if (explosionRadius < 256)
	{
		Spawn(class'ExplosionMedium',,, HitLocation);
		light.size = 4;
	}
	else
	{
		Spawn(class'ExplosionLarge',,, HitLocation);
		light.size = 8;
	}

	// draw a pretty shock ring
	ring = Spawn(class'ShockRing',,, HitLocation, rot(16384,0,0));
	if (ring != None)
		ring.size = explosionRadius / 32.0;
	ring = Spawn(class'ShockRing',,, HitLocation, rot(0,0,0));
	if (ring != None)
		ring.size = explosionRadius / 32.0;
	ring = Spawn(class'ShockRing',,, HitLocation, rot(0,16384,0));
	if (ring != None)
		ring.size = explosionRadius / 32.0;

	// spawn a mark
	s = spawn(class'ScorchMark', Base,, Location-vect(0,0,1)*CollisionHeight, Rotation+rot(16384,0,0));
	if (s != None)
	{
		s.DrawScale = FClamp(explosionDamage/30, 0.1, 3.0);
		s.ReattachDecal();
	}

	// spawn some rocks
	for (i=0; i<explosionDamage/30+1; i++)
		if (FRand() < 0.8)
			spawn(class'Rockchip',,,HitLocation);

	GotoState('Exploding');

	SpawnContents();
}

// ----------------------------------------------------------------------------
// Exploding state
// ----------------------------------------------------------------------------

state Exploding
{
	ignores Explode;

	function Timer()
	{
		local Pawn apawn;
		local float damageRadius;
		local Vector dist;

		if ( Level.NetMode != NM_Standalone )
		{
			damageRadius = (explosionRadius / gradualHurtSteps) * gradualHurtCounter;

			for ( apawn = Level.PawnList; apawn != None; apawn = apawn.nextPawn )
			{
				if ( apawn.IsA('HXPlayerPawn') )
				{
					dist = apawn.Location - Location;
					if ( VSize(dist) < damageRadius )
						HXPlayerPawn(apawn).myProjKiller = Self;
				}
			}
		}
		HurtRadius
		(
			(2 * explosionDamage) / gradualHurtSteps,
			(explosionRadius / gradualHurtSteps) * gradualHurtCounter,
			'Exploded',
			(explosionDamage / gradualHurtSteps) * 100,
			Location
		);
		if (++gradualHurtCounter >= gradualHurtSteps)
		{
			Destroy();
		}
	}

Begin:
	// stagger the HurtRadius outward using Timer()
	// do five separate blast rings increasing in size
	gradualHurtCounter = 1;
	gradualHurtSteps = 5;
	bHidden = True;
	SetCollision(False, False, False);
	SetTimer(0.5/float(gradualHurtSteps), True);
}


// ----------------------------------------------------------------------------
// this is our normal, just sitting there state
// ----------------------------------------------------------------------------

auto state Active
{
	function TakeDamage(int Damage, Pawn EventInstigator, vector HitLocation, vector Momentum, name DamageType)
	{
		local float avg;

		if (bStatic || bInvincible)
			return;

		if ((DamageType == 'TearGas') || (DamageType == 'PoisonGas') || (DamageType == 'Radiation'))
			return;

		if ((DamageType == 'EMP') || (DamageType == 'NanoVirus') || (DamageType == 'Shocked'))
			return;

		if (DamageType == 'HalonGas')
			ExtinguishFire();

		if ((DamageType == 'Burned') || (DamageType == 'Flamed'))
		{
			if (bExplosive)		// blow up if we are hit by fire
				HitPoints = 0;
			else if (bFlammable && !Region.Zone.bWaterZone)
			{
				GotoState('Burning');
				return;
			}
		}

		if (Damage >= minDamageThreshold)
			HitPoints -= Damage;
		else
		{
			// sabot damage at 50%
			// explosion damage at 25%
			if (damageType == 'Sabot')
				HitPoints -= Damage * 0.5;
			else if (damageType == 'Exploded')
				HitPoints -= Damage * 0.25;
		}

		if (HitPoints > 0)		// darken it to show damage (from 1.0 to 0.1 - don't go completely black)
		{
			ResetScaleGlow();
		}
		else	// destroy it!
		{
			DropThings();

			// clear the event to keep Destroyed() from triggering the event
			Event = '';
			avg = (CollisionRadius + CollisionHeight) / 2;
			Instigator = EventInstigator;
			if (Instigator != None)
				MakeNoise(1.0);

			if (fragType == class'WoodFragment')
			{
				if (avg > 20)
					PlaySound(Sound'WoodBreakLarge', SLOT_Misc,,, 512);
				else
					PlaySound(Sound'WoodBreakSmall', SLOT_Misc,,, 512);
				AISendEvent('LoudNoise', EAITYPE_Audio, , 512);
			}

			// if we have been blown up, then destroy our contents
			// CNN - don't destroy contents now
//			if (DamageType == 'Exploded')
//			{
//				Contents = None;
//				Content2 = None;
//				Content3 = None;
//			}

			if (bExplosive)
			{
				Frag(fragType, Momentum * explosionRadius / 4, avg/20.0, avg/5 + 1);
				Explode(HitLocation);
			}
			else
				Frag(fragType, Momentum / 10, avg/20.0, avg/5 + 1);
		}
	}
}

// ----------------------------------------------------------------------------
// ExtinguishFire()
// ----------------------------------------------------------------------------

function ExtinguishFire()
{
	local Fire f;

	if (IsInState('Burning'))
	{
		foreach BasedActors(class'Fire', f)
			f.Destroy();

		GotoState('Active');
	}
}

// ----------------------------------------------------------------------------
// state Burning
//
// We are burning and slowly taking damage
// ----------------------------------------------------------------------------

state Burning
{
	function TakeDamage(int Damage, Pawn EventInstigator, vector HitLocation, vector Momentum, name DamageType)
	{
		local float avg;

		if ((DamageType == 'TearGas') || (DamageType == 'PoisonGas') || (DamageType == 'Radiation'))
			return;

		if ((DamageType == 'EMP') || (DamageType == 'NanoVirus') || (DamageType == 'Shocked'))
			return;

		if (DamageType == 'HalonGas')
			ExtinguishFire();

		// if we are already burning, we can't take any more damage
		if ((DamageType == 'Burned') || (DamageType == 'Flamed'))
		{
			HitPoints -= Damage/2;
		}
		else
		{
			if (Damage >= minDamageThreshold)
				HitPoints -= Damage;
		}

		if (bExplosive)
			HitPoints = 0;

		if (HitPoints > 0)		// darken it to show damage (from 1.0 to 0.1 - don't go completely black)
		{
			ResetScaleGlow();
		}
		else	// destroy it!
		{
			DropThings();

			// clear the event to keep Destroyed() from triggering the event
			Event = '';
			avg = (CollisionRadius + CollisionHeight) / 2;
			Frag(fragType, Momentum / 10, avg/20.0, avg/5 + 1);
			Instigator = EventInstigator;
			if (Instigator != None)
				MakeNoise(1.0);

			// if we have been blown up, then destroy our contents
			if (bExplosive)
			{
				Contents = None;
				Content2 = None;
				Content3 = None;
				Explode(HitLocation);
			}
		}
	}

	// continually burn and do damage
	function BurnTimer()
	{
		TakeDamage( 2, None, Location, vect(0,0,0), 'Burned' );

		// Reset Timer.
		BurnTime = 1.0;
	}

	function BeginState()
	{
		local Fire f;
		local int i;
		local vector loc;

		for (i=0; i<8; i++)
		{
			loc.X = 0.9*CollisionRadius * (1.0-2.0*FRand());
			loc.Y = 0.9*CollisionRadius * (1.0-2.0*FRand());
			loc.Z = 0.9*CollisionHeight * (1.0-2.0*FRand());
			loc += Location;
			f = Spawn(class'Fire', Self,, loc);
			if (f != None)
			{
				f.DrawScale = FRand() + 1.0;
				f.LifeSpan = Flammability;

				// turn off the Sound and lights for all but the first one
				if (i > 0)
				{
					f.AmbientSound = None;
					f.LightType = LT_None;
				}

				// turn on/off extra fire and smoke
				if (FRand() < 0.5)
					f.smokeGen.Destroy();
				if (FRand() < 0.5)
					f.AddFire(1.5);
			}
		}

		// Set the burn timer.
		BurnTime = 1.0;
	}
}

// ----------------------------------------------------------------------------
// Frob()
//
// If we are frobbed, try to invoke conversation, if if failed
// trigger our event.
// ----------------------------------------------------------------------------

function Frob( Actor Frobber, Inventory FrobWith )
{
	local HXPlayerPawn PlayerFrobber;
	local Actor Actor;

	PlayerFrobber = HXPlayerPawn(Frobber);

	// First check to see if there's a conversation associated with this 
	// decoration.  If so, trigger the conversation instead of triggering
	// the event for this decoration
	if ( PlayerFrobber!=None && !bIsInformationDevice )
		if ( Game.StartConversation( PlayerFrobber, Self, IM_Frob ) )
			return;

	// Trigger event if we aren't hackable
	if ( Event!='' && !IsA('HXHackableDevices') )
		foreach AllActors( Class 'Actor', Actor, Event )
			Actor.Trigger( Self, Pawn(Frobber) );
}

// ----------------------------------------------------------------------------
// Frag()
//
// copied from Engine.Decoration
// modified so fragments will smoke	and use the skin of their parent object
// ----------------------------------------------------------------------------

function Frag(class<fragment> FragType, vector Momentum, float DSize, int NumFrags) 
{
	local int i;
	local actor A, Toucher;
	local DeusExFragment s;

	if ( bOnlyTriggerable )
		return; 
	if (Event!='')
		foreach AllActors( class 'Actor', A, Event )
			A.Trigger( Toucher, pawn(Toucher) );

	SpawnContents();

	if ( Region.Zone.bDestructive )
	{
		Destroy();
		return;
	}
	for (i=0 ; i<NumFrags ; i++) 
	{
		s = DeusExFragment(Spawn(FragType, Owner));
		if (s != None)
		{
			s.Instigator = Instigator;
			s.CalcVelocity(Momentum,0);
			s.DrawScale = DSize*0.5+0.7*DSize*FRand();
			s.Skin = GetMeshTexture();
			if (bExplosive)
				s.bSmoking = True;
		}
	}

	if (!bExplosive)
		Destroy();
}

// ----------------------------------------------------------------------------
// Vanish()
//
// Intended to be called if Actor should just disappear without leaving
// any trace (e.g. setting flags, trigger, spawn rats, etc.).
// ----------------------------------------------------------------------------

simulated function Vanish()
{
	bVanished = true;
	Destroy();
}

// ----------------------------------------------------------------------------
// Destroyed()
// ----------------------------------------------------------------------------

simulated function Destroyed()
{
	local Actor Actor;

	if ( RenderProxy!=None )
	{
		RenderProxy.Destroy();
		RenderProxy = None;
	}
	if ( SmokeProxy!=None )
	{
		SmokeProxy.Destroy();
		SmokeProxy = None;
	}

	if ( Role==ROLE_Authority )
	{
		if ( FlyGenerator!=None )
		{
			if ( !Level.bStartup )
				FlyGenerator.Burst(); // Burst should only happen if crushed.
			FlyGenerator.StopGenerator(); // This doesn't destroy it directly it seems. FIX-ME.
			FlyGenerator = None;
		}

		StopPushSound();

		// Pass a message to conPlay, if it exists in the player, that 
		// this object has been destroyed.  This is used to prevent 
		// bad things from happening in converseations.
		if ( Level.NetMode!=NM_Client && Level.Game.IsA('HXGameInfo') )
			if ( Game.ConPlay!=None )
				Game.ConPlay.ActorDestroyed(Self);

		// Containers won't trigger (Why?).
		if ( !bIsContainer )
			if( Event!='' && !Level.bStartup && !bVanished )
				foreach AllActors( Class 'Actor', Actor, Event )
					Actor.Trigger( Self, None );
	}
}

// ----------------------------------------------------------------------------
// Trigger()
//
// If we are triggered and explosive, then explode.
// ----------------------------------------------------------------------------

function Trigger( Actor Other, Pawn EventInstigator )
{
	if ( bExplosive )
	{
		Explode( Location );
		Instigator = EventInstigator;
		TakeDamage( 1000, EventInstigator, Location, Vect(0,0,1)*900, 'Exploded' );
	}
}

// ----------------------------------------------------------------------------
// SpawnContents()
//
// Overriden by Containers.
// ----------------------------------------------------------------------------

function SpawnContents()
{
	local Actor Dropped;
	local Class<Inventory> TempClass;

	if( !bContentsSpawned && Contents!=None && !Level.bStartup )
	{
		TempClass = Contents;
		if ( Content2!=None && FRand()<0.3 )
			TempClass = Content2;
		if ( Content3!=None && FRand()<0.3 )
			TempClass = Content3;

		Dropped = Spawn( TempClass );
		Dropped.RemoteRole = ROLE_DumbProxy;
		Dropped.SetPhysics( PHYS_Falling );
		Dropped.bCollideWorld = True;
		//if ( Inventory(Dropped)!=None )
			//Inventory(Dropped).GotoState( 'Pickup', 'Dropped' );
		Dropped.GotoState( 'Pickup', 'Dropped' );
	}

	bContentsSpawned = true;
}

// ----------------------------------------------------------------------------
// StartPushSound()
//
// Notes:
//  * I increased sound and event radius, but turned down volume.
//  * Amount is intented for velscale passed in, to scale the event and
//    sound based on how fast we push.
//  * Maybe add ability to change the pushing sound/event scale, but for
//    now I don't know if stopping the event and firing it again would
//    not cause issues.
// ----------------------------------------------------------------------------

function StartPushSound( float Amount )
{
	local float ClampedAmount;

	if ( !bPushSoundPlaying )
	{
		ClampedAmount = FClamp( Amount, 0.25, 1.00 ) * 0.66;

		PlaySound( PushSound, SLOT_Misc, ClampedAmount,, 384 );
		AIStartEvent( 'LoudNoise', EAITYPE_Audio, ClampedAmount, 384 );
		bPushSoundPlaying = True;
	}
}

// ----------------------------------------------------------------------------
// StopPushSound()
//
// Note: Maybe reset Instigator.
// ----------------------------------------------------------------------------

function StopPushSound()
{
	if ( bPushSoundPlaying )
	{
		StopSoundSlot( SLOT_Misc );
		AIEndEvent( 'LoudNoise', EAITYPE_Audio );
		bPushSoundPlaying = False;
	}
}

// ----------------------------------------------------------------------------
// EnterWorld()
// ----------------------------------------------------------------------------

function EnterWorld()
{
	if ( !bInWorld )
	{
		bInWorld    = true;
		bHidden     = Default.bHidden;
		bDetectable = Default.bDetectable;

		AmbientSound = WorldAmbientSound;
		//SetLocation( WorldPosition );

		SetCollision( Default.bCollideActors, Default.bBlockActors, Default.bBlockPlayers );
		bCollideWorld = Default.bCollideWorld;

		SetPhysics( Default.Physics );
	}
}

// ----------------------------------------------------------------------------
// LeaveWorld()
// ----------------------------------------------------------------------------

function LeaveWorld()
{
	if ( bInWorld )
	{
		bInWorld          = false;
		bHidden           = true;
		bDetectable       = false;

		WorldAmbientSound = AmbientSound;
		AmbientSound      = None;
		//WorldPosition     = Location;

		SetCollision( false, false, false );
		bCollideWorld = false;

		SetPhysics( PHYS_None );
		//SetLocation(Location+vect(0,0,20000));  // move it out of the way
	}
}

// ----------------------------------------------------------------------------
// StartRolling() -- Stub, implemented by Cart.
// ----------------------------------------------------------------------------

function StartRolling( Vector RollVelocity );

// ----------------------------------------------------------------------------
// InitTrash () -- Stub, Implemented by Trash.
// ----------------------------------------------------------------------------

simulated function InitTrash( bool bInTumble, bool bInWind );

// ----------------------------------------------------------------------
// ----------------------------------------------------------------------

defaultproperties
{
	bSpawning=True
	RenderIteratorClass=HXDecorationRenderIterator
	SwimOmega=1.570796
	SwimAmp=(Pitch=512,Yaw=256,Roll=512)
	bContentsSpawned=False
	HitPoints=20
	FragType=MetalFragment
	Flammability=30.0
	ExplosionDamage=100
	ExplosionRadius=768.0
	bHighlight=True
	bClientHighlight=True
	ItemArticle="a"
	ItemName="DEFAULT DECORATION NAME - REPORT THIS AS A BUG"
	bPushable=True
	PushSound=PushMetal
	bStatic=False
	bTravel=True
	Physics=PHYS_Falling
	DrawType=DT_Mesh
	bCollideActors=True
	bCollideWorld=True
	bBlockActors=True
	bBlockPlayers=True
	bInWorld=True
	CollisionRadius=22.00
	//CollisionHeight=22.00
	CollisionHeight=21.25
}
