//=============================================================================
// HXElectricityEmitter.
//
// Each client will see different beams, but the effect is too fast
// to sync it properly over network.
//
// TODO:
//  * Well, big cleanup.
//  * Copy BeamTexture over regular? As it's propagation can be delayed
//    in network play.
//=============================================================================
class HXElectricityEmitter extends HXEffects;

// LaserEmitter.
var HXLaserSpot spot[2];			// max of 2 reflections
var bool bIsOn;
var actor HitActor;
var bool bFrozen;				// are we out of the player's sight?
var bool bRandomBeam;
//var bool bBlueBeam;				// is this beam blue?
var bool bHiddenBeam;			// is this beam hidden?

var HXLaserProxy proxy;

// ElectricityEmitter.
var() float   RandomAngle;  // Random amount to change yaw/pitch.
var() int     DamageAmount; // How much damage does this do?
var() float   DamageTime;   // How often does this do damage?
var() Texture BeamTexture;  // Texture for beam.
var() bool    bInitiallyOn; // Is this initially on?
var() bool    bFlicker;     // Randomly flicker on and off?
var() bool    bEmitLight;   // Should I cast light, also?
var() float   FlickerTime;  // How often to check for random flicker?

var Actor LastHitActor;
var float LastDamageTime;
var float LastFlickerTime;
var bool  bAlreadyInitialized;		// has this item been init'ed yet?

// HXElectricityEmitter.
var bool bIsLocalOn;

// ----------------------------------------------------------------------------
// Network Replication.
// ----------------------------------------------------------------------------

replication
{
	// Initial Server to Client.
	reliable if ( Role==ROLE_Authority && bNetInitial )
		BeamTexture;

	// Config.
	reliable if (Role == ROLE_Authority)
		RandomAngle, DamageTime /* Why? */, bInitiallyOn, bFlicker, FlickerTime, bEmitLight;

	// State.
	reliable if (Role == ROLE_Authority)
		bIsOn;
}

// ----------------------------------------------------------------------------
// Succeeds()
//
// Intended to be called by Mutator code if Self was spawned as a successor
// to Other. Super call recommended.
// ----------------------------------------------------------------------------

function Succeeds( Actor Other )
{
	local ElectricityEmitter OtherElectricityEmitter;

	Super.Succeeds( Other );

	// Special ElectricityEmitter Init.
	OtherElectricityEmitter = ElectricityEmitter(Other);
	if ( OtherElectricityEmitter!=None )
	{
		// ElectricityEmitter.
		RandomAngle  = OtherElectricityEmitter.RandomAngle;
		DamageAmount = OtherElectricityEmitter.DamageAmount;
		DamageTime   = OtherElectricityEmitter.DamageTime;
		BeamTexture  = OtherElectricityEmitter.BeamTexture;
		bInitiallyOn = OtherElectricityEmitter.bInitiallyOn;
		bFlicker     = OtherElectricityEmitter.bFlicker;
		bEmitLight   = OtherElectricityEmitter.bEmitLight;
		FlickerTime  = OtherElectricityEmitter.FlickerTime;
	}
}

// ----------------------------------------------------------------------------
// PostBeginPlay()
// ----------------------------------------------------------------------------

simulated function PostBeginPlay()
{
	Super.PostBeginPlay();

	if ( bDeleteMe || IsInRelevant() )
		return;

	if ( Role==ROLE_Authority )
	{
		// Create our proxy laser beam
		if ( Proxy==None )
			Proxy = Spawn( class'HXLaserProxy',,, Location, Rotation );

		if ( Proxy!=None )
		{
			Proxy.Skin       = BeamTexture;
		}

		//if ( bAlreadyInitialized )
			//return;

		if ( bInitiallyOn )
		{
			bIsOn = False;
			TurnOn();
		}
		else
		{
			bIsOn = True;
			TurnOff();
		}

		//bAlreadyInitialized = True;
	}
}

// ----------------------------------------------------------------------------
// PostNetBeginPlay()
// ----------------------------------------------------------------------------

simulated function PostNetBeginPlay()
{
	Super.PostNetBeginPlay();

	// create our proxy laser beam
	if ( proxy == None )
		proxy = Spawn(class'HXLaserProxy',,, Location, Rotation);

	if ( proxy != None )
	{
		proxy.Skin = BeamTexture;
	}

	//if ( bAlreadyInitialized )
		//return;

	if ( bInitiallyOn )
	{
		bIsLocalOn = False;
		TurnOn();
	}
	else
	{
		bIsLocalOn = True;
		TurnOff();
	}

	//bAlreadyInitialized = True;
}

// ----------------------------------------------------------------------------
// Destroyed()
// ----------------------------------------------------------------------------

simulated function Destroyed()
{
	TurnOff();

	if (proxy != None)
	{
		proxy.Destroy();
		proxy = None;
	}

	Super.Destroyed();
}

// ----------------------------------------------------------------------------
// CalcTrace()
// ----------------------------------------------------------------------------

simulated function CalcTrace(float deltaTime)
{
	local vector StartTrace, EndTrace, HitLocation, HitNormal, loc;
	local Rotator rot;
	local actor target;
	local int texFlags;
	local name texName, texGroup;
	local Texture Tex;

	if ( !bHiddenBeam )
	{
		// set up the random beam stuff
		rot.Pitch = Int((0.5 - FRand()) * RandomAngle);
		rot.Yaw = Int((0.5 - FRand()) * RandomAngle);
		rot.Roll = Int((0.5 - FRand()) * RandomAngle);

		StartTrace = Location;
		EndTrace = Location + 5000 * vector(Rotation + rot);
		HitActor = None;

		//foreach TraceTexture(class'Actor', target, texName, texGroup, texFlags, HitLocation, HitNormal, EndTrace, StartTrace)
		foreach TraceTextures(class'Actor', target, Tex, texName, texGroup, texFlags, HitLocation, HitNormal, EndTrace, StartTrace)
		{
			if ((target.DrawType == DT_None) || target.bHidden)
			{
				// do nothing - keep on tracing
			}
			else if ((target == Level) || target.IsA('Mover'))
			{
				break;
			}
			else
			{
				HitActor = target;
				break;
			}
		}

		LastDamageTime += deltaTime;

		// Only do dmg on server.
		if ( Role == ROLE_Authority )
		{
			// shock whatever gets in the beam
			if ((HitActor != None) && (LastDamageTime >= DamageTime))
			{
				HitActor.TakeDamage(DamageAmount, Instigator, HitLocation, vect(0,0,0), 'Shocked');
				LastDamageTime = 0;
			}
		}

		if (HXElectricityIterator(RenderInterface) != None)
			HXElectricityIterator(RenderInterface).AddBeam(0, Location, Rotation + rot, VSize(Location - HitLocation));
	}
}

// ----------------------------------------------------------------------------
// TurnOn()
// ----------------------------------------------------------------------------

simulated function TurnOn()
{
	if ( Role==ROLE_Authority )
	{
		if ( !bIsOn )
		{
			bIsOn = True;
			HitActor = None;
			CalcTrace(0.0);
			if (!bHiddenBeam)
				proxy.bHidden = False;
			SoundVolume = 128;
		}
	}
	else
	{
		if ( !bIsLocalOn )
		{
			bIsLocalOn = True;
			HitActor = None;
			CalcTrace(0.0);
			if (!bHiddenBeam)
				proxy.bHidden = False;
			SoundVolume = 128;
		}	
	}

	if ( bEmitLight )
		LightType = LT_Steady;
}

// ----------------------------------------------------------------------------
// TurnOff()
// ----------------------------------------------------------------------------

simulated function TurnOff()
{
	local int i;

	if ( Role==ROLE_Authority )
	{
		if ( bIsOn )
		{
			for ( i=0; i<ArrayCount(spot); i++ )
			{
				if ( spot[i] != None )
				{
					spot[i].Destroy();
					spot[i] = None;
					if ( HXElectricityIterator(RenderInterface) != None )
						HXElectricityIterator(RenderInterface).DeleteBeam(i);
				}
			}

			HitActor = None;
			bIsOn = False;
			if ( !bHiddenBeam )
				proxy.bHidden = True;
			SoundVolume = 0;
		}
	}
	else
	{
		if ( bIsLocalOn )
		{
			for ( i = 0; i < ArrayCount(spot); i++ )
			{
				if ( spot[i] != None )
				{
					spot[i].Destroy();
					spot[i] = None;
					if ( HXElectricityIterator(RenderInterface) != None )
						HXElectricityIterator(RenderInterface).DeleteBeam(i);
				}
			}

			HitActor = None;
			bIsLocalOn = False;
			if ( !bHiddenBeam )
				proxy.bHidden = True;
			SoundVolume = 0;
		}
	}

	if (bEmitLight)
		LightType = LT_None;

	if (HXElectricityIterator(RenderInterface) != None)
		HXElectricityIterator(RenderInterface).DeleteBeam(0);
}

// ----------------------------------------------------------------------------
// Tick()
// ----------------------------------------------------------------------------

simulated function Tick(float deltaTime)
{
	local HXPlayerPawn player;

	// check for visibility
	player = HXPlayerPawn(GetPlayerPawn());

	// Make local status match the replicated one.
	if ( Role < ROLE_Authority )
	{
		if ( bIsOn && !bIsLocalOn )
		{
			TurnOn();
		}
		else if ( !bIsOn && bIsLocalOn )
		{
			TurnOff();
		}
	}

	if ( ( bIsOn && Role == ROLE_Authority )
		|| ( bIsLocalOn && Role < ROLE_Authority ) )
	{
		// if we are a weapon's laser sight, do not freeze us
		//if ((Owner != None) && (Owner.IsA('Weapon') || Owner.IsA('ScriptedPawn')))
			//bFrozen = False;
		//else if (proxy != None)
		if (proxy != None)
		{
			// if we are close, say 60 feet
			if (proxy.DistanceFromPlayer < 960)
				bFrozen = False;
			else
			{
				// can the player see the generator?
				if (proxy.LastRendered() <= 2.0)
					bFrozen = False;
				else
					bFrozen = True;
			}
		}
		else
			bFrozen = True;

		if (bFrozen)
			return;

		CalcTrace(deltaTime);

		if ( !bFrozen && bFlicker )
		{
			LastFlickerTime += deltaTime;

			if (LastFlickerTime >= FlickerTime)
			{
				LastFlickerTime = 0;
				if (FRand() < 0.5)
				{
					SetHiddenBeam(True);
					SoundVolume = 0;
					if (bEmitLight)
						LightType = LT_None;
				}
				else
				{
					SetHiddenBeam(False);
					SoundVolume = 128;
					if (bEmitLight)
						LightType = LT_Steady;
				}
			}
		}

	}
}

// ----------------------------------------------------------------------------
// Trigger()
//
// if we are triggered, turn us on
// ----------------------------------------------------------------------------

function Trigger(Actor Other, Pawn Instigator)
{
	Super.Trigger(Other, Instigator);

	TurnOn();
}

// ----------------------------------------------------------------------------
// UnTrigger()
//
// if we are untriggered, turn us off
// ----------------------------------------------------------------------------

function UnTrigger(Actor Other, Pawn Instigator)
{
	Super.UnTrigger(Other, Instigator);

	TurnOff();
}

// ----------------------------------------------------------------------------
// SetHiddenBeam()
// ----------------------------------------------------------------------------

simulated function SetHiddenBeam(bool bHide)
{
	bHiddenBeam = bHide;
	if (proxy != None)
		proxy.bHidden = bHide;
}

// ----------------------------------------------------------------------------
// ----------------------------------------------------------------------------

//DrawType=DT_Sprite

defaultproperties
{
	RandomAngle=8192.000000
	DamageAmount=2
	DamageTime=0.200000
	BeamTexture=FireTexture'Effects.Electricity.Nano_SFX'
	bInitiallyOn=True
	bFlicker=True
	FlickerTime=0.020000
	bEmitLight=True
	bRandomBeam=True
	bDirectional=True
	DrawType=DT_None
	Texture=Texture'Engine.S_Inventory'
	SoundRadius=64
	AmbientSound=Sound'Ambient.Ambient.Electricity4'
	LightBrightness=128
	LightHue=150
	LightSaturation=32
	LightRadius=6
	CollisionRadius=40.000000
	CollisionHeight=40.000000
	RenderIteratorClass=Class'HX.HXElectricityIterator'
	bReplicateRotation=True
}
