//=============================================================================
// HXLaserBeamTrigger.
//
// Newly introduced baseclass for HXLaserTrigger and HXBeamTrigger.
//=============================================================================
class HXLaserBeamTrigger extends HXTrigger;

//#exec OBJ LOAD FILE=HXLaser
#exec OBJ LOAD FILE=Textures\HXLaser.utx PACKAGE=HX.Laser

// HXLaserEmitter. Original Mesh had MESHMAP NUM=1 masked.
#EXEC Mesh Import MESH=HXLaserEmitter ANIVFILE="Models\HXLaserEmitter_a.3d" DATAFILE="Models\HXLaserEmitter_d.3d"
#EXEC MESH ORIGIN MESH=HXLaserEmitter YAW=64
#EXEC MESH LODPARAMS MESH=HXLaserEmitter STRENGTH=0.500000
#EXEC MESH SEQUENCE MESH=HXLaserEmitter SEQ=All STARTFRAME=0 NUMFRAMES=1
#EXEC MESH SEQUENCE MESH=HXLaserEmitter SEQ=Still STARTFRAME=0 NUMFRAMES=1
#EXEC MESHMAP SCALE MESHMAP=HXLaserEmitter X=0.00390625 Y=0.00390625 Z=0.00390625
#EXEC MESHMAP SETTEXTURE MESHMAP=HXLaserEmitter NUM=0 TEXTURE=LaserEmitterTex1
#EXEC MESHMAP SETTEXTURE MESHMAP=HXLaserEmitter NUM=1 TEXTURE=HXLaserSpot1

// Configuration.
var() bool bIsOn;               // !! Though it was/is configurable, PreBeginPlay will always enabled it !!

var bool   bConfused;           // Used when hit by EMP.
var float  ConfusionTimer;      // How long until trigger resumes normal operation.
var float  ConfusionDuration;   // How long does EMP hit last?
var float  ConfusionFlipTimer;  // Used to lock flipping effect to 30 Hz.
var int    HitPoints;
var int    MinDamageThreshold;

var class<HXLaserEmitter> LaserEmitterClass; // Per subclass configurable LaserEmitterClass.
var HXLaserEmitter Emitter;

// ----------------------------------------------------------------------------
// Networking replication.
// ----------------------------------------------------------------------------

replication
{
	// Server to client.
	reliable if ( Role==ROLE_Authority )
		bIsOn, bConfused;
}

// ----------------------------------------------------------------------------
// BeginPlay()
// ----------------------------------------------------------------------------

function BeginPlay()
{
	Super.BeginPlay();

	if ( bDeleteMe )
		return;

	if ( Emitter==None )
		Emitter = Spawn( LaserEmitterClass );

	bIsOn        = true; // Sic!

	if ( bIsOn )
		TurnOnEmitter();
}

// ----------------------------------------------------------------------------
// PostNetBeginPlay()
// ----------------------------------------------------------------------------

simulated function PostNetBeginPlay()
{
	Super.PostNetBeginPlay();

	if ( Emitter==None )
		Emitter = Spawn( LaserEmitterClass );

	// We can start with turned on emitter for bConfused as well.
	if ( bIsOn )
		TurnOnEmitter();
}

// ----------------------------------------------------------------------------
// Destroyed()
// ----------------------------------------------------------------------------

simulated function Destroyed()
{
	if ( Emitter!=None )
	{
		Emitter.Destroy();
		Emitter = None;
	}

	Super.Destroyed();
}

// ----------------------------------------------------------------------------
// Tick()
// ----------------------------------------------------------------------------

simulated function Tick( float DeltaTime )
{
	local Actor A, HitActor;
	local HXAdaptiveArmor Armor;
	local bool bTrigger;
	local float Chance;

	// If we've been EMP'ed, act confused.
	if ( bIsOn )
	{
		if ( bConfused )
		{
			ConfusionTimer += DeltaTime;

			// Randomly turn on/off the beam. But don't go above 30 Hz.
			if ( ConfusionTimer>ConfusionFlipTimer )
			{
				// Increase chance towards end of ConfusionDuration.
				Chance = (ConfusionTimer/ConfusionDuration);
				Chance *= Chance; // Quadratic ramp.

				if ( FRand()<Chance )
					TurnOnEmitter();
				else
					TurnOffEmitter();

				ConfusionFlipTimer = ConfusionTimer+0.033;
			}

			if ( ConfusionTimer>ConfusionDuration )
			{
				TurnOnEmitter();

				if ( Role==ROLE_Authority )
					bConfused = false;
			}
		}
		else // !bConfused
		{
			// Make sure it's on.
			TurnOnEmitter();

			// If not confused reset the timer for both client and server.
			ConfusionTimer     = 0.0;
			ConfusionFlipTimer = 0.0;

			// Do hit detection and stuff.
			if ( Role==ROLE_Authority )
			{
				if ( Emitter!=None )
				{
					// Lets not do this anymore. --han
					//Emitter.SetLocation( Location );
					//Emitter.SetRotation( Rotation );

					HitActor = Emitter.HitActor;

					if ( HitActor!=None && IsRelevant(HitActor) )
					{
						bTrigger = true;

						if ( bTrigger )
						{
							BeamTriggered( HitActor );
						}
					}
				}
			}
		}
	}
	else // !bIsOn.
	{
		// Make sure it's off.
		TurnOffEmitter();
	}
}

// ----------------------------------------------------------------------------
// IsRelevant()
//
// See whether the other actor is relevant to this trigger.
// ----------------------------------------------------------------------------

function bool IsRelevant( Actor Other )
{
	local HXAdaptiveArmor Armor;
	local HXPlayerPawn PP;

	if ( !Super.IsRelevant(Other) )
		return false;

	// In case of a PlayerPawn check for AdaptiveArmor.
	PP = HXPlayerPawn(Other);
	if ( PP!=None && PP.HXUsingChargedPickup(class'HXAdaptiveArmor') )
		return false;

	return true;
}

// ----------------------------------------------------------------------------
// BeamTriggered()
//
// A new releveant Actor ran into the beam.
// ----------------------------------------------------------------------------

function BeamTriggered( Actor Other )
{
	// Play "beam broken" sound serverside.
	PlaySound( Sound'Beep2',,,, 1280.0, 3.0 );
}

// ----------------------------------------------------------------------------
// TurnOnEmitter()
// ----------------------------------------------------------------------------

simulated function TurnOnEmitter()
{
	MultiSkins[1] = default.MultiSkins[1];

	if ( Emitter!=None )
		Emitter.TurnOn();
}

// ----------------------------------------------------------------------------
// TurnOffEmitter()
// ----------------------------------------------------------------------------

simulated function TurnOffEmitter()
{
	MultiSkins[1] = Texture'BlackMaskTex';

	if ( Emitter!=None )
		Emitter.TurnOff();
}

// ----------------------------------------------------------------------------
// Trigger()
// ----------------------------------------------------------------------------

// If we are triggered, turn us on.
function Trigger( Actor Other, Pawn Instigator )
{
	if ( bConfused ) // Does this make sense? --han
		return;

	if ( !bIsOn )
	{
		bIsOn        = true;

		TurnOnEmitter();
	}

	Super.Trigger( Other, Instigator );
}

// ----------------------------------------------------------------------------
// UnTrigger()
//
// If we are untriggered, turn us off.
// ----------------------------------------------------------------------------

function UnTrigger( Actor Other, Pawn Instigator )
{
	if ( bConfused ) // Does this make sense? --han
		return;

	if ( bIsOn )
	{
		bIsOn        = false;

		TurnOffEmitter();
	}

	Super.UnTrigger( Other, Instigator );
}

// ----------------------------------------------------------------------------
// Succeeds()
//
// Intended to be called by Mutator code if Self was spawned as a successor
// to Other. Super call recommended.
// ----------------------------------------------------------------------------

function Succeeds( Actor Other )
{
	// Call Super.
	Super.Succeeds( Other );

	// Special LaserBeamTrigger Init.
	if ( Other!=None )
	{
		bIsOn = bool(Other.GetPropertyText("bIsOn"));

		// Quirk for fixing up mesh.
		if ( Other.Mesh==Mesh'LaserEmitter' )
		{
			// HXLaserEmitter.
			Mesh = Default.Mesh;	
		}

		// HXLaserSpot1 (red) or HXLaserSpot2 (blue). Set this always as this gets flipped by code.
		MultiSkins[1] = Default.MultiSkins[1];
	}
}

// ----------------------------------------------------------------------------
// Touch()
//
// Does nothing when touched.
// ----------------------------------------------------------------------------

function Touch( Actor Other );

// ----------------------------------------------------------------------------
// TakeDamage()
// ----------------------------------------------------------------------------

function TakeDamage( int Damage, Pawn EventInstigator, Vector HitLocation, Vector Momentum, Name DamageType )
{
	local MetalFragment MetalFragment;

	switch ( DamageType )
	{
		case 'EMP':
			// Prolong confused time if already confused.
			ConfusionTimer     = 0.0;
			ConfusionFlipTimer = 0.0;

			if ( !bConfused )
			{
				bConfused = true;
				PlaySound( Sound'EMPZap', SLOT_None,,, 1280.0 );
			}
			break;

		case 'Exploded':
		case 'Shot':

		// Added. --han
		case 'AutoShot':
		case 'Sabot':
			if ( Damage>=MinDamageThreshold )
				HitPoints -= Damage;

			if ( HitPoints<=0 )
			{
				MetalFragment = Spawn( class'MetalFragment', Owner );
				if ( MetalFragment!=None )
				{
					MetalFragment.Instigator = EventInstigator;
					MetalFragment.CalcVelocity( Momentum,0 );
					MetalFragment.DrawScale = 0.5*FRand();
					MetalFragment.Skin = GetMeshTexture();
				}

				Destroy();
			}
			break;

		default:
			break;
	}
}

// ----------------------------------------------------------------------------
// ----------------------------------------------------------------------------

defaultproperties
{
	LaserEmitterClass=HXLaserEmitter
	bIsOn=True
	ConfusionDuration=10.0
	HitPoints=50
	MinDamageThreshold=50
	TriggerType=TT_AnyProximity
	bHidden=False
	bDirectional=True
	DrawType=DT_Mesh
	Mesh=HXLaserEmitter
	MultiSkins(1)=HXLaserSpot1
	CollisionRadius=2.5
	CollisionHeight=2.5
	RemoteRole=ROLE_SimulatedProxy
}
