//=============================================================================
// HXLaserEmitter.
//=============================================================================
class HXLaserEmitter extends HXEffects;

#exec OBJ LOAD FILE=Effects

var HXLaserSpot spot[2];			// max of 2 reflections
var bool bIsOn;
var actor HitActor;
var bool bFrozen;				// are we out of the player's sight?
var bool bRandomBeam;
var bool bHiddenBeam;			// is this beam hidden?

var HXLaserProxy proxy;

// Properties for configuring beam color in subclasses.
var class<HXLaserProxy> LaserProxyClass;
var class<HXLaserSpot>  LaserSpotClass;

// ----------------------------------------------------------------------------
// Network Replication.
// ----------------------------------------------------------------------------

replication
{
	//Variables server to client
	reliable if (Role == ROLE_Authority)
		bIsOn;
}

function PostBeginPlay()
{
	Super.PostBeginPlay();

	// create our proxy laser beam
	if (proxy == None)
		proxy = Spawn( LaserProxyClass,,, Location, Rotation );
}

simulated function Destroyed()
{
	TurnOff();

	if (proxy != None)
	{
		proxy.Destroy();
		proxy = None;
	}

	Super.Destroyed();
}

function CalcTrace(float deltaTime)
{
	local vector StartTrace, EndTrace, HitLocation, HitNormal, Reflection;
	local actor target;
	local int i, texFlags;
	local name texName, texGroup;
	local Texture Tex;

	StartTrace = Location;
	EndTrace = Location + 5000 * vector(Rotation);
	HitActor = None;

	// trace the path of the reflected beam and draw points at each hit
	for (i=0; i<ArrayCount(spot); i++)
	{
		foreach TraceTextures(class'Actor', target, Tex, texName, texGroup, texFlags, HitLocation, HitNormal, EndTrace, StartTrace)
		{
			if ((target.DrawType == DT_None) || target.bHidden)
			{
				// do nothing - keep on tracing
			}
			else if ((target == Level) || target.IsA('Mover'))
			{
				break;
			}
			else
			{
				HitActor = target;
				break;
			}
		}

		// draw first beam
		if (i == 0)
		{
			if (HXLaserIterator(RenderInterface) != None)
				HXLaserIterator(RenderInterface).AddBeam(i, Location, Rotation, VSize(Location - HitLocation));
		}
		else
		{
			if (HXLaserIterator(RenderInterface) != None)
				HXLaserIterator(RenderInterface).AddBeam(i, StartTrace - HitNormal, Rotator(Reflection), VSize(StartTrace - HitLocation - HitNormal));
		}

		if (spot[i] == None)
		{
			spot[i] = Spawn( LaserSpotClass, Self, , HitLocation, Rotator(HitNormal) );
		}
		else
		{
			spot[i].SetLocation(HitLocation);
			spot[i].SetRotation(Rotator(HitNormal));
		}

		// don't reflect any more if we don't hit a mirror
		// 0x08000000 is the PF_Mirrored flag from UnObj.h
		if ((texFlags & 0x08000000) == 0)
		{
			// kill all of the other spots after this one
			if (i < ArrayCount(spot)-1)
			{
				do
				{
					i++;
					if (spot[i] != None)
					{
						spot[i].Destroy();
						spot[i] = None;
						if (HXLaserIterator(RenderInterface) != None)
							HXLaserIterator(RenderInterface).DeleteBeam(i);
					}
				} until (i>=ArrayCount(spot)-1);
			}

			return;
		}

		Reflection = MirrorVectorByNormal(Normal(HitLocation - StartTrace), HitNormal);
		StartTrace = HitLocation + HitNormal;
		EndTrace = Reflection * 10000;
	}
}

simulated function TurnOn()
{
	if (!bIsOn)
	{
		//Log( Name$".TurnOn()");

		bIsOn = True;
		HitActor = None;
		CalcTrace(0.0);
		if (!bHiddenBeam)
			proxy.bHidden = False;
		//SoundVolume = 128;
		AmbientSound = Default.AmbientSound;

		//bHidden = false;
	}
}

simulated function TurnOff()
{
	local int i;

	if (bIsOn)
	{
		//Log( Name$".TurnOff()");

		for (i=0; i<ArrayCount(spot); i++)
			if (spot[i] != None)
			{
				spot[i].Destroy();
				spot[i] = None;
				if (HXLaserIterator(RenderInterface) != None)
					HXLaserIterator(RenderInterface).DeleteBeam(i);
			}

		HitActor = None;
		bIsOn = False;
		if (!bHiddenBeam)
			proxy.bHidden = True;
		//SoundVolume = 0;
		AmbientSound = None;

		//proxy.Destroy();
		//bHidden = true;
	}
}

function Tick(float deltaTime)
{
	local DeusExPlayer player;

	// check for visibility
	player = DeusExPlayer(GetPlayerPawn());

	if (bIsOn)
	{
		// if we are a weapon's laser sight, do not freeze us
		if ((Owner != None) && (Owner.IsA('Weapon') || Owner.IsA('HXScriptedPawn')))
			bFrozen = False;
		else if (proxy != None)
		{
			// if we are close, say 60 feet
			if (proxy.DistanceFromPlayer < 960)
				bFrozen = False;
			else
			{
				// can the player see the generator?
				if (proxy.LastRendered() <= 2.0)
					bFrozen = False;
				else
					bFrozen = True;
			}
		}
		else
			bFrozen = True;

		if (bFrozen)
			return;

		CalcTrace(deltaTime);
	}
}

function SetHiddenBeam(bool bHide)
{
	bHiddenBeam = bHide;
	if (proxy != None)
		proxy.bHidden = bHide;
}

// ----------------------------------------------------------------------------
// ----------------------------------------------------------------------------

defaultproperties
{
	RemoteRole=ROLE_None
	LaserProxyClass=HXLaserProxy
	LaserSpotClass=HXLaserSpot
	SoundRadius=16
	SoundVolume=128
	AmbientSound=Laser
	CollisionRadius=40.0
	CollisionHeight=40.0
	RenderIteratorClass=HXLaserIterator
}
