//=============================================================================
// HXLaserTrigger.
//=============================================================================
class HXLaserTrigger extends HXLaserBeamTrigger;

// Configuration.
var() bool bNoAlarm;           // If true, does NOT sound alarm.

var float  LastAlarmTime;      // Last time the alarm was sounded.
var int    AlarmTimeout;       // How long before the alarm silences itself.
var Actor  TriggerActor;       // Actor which last triggered the alarm.
var Vector ActorLocation;      // Last known location of Actor that triggered alarm.

// ----------------------------------------------------------------------------
// Succeeds()
//
// Intended to be called by Mutator code if Self was spawned as a successor
// to Other. Super call recommended.
// ----------------------------------------------------------------------------

function Succeeds( Actor Other )
{
	local LaserTrigger OtherLaserTrigger;

	// Call Super.
	Super.Succeeds( Other );

	// Special LaserTrigger Init.
	OtherLaserTrigger = LaserTrigger(Other);
	if ( OtherLaserTrigger!=None )
	{
		bNoAlarm = OtherLaserTrigger.bNoAlarm;
	}
}

// ----------------------------------------------------------------------------
// BeginPlay()
// ----------------------------------------------------------------------------

function BeginPlay()
{
	Super.BeginPlay();

	// Turn off the sound if we should.
	if ( Emitter!=None && SoundVolume==0 )
		Emitter.AmbientSound = None;
}

// ----------------------------------------------------------------------------
// BeamTriggered()
//
// A new releveant Actor ran into the beam.
// ----------------------------------------------------------------------------

function BeamTriggered( Actor Other )
{	
	Super.BeamTriggered( Other );

	// Now, the trigger sounds its own alarm
	if ( AmbientSound==None )
	{
		TriggerActor  = Other;
		ActorLocation = Other.Location-vect(0,0,1)*(Other.CollisionHeight-1.0);

		BeginAlarm();
	}
}

// ----------------------------------------------------------------------------
// IsRelevant()
//
// See whether the other actor is relevant to this trigger.
// ----------------------------------------------------------------------------

function bool IsRelevant( Actor Other )
{
	if ( bNoAlarm )
		return false;

	if ( Super.IsRelevant(Other) )
		return true;

	// TT_PlayerProximity/TT_PawnProximity actually works with decorations, too.
	if ( TriggerType==TT_PlayerProximity || TriggerType==TT_PawnProximity )
	{
		if ( Other.IsA('Decoration') )
			return true;

		// As per strawpoll vote.
		if ( Other.IsA('Inventory') )
			return true;
	}

	return false;
}

// ----------------------------------------------------------------------------
// Tick()
// ----------------------------------------------------------------------------

simulated function Tick( float DeltaTime )
{
	if ( Role==ROLE_Authority )
	{
		// Shut off the alarm if the timeout has expired, do this before super call,
    // as super can reenable alarm the very same tick this way.
		if ( LastAlarmTime!=0 )
			if ( Level.TimeSeconds-LastAlarmTime>=AlarmTimeout )
				EndAlarm();
	}

	Super.Tick( DeltaTime );
}

// ----------------------------------------------------------------------------
// UnTrigger()
//
// If we are untriggered, turn us off.
// ----------------------------------------------------------------------------

function UnTrigger( Actor Other, Pawn Instigator )
{
	if ( bIsOn && !bConfused )
	{
		EndAlarm();
	}

	Super.UnTrigger( Other, Instigator );
}

// ----------------------------------------------------------------------------
// BeginAlarm()
// ----------------------------------------------------------------------------

function BeginAlarm()
{
	AmbientSound = Sound'Klaxon2';
	SoundVolume = 128;
	SoundRadius = 64;
	LastAlarmTime = Level.TimeSeconds;
	AIStartEvent( 'Alarm', EAITYPE_Audio, SoundVolume/255.0, 25*(SoundRadius+1) ); // Should be fine to call multiple times with the same values without having an effect.

	// make sure we can't go into stasis while we're alarming
	bStasis = false;
}

// ----------------------------------------------------------------------------
// EndAlarm()
// ----------------------------------------------------------------------------

function EndAlarm()
{
	AmbientSound = None;
	LastAlarmTime = 0;
	AIEndEvent( 'Alarm', EAITYPE_Audio );

	// reset our stasis info
	bStasis = default.bStasis;
}

// ----------------------------------------------------------------------------
// ----------------------------------------------------------------------------

defaultproperties
{
	LaserEmitterClass=HXRedLaserEmitter
	MultiSkins(1)=HXLaserSpot1
	AlarmTimeout=30
}
