//=============================================================================
// HXMenuScreenCustomizeKeys.
//
// Notes:
//  * WinDrvExt uses Unreal Tournament 2004 Key/Button/Axis mapping.
//		- IK_UnknownC1 ~ IK_Mouse4,
//		- IK_UnknownC2 ~ IK_Mouse5
//
// Todo:
//  * Evaluate IK_Mouse4/IK_Mouse5 behaviour in RawMouseButtonPressed().
//
// Revision History:
//  * Added support for XButton 1/2.
//  * Removed the need to match the arraysizes to the amout of defprops.
//  * Removed obsolete DXMP keybindings.
//=============================================================================
class HXMenuScreenCustomizeKeys expands HXMenuUIScreenWindow;

var MenuUIListHeaderButtonWindow BtnHeaderAction;
var MenuUIListHeaderButtonWindow BtnHeaderAssigned;
var MenuUIListWindow						 LstKeys;
var MenuUIScrollAreaWindow			 WinScroll;

struct S_KeyDisplayItem
{
	var EInputKey 			 InputKey;
	var localized String DisplayName;
};

var localized S_KeyDisplayItem KeyDisplayNames[128];
var localized string					 FunctionText[96];
var string										 MenuValues1[96];
var string										 MenuValues2[96];
var string										 AliasNames[96];
var string										 PendingCommands[100];

var int  Pending;
var int  Selection; 	
var bool bWaitingForInput;

var localized string NoneText;
var localized string StrHeaderActionLabel;
var localized string StrHeaderAssignedLabel;
var localized string WaitingHelpText;
var localized string InputHelpText;
var localized string ReassignedFromLabel;

// ----------------------------------------------------------------------
// InitWindow()
//
// Initialize the Window
// ----------------------------------------------------------------------

event InitWindow()
{
	Super.InitWindow();

	Pending = 0;
	Selection = -1;
	bWaitingForInput = False;
	BuildKeyBindings();

	CreateKeyList();
	CreateHeaderButtons();
	PopulateKeyList();
	ShowHelp( WaitingHelpText );
}

// ----------------------------------------------------------------------
// VirtualKeyPressed() 
// ----------------------------------------------------------------------

event bool VirtualKeyPressed( EInputKey Key, bool bRepeat )
{
	if ( !bWaitingForInput ) 
	{
		// If the user presses [Delete] or [Backspace], then clear this setting.
		switch ( Key )
		{
			case IK_Delete:
			case IK_Backspace:
				ClearFunction();
				return True;
				break;

			default:
				return Super.VirtualKeyPressed( Key, bRepeat );
				break;
		}
	}

	// First check to see if we're waiting for the user to select a Keyboard or mouse/joystick Button to override. 
	WaitingForInput( False );
                                                                                
	ProcessKeySelection( Key, Mid(string(GetEnum(enum'EInputKey',Key)),3), GetKeyDisplayName(Key) );

	return True;
}

// ----------------------------------------------------------------------
// RawMouseButtonPressed()
// ----------------------------------------------------------------------

event bool RawMouseButtonPressed( float PointX, float PointY, EInputKey Button, EInputState iState )
{
	if ( !bWaitingForInput )
		return False;

	if ( iState!=IST_Release )
		return True;

	// First check to see if we're waiting for the user to select a 
	// Keyboard or mouse/joystick Button to override. 
	//
	// Ignore everything but mouse Button and wheel presses
	switch ( Button )
	{
		case IK_MouseWheelUp:
		case IK_MouseWheelDown:
		case IK_LeftMouse:
		case IK_RightMouse:
		case IK_MiddleMouse:
		case IK_UnknownC1: // IK_Mouse4.
		case IK_UnknownC2: // IK_Mouse5.
			ProcessKeySelection( Button, Mid(string(GetEnum(enum'EInputKey',Button)),3), GetKeyDisplayName(Button));
			WaitingForInput(False);
			break;
	}

	return True;
}

// ----------------------------------------------------------------------
// ListRowActivated()
//
// User double-clicked on one of the rows, meaning he/she/it wants 
// to redefine one of the functions
// ----------------------------------------------------------------------

event bool ListRowActivated(window list, int RowId)
{
	// Show help.
	ShowHelp( InputHelpText );
	Selection = LstKeys.RowIdToIndex( RowId );
	WaitingForInput( True );
	return True;
}

// ----------------------------------------------------------------------
// WaitingForInput()
// ----------------------------------------------------------------------

function WaitingForInput( bool bWaiting )
{
 	if ( bWaiting )
	{
		ShowHelp( InputHelpText );
		SetSelectability( True );
		SetFocusWindow( Self );
		GrabMouse();
		Root.LockMouse( True, False );
		Root.ShowCursor( False );
	}
	else
	{
		ShowHelp( WaitingHelpText );
		SetSelectability( False );
		UngrabMouse();
		Root.LockMouse( False, False );
		Root.ShowCursor( True );

		// Set the focus back to the list.
		SetFocusWindow( LstKeys );
	}

	bWaitingForInput = bWaiting;
}

// ----------------------------------------------------------------------
// SaveSettings()
// ----------------------------------------------------------------------

function SaveSettings()
{
	ProcessPending();
}

// ----------------------------------------------------------------------
// ClearFunction()
// ----------------------------------------------------------------------

function ClearFunction()
{
	local int RowId;
	local int RowIndex;

	RowId = LstKeys.GetSelectedRow();

	if ( RowId!=0 )
	{
		RowIndex = LstKeys.RowIdToIndex( RowId );

		if ( MenuValues2[RowIndex]!="" && CanRemapKey(MenuValues2[RowIndex]) )
		{
			AddPending("SET ini:Engine.Engine.Input " $ GetKeyFromDisplayName(MenuValues2[RowIndex]));
			MenuValues2[RowIndex] = "";
		}

		if ( MenuValues1[RowIndex]!="" && CanRemapKey(MenuValues1[RowIndex]) )
		{
			AddPending("SET ini:Engine.Engine.Input " $ GetKeyFromDisplayName(MenuValues1[RowIndex]));
			MenuValues1[RowIndex] = MenuValues2[RowIndex];
			MenuValues2[RowIndex] = "";
		}

		// Update the buttons.
		RefreshKeyBindings();
	}
}

// ----------------------------------------------------------------------
// BuildKeyBindings()
// ----------------------------------------------------------------------

function BuildKeyBindings()
{
	local int i, j, Pos;
	local string KeyName;
	local string Alias;

	// First, clear all the existing Keybinding display strings in the MenuValues[1|2] arrays.
	for( i=0; i<ArrayCount(MenuValues1); i++ )
	{
		MenuValues1[i] = "";
		MenuValues2[i] = "";
	}

	// Now loop through all the Keynames and generate human-readable versions of Keys that are mapped.
	for ( i=0; i<255; i++ )
	{
		KeyName = Player.ConsoleCommand ( "KEYNAME " $ i );

		if ( KeyName!="" )
		{
			Alias = Player.ConsoleCommand( "KEYBINDING " $ KeyName );

			if ( Alias!="" )
			{
				Pos = InStr( Alias, " " );
				if ( Pos!=-1 )
					Alias = Left( Alias, Pos );

				for ( j=0; j<ArrayCount(AliasNames); j++ )
				{
					if ( AliasNames[j]!="" && AliasNames[j]==Alias )
					{
						if ( MenuValues1[j] == "" )
							MenuValues1[j] = GetKeyDisplayNameFromKeyName(KeyName);
						else if ( MenuValues2[j] == "" )
							MenuValues2[j] = GetKeyDisplayNameFromKeyName(KeyName);
					}
				}
			}
		}
	}
}

// ----------------------------------------------------------------------
// ProcessPending()
// ----------------------------------------------------------------------

function ProcessPending()
{
	local int i;

	for ( i=0; i<Pending; i++ )
		Player.ConsoleCommand( PendingCommands[i] );
		
	Pending = 0;
}

// ----------------------------------------------------------------------
// AddPending()
// ----------------------------------------------------------------------

function AddPending( String NewCommand )
{
	PendingCommands[Pending] = NewCommand;
	Pending++;
	if ( Pending==100 )
		ProcessPending();
}

// ----------------------------------------------------------------------
// GetKeyFromDisplayName()
// ----------------------------------------------------------------------

function String GetKeyFromDisplayName( String DisplayName )
{
	local int KeyIndex;

	for ( KeyIndex=0; KeyIndex<ArrayCount(KeyDisplayNames); KeyIndex++ )
		if ( KeyDisplayNames[KeyIndex].InputKey!=IK_None && DisplayName==KeyDisplayNames[KeyIndex].DisplayName )
			return Mid( String(GetEnum(enum'EInputKey',KeyDisplayNames[KeyIndex].InputKey)), 3 );

	return DisplayName;
}

// ----------------------------------------------------------------------
// GetKeyDisplayName()
// ----------------------------------------------------------------------

function String GetKeyDisplayName( EInputKey InputKey )
{
	local int KeyIndex;

	for ( KeyIndex=0; KeyIndex<ArrayCount(KeyDisplayNames); KeyIndex++ )
		if ( KeyDisplayNames[KeyIndex].InputKey!=IK_None && InputKey==KeyDisplayNames[KeyIndex].InputKey)
			return KeyDisplayNames[KeyIndex].DisplayName;

	return Mid( String(GetEnum(enum'EInputKey',InputKey)), 3 );
}
	
// ----------------------------------------------------------------------
// GetKeyDisplayNameFromKeyName()
// ----------------------------------------------------------------------

function String GetKeyDisplayNameFromKeyName( String KeyName )
{
	local int KeyIndex;

	for ( KeyIndex=0; KeyIndex<ArrayCount(KeyDisplayNames); KeyIndex++ )
		if ( KeyDisplayNames[KeyIndex].InputKey!=IK_None && Mid(string(GetEnum(enum'EInputKey',KeyDisplayNames[KeyIndex].InputKey)),3)==KeyName )
			return KeyDisplayNames[KeyIndex].DisplayName;

	return KeyName;
}

// ----------------------------------------------------------------------
// CanRemapKey()
// ----------------------------------------------------------------------

function bool CanRemapKey( String KeyName )
{
	return KeyName!="F1" && KeyName!="F2"; // hack - DEUS_EX STM
}

// ----------------------------------------------------------------------
// ProcessKeySelection()
// ----------------------------------------------------------------------

function ProcessKeySelection( int KeyNo, string KeyName, string KeyDisplayName )
{
	local int i;

	// Some Keys CANNOT be assigned:
	//
	// 1.  Escape
	// 2.  Function Keys (used by Augs)
	// 3.  Number Keys (used by Object Belt)
	// 4.  Tilde (used for console)
	// 5.  Pause (used to pause game)
	// 6.  Print Screen (Well, duh)
	//

	// Make sure the user enters a valid Key (Escape and functionKeys can't be assigned)
	switch ( KeyName )
	{
		case "":
		case "Escape":
		case "Tilde":
		case "PrintScrn":
		case "Pause":
			return;
			break;

		default:
			// Function Keys and 0-9
			if ( (KeyNo>=0x70 && KeyNo<=0x81) || (KeyNo>=48 && KeyNo<=57) )
				return;
			break;
	}

	// Don't waste our time if this Key is already assigned here
	if ( MenuValues1[Selection]==KeyDisplayName || MenuValues2[Selection]==KeyDisplayName )
	   return;

	// Now check to make sure there are no overlapping assignments.
	for ( i=0; i<ArrayCount(AliasNames); i++ )
	{
		if ( AliasNames[i]!="" )
		{
			if ( MenuValues2[i]==KeyDisplayName )
			{
				ShowHelp( Sprintf(ReassignedFromLabel, KeyDisplayName, FunctionText[i]) );
				AddPending("SET ini:Engine.Engine.Input " $ GetKeyFromDisplayName(MenuValues2[i]) );
				MenuValues2[i] = "";
			}

			if ( MenuValues1[i]==KeyDisplayName )
			{
				ShowHelp( Sprintf(ReassignedFromLabel, KeyDisplayName, FunctionText[i]) );
				AddPending( "SET ini:Engine.Engine.Input " $ GetKeyFromDisplayName(MenuValues1[i]) );
				MenuValues1[i] = MenuValues2[i];
				MenuValues2[i] = "";
			}
		}
	}

	// Now assign the Key, trying the first space if it's empty, but using the second space if necessary.  If both slots
	// are filled, then move the second entry into the first and put the new assignment in the second slot.
	if ( MenuValues1[Selection]=="" ) 
	{
		MenuValues1[Selection] = KeyDisplayName;
	}
	else if ( MenuValues2[Selection]=="" )
	{
		MenuValues2[Selection] = KeyDisplayName;
	}
	else if ( CanRemapKey(MenuValues1[Selection]) )
	{
		// Undo first Key assignment.
		AddPending( "SET ini:Engine.Engine.Input " $ GetKeyFromDisplayName(MenuValues1[Selection]) );
		MenuValues1[Selection] = MenuValues2[Selection];
		MenuValues2[Selection] = KeyDisplayName;
	}
	else if (CanRemapKey(MenuValues2[Selection]))
	{
		// Undo second Key assignment
		AddPending( "SET ini:Engine.Engine.Input " $ GetKeyFromDisplayName(MenuValues2[Selection]) );
		MenuValues2[Selection] = KeyDisplayName;
	}
	AddPending( "SET ini:Engine.Engine.Input "$KeyName$" "$AliasNames[Selection] );

	// Update the buttons.
	RefreshKeyBindings();
}

// ----------------------------------------------------------------------
// CreateKeyList()
//
// Creates the listbox containing the Key bindings
// ----------------------------------------------------------------------

function CreateKeyList()
{
	WinScroll = CreateScrollAreaWindow(WinClient);

	WinScroll.SetPos( 11, 23 );
	WinScroll.SetSize( 369, 268 );

	LstKeys = MenuUIListWindow(WinScroll.clipWindow.NewChild(Class'MenuUIListWindow'));
	LstKeys.EnableMultiSelect( False );
	LstKeys.EnableAutoExpandColumns( False );
	LstKeys.EnableHotKeys( False );

	LstKeys.SetNumColumns( 2 );

	LstKeys.SetColumnWidth( 0, 164 );
	LstKeys.SetColumnType ( 0, COLTYPE_String );
	LstKeys.SetColumnWidth( 1, 205 );
	LstKeys.SetColumnType ( 1, COLTYPE_String );
}

// ----------------------------------------------------------------------
// PopulateKeyList()
// ----------------------------------------------------------------------

function PopulateKeyList()
{
	local int KeyIndex;

	// First erase the old list.
	LstKeys.DeleteAllRows();

	for ( KeyIndex=0; KeyIndex<ArrayCount(AliasNames); KeyIndex++ )
		if ( AliasNames[KeyIndex]!="" )
			LstKeys.AddRow( FunctionText[KeyIndex] $ ";" $ GetInputDisplayText(KeyIndex) );
}

// ----------------------------------------------------------------------
// CreateHeaderButtons()
// ----------------------------------------------------------------------

function CreateHeaderButtons()
{
	BtnHeaderAction   = CreateHeaderButton( 10,  3, 162, StrHeaderActionLabel,   WinClient );
	BtnHeaderAssigned = CreateHeaderButton( 175, 3, 157, StrHeaderAssignedLabel, WinClient );

	BtnHeaderAction.SetSensitivity( False );
	BtnHeaderAssigned.SetSensitivity( False );
}

// ----------------------------------------------------------------------
// GetInputDisplayText()
// ----------------------------------------------------------------------

function String GetInputDisplayText(int KeyIndex)
{
	if ( MenuValues1[KeyIndex]=="" )
		return NoneText;
	else if ( MenuValues2[KeyIndex]!="" )
		return MenuValues1[KeyIndex] $ "," @ MenuValues2[KeyIndex];
	else
		return MenuValues1[KeyIndex];
}

// ----------------------------------------------------------------------
// RefreshKeyBindings()
// ----------------------------------------------------------------------

function RefreshKeyBindings()
{
	local int KeyIndex;
	local int RowId;

	for( KeyIndex=0; KeyIndex<ArrayCount(AliasNames); KeyIndex++ )
	{
		if ( AliasNames[KeyIndex]!="" )
		{
			RowId = LstKeys.IndexToRowId( KeyIndex );
			LstKeys.SetField( RowId, 1, GetInputDisplayText(KeyIndex) );
		}
	}
}

// ----------------------------------------------------------------------
// ResetToDefaults()
// ----------------------------------------------------------------------

function ResetToDefaults()
{
	Pending = 0;
	Selection = -1;
	bWaitingForInput = False;
	BuildKeyBindings();
	PopulateKeyList();
	ShowHelp(WaitingHelpText);
}

// ----------------------------------------------------------------------
// ----------------------------------------------------------------------

defaultproperties
{
	FunctionText(0)="Fire Weapon/Use object in hand"
	FunctionText(1)="Use object in world"
	FunctionText(2)="Drop/Throw Item"
	FunctionText(3)="Put Away Item"
	FunctionText(4)="Move Forward"
	FunctionText(5)="Move Backward"
	FunctionText(6)="Turn Left"
	FunctionText(7)="Turn Right"
	FunctionText(8)="Strafe Left"
	FunctionText(9)="Strafe Right"
	FunctionText(10)="Lean Left"
	FunctionText(11)="Lean Right"
	FunctionText(12)="Jump"
	FunctionText(13)="Crouch"
	FunctionText(14)="Mouse Look"
	FunctionText(15)="Look Up"
	FunctionText(16)="Look Down"
	FunctionText(17)="Center View"
	FunctionText(18)="Walk/Run"
	FunctionText(19)="Toggle Walk/Run"
	FunctionText(20)="Strafe"
	FunctionText(21)="Select Next Belt Item"
	FunctionText(22)="Select Previous Belt Item"
	FunctionText(23)="Reload Weapon"
	FunctionText(24)="Toggle Scope"
	FunctionText(25)="Toggle Laser Sight"
	FunctionText(26)="Activate All Augmentations"
	FunctionText(27)="Deactivate All Augmentations"
	FunctionText(28)="Change Ammo"
	FunctionText(29)="Take Screenshot"
	FunctionText(30)="Activate Inventory Screen"
	FunctionText(31)="Activate Health Screen"
	FunctionText(32)="Activate Augmentations Screen"
	FunctionText(33)="Activate Skills Screen"
	FunctionText(34)="Activate Goals/Notes Screen"
	FunctionText(35)="Activate Conversations Screen"
	FunctionText(36)="Activate Images Screen"
	FunctionText(37)="Activate Logs Screen"
	FunctionText(38)="Quick Save"
	FunctionText(39)="Quick Load"
	FunctionText(40)="Toggle Crosshairs"
	FunctionText(41)="Toggle Hit Display"
	FunctionText(42)="Toggle Compass"
	FunctionText(43)="Toggle Augmentation Display"
	FunctionText(44)="Toggle Object Belt"
	FunctionText(45)="Toggle Ammo Display"
	FunctionText(46)="F3"
	FunctionText(47)="F4"
	FunctionText(48)="F5"
	FunctionText(49)="F6"
	FunctionText(50)="F7"
	FunctionText(51)="F8"
	FunctionText(52)="F9"
	FunctionText(53)="F10"
	FunctionText(54)="F11"
	FunctionText(55)="F12"
	FunctionText(56)="Show MultiPlayer Scores"
	FunctionText(57)="Send Message To All"
	//FunctionText(58)="Send Message To Team"
	//FunctionText(59)="Activate MultiPlayer Skill Menu"
	//FunctionText(60)="Show MultiPlayer Kill Details"
	AliasNames(0)="ParseLeftClick|Fire"
	AliasNames(1)="ParseRightClick"
	AliasNames(2)="DropItem"
	AliasNames(3)="PutInHand"
	AliasNames(4)="MoveForward"
	AliasNames(5)="MoveBackward"
	AliasNames(6)="TurnLeft"
	AliasNames(7)="TurnRight"
	AliasNames(8)="StrafeLeft"
	AliasNames(9)="StrafeRight"
	AliasNames(10)="LeanLeft"
	AliasNames(11)="LeanRight"
	AliasNames(12)="Jump"
	AliasNames(13)="Duck"
	AliasNames(14)="Look"
	AliasNames(15)="LookUp"
	AliasNames(16)="LookDown"
	AliasNames(17)="CenterView"
	AliasNames(18)="Walking"
	AliasNames(19)="ToggleWalk"
	AliasNames(20)="Strafe"
	AliasNames(21)="NextBeltItem"
	AliasNames(22)="PrevBeltItem"
	AliasNames(23)="ReloadWeapon"
	AliasNames(24)="ToggleScope"
	AliasNames(25)="ToggleLaser"
	AliasNames(26)="ActivateAllAugs"
	AliasNames(27)="DeactivateAllAugs"
	AliasNames(28)="SwitchAmmo"
	AliasNames(29)="Shot"
	AliasNames(30)="ShowInventoryWindow"
	AliasNames(31)="ShowHealthWindow"
	AliasNames(32)="ShowAugmentationsWindow"
	AliasNames(33)="ShowSkillsWindow"
	AliasNames(34)="ShowGoalsWindow"
	AliasNames(35)="ShowConversationsWindow"
	AliasNames(36)="ShowImagesWindow"
	AliasNames(37)="ShowLogsWindow"
	AliasNames(38)="QuickSave"
	AliasNames(39)="QuickLoad"
	AliasNames(40)="ToggleCrosshair"
	AliasNames(41)="ToggleHitDisplay"
	AliasNames(42)="ToggleCompass"
	AliasNames(43)="ToggleAugDisplay"
	AliasNames(44)="ToggleObjectBelt"
	AliasNames(45)="ToggleAmmoDisplay"
	AliasNames(46)="DualMapF3"
	AliasNames(47)="DualMapF4"
	AliasNames(48)="DualMapF5"
	AliasNames(49)="DualMapF6"
	AliasNames(50)="DualMapF7"
	AliasNames(51)="DualMapF8"
	AliasNames(52)="DualMapF9"
	AliasNames(53)="DualMapF10"
	AliasNames(54)="DualMapF11"
	AliasNames(55)="DualMapF12"
	AliasNames(56)="ShowScores"
	AliasNames(57)="Talk"
	//AliasNames(58)="TeamTalk"
	//AliasNames(59)="BuySkills"
	//AliasNames(60)="KillerProfile"
	KeyDisplayNames(0)=(InputKey=IK_LeftMouse,DisplayName="Left Mouse Button")
	KeyDisplayNames(1)=(InputKey=IK_RightMouse,DisplayName="Right Mouse Button")
	KeyDisplayNames(2)=(InputKey=IK_MiddleMouse,DisplayName="Middle Mouse Button")
	KeyDisplayNames(3)=(InputKey=IK_CapsLock,DisplayName="CAPS Lock")
	KeyDisplayNames(4)=(InputKey=IK_PageUp,DisplayName="Page Up")
	KeyDisplayNames(5)=(InputKey=IK_PageDown,DisplayName="Page Down")
	KeyDisplayNames(6)=(InputKey=IK_PrintScrn,DisplayName="Print Screen")
	KeyDisplayNames(7)=(InputKey=IK_GreyStar,DisplayName="NumPad Asterisk")
	KeyDisplayNames(8)=(InputKey=IK_GreyPlus,DisplayName="NumPad Plus")
	KeyDisplayNames(9)=(InputKey=IK_GreyMinus,DisplayName="NumPad Minus")
	KeyDisplayNames(10)=(InputKey=IK_GreySlash,DisplayName="NumPad Slash")
	KeyDisplayNames(11)=(InputKey=IK_NumPadPeriod,DisplayName="NumPad Period")
	KeyDisplayNames(12)=(InputKey=IK_NumLock,DisplayName="Num Lock")
	KeyDisplayNames(13)=(InputKey=IK_ScrollLock,DisplayName="Scroll Lock")
	KeyDisplayNames(14)=(InputKey=IK_LShift,DisplayName="Left Shift")
	KeyDisplayNames(15)=(InputKey=IK_RShift,DisplayName="Right Shift")
	KeyDisplayNames(16)=(InputKey=IK_LControl,DisplayName="Left Control")
	KeyDisplayNames(17)=(InputKey=IK_RControl,DisplayName="Right Control")
	KeyDisplayNames(18)=(InputKey=IK_MouseWheelUp,DisplayName="Mouse Wheel Up")
	KeyDisplayNames(19)=(InputKey=IK_MouseWheelDown,DisplayName="Mouse Wheel Down")
	KeyDisplayNames(20)=(InputKey=IK_MouseX,DisplayName="Mouse X")
	KeyDisplayNames(21)=(InputKey=IK_MouseY,DisplayName="Mouse Y")
	KeyDisplayNames(22)=(InputKey=IK_MouseZ,DisplayName="Mouse Z")
	KeyDisplayNames(23)=(InputKey=IK_MouseW,DisplayName="Mouse W")
	KeyDisplayNames(24)=(InputKey=IK_LeftBracket,DisplayName="Left Bracket")
	KeyDisplayNames(25)=(InputKey=IK_RightBracket,DisplayName="Right Bracket")
	KeyDisplayNames(26)=(InputKey=IK_SingleQuote,DisplayName="Single Quote")
	KeyDisplayNames(27)=(InputKey=IK_Joy1,DisplayName="Joystick Button 1")
	KeyDisplayNames(28)=(InputKey=IK_Joy2,DisplayName="Joystick Button 2")
	KeyDisplayNames(29)=(InputKey=IK_Joy3,DisplayName="Joystick Button 3")
	KeyDisplayNames(30)=(InputKey=IK_Joy4,DisplayName="Joystick Button 4")
	KeyDisplayNames(31)=(InputKey=IK_JoyX,DisplayName="Joystick X")
	KeyDisplayNames(32)=(InputKey=IK_JoyY,DisplayName="Joystick Y")
	KeyDisplayNames(33)=(InputKey=IK_JoyZ,DisplayName="Joystick Z")
	KeyDisplayNames(34)=(InputKey=IK_JoyR,DisplayName="Joystick R")
	KeyDisplayNames(35)=(InputKey=IK_JoyU,DisplayName="Joystick U")
	KeyDisplayNames(36)=(InputKey=IK_JoyV,DisplayName="Joystick V")
	KeyDisplayNames(37)=(InputKey=IK_JoyPovUp,DisplayName="Joystick Pov Up")
	KeyDisplayNames(38)=(InputKey=IK_JoyPovDown,DisplayName="Joystick Pov Down")
	KeyDisplayNames(39)=(InputKey=IK_JoyPovLeft,DisplayName="Joystick Pov Left")
	KeyDisplayNames(40)=(InputKey=IK_JoyPovRight,DisplayName="Joystick Pov Right")
	KeyDisplayNames(41)=(InputKey=IK_Ctrl,DisplayName="Control")
	KeyDisplayNames(42)=(InputKey=IK_Left,DisplayName="Left Arrow")
	KeyDisplayNames(43)=(InputKey=IK_Right,DisplayName="Right Arrow")
	KeyDisplayNames(44)=(InputKey=IK_Up,DisplayName="Up Arrow")
	KeyDisplayNames(45)=(InputKey=IK_Down,DisplayName="Down Arrow")
	KeyDisplayNames(46)=(InputKey=IK_Insert,DisplayName="Insert")
	KeyDisplayNames(47)=(InputKey=IK_Home,DisplayName="Home")
	KeyDisplayNames(48)=(InputKey=IK_Delete,DisplayName="Delete")
	KeyDisplayNames(49)=(InputKey=IK_End,DisplayName="End")
	KeyDisplayNames(50)=(InputKey=IK_NumPad0,DisplayName="NumPad 0")
	KeyDisplayNames(51)=(InputKey=IK_NumPad1,DisplayName="NumPad 1")
	KeyDisplayNames(52)=(InputKey=IK_NumPad2,DisplayName="NumPad 2")
	KeyDisplayNames(53)=(InputKey=IK_NumPad3,DisplayName="NumPad 3")
	KeyDisplayNames(54)=(InputKey=IK_NumPad4,DisplayName="NumPad 4")
	KeyDisplayNames(55)=(InputKey=IK_NumPad5,DisplayName="NumPad 5")
	KeyDisplayNames(56)=(InputKey=IK_NumPad6,DisplayName="NumPad 6")
	KeyDisplayNames(57)=(InputKey=IK_NumPad7,DisplayName="NumPad 7")
	KeyDisplayNames(58)=(InputKey=IK_NumPad8,DisplayName="NumPad 8")
	KeyDisplayNames(59)=(InputKey=IK_NumPad9,DisplayName="NumPad 9")
	KeyDisplayNames(60)=(InputKey=IK_Period,DisplayName="Period")
	KeyDisplayNames(61)=(InputKey=IK_Comma,DisplayName="Comma")
	KeyDisplayNames(62)=(InputKey=IK_Backslash,DisplayName="Backslash")
	KeyDisplayNames(63)=(InputKey=IK_Semicolon,DisplayName="Semicolon")
	KeyDisplayNames(64)=(InputKey=IK_Equals,DisplayName="Equals")
	KeyDisplayNames(65)=(InputKey=IK_Slash,DisplayName="Slash")
	KeyDisplayNames(66)=(InputKey=IK_Enter,DisplayName="Enter")
	KeyDisplayNames(67)=(InputKey=IK_Alt,DisplayName="Alt")
	KeyDisplayNames(68)=(InputKey=IK_Backspace,DisplayName="Backspace")
	KeyDisplayNames(69)=(InputKey=IK_Shift,DisplayName="Shift")
	KeyDisplayNames(70)=(InputKey=IK_Space,DisplayName="Space")
	//KeyDisplayNames(71)=(InputKey=IK_Mouse4,DisplayName="Mouse Button 4")
	//KeyDisplayNames(72)=(InputKey=IK_Mouse5,DisplayName="Mouse Button 5")
	KeyDisplayNames(71)=(InputKey=IK_UnknownC1,DisplayName="Mouse Button 4")
	KeyDisplayNames(72)=(InputKey=IK_UnknownC2,DisplayName="Mouse Button 5")
	NoneText="[None]"
	StrHeaderActionLabel="Action"
	StrHeaderAssignedLabel="Assigned Key/Button"
	WaitingHelpText="Select the function you wish to remap and then press [Enter] or Double-Click.  Press [Delete] to remove Key bindings."
	InputHelpText="Please press the Key or Button you wish to assign to this function.  Press [ESC] to cancel."
	ReassignedFromLabel="'%s' reassigned from '%s'"
	ActionButtons(0)=(Align=HALIGN_Right,Action=AB_Cancel)
	ActionButtons(1)=(Align=HALIGN_Right,Action=AB_OK)
	ActionButtons(2)=(Action=AB_Reset)
	Title="Keyboard/Mouse Settings"
	ClientWidth=384
	ClientHeight=366
	ClientTextures(0)=Texture'DeusExUI.UserInterface.MenuCustomizeKeysBackground_1'
	ClientTextures(1)=Texture'DeusExUI.UserInterface.MenuCustomizeKeysBackground_2'
	ClientTextures(2)=Texture'DeusExUI.UserInterface.MenuCustomizeKeysBackground_3'
	ClientTextures(3)=Texture'DeusExUI.UserInterface.MenuCustomizeKeysBackground_4'
	TextureCols=2
	bHelpAlwaysOn=True
	HelpPosY=312
}
