//=============================================================================
// HXMenuUIBorderButtonWindow
//=============================================================================
class HXMenuUIBorderButtonWindow extends HXButtonWindow;

struct S_MenuUIBorderButtonTextures
{
	var Texture Texture;
	var int     Width;
};

var HXPlayerPawn Player;

var String ButtonText;
var bool   bTranslucentText;

// 0 = Normal
// 1 = Depressed :(
// 2 = Focus
// 3 = Insensitive

var S_MenuUIBorderButtonTextures Left_Textures[2];
var S_MenuUIBorderButtonTextures Right_Textures[2];
var S_MenuUIBorderButtonTextures Center_Textures[2];

var Color ButtonFaceColor;
var Color TextColor[4];

// Used when drawing Buttons.
var int TextureIndex;
var int TextOffset;
var int LeftOffset;
var int TextColorIndex;
var int ButtonVerticalOffset;
var int FontBaseLine;
var int FontAcceleratorLineHeight;

// Defaults.
var Font FontButtonText;
var int  VerticalTextMargin;
var int  ButtonHeight;
var int  MinimumButtonWidth;
var int  TextLeftMargin;
var int  MaxTextWidth;
var bool bTranslucent;
var bool bUseTextOffset;
var bool bCenterText;

// Sound Stuff.
var Sound MenuPressSound;
var Sound MenuFocusSound;

// ----------------------------------------------------------------------
// InitWindow()
//
// Initialize the Window
// ----------------------------------------------------------------------

event InitWindow()
{
	Super.InitWindow();

	EnableTextAsAccelerator( false );

	// Get a pointer to the Player
	Player = HXPlayerPawn(GetRootWindow().ParentPawn);

	SetButtonSounds( None, MenuPressSound );
	SetFocusSounds( MenuFocusSound );
	SetSoundVolume( 0.25 );

	SetBaselineData( FontBaseLine, FontAcceleratorLineHeight );

	StyleChanged();
}

// ----------------------------------------------------------------------
// DrawWindow()
// ----------------------------------------------------------------------

event DrawWindow(GC gc)
{	
	SetButtonMetrics();

	// Draw the textures
	if (bTranslucent)
		gc.SetStyle(DSTY_Translucent);
	else
		gc.SetStyle(DSTY_Masked);

	gc.SetTileColor(ButtonFaceColor);

	// Left
	gc.DrawTexture(
		leftOffset, buttonVerticalOffset, 
		Left_Textures[textureIndex].width, buttonHeight, 
		0, 0, 
		Left_Textures[textureIndex].Texture);

	// Center
	gc.DrawPattern(
		leftOffset + Left_Textures[textureIndex].width, buttonVerticalOffset, 
		width - Left_Textures[textureIndex].width - Right_Textures[textureIndex].width - leftOffset, buttonHeight, 
		0, 0,
		Center_Textures[textureIndex].Texture);

	// Right
	gc.DrawTexture(
		width - Right_Textures[textureIndex].width, buttonVerticalOffset, 
		Right_Textures[textureIndex].width, buttonHeight, 
		0, 0, 
		Right_Textures[textureIndex].Texture);

	// Draw the text!
	gc.SetFont(fontButtonText);
	gc.SetTextColor(TextColor[textColorIndex]);
	gc.EnableTranslucentText(bTranslucentText);
	gc.SetVerticalAlignment(VALIGN_Center);

	if (bCenterText)
		gc.SetHorizontalAlignment(HALIGN_Center);
	else
		gc.SetHorizontalAlignment(HALIGN_Left);

	if (bUseTextOffset)
	{
		gc.DrawText(
			textLeftMargin + textOffset, verticalTextMargin + textOffset + buttonVerticalOffset,
			width, height, buttonText);
	}
	else
	{
		gc.DrawText(
			textLeftMargin, verticalTextMargin + buttonVerticalOffset, 
			width, height, buttonText);
	}
}

// ----------------------------------------------------------------------
// SetVerticalOffset()
// ----------------------------------------------------------------------

function SetVerticalOffset( int NewButtonVerticalOffset )
{
	ButtonVerticalOffset = NewButtonVerticalOffset ;
}

// ----------------------------------------------------------------------
// ParentRequestedPreferredSize()
// ----------------------------------------------------------------------

event ParentRequestedPreferredSize(bool bWidthSpecified, out float preferredWidth,
                                   bool bHeightSpecified, out float preferredHeight)
{
	local float clientWidth, clientHeight;
	local float textWidth, textHeight;
	local GC gc;

	gc = GetGC();

	gc.SetFont(fontButtonText);
	gc.GetTextExtent(maxTextWidth, textWidth, textHeight, buttonText);

	preferredWidth  = Max(minimumButtonWidth, Left_Textures[0].width + textWidth + Right_Textures[0].width);
	preferredHeight = buttonHeight + buttonVerticalOffset;

	ReleaseGC(gc);
}

// ----------------------------------------------------------------------
// SetButtonMetrics()
//
// Calculates which set of Textures we're going to use as well as 
// any Text offset (used if the Button is pressed in)
// ----------------------------------------------------------------------

function SetButtonMetrics()
{
	if ( bIsSensitive )
	{
		// Button pressed.
		if ( bButtonPressed )				
		{
			TextureIndex   = 1;
			TextOffset     = 1;
			TextColorIndex = 2;
		}
		// Focus.
		else if ( IsFocusWindow() )
		{
			TextureIndex   = 0;
			TextOffset     = 0;
			TextColorIndex = 1;
		}
		// Normal.
		else
		{
			TextureIndex   = 0;
			TextOffset     = 0;
			TextColorIndex = 0;
		}
	}
	// Disabled.
	else
	{
		TextureIndex   = 0;
		TextOffset     = 0;
		TextColorIndex = 3;
	}
}	

// ----------------------------------------------------------------------
// SetButtonText()
// ----------------------------------------------------------------------

function SetButtonText( String NewText )
{
	ButtonText = NewText;

	SetAcceleratorText( NewText );

	AskParentForReconfigure();
}

// ----------------------------------------------------------------------
// SetCenterText()
// ----------------------------------------------------------------------

function SetCenterText( bool bNewCenter )
{
	bCenterText = bNewCenter;
}

// ----------------------------------------------------------------------
// StyleChanged()
// ----------------------------------------------------------------------

event StyleChanged()
{
	local ColorTheme Theme;

	Theme = Player.ThemeManager.GetCurrentMenuColorTheme();

	ButtonFaceColor = Theme.GetColorFromName('MenuColor_ButtonFace');
	TextColor[0]    = Theme.GetColorFromName('MenuColor_ButtonTextNormal');
	TextColor[1]    = Theme.GetColorFromName('MenuColor_ButtonTextFocus');
	TextColor[2]    = TextColor[1];
	TextColor[3]    = Theme.GetColorFromName('MenuColor_ButtonTextDisabled');
}

// ----------------------------------------------------------------------
// ----------------------------------------------------------------------

defaultproperties
{
	MenuPressSound=Menu_Press
	MenuFocusSound=Menu_Focus
	ButtonFaceColor=(R=255,G=255,B=255)
	TextColor(0)=(R=200,G=200,B=200)
	TextColor(1)=(R=255,G=255,B=255)
	TextColor(2)=(R=255,G=255,B=255)
	TextColor(3)=(R=50,G=50,B=50)
	FontBaseLine=1
	FontAcceleratorLineHeight=1
	FontButtonText=Font'DeusExUI.FontMenuSmall'
	VerticalTextMargin=1
	MaxTextWidth=200
	bUseTextOffset=True
}
