//=============================================================================
// HXMenuUITitleWindow
//=============================================================================
class HXMenuUITitleWindow extends HXWindow;

// Constraints.
var int	TitleHeight;
var int TitleMinWidth;
var int TitleLeftMargin;
var int TitleRightMargin;
var int LeftWidth;
var int RightWidth;
var int CenterWidth;
var int LeftBottomWidth;
var int LeftBottomHeight;
var int TextMaxWidth;

// Fonts.
var Font TitleFont;

// Colors.
var Color TitleColor;
var Color TitleColorText;
var Color BubbleColor;

// Textures.
var Texture IconTexture;
var Texture BarTextureLeft;
var Texture BarTextureCenter;
var Texture BarTextureRight;
var Texture BarTextureLeftBottom;
var Texture BubbleTextureLeft;
var Texture BubbleTextureCenter;
var Texture BubbleTextureRight;

// Text.
var String TitleText;

// Windows.
var HXWindow IconWindow;

// ----------------------------------------------------------------------------
// InitWindow()
//
// Initialize the Window
// ----------------------------------------------------------------------------

event InitWindow()
{
	Super.InitWindow();

	CreateIconWindow();
}

// ----------------------------------------------------------------------------
// DrawWindow()
//
// The Title is drawn as several pieces.
// 1.  Left piece
// 2.  Left bottom piece
// 3.  Middle (tiling)
// 4.  Right
// 5.  Icon (draws itself)
// 6.  Title text
// ----------------------------------------------------------------------------

event DrawWindow( GC Context )
{
	local float BarLeftRansomU, BarLeftRansomV, BarCenterRansomV, BarRightRansomU, BarRightRansomV, BarLeftBottomRansomU, BarLeftBottomRansomV;
	local float BubbleLeftRansomU, BubbleLeftRansomV, BubbleCenterRansomV, BubbleRightRansomU, BubbleRightRansomV;

	// Calculate Modifier for DrawStrechedTexture.
	BarLeftRansomU       = float(LeftWidth)       /BarTextureLeft.USize;
	BarLeftRansomV       = float(TitleHeight)     /BarTextureLeft.VSize;
	BarCenterRansomV     = float(TitleHeight)     /BarTextureCenter.VSize;
	BarRightRansomU      = float(RightWidth)      /BarTextureRight.USize;
	BarRightRansomV      = float(TitleHeight)     /BarTextureRight.VSize;
	BarLeftBottomRansomU = float(LeftBottomWidth) /BarTextureLeftBottom.USize;
	BarLeftBottomRansomV = float(LeftBottomHeight)/BarTextureLeftBottom.VSize;
	BubbleLeftRansomU    = float(LeftWidth)       /BubbleTextureLeft.USize;
	BubbleLeftRansomV    = float(TitleHeight)     /BubbleTextureLeft.VSize;
	BubbleCenterRansomV  = float(TitleHeight)     /BubbleTextureCenter.VSize;
	BubbleRightRansomU   = float(RightWidth)      /BubbleTextureRight.USize;
	BubbleRightRansomV   = float(TitleHeight)     /BubbleTextureRight.VSize;

	// Draw the Title Background textures.
	Context.SetStyle( DSTY_Masked );
	Context.SetTileColor( TitleColor );

	// Left.
	Context.DrawStretchedTexture( 0, 0, LeftWidth, TitleHeight, 0, 0, BarLeftRansomU*BarTextureLeft.USize, BarLeftRansomV*BarTextureLeft.VSize, BarTextureLeft );

	// Left Bottom.
	Context.DrawStretchedTexture( 0, TitleHeight-1, LeftBottomWidth, LeftBottomHeight, 0, 0, BarLeftBottomRansomU*BarTextureLeftBottom.USize, BarLeftBottomRansomV*BarTextureLeftBottom.VSize, BarTextureLeftBottom );

	// Center
	Context.DrawStretchedTexture( LeftWidth, 0, Width-LeftWidth-RightWidth, TitleHeight, 0, 0, BarTextureCenter.USize, BarCenterRansomV*BarTextureCenter.VSize, BarTextureCenter );

	// Right
	Context.DrawStretchedTexture( Width-RightWidth, 0, RightWidth, TitleHeight, 0, 0, BarRightRansomU*BarTextureRight.USize, BarRightRansomV*BarTextureRight.VSize, BarTextureRight );

	// Now draw the Bubble textures.
	Context.SetStyle( DSTY_Masked );
	Context.SetTileColor( BubbleColor );

	// Left
	Context.DrawStretchedTexture( 0, 0, LeftWidth, TitleHeight, 0, 0, BubbleLeftRansomU*BubbleTextureLeft.USize, BubbleLeftRansomV*BubbleTextureLeft.VSize, BubbleTextureLeft );

	// Center
	Context.DrawStretchedTexture( LeftWidth, 0, Width-LeftWidth-RightWidth, TitleHeight, 0, 0, BubbleTextureCenter.USize, BubbleCenterRansomV*BubbleTextureCenter.VSize, BubbleTextureCenter );

	// Right
	Context.DrawStretchedTexture( Width-RightWidth, 0, RightWidth, TitleHeight, 0, 0, BubbleRightRansomU*BubbleTextureRight.USize, BubbleRightRansomV*BubbleTextureRight.VSize, BubbleTextureRight );

	// Draw the text!
	Context.SetFont( TitleFont );
	Context.SetTextColor( TitleColorText );
	Context.SetHorizontalAlignment( HALIGN_Left );
	Context.SetVerticalAlignment( VALIGN_Center );

	Context.DrawText( TitleLeftMargin, 0, Width-TitleLeftMargin, TitleHeight, TitleText );

	Super.DrawWindow(Context);
}

// ----------------------------------------------------------------------------
// ParentRequestedPreferredSize()
// ----------------------------------------------------------------------------

event ParentRequestedPreferredSize( bool bWidthSpecified, out float PreferredWidth, bool bHeightSpecified, out float PreferredHeight )
{
	local float TextWidth, TextHeight;
	local GC Context;

	Context = GetGC();

	Context.SetFont( TitleFont );
	Context.GetTextExtent( TextMaxWidth, textWidth, textHeight, TitleText );

	PreferredWidth  = Max( TitleMinWidth, TitleLeftMargin+TextWidth+TitleRightMargin );
	PreferredHeight = TitleHeight+LeftBottomHeight;

	ReleaseGC( Context );
}

// ----------------------------------------------------------------------------
// CreateIconWindow()
// ----------------------------------------------------------------------------

function CreateIconWindow()
{
	IconWindow = HXWindow(NewChild( Class'HXWindow' ));
	IconWindow.SetPos( 12, 3 );
	IconWindow.SetSize( 16, 16 );
	IconWindow.SetBackground( IconTexture );
}

// ----------------------------------------------------------------------------
// SetTitle()
// ----------------------------------------------------------------------------

function SetTitle( string InTitle )
{
	TitleText = InTitle;
	AskParentForReconfigure();
}

// ----------------------------------------------------------------------------
// GetOffsetWidths()
// ----------------------------------------------------------------------------

function GetOffsetWidths( out int TitleOffsetX, out int TitleOffsetY )
{
	TitleOffsetX = LeftBottomWidth;
	TitleOffsetY = TitleHeight;
}

// ----------------------------------------------------------------------------
// StyleChanged()
// ----------------------------------------------------------------------------

event StyleChanged()
{
	local ColorTheme Theme;

	Theme = PlayerPawn.ThemeManager.GetCurrentMenuColorTheme();

	// Title colors.
	TitleColor     = Theme.GetColorFromName( 'MenuColor_ButtonFace'      );
	TitleColorText = Theme.GetColorFromName( 'MenuColor_TitleText'       );
	BubbleColor    = Theme.GetColorFromName( 'MenuColor_TitleBackground' );
}

// ----------------------------------------------------------------------------
// ----------------------------------------------------------------------------

defaultproperties
{
	TitleHeight=23
	TitleMinWidth=200
	TitleLeftMargin=31
	TitleRightMargin=45
	TextMaxWidth=400
	TitleFont=FontMenuTitle
	TitleColor=(R=255,G=255,B=255)
	TitleColorText=(R=255,G=255,B=255)
	BubbleColor=(R=255,G=255,B=255)
	IconTexture=MenuIcon_DeusEx
	BarTextureLeft=MenuTitleBar_Left
	BarTextureCenter=MenuTitleBar_Center
	BarTextureRight=MenuTitleBar_Right
	BarTextureLeftBottom=MenuTitleBar_LeftBottom
	BubbleTextureLeft=MenuTitleBubble_Left
	BubbleTextureCenter=MenuTitleBubble_Center
	BubbleTextureRight=MenuTitleBubble_Right
	LeftWidth=87
	RightWidth=100
	CenterWidth=8
	LeftBottomWidth=10
	//LeftBottomWidth=11
	LeftBottomHeight=53
}
