//=============================================================================
// HXMenuUIWindow
//
// Base class for the Menu windows.
//=============================================================================
class HXMenuUIWindow extends HXDeusExBaseWindow
	abstract;

enum EActionButtonEvents
{
	AB_None,
	AB_OK,
	AB_Cancel,
	AB_Reset,
	AB_Previous,
	AB_Other
};

// Constants for MessageBoxMode.
const MB_Exit = 0;

struct S_ActionButtonDefault
{
	var EHAlign Align;
	var EActionButtonEvents Action;
	var localized String Text;
	var String Key;
	var MenuUIActionButtonWindow Button;
};

var localized S_ActionButtonDefault ActionButtons[5];

var int MessageBoxMode;

var HXMenuUITitleWindow           TitleWindow;      // Title bar, outside Client
var HXMenuUIClientWindow          ClientWindow;     // Window that contains all controls
var   MenuUIActionButtonBarWindow ButtonBarWindow;   // Button Bar Window
var   MenuUILeftEdgeWindow        LeftEdgeWindow;
var   MenuUIRightEdgeWindow       RightEdgeWindow;
var   MenuUIHelpWindow            HelpWindow;
var HXMenuUIShadowWindow          ShadowWindow;

// Dragging stuff
var bool	bWindowBeingDragged;
var bool	bAllowWindowDragging;
var float	WindowStartDragX;
var float	WindowStartDragY;

// Defaults
var localized String Title;
var int ClientWidth;
var int ClientHeight;
var int VerticalOffset;
var Texture ClientTextures[6];
var int TextureRows;
var int TextureCols;
var bool bActionButtonBarActive;
var bool bLeftEdgeActive;
var bool bRightEdgeActive;
var bool bUsesHelpWindow;
var bool bHelpAlwaysOn;
var bool bEscapeSavesSettings;
var int  HelpPositionY;
var int  DefaultHelpLeftOffset;
var int  DefaultHelpHeight;
var int  DefaultHelpClientDiffY;

// Shadow stuff.
var class<HXMenuUIShadowWindow> ShadowWindowClass;
var int ShadowOffsetX;
var int ShadowOffsetY;
var int ShadowWidth;
var int ShadowHeight;

// Sound stuff.
var Sound MenuSoundOk;
var Sound MenuSoundCancel;
var Sound MenuSoundActivate;
var Sound MenuSoundPress;

var localized string ButtonLabelOK;
var localized string ButtonLabelCancel;
var localized string ButtonLabelPrevious;
var localized string ButtonLabelResetToDefaults;
var localized string MessageBoxTitle;
var localized string ExitMessage;

// ----------------------------------------------------------------------
// InitWindow()
//
// Initialize the Window
// ----------------------------------------------------------------------

event InitWindow()
{
	Super.InitWindow();

	bSizeParentToChildren = False;   // god damnit

	CreateControls();	
	StyleChanged();

	// Play Menu Activated Sound.
	if ( MenuSoundActivate!=None )
		PlaySound( MenuSoundActivate, 0.25 ); 
}

// ----------------------------------------------------------------------
// CreateControls()
// ----------------------------------------------------------------------

function CreateControls()
{
	CreateShadowWindow();
	CreateTitleWindow();
	CreateClientWindow();
	CreateActionButtonBar();
	CreateActionButtons();
	CreateLeftEdgeWindow();
	CreateRightEdgeWindow();
	CreateHelpWindow();
}

// ----------------------------------------------------------------------
// CreateShadowWindow()
// ----------------------------------------------------------------------

function CreateShadowWindow()
{
	if ( ShadowWindowClass!=None )
	{
		ShadowWindow = HXMenuUIShadowWindow(NewChild(ShadowWindowClass));

		// Store these away so we can properly calculate window sizes/Positions
		ShadowOffsetX = ShadowWindow.ShadowOffsetX;
		ShadowOffsetY = ShadowWindow.ShadowOffsetY;
		ShadowWidth   = ShadowWindow.ShadowWidth;
		ShadowHeight  = ShadowWindow.ShadowHeight;
	}
}

// ----------------------------------------------------------------------
// CreateTitleWindow()
// ----------------------------------------------------------------------

function CreateTitleWindow()
{
	TitleWindow = HXMenuUITitleWindow(NewChild(Class'HXMenuUITitleWindow'));
	SetTitle(Title);
}

// ----------------------------------------------------------------------
// SetTitle()
// ----------------------------------------------------------------------

function SetTitle(String NewTitle)
{
	TitleWindow.SetTitle(NewTitle);
}

// ----------------------------------------------------------------------
// CreateClientWindow()
// ----------------------------------------------------------------------

function CreateClientWindow()
{
	local int iClientTexture;
	local int TitleOffsetX, TitleOffsetY;

	TitleWindow.GetOffsetWidths( TitleOffsetX, TitleOffsetY );

	ClientWindow = HXMenuUIClientWindow(NewChild(Class'HXMenuUIClientWindow'));
	ClientWindow.SetSize( ClientWidth, ClientHeight );
	ClientWindow.SetTextureLayout( TextureCols, TextureRows );

	// Set background textures
	for( iClientTexture=0; iClientTexture<ArrayCount(ClientTextures); iClientTexture++ )
	{
		ClientWindow.SetClientTexture( iClientTexture, ClientTextures[iClientTexture] );
	}
}

// ----------------------------------------------------------------------
// CreateActionButtonBar()
// ----------------------------------------------------------------------

function MenuUIActionButtonBarWindow CreateActionButtonBar()
{
	// Only create if we're supposed to create it.
	if ( bActionButtonBarActive )
		ButtonBarWindow = MenuUIActionButtonBarWindow(NewChild(Class'MenuUIActionButtonBarWindow'));
}

// ----------------------------------------------------------------------
// CreateActionButtons()
// ----------------------------------------------------------------------

function CreateActionButtons()
{
	local int    ButtonIndex;
	local string ButtonText;

	if ( ButtonBarWindow==None )
		return;

	for( ButtonIndex=0; ButtonIndex<ArrayCount(ActionButtons); ButtonIndex++ )
	{
		if ( ActionButtons[ButtonIndex].Action!=AB_None )
		{
			// First get the string
			switch(ActionButtons[ButtonIndex].Action)
			{
				case AB_OK:
					ButtonText = ButtonLabelOK;
					break;

				case AB_Cancel:
					ButtonText = ButtonLabelCancel;
					break;

				case AB_Reset:
					ButtonText = ButtonLabelResetToDefaults;
					break;

				case AB_Previous:
					ButtonText = ButtonLabelPrevious;
					break;

				case AB_Other:
					ButtonText = ActionButtons[ButtonIndex].text;
					break;
			}

			ActionButtons[ButtonIndex].Button = ButtonBarWindow.AddButton( ButtonText, ActionButtons[ButtonIndex].Align );
		}
		else
		{
			break;
		}
	}
}

// ----------------------------------------------------------------------
// CreateLeftEdgeWindow()
// ----------------------------------------------------------------------

function CreateLeftEdgeWindow()
{
	if ( bLeftEdgeActive )
		LeftEdgeWindow = MenuUILeftEdgeWindow(NewChild(Class'MenuUILeftEdgeWindow'));
}

// ----------------------------------------------------------------------
// CreateRightEdgeWindow()
// ----------------------------------------------------------------------

function CreateRightEdgeWindow()
{
	if ( bRightEdgeActive )
	{
		RightEdgeWindow = MenuUIRightEdgeWindow(NewChild(Class'MenuUIRightEdgeWindow'));
		RightEdgeWindow.Lower();
	}
}

// ----------------------------------------------------------------------
// CreateMenuUITab()
// ----------------------------------------------------------------------

function MenuUITabButtonWindow CreateMenuUITab(int TabX, int TabY, String TabButtonText )
{
	local MenuUITabButtonWindow NewButton;

	NewButton = MenuUITabButtonWindow(ClientWindow.NewChild(Class'MenuUITabButtonWindow'));
	NewButton.SetPos( TabX, TabY );
	NewButton.SetButtonText( TabButtonText );

	return NewButton;
}

// ----------------------------------------------------------------------
// CreateHelpWindow()
//
// Optionally creates the Help window, which is displayed at the 
// bottom of the screen when various choice controls are given
// focus.  Can also be forced on/off programmatically.
// ----------------------------------------------------------------------

function CreateHelpWindow()
{
	if ( bUsesHelpWindow )
	{
		HelpWindow = MenuUIHelpWindow(NewChild(Class'MenuUIHelpWindow'));

		if (!bHelpAlwaysOn)
			HelpWindow.Hide();
	}
}

// ----------------------------------------------------------------------
// ParentRequestedPreferredSize()
// ----------------------------------------------------------------------

event ParentRequestedPreferredSize( bool bWidthSpecified, out float PreferredWidth, bool bHeightSpecified, out float PreferredHeight )
{
	local float ClientWidth, ClientHeight;
	local float TitleWidth, TitleHeight;
	local float ButtonBarWidth, ButtonBarHeight;
	local float RightEdgeWidth;

	if ( ClientWindow!=None )
		ClientWindow.QueryPreferredSize( ClientWidth, ClientHeight );

	if ( TitleWindow!=None )
	{
		TitleWidth  = TitleWindow.LeftBottomWidth;
		TitleHeight = TitleWindow.TitleHeight;
	}

	if ( RightEdgeWindow!=None )
		RightEdgeWidth = RightEdgeWindow.RightWidth;

	if ( ButtonBarWindow!=None )
		ButtonBarWindow.QueryPreferredSize( ButtonBarWidth, ButtonBarHeight );

	PreferredWidth  = ClientWidth +TitleWidth +RightEdgeWidth;
	PreferredHeight = ClientHeight+TitleHeight+ButtonBarHeight+VerticalOffset;

	// Take into account Shadow crap.
	if ( PreferredWidth<ShadowWidth )
		PreferredWidth = ShadowWidth;
	if (PreferredHeight<ShadowHeight )
		PreferredHeight = ShadowHeight;
}

// ----------------------------------------------------------------------
// ConfigurationChanged()
// ----------------------------------------------------------------------

function ConfigurationChanged()
{
	local float ClientWidth, ClientHeight;
	local float TitleWidth, TitleHeight, TitleTopHeight, TitleBarWidth;
	local float LeftEdgeWidth, LeftEdgeHeight;
	local float RightEdgeWidth, RightEdgeHeight;
	local float ButtonBarWidth, ButtonBarHeight;
	local float RightEdgeX, RightEdgeY;
	local float LeftEdgeWindowGap;
	local float RightEdgeGap;
	local int TitleOffsetX, TitleOffsetY;

	// Title Window.
	if ( TitleWindow!=None)
		TitleWindow.GetOffsetWidths( TitleOffsetX, TitleOffsetY );

	// Client Window.
	if ( ClientWindow!=None )
	{
		ClientWindow.QueryPreferredSize( ClientWidth, ClientHeight );
		ClientWindow.ConfigureChild( TitleOffsetX+ShadowOffsetX, TitleOffsetY+ShadowOffsetY, ClientWidth, ClientHeight );
	}

	// Title Bar.
	if ( TitleWindow!=None )
	{
		TitleWindow.QueryPreferredSize( TitleBarWidth, TitleHeight );
		TitleWindow.ConfigureChild( ShadowOffsetX, ShadowOffsetY, TitleBarWidth, TitleHeight );
		TitleWidth     = TitleWindow.LeftBottomWidth;
		TitleTopHeight = TitleWindow.TitleHeight;
	}

	// Button Bar.
	if ( ButtonBarWindow!=None )
	{
		// If Right edge active, need to make Button bar wider
		if ( RightEdgeWindow!=None )
		{
			RightEdgeWindow.QueryPreferredSize(RightEdgeWidth, RightEdgeHeight);
			RightEdgeWidth = RightEdgeWindow.RightWidth;
			RightEdgeGap   = 2;
		}

		if ( LeftEdgeWindow!=None )
			LeftEdgeWindowGap = 1;

		ButtonBarWindow.QueryPreferredSize( ButtonBarWidth, ButtonBarHeight );
		ButtonBarWindow.ConfigureChild( TitleWidth+LeftEdgeWindowGap+ShadowOffsetX, TitleTopHeight+ClientHeight+ShadowOffsetY, ClientWidth-1-RightEdgeGap, ButtonBarHeight );
	}

	// Left Edge.
	if ( LeftEdgeWindow != None)
	{
		LeftEdgeWindow.QueryPreferredSize(LeftEdgeWidth, LeftEdgeHeight);
		LeftEdgeWindow.ConfigureChild( TitleWidth-LeftEdgeWidth+ShadowOffsetX, TitleHeight+ShadowOffsetY, LeftEdgeWidth, ClientHeight+ButtonBarHeight-(TitleHeight-TitleTopHeight)-1 );
	}

	// Right Edge.
	if ( RightEdgeWindow!=None )
	{
		RightEdgeWindow.ConfigureChild( TitleBarWidth+1+ShadowOffsetX, TitleTopHeight-4+ShadowOffsetY, ClientWidth-(TitleBarWidth-TitleWidth)-1+RightEdgeWindow.RightWidth, ClientHeight+4+14 );
	}

	// Help Window.
	if ( HelpWindow!=None )
	{
		HelpWindow.ConfigureChild( TitleWidth+DefaultHelpLeftOffset+ShadowOffsetX, TitleTopHeight+HelpPositionY+ShadowOffsetY, ClientWidth-DefaultHelpClientDiffY, DefaultHelpHeight );
	}
}

// ----------------------------------------------------------------------
// ChildRequestedReconfiguration()
// ----------------------------------------------------------------------

function bool ChildRequestedReconfiguration( Window Child )
{
	return False;
}

// ----------------------------------------------------------------------
// VirtualKeyPressed()
//
// Called when a Key is pressed; provides a virtual Key value
// ----------------------------------------------------------------------

event bool VirtualKeyPressed( EInputKey Key, bool bRepeat )
{
	local bool bKeyHandled;
	bKeyHandled = True;

	Super.VirtualKeyPressed(Key, bRepeat);

	if ( IsKeyDown(IK_Alt) || IsKeyDown(IK_Shift) || IsKeyDown(IK_Ctrl) )
		return False;

	switch( Key ) 
	{	
		// Hide the screen if the Escape Key is pressed
		// Temp: Also if the Return Key is pressed
		case IK_Escape:
			if (bEscapeSavesSettings)
			{
				SaveSettings();
				Root.PopWindow();
			}
			else
			{
				CancelScreen();
			}
			break;

		// Enter is the same as clicking OK
		case IK_Enter:
			SaveSettings();
			Root.PopWindow();
			break;

		Default:
			bKeyHandled = False;
	}

	return bKeyHandled;
}

// ----------------------------------------------------------------------
// ButtonActivated()
// ----------------------------------------------------------------------

function bool ButtonActivated( Window ButtonPressed )
{
	local bool bHandled;

	bHandled = False;

	// Check to see if this was an Action Button!
	if ( ButtonPressed.IsA('MenuUIActionButtonWindow') )
		bHandled = ProcessActionButton(MenuUIActionButtonWindow(ButtonPressed));

	if ( bHandled )
		return bHandled;
	else
		return Super.ButtonActivated( ButtonPressed );
}

// ----------------------------------------------------------------------
// MouseMoved()
//
// If we're dragging the window around, move
// ----------------------------------------------------------------------

event MouseMoved( float NewX, float NewY )
{
	if ( bWindowBeingDragged )
		SetPos( X+NewX-WindowStartDragX, Y+NewY-WindowStartDragY );
}

// ----------------------------------------------------------------------
// MouseButtonPressed() 
//
// If the user presses the mouse Button while over the Title bar,
// initiate window dragging.
// ----------------------------------------------------------------------

event bool MouseButtonPressed( float PointX, float PointY, EInputKey Button, int NumClicks )
{
	local float RelX, RelY;

	if ( Button==IK_LeftMouse && NumClicks==1 && (FindWindow(PointX,PointY,RelX,RelY)==TitleWindow || bAllowWindowDragging) )
	{
		bWindowBeingDragged = True;
		WindowStartDragX = PointX;
		WindowStartDragY = PointY;
		GrabMouse();
	}
	else
		bWindowBeingDragged = False;
	return bWindowBeingDragged;  
}

// ----------------------------------------------------------------------
// MouseButtonReleased() 
//
// First check to see if we're dragging the window.  If so, then
// end the drag event.
// ----------------------------------------------------------------------

event bool MouseButtonReleased( float PointX, float PointY, EInputKey Button, int NumClicks )
{
	if ( Button==IK_LeftMouse && bWindowBeingDragged )
		bWindowBeingDragged = False;
	return False;
}

// ----------------------------------------------------------------------
// ShowHelp()
// ----------------------------------------------------------------------

function ShowHelp( String HelpMessage )
{
	if ( HelpWindow!=None )
	{
		HelpWindow.Show();
		HelpWindow.SetText( HelpMessage );
	}
}

// ----------------------------------------------------------------------
// HideHelp()
// ----------------------------------------------------------------------

function HideHelp()
{
	if ( HelpWindow!=None )
		HelpWindow.Hide();
}

// ----------------------------------------------------------------------
// ProcessActionButton()
// ----------------------------------------------------------------------

function bool ProcessActionButton( MenuUIActionButtonWindow ActionButton )
{
	local int  ButtonIndex;
	local bool bHandled;

	bHandled = False;

	// Find out which Button this is in our array.
	for ( ButtonIndex=0; ButtonIndex<arrayCount(ActionButtons); ButtonIndex++ )
	{
		if ( ActionButtons[ButtonIndex].Button==ActionButton )
		{
			switch( ActionButtons[ButtonIndex].Action )
			{
				case AB_OK:
					SaveSettings();
					Root.PopWindow();
					break;

				case AB_Cancel:
					CancelScreen();
					break;

				case AB_Reset:
					ResetToDefaults();
					break;

				case AB_Previous:
					SaveSettings();
					Root.PopWindow();
					break;

				case AB_Other:
					ProcessAction(ActionButtons[ButtonIndex].Key);
					break;		
			}

			bHandled = True;
			break;
		}
	}

	return bHandled;
}

// ----------------------------------------------------------------------
// ProcessAction()
//
// Meant to be called in derived class.
// ----------------------------------------------------------------------

function ProcessAction( String ActionKey )
{
}

// ----------------------------------------------------------------------
// SaveSettings()
//
// Meant to be called in derived class.
// ----------------------------------------------------------------------

function SaveSettings()
{
	// Play OK Sound.
	if ( MenuSoundOk!=None )
		PlaySound( MenuSoundOk, 0.25 );
}

// ----------------------------------------------------------------------
// CancelScreen()
// ----------------------------------------------------------------------

function CancelScreen()
{
	// Play Cancel Sound.
	if ( MenuSoundCancel!=None )
		PlaySound( MenuSoundCancel, 0.25 ); 

	Root.PopWindow();
}

// ----------------------------------------------------------------------
// ResetToDefaults()
//
// Meant to be called in derived class
// ----------------------------------------------------------------------

function ResetToDefaults()
{
}

// ----------------------------------------------------------------------
// BoxOptionSelected()
// ----------------------------------------------------------------------

event bool BoxOptionSelected( Window Button, int ButtonNumber )
{
	// Destroy the msgbox!  
	Root.PopWindow();

	switch ( MessageBoxMode )
	{
		case MB_Exit:
			if ( ButtonNumber==0 ) 
			{
				/* TODO: This is what Unreal Does,
				Player.SaveConfig();
				if ( Level.Game != None )
					Level.Game.SaveConfig();
				*/
				Root.ExitGame();
			}
			break;
	}

	return True;
}

// ----------------------------------------------------------------------
// CreateMenuLabel()
// ----------------------------------------------------------------------

function MenuUILabelWindow CreateMenuLabel( int LabelX, int LabelY, String LabelText, Window WinParent )
{
	local MenuUILabelWindow NewLabel;

	NewLabel = MenuUILabelWindow(WinParent.NewChild(Class'MenuUILabelWindow'));
	NewLabel.SetPos( LabelX, LabelY );
	NewLabel.SetText( LabelText );

	return NewLabel;
}

// ----------------------------------------------------------------------
// CreateMenuHeader()
// ----------------------------------------------------------------------

function MenuUIHeaderWindow CreateMenuHeader(int HeaderX, int HeaderY, String HeaderText, Window WinParent )
{
	local MenuUIHeaderWindow NewLabel;

	NewLabel = MenuUIHeaderWindow(WinParent.NewChild(Class'MenuUIHeaderWindow'));
	NewLabel.SetPos( HeaderX, HeaderY );
	NewLabel.SetText( HeaderText);

	return NewLabel;
}

// ----------------------------------------------------------------------
// CreateMenuEditWindow()
// ----------------------------------------------------------------------

function MenuUIEditWindow CreateMenuEditWindow( int EditX, int EditY, int EditWidth, int MaxSize, Window WinParent )
{
	local MenuUIInfoButtonWindow InfoButton;
	local ClipWindow             ClipName;
	local MenuUIEditWindow       NewEdit;

	// Create an info Button behind this sucker
	InfoButton = MenuUIInfoButtonWindow(WinParent.NewChild(Class'MenuUIInfoButtonWindow'));
	InfoButton.SetPos( EditX, EditY );
	InfoButton.SetWidth( EditWidth);
	InfoButton.SetSensitivity( False);

	ClipName = ClipWindow(ClientWindow.NewChild(Class'ClipWindow'));
	ClipName.SetWidth( EditWidth-8 );
	ClipName.ForceChildSize( False, True );
	ClipName.SetPos( EditX+4, EditY+5 );

	NewEdit = MenuUIEditWindow( ClipName.NewChild(Class'MenuUIEditWindow') );
	NewEdit.SetMaxSize( MaxSize );

	return NewEdit;
}

// ----------------------------------------------------------------------
// CreateHeaderButton()
// ----------------------------------------------------------------------

function MenuUIListHeaderButtonWindow CreateHeaderButton( int PosX, int PosY, int ButtonWidth, String StrLabel, Window WinParent )
{
	local MenuUIListHeaderButtonWindow NewButton;

	NewButton = MenuUIListHeaderButtonWindow(WinParent.NewChild(Class'MenuUIListHeaderButtonWindow'));
	NewButton.SetPos( PosX, PosY );
	NewButton.SetWidth( ButtonWidth );
	NewButton.SetButtonText( StrLabel );

	return NewButton;
}

// ----------------------------------------------------------------------
// CreateScrollAreaWindow()
// ----------------------------------------------------------------------

function MenuUIScrollAreaWindow CreateScrollAreaWindow(Window WinParent)
{
	return MenuUIScrollAreaWindow(WinParent.NewChild(Class'MenuUIScrollAreaWindow'));
}

// ----------------------------------------------------------------------
// EnableActionButton()
//
// Checks the state of the list control and updates the pushButtons
// appropriately
// ----------------------------------------------------------------------

function EnableActionButton( EActionButtonEvents Action, bool bEnable, optional String Key )
{
	local int ButtonIndex;

	// First find our Button to make sure it exists
	for( ButtonIndex=0; ButtonIndex<ArrayCount(ActionButtons); ButtonIndex++ )
	{
		if ( ActionButtons[ButtonIndex].Action==Action )
		{
			if ( Action!=AB_Other || (Action==AB_Other && ActionButtons[ButtonIndex].Key==Key) )
			{
				if ( ActionButtons[ButtonIndex].Button!=None )
					ActionButtons[ButtonIndex].Button.SetSensitivity( bEnable );
				break;
			}
		}
	}
}

// ----------------------------------------------------------------------
// IsActionButtonEnabled()
// ----------------------------------------------------------------------

function bool IsActionButtonEnabled( EActionButtonEvents Action, optional String Key )
{
	local int ButtonIndex;
	local bool bButtonEnabled;

	bButtonEnabled = False;

	// First find our Button to make sure it exists.
	for( ButtonIndex=0; ButtonIndex<ArrayCount(ActionButtons); ButtonIndex++ )
	{
		if ( ActionButtons[ButtonIndex].Action==Action )
		{
			if ( Action!=AB_Other || (Action==AB_Other && ActionButtons[ButtonIndex].Key==Key) )
			{
				if ( ActionButtons[ButtonIndex].Button!=None )
				{
					bButtonEnabled = ActionButtons[ButtonIndex].Button.IsSensitive();
					break;
				}
			}
		}
	}

	return bButtonEnabled;
}

// ======================================================================
// ======================================================================
// Color Scheme Functions
// ======================================================================
// ======================================================================

// ----------------------------------------------------------------------
// StyleChanged()
// ----------------------------------------------------------------------

event StyleChanged()
{
	local ColorTheme Theme;
	local Color CursorColor;

	Theme = Player.ThemeManager.GetCurrentMenuColorTheme();

	CursorColor = Theme.GetColorFromName('MenuColor_Cursor');

	SetDefaultCursor( Texture'DeusExCursor1', Texture'DeusExCursor1_Shadow', 32, 32, CursorColor );
}

// ----------------------------------------------------------------------
// ----------------------------------------------------------------------

Defaultproperties
{
	DefaultHelpLeftOffset=7
	DefaultHelpHeight=27
	DefaultHelpClientDiffY=21
	bUsesHelpWindow=True
	bEscapeSavesSettings=True
	MenuSoundOk=Menu_OK
	MenuSoundCancel=Menu_Cancel
	MenuSoundActivate=Menu_Activate
	MenuSoundPress=Menu_Press
	ButtonLabelOK="|&OK"
	ButtonLabelCancel="|&Cancel"
	ButtonLabelPrevious="|&Previous"
	ButtonLabelResetToDefaults="|&Restore Defaults"
	MessageBoxTitle="Please Confirm"
	ExitMessage="Are you sure you|nwant to exit Deus Ex?"
}
