//=============================================================================
// HXMission15.
//
// Notes:
//  * Check if killing page works with leaving the map at the same time.
//=============================================================================
class HXMission15 expands HXMissionScript;

struct UCData
{
	var Name	      SpawnTag;
	var Class<Pawn> SpawnClass;
	var Name        Tag;
	var Name	      OrderTag;
	var int		      Count;
	var float	      LastKilledTime;
};

var UCData SpawnData[12];
var float JockTimer, PageTimer;

var HXDecoration AugmentedPage;

var Class<HXScriptedPawn> MJ12CommandoClass, MorganEverettClass, TracerTongClass,
                          WaltonSimonsClass;
var Class<HXDecoration>   BlackHelicopterClass, EarthClass, Switch2Class, Fan1Class,
                          BobPageAugmentedClass, LifeSupportBaseClass;

// ----------------------------------------------------------------------------
// LoadClasses()
// ----------------------------------------------------------------------------

function LoadClasses()
{
	local class<Actor> ActorClass;
	local int i;

	Super.LoadClasses();

	// ScriptedPawns.
	MJ12CommandoClass  = LoadScriptedPawnReplacementClass( class'MJ12Commando' );
	MorganEverettClass = LoadScriptedPawnReplacementClass( class'MorganEverett' );
	TracerTongClass    = LoadScriptedPawnReplacementClass( class'TracerTong' );

	// Decorations,
	BlackHelicopterClass  = LoadDecorationReplacementClass( class'BlackHelicopter' );
	LifeSupportBaseClass  = LoadDecorationReplacementClass( class'LifeSupportBase' );
	BobPageAugmentedClass = LoadDecorationReplacementClass( class'BobPageAugmented' );
	Switch2Class          = LoadDecorationReplacementClass( class'Switch2' );
	Fan1Class             = LoadDecorationReplacementClass( class'Fan1' );
	EarthClass            = LoadDecorationReplacementClass( class'Earth' );

	// SpawnData.
	for ( i=0; i<ArrayCount(SpawnData); i++ )
		SpawnData[i].SpawnClass = LoadScriptedPawnReplacementClass( Class<ScriptedPawn>(SpawnData[i].SpawnClass) );
}

// ----------------------------------------------------------------------------
// RandomizeEvents()
//
// Randomizes what happened at this Mission, so one would not start blank
// with what happened in prior mission.
// ----------------------------------------------------------------------------

static function RandomizeEvents( HXGameInfo Game, HXSteve Steve, int MissionNumber, String MapName )
{
	Super.RandomizeEvents( Game, Steve, MissionNumber, MapName );
}

// ----------------------------------------------------------------------------
// FirstFrame()
// 
// Stuff to check at first frame
// ----------------------------------------------------------------------------

function FirstFrame()
{
	local HXInterpolateTrigger InterpolateTrigger;
	local HXGoal Goal;

	Super.FirstFrame();

	//
	// 15_AREA51_BUNKER.
	//
	if ( LocalURL=="15_AREA51_BUNKER" )
	{
		// Unhide a helicopter if a flag is set, and start the countdown to Jock's death.
		if ( Level.NetMode==NM_ListenServer && !GetBoolFlag('Ray_dead') )
		{
			DecorationsEnterWorld( BlackHelicopterClass, 'heli_sabotaged' );
			DecorationsLeaveWorld( BlackHelicopterClass, 'UN_BlackHeli' );

			// Start the new helicopter interpolating.
			foreach AllActors( Class'HXInterpolateTrigger', InterpolateTrigger, 'InterpolateTrigger' )
				InterpolateTrigger.Trigger( Self, None );

			JockTimer = Level.TimeSeconds;

			SetBoolFlag( 'MS_BeginSabotage', True, True, 16 );
		}
	}

	//
	// 15_AREA51_ENTRANCE.
	//
	else if ( LocalURL=="15_AREA51_ENTRANCE" )
	{
		// Force completion of PenetrateBunker goal.
		CompleteGoal( 'PenetrateBunker' );

		// Remove those Goals as the player is likely to have taken an alternate route.
		if ( !IsGoalCompleted('PowerElevator') )
			DeleteGoal( 'PowerElevator' );
		if ( !IsGoalCompleted('EnterBlastDoors') )
			DeleteGoal( 'EnterBlastDoors' );
	}
}

// ----------------------------------------------------------------------------
// NotifyPlayerUp()
// ----------------------------------------------------------------------------

function NotifyPlayerUp( HXPlayerPawn PlayerPawn )
{
	local HXInterpolateTrigger InterpolateTrigger;

	Super.NotifyPlayerUp( PlayerPawn );

	Log( "NotifyPlayerUp()", 'DevMissionScript' );

	// Unhide a helicopter if a flag is set, and start the countdown to Jock's death.
	if ( Level.NetMode!=NM_ListenServer && !GetBoolFlag('MS_BeginSabotage') && !GetBoolFlag('Ray_dead') )
	{
		DecorationsEnterWorld( BlackHelicopterClass, 'heli_sabotaged' );
		DecorationsLeaveWorld( BlackHelicopterClass, 'UN_BlackHeli' );

		// Start the new helicopter interpolating.
		foreach AllActors( Class'HXInterpolateTrigger', InterpolateTrigger, 'InterpolateTrigger' )
			InterpolateTrigger.Trigger( Self, None );

		JockTimer = Level.TimeSeconds;

		SetBoolFlag( 'MS_BeginSabotage', True, True, 16 );
	}
}

// ----------------------------------------------------------------------------
// PreTravel()
// 
// Set flags upon exit of a certain map
// ----------------------------------------------------------------------------

function PreTravel()
{
	Super.PreTravel();
}

// ----------------------------------------------------------------------------
// Timer()
//
// Main state machine for the mission
// ----------------------------------------------------------------------------

function Timer()
{
	local HXScriptedPawn ScriptedPawn;
	local Pawn Pawn;
	local SpawnPoint SpawnPoint;
	local ZoneInfo ZoneInfo;
	local Actor Actor;
	local int Count, i;

	Super.Timer();

	//
	// 15_AREA51_BUNKER
	//
	if ( LocalURL=="15_AREA51_BUNKER" )
	{
		// Destroy Jock's heli on takeoff if a flag is set.
		if ( GetBoolFlag('MS_BeginSabotage') && !GetBoolFlag('MS_JockDead') )
		{
			// 8 seconds after level start, play a datalink.
			if ( !GetBoolFlag('DL_JockDeath_Played') && (Level.TimeSeconds-JockTimer)>= 8.0 )
			{
				StartDataLinkTransmission("DL_JockDeath");
			}

			// 11 seconds after level start, start to destroy Jock's helicopter
			// When the datalink is finished, finish destroying Jock's helicopter
			if ( GetBoolFlag('DL_JockDeath_Played') )
			{
				// spawn the final explosions and lots of debris
				if ( BlackHelicopterClass!=None )
				{
					foreach AllActors( BlackHelicopterClass, Actor, 'heli_sabotaged' )
					{
						Spawn( Class'HXBlackHelicopterFinalExplosionEffect', None, '', Actor.Location );
						Actor.Destroy();
						break;
					}
				}

				// Shake all players view.
				for ( Pawn=Level.PawnList; Pawn!=None; Pawn=Pawn.NextPawn )
					if ( Pawn.bIsPlayer )
						Pawn.ShakeView( 1.0, 1024.0, 32.0 );

				StartDataLinkTransmission("DL_JockDeathTongComment");
				SetBoolFlag('MS_JockDead', True, True, 16);
			}
			else if ( (Level.TimeSeconds-JockTimer)>= 11.0 )
			{
				// Spawn an inital explosion. (Does this happen multiple times?)
				if ( BlackHelicopterClass!=None )
				{
					foreach AllActors( BlackHelicopterClass, Actor, 'heli_sabotaged' )
					{
						Spawn( Class'HXBlackHelicopterExplosionEffect', None, '', Actor.Location );
						break;
					}
				}
			}
		}

		// Turn off pain zone when fan is destroyed.
		if ( !GetBoolFlag('MS_FanDestroyed') )
		{
			if ( !ActorsExist(Fan1Class,'Fan_vertical_shaft_1') )
			{
				foreach AllActors( Class'ZoneInfo', ZoneInfo, 'fan' )
					ZoneInfo.bPainZone = False;
					//if ( ZoneInfo.bPainZone )
						//ZoneInfo.Trigger( None, Self );

				SetBoolFlag( 'MS_FanDestroyed', True, True, 16 );
			}
		}
	}

	//
	// 15_AREA51_ENTRANCE
	//
	else if ( LocalURL=="15_AREA51_ENTRANCE" )
	{
		// Hide the earth and unhide Everett.
		if ( !GetBoolFlag('MS_EverettAppeared') && GetBoolFlag('EverettAppears') )
		{
			HideActors( EarthClass );
			PawnsEnterWorld( MorganEverettClass );

			SetBoolFlag( 'MS_EverettAppeared', True, True, 16 );
		}

		// Unhide the earth and hide Everett.
		if ( !GetBoolFlag('MS_MorganEverettHidden') && GetBoolFlag('M15MeetEverett_Played') )
		{
			UnhideActors( EarthClass );
			PawnsLeaveWorld( MorganEverettClass );

			SetBoolFlag( 'MS_MorganEverettHidden', True, True, 16 );
		}
	}

	//
	// 15_AREA51_FINAL
	//
	else if ( LocalURL=="15_AREA51_FINAL" )
	{
		// Unhide some commandos.
		if ( GetBoolFlag('MeetHelios_Played') && !GetBoolFlag('MS_CommandosUnhidden') )
		{
			PawnsEnterWorld( MJ12CommandoClass, 'commando1' );
			PawnsEnterWorld( MJ12CommandoClass, 'commando2' );
			PawnsEnterWorld( MJ12CommandoClass, 'commando3' );

			SetBoolFlag( 'MS_CommandosUnhidden', True, True, 16 );
		}

		// Hide/Unhide some buttons.
		if ( !GetBoolFlag('MS_ButtonsHidden') && GetBoolFlag('coolantcut') )
		{
			HideActors( Switch2Class, 'gen_switch1_off' );
			HideActors( Switch2Class, 'gen_switch2_off' );
			UnhideActors( Switch2Class, 'gen_switch1_on' );
			UnhideActors( Switch2Class, 'gen_switch2_on' );

			SetBoolFlag( 'MS_ButtonsHidden', True, True, 16 );
		}

		// Hide the earth and unhide Tong.
		if ( !GetBoolFlag('MS_TongAppeared') && GetBoolFlag('TongAppears') )
		{
			HideActors( EarthClass );
			PawnsEnterWorld( TracerTongClass );

			SetBoolFlag( 'MS_TongAppeared', True, True, 16 );
		}

		// Unhide the earth and hide Tong.
		if ( !GetBoolFlag('MS_TracerTongHidden') && GetBoolFlag('M15MeetTong_Played') )
		{
			UnhideActors( EarthClass );
			PawnsLeaveWorld( TracerTongClass );

			SetBoolFlag( 'MS_TracerTongHidden', True, True, 16 );
		}

		// Unhide Paul or Gary.
		if ( !GetBoolFlag('MS_PaulOrGaryAppeared') && GetBoolFlag('PaulAppears') )
		{
			if ( GetBoolFlag('PaulDenton_Dead') )
			{
				PawnsEnterWorld( Class'HXScriptedPawn', 'GarySavage' );
			}
			else
			{
				PawnsEnterWorld( Class'HXScriptedPawn', 'PaulDenton' );
			}

			SetBoolFlag( 'MS_PaulOrGaryAppeared', True, True, 16 );
		}

		// Hide Paul or Gary.
		if ( !GetBoolFlag('MS_PaulOrGaryHidden') && (GetBoolFlag('M15PaulHolo_Played') || GetBoolFlag('M15GaryHolo_Played')) )
		{
			PawnsLeaveWorld( Class'HXScriptedPawn', 'GarySavage' );
			PawnsLeaveWorld( Class'HXScriptedPawn', 'PaulDenton' );

			SetBoolFlag( 'MS_PaulOrGaryHidden', True, True, 16 );
		}
	}

	//
	// 15_AREA51_PAGE
	//
	else if ( LocalURL=="15_AREA51_PAGE" )
	{
		//
		// Check for UC respawing.
		//
		// Count the number of monsters that are still alive.
		foreach AllActors( Class'HXScriptedPawn', ScriptedPawn )
			for ( i=0; i<ArrayCount(SpawnData); i++ )
				if ( ScriptedPawn.Class==SpawnData[i].SpawnClass && ScriptedPawn.Tag==SpawnData[i].Tag )
					SpawnData[i].Count++;

		// Check to see when the last one was killed and set the time correctly.
		for ( i=0; i<ArrayCount(SpawnData); i++ )
			if ( SpawnData[i].Count==0 && SpawnData[i].LastKilledTime==-1 )
				SpawnData[i].LastKilledTime = Level.TimeSeconds;

		// Spawn any monsters which have been missing for 20 seconds.
		for ( i=0; i<ArrayCount(SpawnData); i++ )
		{
			if ( SpawnData[i].SpawnClass!=None && SpawnData[i].Count==0 && (Level.TimeSeconds-SpawnData[i].LastKilledTime)>20 )
			{
				SpawnPoint = GetSpawnPoint( SpawnData[i].SpawnTag ); // TODO: Make this just one time.
				if ( SpawnPoint!=None )
				{
					// Play an effect.
					Spawn( Class'HXUniversalConstructorEffect' );

					// Play a sound.
					SpawnPoint.PlaySound( Sound'Spark1', SLOT_None, 2.0,, 2048.0, 2.0 );

					// spawn the actual monster
					ScriptedPawn = HXScriptedPawn(Spawn( SpawnData[i].SpawnClass, None, SpawnData[i].Tag, SpawnPoint.Location, SpawnPoint.Rotation ));

					if ( ScriptedPawn!=None )
					{
						ScriptedPawn.InitializePawn();
						ScriptedPawn.SetOrders( 'Patrolling', SpawnData[i].OrderTag );
						ScriptedPawn.ChangeAlly( 'Player', -1, True );
						ScriptedPawn.ChangeAlly( 'MJ12',    0, True );

						SpawnData[i].LastKilledTime = -1;
					}
				}
			}

			// Reset the count for the next pass
			SpawnData[i].Count = 0;
		}

		// Play datalinks when devices are frobbed.
		if ( !GetBoolFlag('MS_DL_Played') )
		{
			// Reset count.
			Count = 0;

			// Increase count for each node.
			if ( GetBoolFlag('Node1_Frobbed') ) Count++;
			if ( GetBoolFlag('Node2_Frobbed') ) Count++;
			if ( GetBoolFlag('Node3_Frobbed') ) Count++;
			if ( GetBoolFlag('Node4_Frobbed') ) Count++;

			switch ( Count )
			{
				case 1:
					if ( !GetBoolFlag('DL_Blue1_Played') )
						StartDataLinkTransmission( "DL_Blue1" );
					break;

				case 2:
					if ( !GetBoolFlag('DL_Blue2_Played') )
						StartDataLinkTransmission( "DL_Blue2" );
					break;

				case 3:
					if ( !GetBoolFlag('DL_Blue3_Played') )
						StartDataLinkTransmission( "DL_Blue3" );
					break;

				case 4:
					if ( !GetBoolFlag('DL_Blue4_Played') )
					{
						StartDataLinkTransmission( "DL_Blue4" );
						SetBoolFlag( 'MS_DL_Played', True, True, 16 );
					}
					break;

				default:
					break;
			}
		}

		// Spawn a bunch of explosions when page is dead.
		if ( GetBoolFlag('killpage') && !GetBoolFlag('MS_PageExploding') )
		{
			if ( BobPageAugmentedClass!=None )
				foreach AllActors( BobPageAugmentedClass, Actor )
					AugmentedPage = HXDecoration(Actor);

			PageTimer = Level.TimeSeconds;
			SetBoolFlag( 'MS_PageExploding', True, True, 16 );
		}

		if ( GetBoolFlag('MS_PageExploding') )
		{
			if ( (Level.TimeSeconds-PageTimer)>=3.0 )
			{
				PageExplosionEffects();
			}

			if ( (Level.TimeSeconds-PageTimer)>=6.0 && !GetBoolFlag('MS_PageDestroyed') )
			{
				BlowUpMovers( Class'HXMover', 'platform_pieces' );
				DestroyActors( LifeSupportBaseClass );

				if ( AugmentedPage!=None )
				{
					AugmentedPage.Destroy();
					AugmentedPage = None;
				}

				SetBoolFlag( 'MS_PageDestroyed', True, True, 16 );
			}

			if ( (Level.TimeSeconds-PageTimer)>=9.0 )
			{
				foreach AllActors( Class'Actor', Actor, 'start_endgame' )
					Actor.Trigger( None, None );
			}
		}
	}
}

// ----------------------------------------------------------------------------
// PageExplosionEffects()
//
// Note: Maybe combine the sounds + frags falling into a single effect.
//       Maybe the Pawn.Velocity += Bobble actually does nothing.
// ----------------------------------------------------------------------------

function PageExplosionEffects()
{
	local Sound ExplosionSound;
	local Pawn Pawn;
	local Actor HitActor;
	local DeusExFragment Fragment;
	local Vector Bobble, TraceStart, TraceEnd, HitLocation, HitNormal;
	local float ShakeTime, ShakeRoll, ShakeVert;
	local float Temp, Size;
	local int NumFragments, i;

	// Pick a random explosion size and modify everything accordingly.
	Size      = 0.5 * FRand() + 0.5;
	ShakeTime =   0.5 + Size;
	ShakeRoll = 512.0 + Size * 1024.0;
	ShakeVert =   8.0 + Size *   16.0;
	Bobble    = Vect(300.0,300.0,200.0) + 500.0 * Size * VRand();

	// Scale amount of Fragments per player based on player count.
	if ( Level.Game.NumPlayers>0 )
	{
		Temp = Size * 10.0 / Level.Game.NumPlayers;
		NumFragments = Int(Temp+0.5);
	}
	else
	{
		NumFragments = 1;
	}

	// Pick a sound.
	if ( Size<0.75 )
		ExplosionSound = Sound'MediumExplosion2';
	else if ( Size<0.85 )
		ExplosionSound = Sound'LargeExplosion1';
	else
		ExplosionSound = Sound'LargeExplosion2';

	// Handle PlayerPawns.
	for ( Pawn=Level.PawnList; Pawn!=None; Pawn=Pawn.NextPawn )
	{
		if ( Pawn.bIsPlayer )
		{
			// Play a sound.
			Pawn.PlaySound( ExplosionSound, SLOT_None, 2.0, , 16384 );

			// Shake the view.
			Pawn.ShakeView( ShakeTime, ShakeRoll, ShakeVert );

			// Bobble the player around.
			Pawn.Velocity += Bobble;

			for ( i=0; i<NumFragments; i++ )
			{
				if ( AugmentedPage!=None )
				{
					TraceStart   = AugmentedPage.Location + 64.0 * VRand();
					TraceStart.Z = AugmentedPage.Location.Z;
				}
				else
				{
					TraceStart   = Pawn.Location + 256.0 * VRand();
					TraceStart.Z = Pawn.Location.Z;
				}
				TraceEnd     = TraceStart;
				TraceEnd.Z  += 1024.0;

				HitActor = Trace( HitLocation, HitNormal, TraceEnd, TraceStart, False );
				if ( HitActor==None )
					HitLocation = TraceEnd;

				// Spawn some explosion effects.
				if ( Size<0.75 )
				{
					Spawn( Class'ExplosionMedium', , , HitLocation+8*HitNormal );
				}
				else
				{
					Spawn( Class'ExplosionLarge', , , HitLocation+8*HitNormal );
				}

				// Have random metal fragments fall from the ceiling.
				Fragment = Spawn( Class'MetalFragment', , , HitLocation );
				if ( Fragment!=None )
				{
					Fragment.CalcVelocity( Vect(20000,0,0), 256 );
					Fragment.DrawScale = 0.5 + 2.0 * FRand();
					if ( FRand()<0.25 )
						Fragment.bSmoking = True;
				}
			}
		}
	}
}

// ----------------------------------------------------------------------------
// ----------------------------------------------------------------------------

defaultproperties
{
	PrevMissionScript=Class'HX.HXMission14'
	NextMissionScript=None
	SpawnData(0)=(SpawnTag=UC_spawn1,SpawnClass=Class'DeusEx.SpiderBot2',Tag=spbot1,OrderTag=spiderbot1_0,lastKilledTime=-1.0)
	SpawnData(1)=(SpawnTag=UC_spawn1,SpawnClass=Class'DeusEx.SpiderBot2',Tag=spbot2,OrderTag=spiderbot2_0,lastKilledTime=-1.0)
	SpawnData(2)=(SpawnTag=UC_spawn2,SpawnClass=Class'DeusEx.Gray',Tag=gray_1,OrderTag=gray1_0,lastKilledTime=-1.0)
	SpawnData(3)=(SpawnTag=UC_spawn2,SpawnClass=Class'DeusEx.Gray',Tag=gray_2,OrderTag=gray2_0,lastKilledTime=-1.0)
	SpawnData(4)=(SpawnTag=UC_spawn2,SpawnClass=Class'DeusEx.Gray',Tag=gray_3,OrderTag=gray3_0,lastKilledTime=-1.0)
	SpawnData(5)=(SpawnTag=UC_spawn3,SpawnClass=Class'DeusEx.Karkian',Tag=karkian_1,OrderTag=karkian1_0,lastKilledTime=-1.0)
	SpawnData(6)=(SpawnTag=UC_spawn3,SpawnClass=Class'DeusEx.Karkian',Tag=karkian_2,OrderTag=karkian2_0,lastKilledTime=-1.0)
	SpawnData(7)=(SpawnTag=UC_spawn3,SpawnClass=Class'DeusEx.Karkian',Tag=karkian_3,OrderTag=karkian3_0,lastKilledTime=-1.0)
	SpawnData(8)=(SpawnTag=UC_spawn3,SpawnClass=Class'DeusEx.Greasel',Tag=greasel_1,OrderTag=greasel1_0,lastKilledTime=-1.0)
	SpawnData(9)=(SpawnTag=UC_spawn3,SpawnClass=Class'DeusEx.Greasel',Tag=greasel_2,OrderTag=greasel2_0,lastKilledTime=-1.0)
	SpawnData(10)=(SpawnTag=UC_spawn3,SpawnClass=Class'DeusEx.Greasel',Tag=greasel_3,OrderTag=greasel3_0,lastKilledTime=-1.0)
	SpawnData(11)=(SpawnTag=UC_spawn3,SpawnClass=Class'DeusEx.Greasel',Tag=greasel_4,OrderTag=greasel4_0,lastKilledTime=-1.0)
}
