//=============================================================================
// HXMutator.
//=============================================================================
class HXMutator extends Mutator
	native;

var transient HXGameInfo Game;
var bool bSpawning;

var HXMutator HXNextMutator;

native(3360) static final function Name StringToName( coerce String Str );

//
// Spawn an actor. Returns an actor of the specified class, not
// of class Actor (this is hardcoded in the compiler). Returns None
// if the actor could not be spawned (either the actor wouldn't fit in
// the specified location, or the actor list is full).
// Defaults to spawning at the spawner's location.
//
// HX_NOTE:
// Exposed ULevel::SpawnActor() bNoCollisionFail to uc, but the return
// value is now Actor, and needs to be casted.
//
native(3361) final function Actor SpawnEx
(
	class<Actor>      SpawnClass,
	optional Actor	  SpawnOwner,
	optional Name     SpawnTag,
	optional Vector   SpawnLocation,
	optional Rotator  SpawnRotation,
	optional bool			bSpawnNoCollisionFail,
	optional EPhysics SpawnPhysics
);

native(3362) final function bool DuckCallSucceeds( Actor Successor, Actor Other );

// ----------------------------------------------------------------------------
// Spawned()
// ----------------------------------------------------------------------------

simulated event Spawned()
{
	Super.Spawned();

	Game = HXGameInfo(Level.Game);
}

// ----------------------------------------------------------------------------
// SetInitialState()
//
// Called after PostBeginPlay.
// ----------------------------------------------------------------------------

simulated event SetInitialState()
{
	if ( bDeleteMe /*|| IsInRelevant()*/ )
		return;

	if ( InitialState!='' )
		GotoState( InitialState );
	else
		GotoState( 'Auto' );

	bSpawning = false;
}

// ----------------------------------------------------------------------------
// PostPostBeginPlay()
// ----------------------------------------------------------------------------

simulated event PostPostBeginPlay()
{
	// This may need to be reset, as a new GameInfo is spawned each time.
	Game = HXGameInfo(Level.Game);

	Super.PostPostBeginPlay();
}

// ----------------------------------------------------------------------------
// CheckInventoryClassReplacement()
//
// Conveniance interface for ModifyInventoryClass similiar to CheckReplacement.
// ----------------------------------------------------------------------------

function bool CheckInventoryClassReplacement( out Class<Inventory> InventoryClass )
{
	// Implemented by subclasses. Code is expected to yield the same result for each call.

	return true; // False prevents passing to NextMutator;
}

// ----------------------------------------------------------------------------
// CheckMissionScriptClassReplacement()
//
// Conveniance interface for ModifyMissionScriptClass similiar to
// CheckReplacement.
// ----------------------------------------------------------------------------

// Code is expected to yield the same result for each call.
function bool CheckMissionScriptClassReplacement( out Class<MissionScript> MissionScriptClass, Name LevelName )
{
	// Implemented by subclasse. Code is expected to yield the same result for each call.

	return true; // False prevents passing to NextMutator;
}

// ----------------------------------------------------------------------------
// CheckActorClassReplacement()
//
// Conveniance interface for ModifyActorClass similiar to CheckReplacement.
// ----------------------------------------------------------------------------

// Code is expected to yield the same result for each call.
function bool CheckActorClassReplacement( out Class<Actor> ActorClass )
{
	// Implemented by subclasse. Code is expected to yield the same result for each call.

	return true; // False prevents passing to NextMutator;
}

// ----------------------------------------------------------------------------
// CheckCustomCommand()
//
// Return true to prevent further Mutators and GameInfo from handling the
// command.
// ----------------------------------------------------------------------------

function bool CheckCustomCommand( HXPlayerPawn PlayerPawn, string Command );

// ----------------------------------------------------------------------------
// ModifyInventoryClass()
//
// Modifies class to be used for an Inventory throughout the code
// (e.g. MissionScripts, Conversations, TailMutator...)
// ----------------------------------------------------------------------------

function ModifyInventoryClass( out Class<Inventory> InventoryClass )
{
	local bool bResult;

	// Conveniance interface similiar to CheckReplacement.
	bResult = CheckInventoryClassReplacement( InventoryClass );
	if ( bResult && HXNextMutator!=None )
		HXNextMutator.ModifyInventoryClass( InventoryClass );
}

// ----------------------------------------------------------------------------
// ModifyMissionScriptClass()
//
// Modifies class to be spawned as MissionScript.
// ----------------------------------------------------------------------------

function ModifyMissionScriptClass( out Class<MissionScript> MissionScriptClass, Name LevelName )
{
	local bool bResult;

	// Conveniance interface similiar to CheckReplacement.
	bResult = CheckMissionScriptClassReplacement( MissionScriptClass, LevelName );
	if ( bResult && HXNextMutator!=None )
		HXNextMutator.ModifyMissionScriptClass( MissionScriptClass, LevelName );
}

// ----------------------------------------------------------------------------
// ModifyActorClass()
//
// Deals with everything which is not an Inventory.
// ----------------------------------------------------------------------------

function ModifyActorClass( out Class<Actor> ActorClass )
{
	local bool bResult;

	// Conveniance interface similiar to CheckReplacement.
	bResult = CheckActorClassReplacement( ActorClass );
	if ( bResult && HXNextMutator!=None )
		HXNextMutator.ModifyActorClass( ActorClass );
}

// ----------------------------------------------------------------------------
// ModifyLogin()
//
// Called at the start of GameInfo's ModifyLogin, before GameInfo enforces
// its rules. bSuperRelevant can be set to override any GameInfo ModifyLogin
// logic.
// ----------------------------------------------------------------------------

function ModifyLogin( out class<PlayerPawn> SpawnClass, out string Portal, out string Options, out byte bSuperRelevant )
{
	if ( HXNextMutator!=None )
		HXNextMutator.ModifyLogin( SpawnClass, Portal, Options, bSuperRelevant );
}

// ----------------------------------------------------------------------------
// CustomCommand()
// ----------------------------------------------------------------------------

function bool CustomCommand( HXPlayerPawn PlayerPawn, string Command )
{
	if ( CheckCustomCommand(PlayerPawn,Command) )
		return true;

	if ( HXNextMutator!=None )
		return HXNextMutator.CustomCommand( PlayerPawn, Command );

	return false;
}

// ----------------------------------------------------------------------------
// AddMutator()
//
// Modified to refuse non HXMutators and to also link HXNextMutator list.
// ----------------------------------------------------------------------------

function AddMutator( Mutator M )
{
	// Reject non HXMutators.
	if ( !M.IsA('HXMutator') )
	{
		Warn( "Ignoring non HXMutator" @ M );
		return;
	}

	// Assume NextMutator==HXNextMutator.
	if ( NextMutator==None )
	{
		HXNextMutator = HXMutator(M);
		NextMutator   = M;
	}
	else
	{
		HXNextMutator.AddMutator( M );
	}
}

// ----------------------------------------------------------------------------
// SpawnSuccessor()
//
// Spawns a replacement for Actor of class SpawnClass, when the Actor
// is either !bStatic && !bNoDelete or bEvenSpawnWhenStaticNoDelete is set
// and either Actor.Class!=SpawnClass or bEvenSpawnForSameClass is set.
// ----------------------------------------------------------------------------

function bool SpawnSuccessor( Actor Other, class<Actor> SpawnClass, optional bool bEvenSpawnWhenStaticNoDelete, optional bool bEvenSpawnForSameClass, optional bool bEvenSpawnWhenOwned )
{
	local Actor Successor;

	// Sanity checks.
	if ( Other==None )
	{
		if ( SpawnClass!=None )
			Warn( Sprintf("Other is None (SpawnClass=%s).",SpawnClass.Name) );
		else
			Warn( "Other is None (SpawnClass=None)." );
		return true;
	}
	if ( SpawnClass==None )
	{
		Warn( Sprintf("SpawnClass is None (Other=%s).",Other.Name) );
		return true;
	}

	// Owned Actors usually shouldn't  be touched by Mutators.
	if ( Other.Owner!=None && !bEvenSpawnWhenOwned )
	{
		Log( Sprintf("Keeping owned Actor (Other=%s,Owner=%s).",Other.Name,Other.Owner.Name), 'DevMutator' );
		return true;
	}

	// bStatic and bNoDelete Actors can't be destroyed.
	if ( (Other.bStatic || Other.bNoDelete) && !bEvenSpawnWhenStaticNoDelete )
	{
		if ( Other.bStatic && Other.bNoDelete )
			Log( Sprintf("Keeping bStatic and bNoDelete Actor (Other=%s).",Other.Name), 'DevMutator' );
		else if ( Other.bStatic )
			Log( Sprintf("Keeping bStatic Actor (Other=%s).",Other.Name), 'DevMutator' );
		else
			Log( Sprintf("Keeping bNoDelete Actor (Other=%s).",Other.Name), 'DevMutator' );

		return true;
	}

	// Handle this SpawnClass check here, so callers code can be more compact.
	if ( SpawnClass==Other.Class && !bEvenSpawnForSameClass )
	{
		Log( Sprintf("Keeping Actor with identical Successor SpawnClass (Other=%s,SpawnClass=%s).",Other.Name,SpawnClass.Name), 'DevMutator' );
		return true;
	}

	Successor = SpawnEx( SpawnClass, /*Owner*/ None, Other.Tag, Other.Location, Other.Rotation, /*bNoCollisionFail*/ true, Other.Physics );
	if ( Successor==None )
	{
		Warn( Sprintf("Failed to spawn Successor (Other=%s,SpawnClass=%s).",Other.Name,SpawnClass.Name) );

		// I'm not keeping the Actor if destroyable, as it most likely won't work as desired and is even likely to cause issues.
		return (Other.bStatic || Other.bNoDelete);
	}
	else
	{
		if ( Other.bStatic && Other.bNoDelete )
			Log( Sprintf("Spawned Successor for bStatic and bNoDelete Actor (Other=%s,Successor=%s,SpawnClass=%s).",Other.Name,Successor.Name,SpawnClass.Name), 'DevMutator' );
		else if ( Other.bStatic )
			Log( Sprintf("Spawned Successor for bStatic Actor (Other=%s,Successor=%s,SpawnClass=%s).",Other.Name,Successor.Name,SpawnClass.Name), 'DevMutator' );
		else if ( Other.bNoDelete )
			Log( Sprintf("Spawned Successor for bNoDelete Actor (Other=%s,Successor=%s,SpawnClass=%s).",Other.Name,Successor.Name,SpawnClass.Name), 'DevMutator' );
		else
			Log( Sprintf("Spawned Successor (Other=%s,Successor=%s,SpawnClass=%s).",Other.Name,Successor.Name,SpawnClass.Name), 'DevMutator' );
	}

	// Duck call Succeeds, as it is not a virtual declared in Actor.
	DuckCallSucceeds( Successor, Other );

	// bStatic/bNoDelete Actors can't be removed, so return true for them.
	return (Other.bStatic || Other.bNoDelete);
}

// ----------------------------------------------------------------------------
// SanitizeURL()
//
// Strings .dx map extension and whitespaces from an URL.
// Return true whenever the URL was modified.
// ----------------------------------------------------------------------------

final static function bool SanitizeURL( out string URL )
{
	local string OldURL;

	OldURL = URL;

	ReplaceText( URL, ".dx", "" ); // Strip .dx
	ReplaceText( URL, " ",   "" ); // Strip whitespaces.

	return URL!=OldURL;
}

// ----------------------------------------------------------------------------
// ReplaceText() -- Copied out of UWindowWindow.
// ----------------------------------------------------------------------------

final static function ReplaceText( out string Text, string Replace, string With )
{
	local int i;
	local string Input;
		
	Input = Text;
	Text = "";
	i = InStr( Input, Replace );
	while ( i!=-1 )
	{	
		Text = Text $ Left(Input,i) $ With;
		Input = Mid( Input, i+Len(Replace) );
		i = InStr( Input, Replace );
	}
	Text = Text $ Input;
}

// ----------------------------------------------------------------------------
// ----------------------------------------------------------------------------

defaultproperties
{
	bSpawning=True
}
