//=============================================================================
// HXParticleGenerator.
//=============================================================================
class HXParticleGenerator extends HXEffects;

var() float Frequency;			// what's the chance of spewing a Particle every CheckTime seconds
var() float RiseRate;			// how fast do the Particles rise
var() float EjectSpeed;			// how fast do the Particles get ejected
var() Texture ParticleTexture;	// replacement texture to use (default is smoke)
var() float ParticleLifeSpan;	// how long each Particle lives
var() float ParticleDrawScale;	// draw scale for each Particle
var() bool bParticlesUnlit;		// is each Particle unlit?
var() bool bScale;				// scale each Particle as it rises?
var() bool bFade;				// fade each Particle as it rises?
var() bool bTriggered;			// start by triggering?
var() float SpewTime;			// how long do I spew after I am triggered?
var() bool bRandomEject;		// random eject velocity vector
var() float CheckTime;			// how often should I spit out Particles?
var() bool bTranslucent;		// are these Particles translucent?
var() bool bGravity;			// are these Particles affected by gravity?
var() Sound SpawnSound;			// sound to play when spawned
var() bool bAmbientSound;		// play the ambient sound?
var() int NumPerSpawn;			// number of Particles to spawn per puff
var() name AttachTag;			// attach us to this actor // !! DUBLICATED !!
var() bool bInitiallyOn;		// if triggered, start on instead of off
var() bool bModulated;			// are these Particles modulated?

var bool bSpewing;				// am I spewing?
var bool bFrozen;				// are we out of the player's sight?
var float Time;
var bool bDying;				// don't spew, but continue to update

var vector pLoc;				// Location used for replication, ParticleIterator uses these now
var rotator pRot;				// Rotation used for replication, ParticleIterator uses these now


var ParticleProxy proxy;

replication
{
	reliable if ( ROLE == Role_Authority )
		Frequency, RiseRate, EjectSpeed, ParticleTexture, ParticleLifeSpan, ParticleDrawScale, bParticlesUnlit,
		bScale, bFade, bTriggered, SpewTime, bRandomEject, CheckTime, bTranslucent, bGravity,
		SpawnSound, bAmbientSound, NumPerSpawn, AttachTag, bInitiallyOn, bSpewing, bFrozen, Time, bDying, bModulated,
		//proxy,
		pLoc, pRot;
}

// ----------------------------------------------------------------------------
// Succeeds()
//
// Intended to be called by Mutator code if Self was spawned as a successor
// to Other. Super call recommended.
// ----------------------------------------------------------------------------

function Succeeds( Actor Other )
{
	local ParticleGenerator OtherParticleGenerator;

	Super.Succeeds( Other );

	// Special ParticleGenerator Init.
	OtherParticleGenerator = ParticleGenerator(Other);
	if ( OtherParticleGenerator!=None )
	{
		// ParticleGenerator.
		Frequency         = OtherParticleGenerator.Frequency;
		RiseRate          = OtherParticleGenerator.RiseRate;
		EjectSpeed        = OtherParticleGenerator.EjectSpeed;
		ParticleTexture   = OtherParticleGenerator.ParticleTexture;
		ParticleLifeSpan  = OtherParticleGenerator.ParticleLifeSpan;
		ParticleDrawScale = OtherParticleGenerator.ParticleDrawScale;
		bParticlesUnlit   = OtherParticleGenerator.bParticlesUnlit;
		bScale            = OtherParticleGenerator.bScale;
		bFade             = OtherParticleGenerator.bFade;
		bTriggered        = OtherParticleGenerator.bTriggered;
		SpewTime          = OtherParticleGenerator.SpewTime;
		bRandomEject      = OtherParticleGenerator.bRandomEject;
		CheckTime         = OtherParticleGenerator.CheckTime;
		bTranslucent      = OtherParticleGenerator.bTranslucent;
		bGravity          = OtherParticleGenerator.bGravity;
		SpawnSound        = OtherParticleGenerator.SpawnSound;
		bAmbientSound     = OtherParticleGenerator.bAmbientSound;
		NumPerSpawn       = OtherParticleGenerator.NumPerSpawn;
		AttachTag         = OtherParticleGenerator.AttachTag;
		bInitiallyOn      = OtherParticleGenerator.bInitiallyOn;
		bModulated        = OtherParticleGenerator.bModulated;
	}
}

function Trigger(Actor Other, Pawn EventInstigator)
{
	Super.Trigger(Other, EventInstigator);

	// if we are spewing, turn us off
	if (bSpewing)
	{
		bSpewing = False;
		proxy.bHidden = True;
		if (bAmbientSound && (AmbientSound != None))
			SoundVolume = 0;
	}
	else		// otherwise, turn us on
	{
		bSpewing = True;
		proxy.bHidden = False;
		LifeSpan = SpewTime; // !! What?
		if (bAmbientSound && (AmbientSound != None))
			SoundVolume = 255;
	}
}

function UnTrigger(Actor Other, Pawn EventInstigator)
{
	Super.UnTrigger(Other, EventInstigator);

	// if we are spewing, turn us off
	if (bSpewing)
	{
		bSpewing = False;
		proxy.bHidden = True;
		if (bAmbientSound && (AmbientSound != None))
			SoundVolume = 0;
	}
}

simulated function Tick(float deltaTime)
{
	local int i;

	// Server updates these location and rotation proxies
	if ( Role == ROLE_Authority )
	{
		pLoc = Location;
		pRot = Rotation;
	}

	if (proxy != None)
	{
		// don't freeze if we're dying
		if (bDying)
			bFrozen = False;

		// if we are close, say 20 feet
		else if (proxy.DistanceFromPlayer < 320)
			bFrozen = False;

		// can the player see the generator?
		else if (proxy.LastRendered() <= 2.0)
			bFrozen = False;

		// can the player see our base?
		else if ((Base != None) && (Base != Level) && (Base.LastRendered() <= 2.0))
			bFrozen = False;

		else
			bFrozen = True;
	}
	else
		bFrozen = True;

	// check LifeSpan and see if we need to DelayedDestroy()
	if ((LifeSpan > 0) && (LifeSpan <= 1.0))
	{
		LifeSpan = 0;
		DelayedDestroy();
	}

	// are we frozen
	if (bFrozen)
		return;

	Super.Tick(deltaTime);

	if (proxy != None)
		if (proxy.Texture != ParticleTexture)
			SetProxyData();

	// tick the iterator since Objects don't Tick()
	if (ParticleIterator(RenderInterface) != None)
			ParticleIterator(RenderInterface).Update(deltaTime);

	// don't spew anymore if we're dying
	if (bDying || !bSpewing)
		return;

	// if the owner that I'm attached to is dead, kill me
	if ((AttachTag != '') && (Owner == None))
		Destroy();

	Time += deltaTime;

	if (Time > CheckTime)
	{
		Time = 0;

		if (FRand() <= Frequency)
		{
			if (SpawnSound != None)
				PlaySound(SpawnSound, SLOT_Misc,,, 1024);

			for (i=0; i<NumPerSpawn; i++)
				if (ParticleIterator(RenderInterface) != None)
					ParticleIterator(RenderInterface).AddParticle();
		}
	}
}

//
// Don't actually destroy the generator until after all of the
// Particles have disappeared
//
function function DelayedDestroy()
{
	bDying = True;
	SetBase(None);

	// Last particle will be gone after ParticleLifeSpan, but give it a bit of wiggle room.
	SetTimer( ParticleLifeSpan+1.0, false );
}

function function Timer()
{
	Destroy();
}

simulated function function Destroyed()
{
	if ( Proxy!=None )
	{
		Proxy.Destroy();
		Proxy = None;
	}
	Super.Destroyed();
}

simulated function SetProxyData()
{
	if (proxy != None)
	{
		proxy.bUnlit = bParticlesUnlit;

		if (bModulated)
			proxy.Style = STY_Modulated;
		else if (bTranslucent)
			proxy.Style = STY_Translucent;
		else
			proxy.Style = STY_Masked;

		if (ParticleTexture != None)
			proxy.Texture = ParticleTexture;
	}
}

function PostBeginPlay()
{
	local Actor A;

	Super.PostBeginPlay();

	DrawType = DT_None;

	// Create our proxy Particle.
	if ( Proxy==None )
	{
		Proxy = Spawn( class'ParticleProxy',,, Location, Rotation );
		SetProxyData();
	}

	if ( bTriggered && !bInitiallyOn ) // Well, what a mess, otherwise !bInitiallyOn assumes forced beeing triggered.
	{
		bSpewing = False;
		proxy.bHidden = True;
		if (bAmbientSound && (AmbientSound != None))
			SoundVolume = 0;
	}
	else
	{
		bSpewing = True;
		proxy.bHidden = False;
		//LifeSpan = SpewTime; // !! What?
		if (bAmbientSound && (AmbientSound != None))
			SoundVolume = 255;
	}

	// Attach us to the actor that was tagged.
	if (AttachTag != '')
		foreach AllActors(class'Actor', A, AttachTag)
			if (A != None)
			{
				SetOwner(A);
				SetBase(A);
			}

	pLoc = Location;
	pRot = Rotation;
}

simulated function PostNetBeginPlay()
{
	local Actor A;

	Super.PostNetBeginPlay();

	DrawType = DT_None;

	// Create our proxy Particle.
	if ( Proxy==None )
	{
		Proxy = Spawn( class'ParticleProxy',,, Location, Rotation );
		SetProxyData();
	}

	/*if ( bTriggered && !bInitiallyOn )
	{
		bSpewing = False;
		proxy.bHidden = True;
		SoundVolume = 0;
	}*/
	// This seems to make more sense for the client.
	Proxy.bHidden = !bSpewing;

	// Attach us to the actor that was tagged.
	if (AttachTag != '')
		foreach AllActors(class'Actor', A, AttachTag)
			if (A != None)
			{
				SetOwner(A);
				SetBase(A);
			}

	//pLoc = Location;
	//pRot = Rotation;
}

// ----------------------------------------------------------------------------
// ----------------------------------------------------------------------------

defaultproperties
{
	Frequency=1.000000
	RiseRate=10.000000
	EjectSpeed=10.000000
	ParticleLifeSpan=4.000000
	ParticleDrawScale=0.100000
	bParticlesUnlit=True
	bScale=True
	bFade=True
	CheckTime=0.100000
	bTranslucent=True
	NumPerSpawn=1
	bInitiallyOn=True
	bSpewing=True
	bDirectional=True
	DrawType=DT_Sprite
	Texture=Texture'Engine.S_Inventory'
	CollisionRadius=80.000000
	CollisionHeight=80.000000
	RenderIteratorClass=Class'HX.HXParticleIterator'
}
