//=============================================================================
// HXPickup.
//=============================================================================
class HXPickup extends DeusExPickup
	abstract;

var transient HXGameInfo Game;
var Name PrecessorName;
var bool bSpawning;

var const localized string MsgDropNotAllowed;

var const bool bCanDrop;      // Whether we can be DropItem'ed.

// ----------------------------------------------------------------------------
// Succeeds()
//
// Intended to be called by Mutator code if Self was spawned as a successor
// to Other. Super call recommended.
// ----------------------------------------------------------------------------

function Succeeds( Actor Other )
{
	// Advanced (partial).
	bHidden              = Other.bHidden;
	bOwned               = Other.bOwned;

	// Collision and Blocking.
	bCollideWhenPlacing = Other.bCollideWhenPlacing; // Still used for far moving after spawn.
	bCollideWorld       = true; //Other.bCollideWorld;

	SetCollisionSize( Other.CollisionRadius, Other.CollisionHeight );
	SetCollision( Other.bCollideActors, Other.bBlockActors, Other.bBlockPlayers );

	// Events. 
	Event                = Other.Event;
//Tag                  = Other.Tag;   // Set by Spawn.

	// Object (partial).
	Group                = Other.Group;

	// Filter.
	bDifficulty0         = Other.bDifficulty0;
	bDifficulty1         = Other.bDifficulty1;
	bDifficulty2         = Other.bDifficulty2;
	bDifficulty3         = Other.bDifficulty3;
	bSinglePlayer        = Other.bSinglePlayer;

	// Make sure old Actor has Tag, Event, Group, BindName and BarkBindName removed.
	Other.Tag            = '';
	Other.Event          = '';
	Other.Group          = '';
	Other.BindName       = "";
	Other.BarkBindName   = "";
}

// ----------------------------------------------------------------------------
// IsInRelevant()
// 
// Whether code is currently running during a spawn inside a mutator in which
// case the properties are not yet properly set.
// ----------------------------------------------------------------------------

simulated function bool IsInRelevant()
{
	if ( Level.Game==None ) // Will happen on client too.
		return false;
	return Level.Game.IsInState( 'InIsRelevant' );
}

// ----------------------------------------------------------------------------
// Spawned()
// ----------------------------------------------------------------------------

simulated event Spawned()
{
	Super.Spawned();

	if ( Level.bStartup )
		bGameRelevant = true;

	Game = HXGameInfo(Level.Game);
}

// ----------------------------------------------------------------------------
// PreBeginPlay()
// ----------------------------------------------------------------------------

function PreBeginPlay()
{
	if ( bDeleteMe || IsInRelevant() )
		return;

	// Handle autodestruction if desired. Moved this here because the
	// config properties won't have their correct value set when spawned inside
	// the mutator, so you can't replace them at first pass there.
	if ( !bGameRelevant && Level.NetMode!=NM_Client && !Level.Game.IsRelevant(Self) )
	{
		Destroy();
		return;
	}
}

// ----------------------------------------------------------------------------
// SetInitialState()
//
// Called after PostBeginPlay.
// ----------------------------------------------------------------------------

simulated event SetInitialState()
{
	if ( bDeleteMe || IsInRelevant() )
		return;

	if ( InitialState!='' )
		GotoState( InitialState );
	else
		GotoState( 'Auto' );

	bSpawning = false;
}

// ----------------------------------------------------------------------------
// PostPostBeginPlay()
// ----------------------------------------------------------------------------

event PostPostBeginPlay()
{
	// This may need to be reset, as a new GameInfo is spawned each time.
	Game = HXGameInfo(Level.Game);

	Super(Pickup).PostPostBeginPlay();
}

// ----------------------------------------------------------------------------
// FellOutOfWorld()
// ----------------------------------------------------------------------------

event FellOutOfWorld()
{
	if ( PrecessorName!='' )
	{
		Log( Sprintf("Precessor of falling out of world %s was %s.",Name,PrecessorName), 'FellOutOfWorld' );
	}

	SetPhysics( PHYS_None );
	Destroy();
}

// ----------------------------------------------------------------------------
// CanBeDropped()
//
// Called by HXPlayerPawn.DropItem() to see if we cannot be dropped for some
// reason.
// ----------------------------------------------------------------------------

function bool CanBeDropped( optional bool bSilent )
{
	if ( !bCanDrop )
	{
		if ( !bSilent )
			Pawn(Owner).ClientMessage( Sprintf(MsgDropNotAllowed,ItemName), 'StatusLine' );	
		return false;
	}

	return true;
}

// ----------------------------------------------------------------------------
// GiveTo()
//
// Give this inventory item to a pawn.
// ----------------------------------------------------------------------------

function GiveTo( pawn Other )
{
	// HX_INV
	SetOwner(Other);

	Instigator = Other;
	BecomeItem();
	Other.AddInventory( Self );
	GotoState('Idle2');
}

// ----------------------------------------------------------------------------
// SpawnCopy()
//
// Either give this inventory to player Other, or spawn a copy
// and give it to the player Other, setting up original to be respawned.
// ----------------------------------------------------------------------------

function inventory SpawnCopy( pawn Other )
{
	local inventory Copy;

	if( Level.Game.ShouldRespawn(self) )
	{
		Copy = spawn(Class,Other,,,rot(0,0,0));
		Copy.Tag           = Tag;
		Copy.Event         = Event;

		Copy.Charge				 = Charge;
		HXPickup(Copy).NumCopies = NumCopies;

		GotoState('Sleeping');
		//Destroy();
	}
	else
		Copy = self;

	Copy.RespawnTime = 0.0;
	Copy.bHeldItem = true;
	Copy.GiveTo( Other );
	return Copy;
}

// ----------------------------------------------------------------------------
// Activate()
//
// Toggle Activation of selected Item.
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
// Activate()
// ----------------------------------------------------------------------------

function Activate()
{
	ActivateOn( None );
}

// ----------------------------------------------------------------------------
// ActivateOn()
// ----------------------------------------------------------------------------

function ActivateOn( Actor FrobTarget )
{
	local Pawn PawnOwner;

	if ( !bActivatable )
		return;

	PawnOwner = Pawn(Owner);

	if ( Level.Game.LocalLog!=None )
		Level.Game.LocalLog.LogItemActivate( Self, PawnOwner );
	if (Level.Game.WorldLog != None)
		Level.Game.WorldLog.LogItemActivate( Self, PawnOwner );

	if ( M_Activated!="" )
		PawnOwner.ClientMessage( ItemName$M_Activated, 'StatusLine' );	

	GoToState('Activated');
}

// ----------------------------------------------------------------------------
// InfoTitle() and InfoText()
//
// Text only title and description intended for the info window.
// ----------------------------------------------------------------------------

simulated function string InfoTitle()
{
	return ItemName;
}

simulated function string InfoText()
{
	return Description;
}

// ----------------------------------------------------------------------------
// InventoryCapsString()
//
// Declared in Inventory. Used as a wrapper for retrieving the text for the
// Info window.
// ----------------------------------------------------------------------------

simulated function string InventoryCapsString( Name Property, Pawn Other, Actor Test )
{
	switch ( Property )
	{
		case 'InfoTitle':
			return InfoTitle();
			break;

		case 'InfoText':
			return InfoText();
			break;

		default:
			return "";
			break;
	}
}

//=============================================================================
// state Pickup.
//
// This inventory item is sitting on the ground.
//=============================================================================

auto state Pickup
{
	function BeginState()
	{
		BecomePickup();
		bCollideWorld = true;

		//if ( Level.bStartup )
			//bAlwaysRelevant = true;

		// NumCopies = 1 means we have ONE of this item now DEUS_EX AJY
		//NumCopies = 0;
	}

	function EndState()
	{
		bCollideWorld = false;
		bSleepTouch = false;
	}


	singular function ZoneChange( ZoneInfo NewZone )
	{
		local float splashsize;
		local actor splash;

		if( NewZone.bWaterZone && !Region.Zone.bWaterZone ) 
		{
			splashSize = 0.000025 * Mass * (250 - 0.5 * Velocity.Z);
			if ( NewZone.EntrySound != None )
				PlaySound(NewZone.EntrySound, SLOT_Interact, splashSize);
			if ( NewZone.EntryActor != None )
			{
				splash = Spawn(NewZone.EntryActor); 
				if ( splash != None )
					splash.DrawScale = 2 * splashSize;
			}
		}
	}

	// Validate touch, and if valid trigger event.
	function bool ValidTouch( actor Other )
	{
		local Actor A;

		if( Other.bIsPawn && Pawn(Other).bIsPlayer && (Pawn(Other).Health > 0) && Level.Game.PickupQuery(Pawn(Other), self) )
		{
			if( Event != '' )
				foreach AllActors( class 'Actor', A, Event )
					// Changed to using Self instead of Other, to be able to use this items Tag in triggered MissionScripts.
					//A.Trigger( Other, Other.Instigator );
					A.Trigger( Self, Pawn(Other) );

			return true;
		}
		return false;
	}

	// changed from Touch to Frob - DEUS_EX CNN
	function Frob( Actor Other, Inventory FrobWith )
	{
		local Inventory Copy;

		if ( ValidTouch(Other) ) 
		{
			Copy = SpawnCopy(Pawn(Other));
			if (Level.Game.LocalLog != None)
				Level.Game.LocalLog.LogPickup(Self, Pawn(Other));
			if (Level.Game.WorldLog != None)
				Level.Game.WorldLog.LogPickup(Self, Pawn(Other));
//			if (bActivatable && Pawn(Other).SelectedItem==None) 
//				Pawn(Other).SelectedItem=Copy;
			if ( bActivatable && bAutoActivate && Pawn(Other).bAutoActivate )
				Copy.Activate();
			if ( PickupMessageClass == None )

				Pawn(Other).ClientMessage(PickupMessage @ itemArticle @ itemName, 'Pickup');
			else
				Pawn(Other).ReceiveLocalizedMessage( PickupMessageClass, 0, None, None, Self.Class );
			PlaySound (PickupSound,,2.0);	
			Pickup(Copy).PickupFunction(Pawn(Other));
		}
	}

	// Landed on ground.
	function Landed(Vector HitNormal)
	{
		local rotator newRot;
		newRot = Rotation;
		newRot.pitch = 0;
		SetRotation(newRot);
	}

	// Make sure no pawn already touching (while touch was disabled in sleep).
	function CheckTouching()
	{
		local int i;

		bSleepTouch = false;
		for ( i=0; i<4; i++ )
			if ( (Touching[i] != None) && Touching[i].IsA('Pawn') )
				Touch(Touching[i]);
	}

Begin:
	BecomePickup();
	if ( bRotatingPickup && (Physics != PHYS_Falling) )
		SetPhysics(PHYS_Rotating);

Dropped:
	if( bAmbientGlow )
		AmbientGlow=255;
	if( bSleepTouch )
		CheckTouching();
}

//=============================================================================
// state Activated.
//=============================================================================

state Activated
{
	function Activate()
	{
		ActivateOn( None );
	}

	function ActivateOn( Actor FrobTarget )
	{
		if ( (Pawn(Owner) != None) && Pawn(Owner).bAutoActivate 
			&& bAutoActivate && (Charge>0) )
				return;

		if ( (Pawn(Owner) != None) && (M_Deactivated != "") )
			Pawn(Owner).ClientMessage(ItemName$M_Deactivated);	
		GoToState('DeActivated');	
	}
}

// ----------------------------------------------------------------------------
// ----------------------------------------------------------------------------
// STRING FUNCTIONS
// ----------------------------------------------------------------------------
// ----------------------------------------------------------------------------

// Short form of CarriageReturn. Thats the ION terms for LF though.
static final function String CR()
{
	return Chr(10);
}

// ----------------------------------------------------------------------------
// ----------------------------------------------------------------------------

defaultproperties
{
	FragType=GlassFragment
	CountLabel="COUNT:"
	NumCopies=1
	PickupMessage="You found"
	ItemName="DEFAULT PICKUP NAME - REPORT THIS AS A BUG"
	LandSound=PaperHit1
	bSpawning=True
	bCanDrop=True
	PickupMessageClass=
	//RespawnTime=3153600000.0
	RespawnTime=30.0
	MsgTooMany="You can't carry any more of those"
	MsgDropNotAllowed="You are not allowed to drop your %s"
}
