//=============================================================================
// HXScriptedPawn.
//=============================================================================
class HXScriptedPawn extends ScriptedPawn
	native
	nativereplication
	abstract;

var transient HXGameInfo Game;
var Name PrecessorName;
var bool bSpawning;

var float LipSynchTime; // Used for getting LipSynchTimer() messages .

// Replacement for odd IsA() checks.
var bool bIsRobot; // Prior IsA('Robot')
var bool bIsHuman;
var bool bIsAnimal; // Prior IsA('Animal')
var bool bIsChild;

// Moved out of Robot here.
var() int EMPHitPoints;

var bool bClientInWorld;
var bool bClientHostileTowardsPlayers;

var bool bOn;
var bool bAnimating;

var bool bClientIsSpeaking;		// are we speaking now
var bool bClientWasSpeaking;		// were we speaking last frame?  (should we close our mouth?)

var bool bClientSitting;
var bool bClientDancing;

var name ClientSimulatedAction;
var name LastClientSimulatedAction;

// Player currently frobbing. Used for MedicalBots/RepairBots and later for convos
var HXPlayerPawn curFrobber;

var localized string PawnInUseMsg;

var HXAlarmUnit HXAlarmActor;
var Actor HXConversationActor;

var HXDecoration HXSeatActor;
var HXDecoration HXSeatHack;

// Needed for correct frob display.
var String ClientUnfamiliarName;
var String ClientFamiliarName;
var float ClientLastConEndTime; // HX_NOTE: Server time! Client time might differ.

struct HXNearbyProjectile 
{
	var HXProjectile projectile;
	var vector       location;
	var float        dist;
	var float        range;
};

struct HXNearbyProjectileList
{
	var HXNearbyProjectile list[8];
	var vector             center;
};

//
// Used to keep track of distance to players. Entries is only valid if Player!=None
// and are sorted starting with nearest player.
//
struct PlayerDistance
{
	var HXPlayerPawn Player;
	var float        Distance;
};

var transient PlayerDistance PlayerDistances[32];

// Cached bools for available barks.
var bool bHasIdleBark;
var bool bHasCriticalDamageBark;
var bool bHasAreaSecureBark;
var bool bHasTargetAcquiredBark;
var bool bHasTargetLostBark;
var bool bHasGoingForAlarmBark;
var bool bHasOutOfAmmoBark;
var bool bHasScanningBark;
var bool bHasFutzBark;
var bool bHasOnFireBark;
var bool bHasTearGasBark;
var bool bHasGoreBark;
var bool bHasSurpriseBark;
var bool bHasPreAttackSearchingBark;
var bool bHasPreAttackSightingBark;
var bool bHasPostAttackSearchingBark;
var bool bHasSearchGiveUpBark;
var bool bHasAllianceHostileBark;
var bool bHasAllianceFriendlyBark;

// String to Name conversion.
native(3320) static final function Name StringToName( Coerce string Str );

//
// Fixed and extended version of TraceTexture() which also returns
// the Texture hit (Hint: There are Sound properties on Texture).
//
native(3321) final iterator function TraceTextures
(
	Class<Actor> BaseClass,
	out actor Actor,
	out Texture Tex,
	out name TexName,
	out name TexGroup,
	out int Flags,
	out vector HitLoc,
	out vector HitNorm,
	vector End,
	optional vector Start,
	optional vector Extent
);

//
// First Hit variant of TraceTextures.
// (The same as Trace is to TraceActors)
//
native(3322) final function Actor TraceSingleTexture
(
	Class<Actor> BaseClass,
	out Texture  Texture,
	out Name     TextureName,
	out Name     TextureGroup,
	out int      TextureFlags,
	out Vector   HitLocation,
	out Vector   HitNormal,
	Vector       TraceEnd,
	optional Vector TraceStart,
	optional bool bTraceActors,
	optional Vector TraceExtent
);

//
// Behaves the same FastTrace but returns the
// surface texture if a hit occured.
//
/*native(3323) final function bool FastTextureTrace
(
	Vector          TraceEnd,
	out Texture     Texture,
	out Name        TextureName,
	out Name        TextureGroup,
	out int         TextureFlags,
	optional Vector TraceStart
);*/

//
// Returns true if did not hit non transparent world geometry.
// This might not be exact as this might check a coplanar bsp surface.
// However most windows are on a different plane, so this works
// fine nearly all the time (expect for maybe ugly maps) so it would
// not be worth the additional performance hit.
//
// Code was kindly donated by Smirftsch (oldunreal.com).
//
native(3324) final function bool FastRepTrace
(
	vector          TraceEnd,
	optional vector TraceStart
);

//
// Stops the specified or all sound slots for an Actor.
//
// These functions do work in network. However, ClientHearSound() is NOT
// reliable, so it could happen that the PlaySound() call reaches the client
// after the StopSoundId/StopSoundSlot/StopAllSoundSlots call. So try to avoid 
// starting the sound and stopping it at the same or next few ticks and you
// are safe (the same applies to stop the sound to just play another).
//
native(3225) final function StopSoundSlot( ESoundSlot Slot );
native(3226) final function StopAllSoundSlots();
native simulated event DemoStopSoundSlot( ESoundSlot Slot );
native simulated event DemoStopAllSoundSlots();

// ----------------------------------------------------------------------------
// Network replication.
// ----------------------------------------------------------------------------

replication
{
  // server to client
  reliable if ( Role==ROLE_Authority)
		bClientInWorld, bClientHostileTowardsPlayers,
		bClientIsSpeaking,
		HXSeatActor, HXSeatHack, bClientSitting, bClientDancing,
		ClientSimulatedAction,
		ClientFamiliarName, ClientUnfamiliarName, ClientLastConEndTime;

	reliable if ( Role==ROLE_Authority && bIsRobot )
		EMPHitPoints;

	reliable if( bDemoRecording )
		DemoStopSoundSlot, DemoStopAllSoundSlots;

/*
	// Native replication override for target augmentation.
	// TODO: Change this to a more flexible HealhPercent handling as this would
	//       allow health to be scaled on difficulty.
	reliable if (Role == ROLE_Authority && bIsHuman)
		HealthHead, HealthTorso, HealthLegLeft, HealthLegRight, HealthArmLeft, HealthArmRight;

	// native replication override for head blend anim.
	reliable if (Role == ROLE_Authority && DrawType==DT_Mesh && ((RemoteRole<=ROLE_SimulatedProxy && (!bNetOwner || !bClientAnim)) || bDemoRecording) )
		BlendAnimSequence[3], SimBlendAnim[3], BlendAnimMinRate[3];
*/
}

// ----------------------------------------------------------------------------
// Succeeds()
//
// Intended to be called by Mutator code if Self was spawned as a successor
// to Other. Super call recommended.
// ----------------------------------------------------------------------------

function Succeeds( Actor Other )
{
	local int iInitialAlliances, iInitialInventory;
	local Pawn OtherPawn;
	local ScriptedPawn OtherScriptedPawn;
	local Robot OtherRobot;

	// Save Precessors name aside for debugging purposes.
	PrecessorName        = Other.Name;

	// Advanced.
//bStatic              = Other.bStatic;             // Nope. Defined by defaultproperties.
	bHidden              = Other.bHidden;
//bNoDelete            = Other.bNoDelete;           // Nope. Defined by defaultproperties.
	bCanTeleport         = Other.bCanTeleport;        // Do I want this?
	bIsSecretGoal        = Other.bIsSecretGoal;
	bIsKillGoal          = Other.bIsKillGoal;
	bIsItemGoal          = Other.bIsItemGoal;
	bCollideWhenPlacing  = Other.bCollideWhenPlacing; // Still used by FarMove.
//bTravel              = Other.bTravel;             // Do not mess with this.
//bMovable             = Other.bMovable;            // Do not mess with this.
//bHighDetail          = Other.bHighDetail;         // Nah, do not mess with this.
//bStasis              = Other.bStasis;             // Rather not-ish.
//bForceStasis         = Other.bForceStasis;        // Rather not-ish.
//NetTemporary         = Other.NetTemporary;        // Certainly not.
//bNetOptional         = Other.bNetOptional;        // Certainly not.
  bBlockSight          = Other.bBlockSight;
  bDetectable          = Other.bDetectable;
//bTransient           = Other.bTransient;          // Rather not-ish.
	LifeSpan             = Other.LifeSpan;
	bHiddenEd            = Other.bHiddenEd;           // Editing flag.
	bDirectional         = Other.bDirectional;        // Editing flag.
	bEdShouldSnap        = Other.bEdShouldSnap;       // Editing flag.
//bOwnerNoSee          = Other.bOwnerNoSee;         // Rather not-ish.
//bOnlyOwnerSee        = Other.bOnlyOwnerSee;       // Rather not-ish.
//bAlwaysRelevant      = Other.bAlwaysRelevant;     // Certainly not-ish.
//bGameRelevant        = Other.bGameRelevant;       // Rather uncertain-ish.
	bOwned               = Other.bOwned;

	// Collision.
	//SetCollisionSize( Other.CollisionRadius, Other.CollisionHeight ); // Pawns keep resetting this, so try to not copy it for now.
	SetCollision( Other.bCollideActors, Other.bBlockActors, Other.bBlockPlayers );
	//bCollideWorld	       = Other.bCollideWorld;       // !! This can be an issue to copy. Inventory falls through the floor, etc. !!
	bProjTarget          = Other.bProjTarget;

	// Conversation.
	BindName             = Other.BindName;            // !! Shadowed for ConversationTrigger.
	BarkBindName         = Other.BarkBindName;
	FamiliarName         = Other.FamiliarName;
	UnfamiliarName       = Other.UnfamiliarName;
	ConStartInterval     = Other.ConStartInterval;

	// Display.
	DrawType             = Other.DrawType;
	Style                = Other.Style;
	Sprite               = Other.Sprite;
	Texture              = Other.Texture;
	Skin                 = Other.Skin;
	MultiSkins[0]        = Other.MultiSkins[0];
	MultiSkins[1]        = Other.MultiSkins[1];
	MultiSkins[2]        = Other.MultiSkins[2];
	MultiSkins[3]        = Other.MultiSkins[3];
	MultiSkins[4]        = Other.MultiSkins[4];
	MultiSkins[5]        = Other.MultiSkins[5];
	MultiSkins[6]        = Other.MultiSkins[6];
	MultiSkins[7]        = Other.MultiSkins[7];
	Mesh                 = Other.Mesh;
	DrawScale            = Other.DrawScale;
	ScaleGlow            = Other.ScaleGlow;
	AmbientGlow          = Other.AmbientGlow;
	Fatness              = Other.Fatness;
	bUnlit               = Other.bUnlit;
//bNoSmooth            = Other.bNoSmooth;           // Not replicated.
//bParticles           = Other.bParticles;          // Not replicated.
//bRandomFrame         = Other.bRandomFrame;        // Not replicated.
	bMeshEnviroMap       = Other.bMeshEnviroMap;
//bMeshCurvy           = Other.bMeshCurvy;          // Not replicated.
//VisibilityRadius     = Other.VisibilityRadius;    // Not replicated.
//VisibilityHeight     = Other.VisibilityHeight;    // Not replicated.
//bShadowCast          = Other.bShadowCast;         // Not replicated.
	AnimSequence         = Other.AnimSequence;        // Preset?
	AnimFrame            = Other.AnimFrame;           // Preset? Implicit replicated?
	AnimRate             = Other.AnimRate;            // Preset? Implicit replicated?
//LODBias              = Other.LODBias;             // Not replicated.
//RenderIteratorClass  = Other.RenderIteratorClass; // Not replicated.

	// Events. 
	Event                = Other.Event;
//Tag                  = Other.Tag;   // Set by Spawn.

	// Filter.
	bDifficulty0         = Other.bDifficulty0;
	bDifficulty1         = Other.bDifficulty1;
	bDifficulty2         = Other.bDifficulty2;
	bDifficulty3         = Other.bDifficulty3;
	bSinglePlayer        = Other.bSinglePlayer;
	bNet                 = Other.bNet;
	bNetSpecial          = Other.bNetSpecial;
	OddsOfAppearing      = Other.OddsOfAppearing;

	// LightColor.
	LightBrightness      = Other.LightBrightness;
	LightHue             = Other.LightHue;
	LightSaturation      = Other.LightSaturation;

	// Lighting.
	LightType            = Other.LightType;
	LightEffect          = Other.LightEffect;
	LightRadius          = Other.LightRadius;
	LightPeriod          = Other.LightPeriod;
	LightPhase           = Other.LightPhase;
//LightCone            = Other.LightCone;        // Not replicated.
	VolumeBrightness     = Other.VolumeBrightness;
	VolumeRadius         = Other.VolumeRadius;
//VolumeFog            = Other.VolumeFog;        // Not replicated.
	bSpecialLit          = Other.bSpecialLit;
//bActorShadows        = Other.bActorShadows;    // Not replicated.
	bCorona              = Other.bCorona;          // Not replicated.
	bLensFlare           = Other.bLensFlare;       // Not replicated.

	// Movement.
//Location            = Other.Location; // Set by Spawn.
//Rotation            = Other.Rotation; // Set by Spawn.
//Velocity            = Other.Velocity; // Nah.
//SetPhysics( Other.Physics );          // Tricky-ish. I'm now using SpawnPhysics though.
	AttachTag           = Other.AttachTag;
	bBounce             = Other.bBounce;
	bFixedRotationDir   = Other.bFixedRotationDir;
	bRotateToDesired    = Other.bRotateToDesired;
	Mass                = Other.Mass;
	Buoyancy            = Other.Buoyancy;
	RotationRate        = Other.RotationRate;
	DesiredRotation     = Other.DesiredRotation;

	// Networking.
//RemoteRole           = Other.RemoteRole;         // I certainly
//NetPriority          = Other.NetPriority;        // won't use
//NetUpdateFrequency   = Other.NetUpdateFrequency; // these
//RelevantRadius       = Other.RelevantRadius;     // values. --han

	// Object.
	InitialState         = Other.InitialState; // Does this even work out of the box?
	Group                = Other.Group;

	// Smell.
	SmellClass           = Other.SmellClass;

	// Sound.
	SoundRadius          = Other.SoundRadius;
	SoundVolume          = Other.SoundVolume;
	SoundPitch           = Other.SoundPitch;
	AmbientSound         = Other.AmbientSound;
//TransientSoundVolume = Other.TransientSoundVolume; // Not replicated.
//TransientSoundRadius = Other.TransientSoundRadius; // Not replicated.

	// Make sure old Actor has Tag, Event, Group, BindName and BarkBindName removed.
	Other.Tag            = '';
	Other.Event          = '';
	Other.Group          = '';
	Other.BindName       = "";
	Other.BarkBindName   = "";

	// Remove Collision of old Actor if bStatic or bNoDelete.
	if ( Other.bStatic || Other.bNoDelete )
	{
		Other.SetCollision( false, false, false );
	}

	// Special Pawn Init.
	OtherPawn = Pawn(Other);
	if ( OtherPawn!=None )
	{
		bCanStrafe                 = OtherPawn.bCanStrafe;
		bFixedStart                = OtherPawn.bFixedStart;
		MeleeRange                 = OtherPawn.MeleeRange;
		GroundSpeed                = OtherPawn.GroundSpeed;
		WaterSpeed                 = OtherPawn.WaterSpeed;
		AirSpeed                   = OtherPawn.AirSpeed;
		AccelRate                  = OtherPawn.AccelRate;
		JumpZ                      = OtherPawn.JumpZ;
		MaxStepHeight              = OtherPawn.MaxStepHeight;
		AirControl                 = OtherPawn.AirControl;
		Visibility                 = OtherPawn.Visibility;
		SightRadius                = OtherPawn.SightRadius;
		PeripheralVision           = OtherPawn.PeripheralVision;
		HearingThreshold           = OtherPawn.HearingThreshold;
		BaseEyeHeight              = OtherPawn.BaseEyeHeight;
		FovAngle                   = OtherPawn.FovAngle;
		Health                     = OtherPawn.Health;
		SelectionMesh              = OtherPawn.SelectionMesh;
		SpecialMesh                = OtherPawn.SpecialMesh;
		ReducedDamageType          = OtherPawn.ReducedDamageType;
		ReducedDamagePct           = OtherPawn.ReducedDamagePct;
		DropWhenKilled             = OtherPawn.DropWhenKilled;
		UnderWaterTime             = OtherPawn.UnderWaterTime;

		SetPropertyText( "AttitudeToPlayer", OtherPawn.GetPropertyText("AttitudeToPlayer") );
		SetPropertyText( "Intelligence",     OtherPawn.GetPropertyText("Intelligence"    ) );

		Skill                      = OtherPawn.Skill;
		Land                       = OtherPawn.Land;
		WaterStep                  = OtherPawn.WaterStep;
		CombatStyle                = OtherPawn.CombatStyle;
		AlarmTag                   = OtherPawn.AlarmTag;
		SharedAlarmTag             = OtherPawn.SharedAlarmTag;
		MenuName                   = OtherPawn.MenuName;
		NameArticle                = OtherPawn.NameArticle;
		VoicePitch                 = OtherPawn.VoicePitch;
		VoiceType                  = OtherPawn.VoiceType;
	//PlayerReplicationInfoClass = OtherPawn.PlayerReplicationInfoClass; // Nope.
		HealthHead                 = OtherPawn.HealthHead;
		HealthTorso                = OtherPawn.HealthTorso;
		HealthTorso                = OtherPawn.HealthTorso;
		HealthLegLeft              = OtherPawn.HealthLegLeft;
		HealthLegRight             = OtherPawn.HealthLegRight;
		HealthArmLeft              = OtherPawn.HealthArmLeft;
		HealthArmRight             = OtherPawn.HealthArmRight;
		AIHorizontalFov            = OtherPawn.AIHorizontalFov;
		AspectRatio                = OtherPawn.AspectRatio;
		AngularResolution          = OtherPawn.AngularResolution;
		VisibilityThreshold        = OtherPawn.VisibilityThreshold;
		SmellThreshold             = OtherPawn.SmellThreshold;
		Alliance                   = OtherPawn.Alliance;

		// Special quirk for bIsFemale and bIsChild hit and death sounds were set by script before.
		if ( OtherPawn.HitSound1!=OtherPawn.default.HitSound1 ) HitSound1 = OtherPawn.HitSound1;
		if ( OtherPawn.HitSound2!=OtherPawn.default.HitSound2 ) HitSound2 = OtherPawn.HitSound2;
		if ( OtherPawn.Die      !=OtherPawn.default.Die       ) Die       = OtherPawn.Die;

		// Special ScriptedPawn Init.
		OtherScriptedPawn = ScriptedPawn(Other);
		if ( OtherScriptedPawn!=None )
		{
			BaseAccuracy        = OtherScriptedPawn.BaseAccuracy;
			MaxRange            = OtherScriptedPawn.MaxRange;
			MinRange            = OtherScriptedPawn.MinRange;
			MinHealth           = OtherScriptedPawn.MinHealth;
			RandomWandering     = OtherScriptedPawn.RandomWandering;
			CarcassType         = OtherScriptedPawn.CarcassType;
			Orders              = OtherScriptedPawn.Orders;
			OrderTag            = OtherScriptedPawn.OrderTag;
			HomeTag             = OtherScriptedPawn.HomeTag;
			HomeExtent          = OtherScriptedPawn.HomeExtent;
			ProjectileSpeed     = OtherScriptedPawn.ProjectileSpeed;
			ClotPeriod          = OtherScriptedPawn.ClotPeriod;
			bKeepWeaponDrawn    = OtherScriptedPawn.bKeepWeaponDrawn;
			bShowPain           = OtherScriptedPawn.bShowPain;
			bCanSit             = OtherScriptedPawn.bCanSit;
			bAlwaysPatrol       = OtherScriptedPawn.bAlwaysPatrol;
			bPlayIdle           = OtherScriptedPawn.bPlayIdle;
			bLeaveAfterFleeing  = OtherScriptedPawn.bLeaveAfterFleeing;
			bLikesNeutral       = OtherScriptedPawn.bLikesNeutral;
			bUseFirstSeatOnly   = OtherScriptedPawn.bUseFirstSeatOnly;
			bCower              = OtherScriptedPawn.bCower;
			bImportant          = OtherScriptedPawn.bImportant;
			bInvincible         = OtherScriptedPawn.bInvincible;
			bAvoidAim           = OtherScriptedPawn.bAvoidAim;
			bAimForHead         = OtherScriptedPawn.bAimForHead;
			bDefendHome         = OtherScriptedPawn.bDefendHome;
			bUseFallbackWeapons = OtherScriptedPawn.bUseFallbackWeapons;
			bHateHacking        = OtherScriptedPawn.bHateHacking;
			bHateWeapon         = OtherScriptedPawn.bHateWeapon;
			bHateShot           = OtherScriptedPawn.bHateShot;
			bHateInjury         = OtherScriptedPawn.bHateInjury;
			bHateIndirectInjury = OtherScriptedPawn.bHateIndirectInjury;
			bHateCarcass        = OtherScriptedPawn.bHateCarcass;
			bHateDistress       = OtherScriptedPawn.bHateDistress;
			bReactFutz          = OtherScriptedPawn.bReactFutz;
			bReactPresence      = OtherScriptedPawn.bReactPresence;
			bReactLoudNoise     = OtherScriptedPawn.bReactLoudNoise;
			bReactAlarm         = OtherScriptedPawn.bReactAlarm;
			bReactShot          = OtherScriptedPawn.bReactShot;
			bReactCarcass       = OtherScriptedPawn.bReactCarcass;
			bReactDistress      = OtherScriptedPawn.bReactDistress;
			bReactProjectiles   = OtherScriptedPawn.bReactProjectiles;
			bFearHacking        = OtherScriptedPawn.bFearHacking;
			bFearWeapon         = OtherScriptedPawn.bFearWeapon;
			bFearShot           = OtherScriptedPawn.bFearShot;
			bFearInjury         = OtherScriptedPawn.bFearInjury;
			bFearIndirectInjury = OtherScriptedPawn.bFearIndirectInjury;
			bFearCarcass        = OtherScriptedPawn.bFearCarcass;
			bFearDistress       = OtherScriptedPawn.bFearDistress;
			bFearAlarm          = OtherScriptedPawn.bFearAlarm;
			bFearProjectiles    = OtherScriptedPawn.bFearProjectiles;
			bEmitDistress       = OtherScriptedPawn.bEmitDistress;

			SetPropertyText( "RaiseAlarm", OtherScriptedPawn.GetPropertyText("RaiseAlarm") );

			bMustFaceTarget     = OtherScriptedPawn.bMustFaceTarget;
			FireAngle           = OtherScriptedPawn.FireAngle;
			FireElevation       = OtherScriptedPawn.FireElevation;
			MaxProvocations     = OtherScriptedPawn.MaxProvocations;

			for ( iInitialAlliances=0; iInitialAlliances<ArrayCount(InitialAlliances); iInitialAlliances++ )
			{
				InitialAlliances[iInitialAlliances].AllianceName  = OtherScriptedPawn.InitialAlliances[iInitialAlliances].AllianceName;
				InitialAlliances[iInitialAlliances].AllianceLevel = OtherScriptedPawn.InitialAlliances[iInitialAlliances].AllianceLevel;
				InitialAlliances[iInitialAlliances].bPermanent    = OtherScriptedPawn.InitialAlliances[iInitialAlliances].bPermanent;
			}

			BaseAssHeight       = OtherScriptedPawn.BaseAssHeight;
			EnemyTimeout        = OtherScriptedPawn.EnemyTimeout;
			bTickVisibleOnly    = OtherScriptedPawn.bTickVisibleOnly;
			bInWorld            = OtherScriptedPawn.bInWorld;
			bHighlight          = OtherScriptedPawn.bHighlight;
			bHokeyPokey         = OtherScriptedPawn.bHokeyPokey;

			for ( iInitialInventory=0; iInitialInventory<ArrayCount(InitialInventory); iInitialInventory++ )
			{
				InitialInventory[iInitialInventory].Inventory = OtherScriptedPawn.InitialInventory[iInitialInventory].Inventory;
				InitialInventory[iInitialInventory].Count     = OtherScriptedPawn.InitialInventory[iInitialInventory].Count;
			}

			WalkSound           = OtherScriptedPawn.WalkSound;

			// Keep this.. thing for now until addressing pawn shadow code. --han
			bClientInWorld = bInWorld;

			// Set these from here to the correct value
			ClientFamiliarName   = FamiliarName;
			ClientUnfamiliarName = UnfamiliarName;

			// Set bImportant to false to avoid warnings. Why is this needed? --han
			OtherScriptedPawn.bImportant = false;

			// Prevent OtherPawn from spawning a rogue shadow
			OtherScriptedPawn.bHasShadow = false;

			// Special ScriptedPawn Init.
			OtherRobot = Robot(Other);
			if ( OtherRobot!=None )
			{
				EMPHitPoints = OtherRobot.EMPHitPoints; // Moved off Robot.
			}
		}
	}
}

// ----------------------------------------------------------------------------
// IsInRelevant()
// 
// Whether code is currently running during a spawn inside a mutator in which
// case the properties are not yet properly set.
// ----------------------------------------------------------------------------

simulated function bool IsInRelevant()
{
	if ( Level.Game==None ) // Will happen on client too.
		return false;
	return Level.Game.IsInState( 'InIsRelevant' );
}

// ----------------------------------------------------------------------------
// Spawned()
// ----------------------------------------------------------------------------

simulated event Spawned()
{
	if ( Level.bStartup )
		bGameRelevant = true;

	Game = HXGameInfo(Level.Game);
}

// ----------------------------------------------------------------------------
// PreBeginPlay()
// ----------------------------------------------------------------------------

simulated function PreBeginPlay()
{
	local class<Actor> ActorClass;
	local int iInitialInventory;

	if ( bDeleteMe || IsInRelevant() )
		return;

	// Handle autodestruction if desired. Moved this here because the
	// config properties won't have their correct value set when spawned inside
	// the mutator, so you can't replace them at first pass there.
	if ( !bGameRelevant && Level.NetMode!=NM_Client && !Level.Game.IsRelevant(Self) )
	{
		Destroy();
		return;
	}

	// Add to Level.PawnList.
	AddPawn();

	if ( Level.NetMode!=NM_Client )
	{
		if ( DropWhenKilled!=None )
		{
			Game.ModifyInventoryClass( DropWhenKilled );
		}
		if ( CarcassType!=None )
		{
			ActorClass = CarcassType;
			Game.ModifyActorClass( ActorClass );
			CarcassType = Class<Carcass>(ActorClass);
		}
		for ( iInitialInventory=0; iInitialInventory<ArrayCount(InitialInventory); iInitialInventory++ )
			if ( InitialInventory[iInitialInventory].Inventory!=None )
				Game.ModifyInventoryClass( InitialInventory[iInitialInventory].Inventory );

		// Added angular size computation - DEUS_EX STM
		MinAngularSize = tan(AngularResolution*0.5*Pi/180.0);
		MinAngularSize *= MinAngularSize;

		// Set instigator to self.
		Instigator = Self;
		DesiredRotation = Rotation;
		SightCounter = 0.2 * FRand();  //offset randomly 
		if ( Level.Game != None )
			Skill += Level.Game.Difficulty; 
		Skill = FClamp(Skill, 0, 3);
		PreSetMovement();

		if ( DrawScale!=Default.Drawscale )
		{
			SetCollisionSize(CollisionRadius*DrawScale/Default.DrawScale, CollisionHeight*DrawScale/Default.DrawScale);
			Health = Health * DrawScale/Default.DrawScale;
		}

		if ( BaseEyeHeight==0 )
			EyeHeight = 0.8*CollisionHeight;
		else
			EyeHeight = BaseEyeHeight;

		// Vary monster fatness slightly if at default.
		if ( Fatness==0 )
			Fatness = 120+Rand(8)+Rand(8);

		if ( MenuName=="" )
			MenuName = GetItemName( string(Class) );

		if ( SelectionMesh=="" )
			SelectionMesh = string(Mesh);

		// create our shadow.
		CreateShadow();

		// Set our alliance.
		SetAlliance( Alliance );

		// Set up callbacks.
		UpdateReactionCallbacks();

		// Unreal Gold...
		// Has 200 / 0.15
		//SightRadius			 += 200 * Level.Game.Difficulty;
		//RotationRate.Yaw *= 1 + ( 0.20 * Level.Game.Difficulty );
		// HearingThreshold?
		// PeripheralVision?

		// Why? --han
		GenerateTotalHealth();
	}
}

// ----------------------------------------------------------------------------
// PostBeginPlay()
// ----------------------------------------------------------------------------

simulated function PostBeginPlay()
{
	if ( bDeleteMe || IsInRelevant() )
		return;

	if ( Role==ROLE_Authority )
	{
		SplashTime = 0;

		// Set up pain timer
		if ( Region.Zone.bPainZone || HeadRegion.Zone.bPainZone || FootRegion.Zone.bPainZone )
			PainTime = 5.0;
		else if ( HeadRegion.Zone.bWaterZone )
			PainTime = UnderWaterTime;

		// Handle holograms.
		if ((Style != STY_Masked) && (Style != STY_Normal))
		{
			SetSkinStyle(Style, None);
			SetCollision(false, false, false);
			KillShadow();
			bHasShadow = False;
		}

		InitializePawn();
	}
}

// ----------------------------------------------------------------------------
// SetInitialState()
//
// Called after PostBeginPlay.
// ----------------------------------------------------------------------------

simulated event SetInitialState()
{
	if ( bDeleteMe || IsInRelevant() )
		return;

	if ( InitialState!='' )
		GotoState( InitialState );
	else
		GotoState( 'Auto' );

	bSpawning = false;
}

// ----------------------------------------------------------------------------
// PostPostBeginPlay()
// ----------------------------------------------------------------------------

simulated function PostPostBeginPlay()
{
	if ( bDeleteMe || IsInRelevant() )
		return;

	// This may need to be reset, as a new GameInfo is spawned each time.
	Game = HXGameInfo(Level.Game);

	if ( Role==ROLE_Authority )
	{
		// ScriptedPawns can be client side (Fly, etc.), so be careful to not invoke ConBindEvents.
		if ( Level.NetMode!=NM_Client )
		{
			// Bind any conversation events to this ScriptedPawn.
			ConBindEvents();

			// Cache availibility of barks.
			CacheBarks();
		}

		// Make sure it gets updated at map reload.
		UpdateReactionCallbacks();
	}
}

// ----------------------------------------------------------------------------
// PostNetBeginPlay()
// ----------------------------------------------------------------------------

simulated function PostNetBeginPlay()
{
	local Pawn Pawn;

	if ( Role==ROLE_SimulatedProxy )
	{
		//Log("HXScriptedPawn.PostNetBeginPlay() with Role == ROLE_SimulatedProxy called.");

		if (bHasShadow && bClientInWorld)
			if (Shadow == None)
				Shadow = Spawn(class'Shadow', Self,, Location-vect(0,0,1)*CollisionHeight, rot(16384,0,0));
	}
}

// ----------------------------------------------------------------------------
// Destroyed()
// ----------------------------------------------------------------------------

simulated function Destroyed()
{
	if ( CurFrobber!=None )
		CurFrobber.CloseThisPawn( Self );

	// Pass a message to conPlay, if it exists in the gameinfo, that 
	// this pawn has been destroyed.  This is used to prevent 
	// bad things from happening in converseations.

	if ( Level.NetMode!=NM_Client && Level.Game.IsA('HXGameInfo') )
		if ( HXGameInfo(Level.Game).ConPlay!=None )
			HXGameInfo(Level.Game).ConPlay.ActorDestroyed( Self );

	Super(Pawn).Destroyed();
}

// ----------------------------------------------------------------------------
// FellOutOfWorld()
// ----------------------------------------------------------------------------

event FellOutOfWorld()
{
	if ( PrecessorName!='' )
	{
		Log( Sprintf("Precessor of falling out of world %s was %s.",Name,PrecessorName), 'FellOutOfWorld' );
	}

	if ( Level.NetMode!=NM_Client )
	{
		// HX_NOTE: EVIL HACK!
		if ( !bInWorld )
		{
			Log( "Saved " $ Self $ " from falling out of the world." );

			SetLocation( WorldPosition + vect(0,0,20000) );
			SetPhysics(PHYS_None);
		}
		else
		{
			Health = -1;
			SetPhysics(PHYS_None);
			Weapon = None;
			Died( None, 'Fell', Location );
		}
	}
}

// ----------------------------------------------------------------------------
// CacheBarks()
//
// Caches avaibility of the certain bark parts, to be able to cancel out
// early.
// ----------------------------------------------------------------------------

function CacheBarks()
{
	bHasIdleBark                = Game.HasAIBarkConversation( Self, BM_Idle );
	bHasCriticalDamageBark      = Game.HasAIBarkConversation( Self, BM_CriticalDamage );
	bHasAreaSecureBark          = Game.HasAIBarkConversation( Self, BM_AreaSecure );
	bHasTargetAcquiredBark      = Game.HasAIBarkConversation( Self, BM_TargetAcquired );
	bHasTargetLostBark          = Game.HasAIBarkConversation( Self, BM_TargetLost );
	bHasGoingForAlarmBark       = Game.HasAIBarkConversation( Self, BM_GoingForAlarm );
	bHasOutOfAmmoBark           = Game.HasAIBarkConversation( Self, BM_OutOfAmmo );
	bHasScanningBark            = Game.HasAIBarkConversation( Self, BM_Scanning );
	bHasFutzBark                = Game.HasAIBarkConversation( Self, BM_Futz );
	bHasOnFireBark              = Game.HasAIBarkConversation( Self, BM_OnFire );
	bHasTearGasBark             = Game.HasAIBarkConversation( Self, BM_TearGas );
	bHasGoreBark                = Game.HasAIBarkConversation( Self, BM_Gore );
	bHasSurpriseBark            = Game.HasAIBarkConversation( Self, BM_Surprise );
	bHasPreAttackSearchingBark  = Game.HasAIBarkConversation( Self, BM_PreAttackSearching );
	bHasPreAttackSightingBark   = Game.HasAIBarkConversation( Self, BM_PreAttackSighting );
	bHasPostAttackSearchingBark = Game.HasAIBarkConversation( Self, BM_PostAttackSearching );
	bHasSearchGiveUpBark        = Game.HasAIBarkConversation( Self, BM_SearchGiveUp );
	bHasAllianceHostileBark     = Game.HasAIBarkConversation( Self, BM_AllianceHostile );
	bHasAllianceFriendlyBark    = Game.HasAIBarkConversation( Self, BM_AllianceFriendly );
}

// ----------------------------------------------------------------------------
// ----------------------------------------------------------------------------
// GENERAL UTILITIES
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
// InitializePawn()
// ----------------------------------------------------------------------------

function InitializePawn()
{
	if (!bInitialized)
	{
		// maybe put it here will help
		bInitialized = true;

		InitializeInventory();
		InitializeAlliances();
		InitializeHomeBase();

		BlockReactions();

		if (Alliance != '')
			ChangeAlly(Alliance, 1.0, true);

		if (!bInWorld)
		{
			// tricky
			bInWorld = true;
			LeaveWorld();
		}

		// hack!
		animTimer[1] = 20.0;
		PlayTurnHead(LOOK_Forward, 1.0, 0.0001);

		//bInitialized = true;
	}
}

// ----------------------------------------------------------------------------
// InitializeInventory()
// ----------------------------------------------------------------------------

function InitializeInventory()
{
	local int       i, j;
	local Inventory inv;
	local Weapon    weapons[8];
	local int       weaponCount;
	local Weapon    firstWeapon;

	// Add initial inventory items
	weaponCount = 0;
	for (i=0; i<8; i++)
	{
		if ((InitialInventory[i].Inventory != None) && (InitialInventory[i].Count > 0))
		{
			firstWeapon = None;
			for (j=0; j<InitialInventory[i].Count; j++)
			{
				inv = None;
				if (Class<Ammo>(InitialInventory[i].Inventory) != None)
				{
					inv = FindInventoryType(InitialInventory[i].Inventory);
					if (inv != None)
					{
						//Ammo(inv).AmmoAmount += Class<Ammo>(InitialInventory[i].Inventory).default.AmmoAmount; // This allowed more than max ammo. --ha
						Ammo(Inv).AddAmmo( Class<Ammo>(InitialInventory[i].Inventory).default.AmmoAmount );
					}
				}
				if (inv == None)
				{
					inv = spawn(InitialInventory[i].Inventory, self);
					if (inv != None)
					{
						inv.InitialState='Idle2';
						inv.GiveTo(Self);
						inv.SetBase(Self);
						if ((firstWeapon == None) && (Weapon(inv) != None))
							firstWeapon = Weapon(inv);
					}
				}
			}
			if (firstWeapon != None)
				weapons[WeaponCount++] = firstWeapon;
		}
	}
	for (i=0; i<weaponCount; i++)
	{
		if ((weapons[i].AmmoType == None) && (weapons[i].AmmoName != None) &&
			(weapons[i].AmmoName != Class'HXAmmoNone'))
		{
			weapons[i].AmmoType = Ammo(FindInventoryType(weapons[i].AmmoName));
			if (weapons[i].AmmoType == None)
			{
				weapons[i].AmmoType = spawn(weapons[i].AmmoName);
				weapons[i].AmmoType.InitialState='Idle2';
				weapons[i].AmmoType.GiveTo(Self);
				weapons[i].AmmoType.SetBase(Self);
			}
		}
	}

	SetupWeapon(false);
}





// ----------------------------------------------------------------------------
// GetCarcassData()
// ----------------------------------------------------------------------------

function bool GetCarcassData(actor sender, out Name killer, out Name alliance,
                             out Name CarcassName, optional bool bCheckName)
{
	local DeusExPlayer  dxPlayer;
	local HXCarcass carcass;
	local POVCorpse     corpseItem;
	local bool          bCares;
	local bool          bValid;

	alliance = '';
	killer   = '';

	bValid   = false;
	dxPlayer = DeusExPlayer(sender);
	carcass  = HXCarcass(sender);
	if (dxPlayer != None)
	{
		corpseItem = POVCorpse(dxPlayer.inHand);
		if (corpseItem != None)
		{
			if (corpseItem.bEmitCarcass)
			{
				alliance    = corpseItem.Alliance;
				killer      = corpseItem.KillerAlliance;
				CarcassName = corpseItem.CarcassName;
				bValid      = true;
			}
		}
	}
	else if (carcass != None)
	{
		if (carcass.bEmitCarcass)
		{
			alliance    = carcass.Alliance;
			killer      = carcass.KillerAlliance;
			CarcassName = carcass.CarcassName;
			bValid      = true;
		}
	}

	bCares = false;
	if (bValid && (!bCheckName || !HaveSeenCarcass(CarcassName)))
	{
		if (bFearCarcass)
			bCares = true;
		else
		{
			if (GetAllianceType(alliance) == ALLIANCE_Friendly)
			{
				if (bHateCarcass)
					bCares = true;
				else if (bReactCarcass)
				{
					if (GetAllianceType(killer) == ALLIANCE_Hostile)
						bCares = true;
				}
			}
		}
	}

	return bCares;
}

// ----------------------------------------------------------------------------
// ReactToProjectiles()
// ----------------------------------------------------------------------------

function ReactToProjectiles(Actor projectileActor)
{
	local HXProjectile dxProjectile;
	local Pawn         instigator;

	if ((bFearProjectiles || bReactProjectiles) && bLookingForProjectiles)
	{
		dxProjectile = HXProjectile(projectileActor);
		if ((dxProjectile == None) || HXIsProjectileDangerous(dxProjectile))
		{
			instigator = Pawn(projectileActor);
			if (instigator == None)
				instigator = projectileActor.Instigator;
			if (instigator != None)
			{
				if (bFearProjectiles)
					IncreaseFear(instigator, 2.0);
				if (SetEnemy(instigator))
				{
					SetDistressTimer();
					HandleEnemy();
				}
				else if (bFearProjectiles && IsFearful())
				{
					SetDistressTimer();
					SetEnemy(instigator, , true);
					GotoState('Fleeing');
				}
				else if (bAvoidHarm)
					SetState('AvoidingProjectiles');
			}
		}
	}
}

// ----------------------------------------------------------------------------
// GenerateTotalHealth()
//
// this will calculate a weighted average of all of the body parts
// and put that value in the generic Health
// NOTE: head and torso are both critical
// ----------------------------------------------------------------------------

function GenerateTotalHealth()
{
	local float limbDamage, headDamage, torsoDamage;

	if (!bInvincible)
	{
		// Scoring works as follows:
		// Disabling the head (100 points damage) will kill you.
		// Disabling the torso (100 points damage) will kill you.
		// Disabling 2 1/2 limbs (250 points damage) will kill you.
		// Combinations can also do you in -- 50 points damage to the head
		// and 125 points damage to the limbs, for example.

		// Note that this formula can produce numbers less than zero, so we'll clamp our
		// health value...

		// Compute total limb damage
		limbDamage  = 0;
		if (Default.HealthLegLeft > 0)
			limbDamage += float(Default.HealthLegLeft-HealthLegLeft)/Default.HealthLegLeft;
		if (Default.HealthLegRight > 0)
			limbDamage += float(Default.HealthLegRight-HealthLegRight)/Default.HealthLegRight;
		if (Default.HealthArmLeft > 0)
			limbDamage += float(Default.HealthArmLeft-HealthArmLeft)/Default.HealthArmLeft;
		if (Default.HealthArmRight > 0)
			limbDamage += float(Default.HealthArmRight-HealthArmRight)/Default.HealthArmRight;
		limbDamage *= 0.4;  // 2 1/2 limbs disabled == death

		// Compute total head damage
		headDamage  = 0;
		if (Default.HealthHead > 0)
			headDamage  = float(Default.HealthHead-HealthHead)/Default.HealthHead;

		// Compute total torso damage
		torsoDamage = 0;
		if (Default.HealthTorso > 0)
			torsoDamage = float(Default.HealthTorso-HealthTorso)/Default.HealthTorso;

		// Compute total health, relative to original health level
		Health = FClamp(Default.Health - ((limbDamage+headDamage+torsoDamage)*Default.Health), 0.0, Default.Health);
	}
	else
	{
		// Pawn is invincible - reset health to defaults
		HealthHead     = Default.HealthHead;
		HealthTorso    = Default.HealthTorso;
		HealthArmLeft  = Default.HealthArmLeft;
		HealthArmRight = Default.HealthArmRight;
		HealthLegLeft  = Default.HealthLegLeft;
		HealthLegRight = Default.HealthLegRight;
		Health         = Default.Health;
	}
}


// ----------------------------------------------------------------------------
// PutInWorld()
// ----------------------------------------------------------------------------

function PutInWorld(bool bEnter)
{
	//Log( Self $ ".PutInWorld( " $ bEnter $ " )" );
	//Log( "bInWorld = " $ bInWorld );

	if (bInWorld && !bEnter)
	{
		bInWorld            = false;
		GotoState('Idle');
		bHidden             = true;
		bDetectable         = false;
		WorldPosition       = Location;
		bWorldCollideActors = bCollideActors;
		bWorldBlockActors   = bBlockActors;
		bWorldBlockPlayers  = bBlockPlayers;
		SetPhysics(PHYS_None);
		SetCollision(false, false, false);
		bCollideWorld       = false;
		KillShadow();

		// nice way to hide/show shadow on client
		//RemoteRole = ROLE_None;
		SetLocation(Location+vect(0,0,20000));  // move it out of the way

		ClientSimulatedAction = '';
	}
	else if (!bInWorld && bEnter)
	{
		bInWorld    = true;
		bHidden     = Default.bHidden;
		bDetectable = Default.bDetectable;
		SetLocation(WorldPosition);
		SetCollision(bWorldCollideActors, bWorldBlockActors, bWorldBlockPlayers);
		bCollideWorld = Default.bCollideWorld;
		SetMovementPhysics();
		CreateShadow();

		// nice way to hide/show shadow on client
		//RemoteRole = ROLE_SimulatedProxy;

		//Log( "Orders = " $ Orders );
		//Log( "OrderTag = " $ OrderTag );
		//Log( "HXSeatActor = " $ HXSeatActor );
		//Log( "HXSeatHack = " $ HXSeatHack );
		//Log( "bSeatHackUsed = " $ bSeatHackUsed );

		//Log( GetStateName() );

		bSeatHackUsed = False;

		FollowOrders();

		//Log( GetStateName() );
	}
}


// ----------------------------------------------------------------------
// MakePawnIgnored()
// ----------------------------------------------------------------------

function MakePawnIgnored(bool bNewIgnore)
{
	if (bNewIgnore)
	{
		bIgnore = bNewIgnore;
		// to restore original behavior, uncomment the next line
		//bDetectable = !bNewIgnore;
	}
	else
	{
		bIgnore = Default.bIgnore;
		// to restore original behavior, uncomment the next line
		//bDetectable = Default.bDetectable;
	}

}


// ----------------------------------------------------------------------
// EnableCollision() [for sitting state]
// ----------------------------------------------------------------------

function EnableCollision(bool bSet)
{
	EnableShadow(bSet);

	if (bSet)
		SetCollision(Default.bCollideActors, Default.bBlockActors, Default.bBlockPlayers);
	else
		SetCollision(True, False, True);
}


// ----------------------------------------------------------------------
// SetBasedPawnSize()
// ----------------------------------------------------------------------

function bool SetBasedPawnSize(float newRadius, float newHeight)
{
	local float  oldRadius, oldHeight;
	local bool   bSuccess;
	local vector centerDelta;
	local float  deltaEyeHeight;

	if (newRadius < 0)
		newRadius = 0;
	if (newHeight < 0)
		newHeight = 0;

	oldRadius = CollisionRadius;
	oldHeight = CollisionHeight;

	if ((oldRadius == newRadius) && (oldHeight == newHeight))
		return true;

	centerDelta    = vect(0, 0, 1)*(newHeight-oldHeight);
	deltaEyeHeight = GetDefaultCollisionHeight() - Default.BaseEyeHeight;

	bSuccess = false;
	if ((newHeight <= CollisionHeight) && (newRadius <= CollisionRadius))  // shrink
	{
		SetCollisionSize(newRadius, newHeight);
		if (Move(centerDelta))
			bSuccess = true;
		else
			SetCollisionSize(oldRadius, oldHeight);
	}
	else
	{
		if (Move(centerDelta))
		{
			SetCollisionSize(newRadius, newHeight);
			bSuccess = true;
		}
	}

	if (bSuccess)
	{
		PrePivotOffset  = vect(0, 0, 1)*(GetDefaultCollisionHeight()-newHeight);
		PrePivot        -= centerDelta;
		DesiredPrePivot -= centerDelta;
		BaseEyeHeight   = newHeight - deltaEyeHeight;
	}

	return (bSuccess);
}


// ----------------------------------------------------------------------
// ResetBasedPawnSize()
// ----------------------------------------------------------------------

function ResetBasedPawnSize()
{
	SetBasedPawnSize(Default.CollisionRadius, GetDefaultCollisionHeight());
}


// ----------------------------------------------------------------------
// GetDefaultCollisionHeight()
// ----------------------------------------------------------------------

function float GetDefaultCollisionHeight()
{
	return default.CollisionHeight;
}

// ----------------------------------------------------------------------
// GetCrouchHeight()
// ----------------------------------------------------------------------

function float GetCrouchHeight()
{
	// Original code for FakeShrinking.
	//return default.CollisionHeight*0.65;

	// This would reproduce the original values, but this wouldn't scale properly with different DrawScales.
	//return (default.CollisionHeight+4.50)*0.65;

	// Match above result at CollisionHeight==43.0 (typical male mesh) but scale correctly.
	return default.CollisionHeight*0.718023;
}

// ----------------------------------------------------------------------
// GetSitHeight()
// ----------------------------------------------------------------------

function float GetSitHeight()
{
	return (GetDefaultCollisionHeight()+(BaseAssHeight*0.5));
}


// ----------------------------------------------------------------------------
// GetFloorMaterial()
//
// Gets the name of the texture group that we are standing on
// ----------------------------------------------------------------------------

function name GetFloorMaterial()
{
	local vector EndTrace, HitLocation, HitNormal;
	local actor target;
	local int texFlags;
	local name texName, texGroup;
	local Texture Tex;

	// trace down to our feet
	EndTrace = Location - CollisionHeight * 2 * vect(0,0,1);

	foreach TraceTextures(class'Actor', target, Tex, texName, texGroup, texFlags, HitLocation, HitNormal, EndTrace)
	{
		if ( target == Level )
			break;
	}

	return texGroup;
}

// ----------------------------------------------------------------------------
// PlayFootStep()
//
// Plays footstep sounds based on the texture group
// (yes, I know this looks nasty -- I'll have to figure out a cleaner way to do this)
// ----------------------------------------------------------------------------
/*
function PlayFootStep()
{
	local Sound stepSound;
	local float rnd;
	local name mat;
	local float speedFactor, massFactor;
	local float volume, pitch, range;
	local float radius, maxRadius;
	local float volumeMultiplier;
	local float randomPlaybackVolumeMultiplier;

	local Pawn P;
	local float shakeRadius, shakeMagnitude;
	local float playerDist;

	rnd = FRand();
	mat = GetFloorMaterial();

	volumeMultiplier = 1.0;
	if (WalkSound == None)
	{
		if (FootRegion.Zone.bWaterZone)
		{
			if (rnd < 0.33)
				stepSound = Sound'WaterStep1';
			else if (rnd < 0.66)
				stepSound = Sound'WaterStep2';
			else
				stepSound = Sound'WaterStep3';
		}
		else
		{
			switch(mat)
			{
				case 'Textile':
				case 'Paper':
					volumeMultiplier = 0.7;
					if (rnd < 0.25)
						stepSound = Sound'CarpetStep1';
					else if (rnd < 0.5)
						stepSound = Sound'CarpetStep2';
					else if (rnd < 0.75)
						stepSound = Sound'CarpetStep3';
					else
						stepSound = Sound'CarpetStep4';
					break;

				case 'Foliage':
				case 'Earth':
					volumeMultiplier = 0.6;
					if (rnd < 0.25)
						stepSound = Sound'GrassStep1';
					else if (rnd < 0.5)
						stepSound = Sound'GrassStep2';
					else if (rnd < 0.75)
						stepSound = Sound'GrassStep3';
					else
						stepSound = Sound'GrassStep4';
					break;

				case 'Metal':
				case 'Ladder':
					volumeMultiplier = 1.0;
					if (rnd < 0.25)
						stepSound = Sound'MetalStep1';
					else if (rnd < 0.5)
						stepSound = Sound'MetalStep2';
					else if (rnd < 0.75)
						stepSound = Sound'MetalStep3';
					else
						stepSound = Sound'MetalStep4';
					break;

				case 'Ceramic':
				case 'Glass':
				case 'Tiles':
					volumeMultiplier = 0.7;
					if (rnd < 0.25)
						stepSound = Sound'TileStep1';
					else if (rnd < 0.5)
						stepSound = Sound'TileStep2';
					else if (rnd < 0.75)
						stepSound = Sound'TileStep3';
					else
						stepSound = Sound'TileStep4';
					break;

				case 'Wood':
					volumeMultiplier = 0.7;
					if (rnd < 0.25)
						stepSound = Sound'WoodStep1';
					else if (rnd < 0.5)
						stepSound = Sound'WoodStep2';
					else if (rnd < 0.75)
						stepSound = Sound'WoodStep3';
					else
						stepSound = Sound'WoodStep4';
					break;

				case 'Brick':
				case 'Concrete':
				case 'Stone':
				case 'Stucco':
				default:
					volumeMultiplier = 0.7;
					if (rnd < 0.25)
						stepSound = Sound'StoneStep1';
					else if (rnd < 0.5)
						stepSound = Sound'StoneStep2';
					else if (rnd < 0.75)
						stepSound = Sound'StoneStep3';
					else
						stepSound = Sound'StoneStep4';
					break;
			}
		}
	}
	else
		stepSound = WalkSound;

	// compute sound volume, range and pitch, based on mass and speed
	speedFactor = VSize(Velocity)/120.0;
	massFactor  = Mass/150.0;
	radius      = 768.0;
	maxRadius   = 2048.0;
//	volume      = (speedFactor+0.2)*massFactor;
//	volume      = (speedFactor+0.7)*massFactor;
	volume      = massFactor*1.5;
	range       = radius * volume;
	pitch       = (volume+0.5);
	volume      = 1.0;
	range       = FClamp(range, 0.01, maxRadius);
	pitch       = FClamp(pitch, 1.0, 1.5);

	// play the sound and send an AI event
	// HX: slightly vary playback volume
	randomPlaybackVolumeMultiplier = 0.9 + 0.2 * FRand();
	PlaySound(stepSound, SLOT_Interact, volume*randomPlaybackVolumeMultiplier, , range, pitch);
	AISendEvent('LoudNoise', EAITYPE_Audio, volume*volumeMultiplier, range*volumeMultiplier);

	// Shake the camera when heavy things tread
	if (Mass > 400)
	{
		// HX: shake through player list
		for( P = Level.PawnList; P != None; P = P.nextPawn )
		{
			if( P.bIsPlayer )
			{
				playerDist = VSize( P.Location - Location );
				shakeRadius = FClamp((Mass-400)/600, 0, 1.0) * (range*0.5);
				shakeMagnitude = FClamp((Mass-400)/1600, 0, 1.0);
				shakeMagnitude = FClamp(1.0-(playerDist/shakeRadius), 0, 1.0) * shakeMagnitude;
				if (shakeMagnitude > 0)
					HXPlayerPawn(P).ClientJoltView(shakeMagnitude);
			}
		}
	}
}
*/

// ----------------------------------------------------------------------------
// PlayFootStep()
// ----------------------------------------------------------------------------

/*simulated */function PlayFootStep()
{
	local Sound StepSound;
	local float SpeedFactor, MassFactor;
	local float Volume, AIVolumeMul;
	local float Pitch, Range, Radius, MaxRadius;
	local float ShakeRadius, ShakeMagnitude, PlayerDist;
	local Pawn Pawn;

	StepSound = FootStepSound( Volume, AIVolumeMul );

	// compute sound volume, range and pitch, based on mass and speed
	if ( Physics==PHYS_Swimming )
		SpeedFactor = WaterSpeed/120.0;
	else
		SpeedFactor = VSize(Velocity)/120.0;

	MassFactor  = Mass/150.0;
	Radius      = 768.0;
	MaxRadius   = 2048.0;
	//Volume      = (SpeedFactor+0.2)*MassFactor;
	//Volume      = (SpeedFactor+0.7)*MassFactor;
	Volume      = MassFactor * 1.5;
	Range       = Radius * Volume;
	Pitch       = Volume + 0.5;
	Volume      = 1.0;
	Range       = FClamp( Range, 0.01, MaxRadius );
	Pitch       = FClamp( Pitch, 1.0, 1.5 );

	// Play the sound and send an AI event.
	// HX: slightly vary playback volume.
	PlaySound( StepSound, SLOT_Interact, Volume*(0.9 + 0.2 * FRand()), , Range, Pitch );
	AISendEvent( 'LoudNoise', EAITYPE_Audio, Volume*AIVolumeMul, Range*AIVolumeMul );

	// Shake the camera when heavy things tread
	if ( Mass>400 )
	{
		// HX: shake through player list.
		for ( Pawn=Level.PawnList; Pawn!=None; Pawn=Pawn.NextPawn )
		{
			if ( Pawn.bIsPlayer )
			{
				PlayerDist     = VSize(Pawn.Location-Location);
				ShakeRadius    = FClamp((Mass-400)/600, 0, 1.0) * (Range*0.5);
				ShakeMagnitude = FClamp((Mass-400)/1600, 0, 1.0);
				ShakeMagnitude = FClamp(1.0-(PlayerDist/ShakeRadius),0.0,1.0) * ShakeMagnitude;
				if ( ShakeMagnitude>0 )
					HXPlayerPawn(Pawn).ClientJoltView( ShakeMagnitude );
			}
		}
	}
}

// ----------------------------------------------------------------------------
// FootStepSound()
//
// Tries to figure out which footstep sound to use.
//
// Potential Additional Footstep Sounds:
//	* MoverSFX.StallDoorClose
//  * DeusExSounds.KarkianFootstep
//  * DeusExSounds.GreaselFootstep
//  * DeusExSounds.GrayFootstep
//
// Note:
//  * Maybe add FootSteps for Fragment.
// ----------------------------------------------------------------------------

simulated function Sound FootStepSound( out float Volume, out float AIVolumeMul )
{
	local Vector       End, HitLoc, HitNorm;
	local Actor        TraceTarget;
	local int          TexFlags;
	local Name         TexName, TexGroup;
	local Texture      Tex;
	//local Mover        Mover;
	local Decoration   Decoration;
	local Carcass      Carcass;
	local float        Rnd;

	Volume	    = 1.0;
	AIVolumeMul = 1.0;

	// Handle Water
	if ( Physics==PHYS_Swimming )
		return SwimmingStepSound( Volume, AIVolumeMul );
	if ( FootRegion.Zone.bWaterZone )
		return WaterStepSound( Volume, AIVolumeMul );

	// Prefer WalkSound over any texture speficic sound.
	if ( WalkSound!=None )
		return WalkSound;

	// trace down to our feet
	End = Location - CollisionHeight * 2 * vect(0,0,1);

	foreach TraceTextures(Class'Actor', TraceTarget, Tex, TexName, TexGroup, TexFlags, HitLoc, HitNorm, End )
	{
		// Vanilla behaviour for Level Geometry.
		//if ( TraceTarget==Level )
		if ( TraceTarget==Level || Mover(TraceTarget)!=None )
		{
			// ZonePortals, etc.
			if ( Texture==None && TexFlags==0 )
				continue;
			return TextureGroupStepSound( Volume, AIVolumeMul, Tex, TexGroup );
		}

		// Carcass.
		Carcass = Carcass(TraceTarget);
		if ( Carcass!=None ) 
			return CarcassStepSound( Volume, AIVolumeMul, Carcass );

		// Choose Decoration FootStepSound based on FragType.
		Decoration = Decoration(TraceTarget);
		if ( Decoration!=None )
			return DecorationStepSound( Volume, AIVolumeMul, Decoration );

		// Not yet supported.
		//Mover = Mover(TraceTarget);
		//if ( Mover != None )
			//;
	}
	return FallbackStepSound( Volume, AIVolumeMul );
}

// ----------------------------------------------------------------------------
// TextureGroupStepSound()
//
// Used for Bsp based footsteps.
// ----------------------------------------------------------------------------

simulated function Sound TextureGroupStepSound( out float Volume, out float AIVolumeMul, Texture Texture, Name TextureGroup )
{
	local float Rnd;

	// Set Volume and VolumeMul first.
	switch ( TextureGroup )
	{
		case 'Metal':
		case 'Ladder':
			AIVolumeMul = 1.0;
			break;
		case 'Foliage':
		case 'Earth':
			AIVolumeMul = 0.6;
			break;
		default:
			AIVolumeMul = 0.7;
			break;
	}

	// Use Texture.FootstepSound if available.
	if ( Texture!=None && Texture.FootstepSound!=None )
		return Texture.FootstepSound;

	Rnd = FRand();
	switch ( TextureGroup )
	{
		case 'Textile':
		case 'Paper':
			Volume *= 0.8;
		case 'Plastic':
			AIVolumeMul = 0.4;
			if (Rnd < 0.25) return Sound'CarpetStep1';
			if (Rnd < 0.5)  return Sound'CarpetStep2';
			if (Rnd < 0.75) return Sound'CarpetStep3';
			else            return Sound'CarpetStep4';
			break;

		case 'Foliage':
		case 'Earth':
			AIVolumeMul = 0.55;
			if (Rnd < 0.25) return Sound'GrassStep1';
			if (Rnd < 0.5)  return Sound'GrassStep2';
			if (Rnd < 0.75) return Sound'GrassStep3';
			else            return Sound'GrassStep4';
			break;

		case 'Metal':
		case 'Ladder':
			AIVolumeMul = 1.0;
			if (Rnd < 0.25) return Sound'MetalStep1';
			if (Rnd < 0.5)  return Sound'MetalStep2';
			if (Rnd < 0.75) return Sound'MetalStep3';
			else            return Sound'MetalStep4';
			break;

		case 'Ceramic':
		case 'Glass':
		case 'Tiles':
			AIVolumeMul = 0.7;
			if (Rnd < 0.25) return Sound'TileStep1';
			if (Rnd < 0.5)  return Sound'TileStep2';
			if (Rnd < 0.75) return Sound'TileStep3';
			else            return Sound'TileStep4';
			break;

		case 'Wood':
			AIVolumeMul = 0.7;
			if (Rnd < 0.25) return Sound'WoodStep1';
			if (Rnd < 0.5)  return Sound'WoodStep2';
			if (Rnd < 0.75) return Sound'WoodStep3';
			else            return Sound'WoodStep4';
			break;

		case 'Brick':
		case 'Concrete':
		case 'Stone':
		case 'Stucco':
		default:
			AIVolumeMul = 0.7;
			if (Rnd < 0.25) return Sound'StoneStep1';
			if (Rnd < 0.5)  return Sound'StoneStep2';
			if (Rnd < 0.75) return Sound'StoneStep3';
			else            return Sound'StoneStep4';
			break;
	}
}

// ----------------------------------------------------------------------------
// CarcassStepSound()
//
// StepSound for walking over carcass.
// ----------------------------------------------------------------------------

simulated function Sound CarcassStepSound( out float Volume, out float AIVolumeMul, Carcass Carcass )
{
	AIVolumeMul = 0.5;
	Volume      = 0.5;
	return Sound'KarkianFootstep';
}

// ----------------------------------------------------------------------------
// DecorationStepSound()
//
// StepSound for walking over carcass.
// ----------------------------------------------------------------------------

simulated function Sound DecorationStepSound( out float Volume, out float AIVolumeMul, Decoration Decoration )
{
	local DeusExDecoration DeusExDecoration;
	local HXDecoration HXDecoration;
	local Class<Fragment> FragmentClass;

	HXDecoration = HXDecoration(Decoration);
	if ( HXDecoration!=None )
	{
		FragmentClass = HXDecoration.FragType;
	}
	else
	{
		DeusExDecoration = DeusExDecoration(Decoration);
		if ( DeusExDecoration!=None )
			FragmentClass = DeusExDecoration.FragType;
	} 

	if ( FragmentClass==None )
	{
		Log( Decoration@"has empty FragType. Report this as a Bug." );
		//return Sound'TouchTone11';
		return FallbackStepSound( Volume, AIVolumeMul );
	}

	switch ( FragmentClass.Name ) 
	{
		case 'GlassFragment':		return TextureGroupStepSound( Volume, AIVolumeMul, None, 'Glass' );
		case 'MetalFragment':		return TextureGroupStepSound( Volume, AIVolumeMul, None, 'Metal' );
		case 'PaperFragment':
			AIVolumeMul = 0.3;
			Volume      = 0.8;
			return Sound'StallDoorClose';
		case 'PlasticFragment':	return TextureGroupStepSound( Volume, AIVolumeMul, None, 'Textile' );
		case 'WoodFragment':		return TextureGroupStepSound( Volume, AIVolumeMul, None, 'Wood' );
		case 'Rockchip':				return TextureGroupStepSound( Volume, AIVolumeMul, None, 'Concrete' );
		case 'FleshFragment':
			AIVolumeMul = 0.5;
			Volume      = 0.5;
			return Sound'KarkianFootstep';
		default:
			Log( "Unhandled FragmentClass="$FragmentClass$" in GetFootStepGroup() Report this as a Bug." );
			//return Sound'TouchTone5';
			return FallbackStepSound( Volume, AIVolumeMul );
	}
}

// ----------------------------------------------------------------------------
// FallbackStepSound()
// ----------------------------------------------------------------------------

simulated function Sound FallbackStepSound( out float Volume, out float AIVolumeMul )
{
	return TextureGroupStepSound( Volume, AIVolumeMul, None, 'Brick' );
}

// ----------------------------------------------------------------------------
// SwimmingStepSound()
// ----------------------------------------------------------------------------

simulated function Sound SwimmingStepSound( out float Volume, out float AIVolumeMul )
{
	AIVolumeMul = 0.5;
	if ( FRand()<0.5 )
		return Sound'Swimming';
	else
		return Sound'Treading';
}

// ----------------------------------------------------------------------------
// WaterStepSound()
// ----------------------------------------------------------------------------

simulated function Sound WaterStepSound( out float Volume, out float AIVolumeMul )
{
	local float Rnd;
	AIVolumeMul = 1.0;
	Rnd = FRand();
	if ( Rnd<0.33 )
		return Sound'WaterStep1';
	if ( Rnd<0.66 ) 
		return Sound'WaterStep2';
	else
		return Sound'WaterStep3';
}

// ----------------------------------------------------------------------------
// HandleAlarm()
// ----------------------------------------------------------------------------

function HandleAlarm(Name event, EAIEventState state, XAIParams params)
{
	// React, Fear

	local HXAlarmUnit      alarm;
	local HXLaserTrigger   laser;
	local HXSecurityCamera camera;
	local HXComputers      computer;
	local Pawn           alarmInstigator;
	local vector         alarmLocation;

	if (state == EAISTATE_Begin || state == EAISTATE_Pulse)
	{
		alarmInstigator = None;
		alarm    = HXAlarmUnit(params.bestActor);
		laser    = HXLaserTrigger(params.bestActor);
		camera   = HXSecurityCamera(params.bestActor);
		computer = HXComputers(params.bestActor);
		if (alarm != None)
		{
			alarmInstigator = alarm.alarmInstigator;
			alarmLocation   = alarm.alarmLocation;
		}
		else if (laser != None)
		{
			alarmInstigator = Pawn(laser.triggerActor);
			if (alarmInstigator == None)
				alarmInstigator = laser.triggerActor.Instigator;
			alarmLocation   = laser.actorLocation;
		}
		else if (camera != None)
		{
			//alarmInstigator = GetPlayerPawn();  // player is implicit for cameras
			//Log( "HXScriptedPawn.HandleAlarm() get rid of GetPlayerPawn()" );
			alarmInstigator = camera.alarmInstigator;
			alarmLocation   = camera.playerLocation;
		}
		else if (computer != None)
		{
			//alarmInstigator = GetPlayerPawn();  // player is implicit for computers
			//Log( "HXScriptedPawn.HandleAlarm() get rid of GetPlayerPawn()" );
			alarmInstigator = computer.alarmInstigator;
			alarmLocation   = computer.Location;
		}

		if (bFearAlarm)
		{
			IncreaseFear(alarmInstigator, 2.0);
			if (IsFearful())
			{
				SetDistressTimer();
				SetEnemy(alarmInstigator, , true);
				GotoState('Fleeing');
			}
		}

		if (alarmInstigator != None)
		{
			if (alarmInstigator.Health > 0)
			{
				if (IsValidEnemy(alarmInstigator))
				{
					AlarmTimer = 120;
					SetDistressTimer();
					SetSeekLocation(alarmInstigator, alarmLocation, SEEKTYPE_Sound);
					HandleEnemy();
				}
			}
		}
	}
}

// ----------------------------------------------------------------------------
// PlayDyingSound()
// ----------------------------------------------------------------------------

function PlayDyingSound()
{
	SetDistressTimer();
	PlaySound(Die, SLOT_Pain,,,, RandomPitch());
	AISendEvent('LoudNoise', EAITYPE_Audio);
	if (bEmitDistress)
		AISendEvent('Distress', EAITYPE_Audio);
}

// ----------------------------------------------------------------------------
// PlayIdleSound()
// ----------------------------------------------------------------------------

function PlayIdleSound()
{
	if ( bHasIdleBark )
		HXGameInfo(Level.Game).StartAIBarkConversation( Self, BM_Idle );
}

// ----------------------------------------------------------------------------
// PlayScanningSound()
// ----------------------------------------------------------------------------

function PlayScanningSound()
{
	if ( bHasScanningBark )
		HXGameInfo(Level.Game).StartAIBarkConversation( Self, BM_Scanning );
}

// ----------------------------------------------------------------------------
// PlayPreAttackSearchingSound()
// ----------------------------------------------------------------------------

function PlayPreAttackSearchingSound()
{
	if ( bHasPreAttackSearchingBark && SeekPawn!=None && SeekPawn.bIsPlayer )
		HXGameInfo(Level.Game).StartAIBarkConversation( Self, BM_PreAttackSearching );
}

// ----------------------------------------------------------------------------
// PlayPreAttackSightingSound()
// ----------------------------------------------------------------------------

function PlayPreAttackSightingSound()
{
	if ( bHasPreAttackSightingBark && SeekPawn!=None && SeekPawn.bIsPlayer )
		HXGameInfo(Level.Game).StartAIBarkConversation( Self, BM_PreAttackSighting );
}

// ----------------------------------------------------------------------------
// PlayPostAttackSearchingSound()
// ----------------------------------------------------------------------------

function PlayPostAttackSearchingSound()
{
	if ( bHasPostAttackSearchingBark && SeekPawn!=None && SeekPawn.bIsPlayer )
		HXGameInfo(Level.Game).StartAIBarkConversation( Self, BM_PostAttackSearching );
}

// ----------------------------------------------------------------------------
// PlayTargetAcquiredSound()
// ----------------------------------------------------------------------------

function PlayTargetAcquiredSound()
{
	if ( bHasTargetAcquiredBark && Enemy!=None && Enemy.bIsPlayer )
		HXGameInfo(Level.Game).StartAIBarkConversation( Self, BM_TargetAcquired );
}

// ----------------------------------------------------------------------------
// PlayTargetLostSound()
// ----------------------------------------------------------------------------

function PlayTargetLostSound()
{
	if ( bHasTargetLostBark && SeekPawn!=None && SeekPawn.bIsPlayer )
		HXGameInfo(Level.Game).StartAIBarkConversation( Self, BM_TargetLost );
}

// ----------------------------------------------------------------------------
// PlaySearchGiveUpSound()
// ----------------------------------------------------------------------------

function PlaySearchGiveUpSound()
{
	if ( bHasSearchGiveUpBark && SeekPawn!=None && SeekPawn.bIsPlayer )
		HXGameInfo(Level.Game).StartAIBarkConversation( Self, BM_SearchGiveUp );
}

// ----------------------------------------------------------------------------
// PlayNewTargetSound()
// ----------------------------------------------------------------------------

function PlayNewTargetSound()
{
	// someday...
}

// ----------------------------------------------------------------------------
// PlayGoingForAlarmSound()
// ----------------------------------------------------------------------------

function PlayGoingForAlarmSound()
{
	if ( bHasGoingForAlarmBark && Enemy!=None && Enemy.bIsPlayer )
		HXGameInfo(Level.Game).StartAIBarkConversation( Self, BM_GoingForAlarm );
}

// ----------------------------------------------------------------------------
// PlayOutOfAmmoSound()
// ----------------------------------------------------------------------------

function PlayOutOfAmmoSound()
{
	if ( bHasOutOfAmmoBark )
		HXGameInfo(Level.Game).StartAIBarkConversation( Self, BM_OutOfAmmo );
}

// ----------------------------------------------------------------------------
// PlayCriticalDamageSound()
// ----------------------------------------------------------------------------

function PlayCriticalDamageSound()
{
	if ( bHasCriticalDamageBark && Enemy!=None && Enemy.bIsPlayer )
		HXGameInfo(Level.Game).StartAIBarkConversation( Self, BM_CriticalDamage );
}

// ----------------------------------------------------------------------------
// PlayAreaSecureSound()
// ----------------------------------------------------------------------------

function PlayAreaSecureSound()
{
	if ( bHasAreaSecureBark && Enemy!=None && Enemy.bIsPlayer )
		HXGameInfo(Level.Game).StartAIBarkConversation( Self, BM_AreaSecure );
}

// ----------------------------------------------------------------------------
// PlayFutzSound()
// ----------------------------------------------------------------------------

function PlayFutzSound()
{
	if ( bHasFutzBark )
		HXGameInfo(Level.Game).StartAIBarkConversation( Self, BM_Futz );
}

// ----------------------------------------------------------------------------
// PlayOnFireSound()
// ----------------------------------------------------------------------------

function PlayOnFireSound()
{
	if ( bHasOnFireBark )
		HXGameInfo(Level.Game).StartAIBarkConversation( Self, BM_OnFire );
}

// ----------------------------------------------------------------------------
// PlayTearGasSound()
// ----------------------------------------------------------------------------

function PlayTearGasSound()
{
	if ( bHasTearGasBark )
		HXGameInfo(Level.Game).StartAIBarkConversation( Self, BM_TearGas );
}

// ----------------------------------------------------------------------------
// PlayCarcassSound()
// ----------------------------------------------------------------------------

function PlayCarcassSound()
{
	if ( bHasGoreBark && SeekPawn!=None && SeekPawn.bIsPlayer )
		HXGameInfo(Level.Game).StartAIBarkConversation( Self, BM_Gore );
}

// ----------------------------------------------------------------------------
// PlaySurpriseSound()
// ----------------------------------------------------------------------------

function PlaySurpriseSound()
{
	if ( bHasSurpriseBark && Enemy!=None && Enemy.bIsPlayer )
		HXGameInfo(Level.Game).StartAIBarkConversation( Self, BM_Surprise );
}

// ----------------------------------------------------------------------------
// PlayAllianceHostileSound()
// ----------------------------------------------------------------------------

function PlayAllianceHostileSound()
{
	if ( bHasAllianceHostileBark && Enemy!=None && Enemy.bIsPlayer )
		HXGameInfo(Level.Game).StartAIBarkConversation( Self, BM_AllianceHostile );
}

// ----------------------------------------------------------------------------
// PlayAllianceFriendlySound()
// ----------------------------------------------------------------------------

function PlayAllianceFriendlySound()
{
	if ( bHasAllianceFriendlyBark && Enemy!=None && Enemy.bIsPlayer )
		HXGameInfo(Level.Game).StartAIBarkConversation( Self, BM_AllianceFriendly );
}

// ----------------------------------------------------------------------------
// IsDoor()
// ----------------------------------------------------------------------------

function bool IsDoor(Actor door, optional bool bWarn)
{
	local bool        bIsDoor;
	local HXMover dxMover;

	bIsDoor = false;

	dxMover = HXMover(door);
	if (dxMover != None)
	{
		if (dxMover.NumKeys > 1)
		{
			if (dxMover.bIsDoor)
				bIsDoor = true;
			/*
			else if (bWarn)  // hack for now
				log("WARNING: NPC "$self$" trying to use door "$dxMover$", but bIsDoor flag is False");
			*/
		}
	}

	return bIsDoor;
}


// ----------------------------------------------------------------------------
// CheckOpenDoor()
// ----------------------------------------------------------------------------

function CheckOpenDoor(vector HitNormal, actor Door, optional name Label)
{
	local HXMover dxMover;

	dxMover = HXMover(Door);
	if (dxMover != None)
	{
		if (bCanOpenDoors && !IsDoor(dxMover) && dxMover.bBreakable)  // break glass we walk into
		{
			dxMover.TakeDamage(200, self, dxMover.Location, Velocity, 'Shot');
			return;
		}

		if (dxMover.bInterpolating && (dxMover.MoverEncroachType == ME_IgnoreWhenEncroach))
			return;

		if (bCanOpenDoors && bInterruptState && !bInTransientState && IsDoor(dxMover, true))
		{
			if (Label == '')
				Label = 'Begin';
			if (GetStateName() != 'OpeningDoor')
				SetNextState(GetStateName(), 'ContinueFromDoor');
			Target = Door;
			destLoc = HitNormal;
			GotoState('OpeningDoor', 'BeginHitNormal');
		}
		else if ((Acceleration != vect(0,0,0)) && (Physics == PHYS_Walking) &&
		         (TurnDirection == TURNING_None))
			Destination = Location;
	}
}


// ----------------------------------------------------------------------------
// EncroachedBy()
// ----------------------------------------------------------------------------

event EncroachedBy( actor Other )
{
	// overridden so indestructable NPCs aren't InstaGibbed by stupid movement code
}


// ----------------------------------------------------------------------------
// EncroachedByMover()
// ----------------------------------------------------------------------------

function EncroachedByMover(Mover encroacher)
{
	local HXMover dxMover;

	dxMover = HXMover(encroacher);
	if (dxMover != None)
		if (!dxMover.bInterpolating && IsDoor(dxMover))
			FrobDoor(dxMover);
}


// ----------------------------------------------------------------------------
// FrobDoor()
// ----------------------------------------------------------------------------

function bool FrobDoor(actor Target)
{
	local HXMover      dxMover;
	local HXMover      triggerMover;
	local DeusExDecoration trigger;
	local float            dist;
	local DeusExDecoration bestTrigger;
	local float            bestDist;
	local bool             bDone;

	bDone = false;

	dxMover = HXMover(Target);
	if (dxMover != None)
	{
		bestTrigger = None;
		bestDist    = 10000;
		foreach AllActors(Class'DeusExDecoration', trigger)
		{
			if (dxMover.Tag == trigger.Event)
			{
				dist = VSize(Location - trigger.Location);
				if ((bestTrigger == None) || (bestDist > dist))
				{
					bestTrigger = trigger;
					bestDist    = dist;
				}
			}
		}
		if (bestTrigger != None)
		{
			foreach AllActors(Class'HXMover', triggerMover, dxMover.Tag)
				triggerMover.Trigger(bestTrigger, self);
			bDone = true;
		}
		else if (dxMover.bFrobbable)
		{
			if ((dxMover.WaitingPawn == None) ||
			    (dxMover.WaitingPawn == self))
			{
				dxMover.Frob(self, None);
				bDone = true;
			}
		}

		if (bDone)
			dxMover.WaitingPawn = self;
	}
	return bDone;

}


// ----------------------------------------------------------------------------
// EndConversation()
// ----------------------------------------------------------------------------

function EndConversation()
{
	Super.EndConversation();

	if ((GetStateName() == 'Conversation') || (GetStateName() == 'FirstPersonConversation'))
	{
		bConversationEndedNormally = True;

		if (!bConvEndState)
			FollowOrders();
	}

	bInConversation = False;

	ClientLastConEndTime = Level.TimeSeconds;
}

// ----------------------------------------------------------------------------
// Died()
// ----------------------------------------------------------------------------

function Died(pawn Killer, name damageType, vector HitLocation)
{
	//local DeusExPlayer player;
	local name flagName;

	// Set a flag when NPCs die so we can possibly check it later
	//player = DeusExPlayer(GetPlayerPawn());

	ExtinguishFire();

	// set the instigator to be the killer
	Instigator = Killer;

	//if (player != None)
	//{
		// Abort any conversation we may have been having with the NPC
		if (bInConversation)
			HXGameInfo(Level.Game).AbortConversation();
			//player.AbortConversation();
	//}

	// Abort any barks we may have been playing
	if (HXGameInfo(Level.Game).BarkManager != None)
		HXGameInfo(Level.Game).BarkManager.ScriptedPawnDied(Self);

	Super(Pawn).Died(Killer, damageType, HitLocation);  // bStunned is set here

	//if (player != None)
	//{
		if (bImportant)
		{
			flagName = StringToName(BindName$"_Dead");
			HXGameInfo(Level.Game).Steve.FlagBase.SetBool(flagName, True);
			HXGameInfo(Level.Game).FlagReplicationInfo.SetBoolFlag( flagname, int(True) );

			// make sure the flag never expires
			HXGameInfo(Level.Game).Steve.FlagBase.SetExpiration(flagName, FLAG_Bool, 0);

			if (bStunned)
			{
				flagName = StringToName(BindName$"_Unconscious");
				HXGameInfo(Level.Game).Steve.FlagBase.SetBool(flagName, True);
				HXGameInfo(Level.Game).FlagReplicationInfo.SetBoolFlag( flagname, int(True) );

				// make sure the flag never expires
				HXGameInfo(Level.Game).Steve.FlagBase.SetExpiration(flagName, FLAG_Bool, 0);
			}
		}
	//}
}

// ----------------------------------------------------------------------------
// KillMessage()
// ----------------------------------------------------------------------------

function string KillMessage( name damageType, pawn Other )
{
	local string Message;

	// TODO add localized messages
	switch ( damageType )
	{
		case 'Shot':
		case 'AutoShot':
		case 'Sabot':
			Message = " was shot by ";
			break;

		case 'Burned':
		case 'Flamed':
			Message = " was incinerated by ";

		case 'Exploded':
			Message = " was blown up by ";

		default:
			Message = " was killed by ";
			break;
	}

	return ( Message $ namearticle $ FamiliarName );
}

// ----------------------------------------------------------------------------
// CloseOut()
// ----------------------------------------------------------------------------

function CloseOut()
{
}

// ----------------------------------------------------------------------------
// state OpeningDoor
//
// Open a door.
// ----------------------------------------------------------------------------

state OpeningDoor
{
	ignores EnemyNotVisible;

	function SetFall()
	{
		StartFalling(NextState, NextLabel);
	}

	function HitWall(vector HitNormal, actor Wall)
	{
		if (Physics == PHYS_Falling)
			return;
		Global.HitWall(HitNormal, Wall);
		if (Target == Wall)
			CheckOpenDoor(HitNormal, Wall);
	}

	function bool DoorEncroaches()
	{
		local bool        bEncroaches;
		local HXMover dxMover;

		bEncroaches = true;
		dxMover = HXMover(Target);
		if (dxMover != None)
		{
			if (IsDoor(dxMover) && (dxMover.MoverEncroachType == ME_IgnoreWhenEncroach))
				bEncroaches = false;
		}

		return bEncroaches;
	}

	function FindBackupPoint()
	{
		local vector hitNorm;
		local rotator rot;
		local vector center;
		local vector area;
		local vector relPos;
		local float  distX, distY;
		local float  dist;

		hitNorm = Normal(destLoc);
		rot = Rotator(hitNorm);
		HXMover(Target).ComputeMovementArea(center, area);
		area.X += CollisionRadius + 30;
		area.Y += CollisionRadius + 30;
		//area.Z += CollisionHeight + 30;
		relPos = Location - center;
		if ((relPos.X < area.X) && (relPos.X > -area.X) &&
		    (relPos.Y < area.Y) && (relPos.Y > -area.Y))
		{
			// hack
			if (hitNorm.Y == 0)
				hitNorm.Y = 0.00000001;
			if (hitNorm.X == 0)
				hitNorm.X = 0.00000001;
			if (hitNorm.X > 0)
				distX = (area.X - relPos.X)/hitNorm.X;
			else
				distX = (-area.X - relPos.X)/hitNorm.X;
			if (hitNorm.Y > 0)
				distY = (area.Y - relPos.Y)/hitNorm.Y;
			else
				distY = (-area.Y - relPos.Y)/hitNorm.Y;
			dist = FMin(distX, distY);
			if (dist < 45)
				dist = 45;
			else if (dist > 700)
				dist = 700;  // sanity check
			if (!AIDirectionReachable(Location, rot.Yaw, rot.Pitch, 40, dist, destLoc))
				destLoc = Location;
		}
		else
			destLoc = Location;
	}

	function vector FocusDirection()
	{
		return (Vector(Rotation)*30+Location);
	}

	function StopWaiting()
	{
		GotoState('OpeningDoor', 'DoorOpened');
	}

	function AnimEnd()
	{
		PlayWaiting();
	}

	function BeginState()
	{
		StandUp();
		Disable('AnimEnd');
		bCanJump = false;
		BlockReactions();
		bStasis = False;
		bInTransientState = True;
		EnableCheckDestLoc(false);
	}

	function EndState()
	{
		EnableCheckDestLoc(false);
		bAcceptBump = True;

		if (JumpZ > 0)
			bCanJump = true;

		ResetReactions();
		bStasis = True;
		bInTransientState = false;
	}

Begin:
	destLoc = vect(0,0,0);

BeginHitNormal:
	Acceleration = vect(0,0,0);
	FindBackupPoint();

	if (!DoorEncroaches())
		if (!FrobDoor(Target))
			Goto('DoorOpened');	
	PlayRunning();
	StrafeTo(destLoc, FocusDirection());
	if (DoorEncroaches())
		if (!FrobDoor(Target))
			Goto('DoorOpened');
	PlayWaiting();
	Sleep(5.0);

DoorOpened:
	if (HasNextState())
		GotoNextState();
	else
		FollowOrders();  // THIS IS BAD!!!

}


// ----------------------------------------------------------------------------
// HXAddProjectileToList()
// ----------------------------------------------------------------------------

function HXAddProjectileToList(out HXNearbyProjectileList projList,
                             HXProjectile proj, vector projPos,
                             float dist, float range)
{
	local int   count;
	local int   pos;
	local int   bestPos;
	local float worstDist;

	bestPos   = -1;
	worstDist = dist;
	pos       = 0;
	while (pos < ArrayCount(projList.list))
	{
		if (projList.list[pos].projectile == None)
		{
			bestPos = pos;
			break;  // short-circuit loop
		}
		else
		{
			if (worstDist < projList.list[pos].dist)
			{
				worstDist = projList.list[pos].dist;
				bestPos   = pos;
			}
		}

		pos++;
	}

	if (bestPos >= 0)
	{
		projList.list[bestPos].projectile = proj;
		projList.list[bestPos].location   = projPos;
		projList.list[bestPos].dist       = dist;
		projList.list[bestPos].range      = range;
	}

}

// ----------------------------------------------------------------------------
// HXIsProjectileDangerous()
// ----------------------------------------------------------------------------

function bool HXIsProjectileDangerous(HXProjectile projectile)
{
	local bool bEvil;

	if (projectile.IsA('HXCloud'))
		bEvil = true;
	else if (projectile.IsA('HXThrownProjectile'))
	{
		if (projectile.IsA('HXSpyBot'))
			bEvil = false;
		else if ((HXThrownProjectile(projectile) != None) && (HXThrownProjectile(projectile).bProximityTriggered))
			bEvil = false;
		else
			bEvil = true;
	}
	else
		bEvil = false;

	return (bEvil);

}

// ----------------------------------------------------------------------------
// HXGetProjectileList()
// ----------------------------------------------------------------------------

function int HXGetProjectileList(out HXNearbyProjectileList projList, vector Location)
{
	local float            dist;
	local int              count;
	local HXProjectile curProj;
	local HXThrownProjectile throwProj;
	local HXCloud            cloudProj;
	local vector           HitNormal, HitLocation;
	local vector           extent;
	local vector           traceEnd;
	local Actor            hitActor;
	local float            range;
	local vector           pos;
	local float            time;
	local float            maxTime;
	local float            elasticity;
	local int              i;
	local bool             bValid;

	for (i=0; i<ArrayCount(projList.list); i++)
		projList.list[i].projectile = None;
	projList.center = Location;

	maxTime = 2.0;
	foreach RadiusActors(Class'HXProjectile', curProj, 1000)
	{
		if (HXIsProjectileDangerous(curProj))
		{
			throwProj = HXThrownProjectile(curProj);
			cloudProj = HXCloud(curProj);
			extent   = vect(1,1,0)*curProj.CollisionRadius;
			extent.Z = curProj.CollisionHeight;

			range    = VSize(extent);
			if (curProj.bExplodes)
				if (range < curProj.blastRadius)
					range = curProj.blastRadius;
			if (cloudProj != None)
				if (range < cloudProj.cloudRadius)
					range = cloudProj.cloudRadius;
			range += CollisionRadius+60;

			if (throwProj != None)
				elasticity = throwProj.Elasticity;
			else
				elasticity = 0.2;

			bValid = true;
			if (throwProj != None)
				if (throwProj.bProximityTriggered)  // HACK!!!
					bValid = false;

			if (((curProj.Physics == PHYS_Falling) || (curProj.Physics == PHYS_Projectile) || (curProj.Physics == PHYS_None)) &&
			    bValid)
			{
				pos = curProj.Location;
				dist = VSize(Location - curProj.Location);
				HXAddProjectileToList(projList, curProj, pos, dist, range);
				if (curProj.Physics == PHYS_Projectile)
				{
					traceEnd = curProj.Location + curProj.Velocity*maxTime;
					hitActor = Trace(HitLocation, HitNormal, traceEnd, curProj.Location, true, extent);
					if (hitActor == None)
						pos = traceEnd;
					else
						pos = HitLocation;
					dist = VSize(Location - pos);
					HXAddProjectileToList(projList, curProj, pos, dist, range);
				}
				else if (curProj.Physics == PHYS_Falling)
				{
					time = ParabolicTrace(pos, curProj.Velocity, curProj.Location, true, extent, maxTime,
					                      elasticity, curProj.bBounce, 60);
					if (time > 0)
					{
						dist = VSize(Location - pos);
						HXAddProjectileToList(projList, curProj, pos, dist, range);
					}
				}
			}
		}
	}

	count = 0;
	for (i=0; i<ArrayCount(projList.list); i++)
		if (projList.list[i].projectile != None)
			count++;

	return (count);

}

// ----------------------------------------------------------------------------
// HXIsLocationDangerous()
// ----------------------------------------------------------------------------

function bool HXIsLocationDangerous(HXNearbyProjectileList projList,
                                  vector Location)
{
	local bool  bDanger;
	local int   i;
	local float dist;

	bDanger = false;
	for (i=0; i<ArrayCount(projList.list); i++)
	{
		if (projList.list[i].projectile == None)
			break;
		if (projList.center == Location)
			dist = projList.list[i].dist;
		else
			dist = VSize(projList.list[i].location - Location);
		if (dist < projList.list[i].range)
		{
			bDanger = true;
			break;
		}
	}

	return (bDanger);
}


// ----------------------------------------------------------------------------
// HXComputeAwayVector()
// ----------------------------------------------------------------------------

function vector HXComputeAwayVector(HXNearbyProjectileList projList)
{
	local vector          awayVect;
	local vector          tempVect;
	local rotator         tempRot;
	local int             i;
	local float           dist;
	local int             yaw;
	local int             absYaw;
	local int             bestYaw;
	local NavigationPoint navPoint;
	local NavigationPoint bestPoint;
	local float           segmentDist;

	segmentDist = GroundSpeed*0.5;

	awayVect = vect(0, 0, 0);
	for (i=0; i<ArrayCount(projList.list); i++)
	{
		if ((projList.list[i].projectile != None) &&
		    (projList.list[i].dist < projList.list[i].range))
		{
			tempVect = projList.list[i].location - projList.center;
			if (projList.list[i].dist > 0)
				tempVect /= projList.list[i].dist;
			else
				tempVect = VRand();
			awayVect -= tempVect;
		}
	}

	if (awayVect != vect(0, 0, 0))
	{
		awayVect += Normal(Velocity*vect(1,1,0))*0.9;  // bias to direction already running
		yaw = Rotator(awayVect).Yaw;
		bestPoint = None;
		foreach ReachablePathnodes(Class'NavigationPoint', navPoint, None, dist)
		{
			tempRot = Rotator(navPoint.Location - Location);
			absYaw = (tempRot.Yaw - Yaw) & 65535;
			if (absYaw > 32767)
				absYaw -= 65536;
			absYaw = Abs(absYaw);
			if ((bestPoint == None) || (bestYaw > absYaw))
			{
				bestPoint = navPoint;
				bestYaw = absYaw;
			}
		}
		if (bestPoint != None)
			awayVect = bestPoint.Location-Location;
		else
		{
			tempRot = Rotator(awayVect);
			tempRot.Pitch += Rand(7282)-3641;   // +/- 20 degrees
			tempRot.Yaw   += Rand(7282)-3641;   // +/- 20 degrees
			awayVect = Vector(tempRot)*segmentDist;
		}
	}
	else
		awayVect = VRand()*segmentDist;

	return (awayVect);

}

// ----------------------------------------------------------------------------
// ----------------------------------------------------------------------------
// ATTACKING FUNCTIONS
// ----------------------------------------------------------------------------
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
// AISafeToShoot()
// ----------------------------------------------------------------------------

function bool AISafeToShoot(out Actor hitActor, vector traceEnd, vector traceStart,
                            optional vector extent, optional bool bIgnoreLevel)
{
	local Actor            traceActor;
	local Vector           hitLocation;
	local Vector           hitNormal;
	local Pawn             tracePawn;
	local DeusExDecoration traceDecoration;
	local HXMover      traceMover;
	local bool             bSafe;

	// Future improvement:
	// Ideally, this should use the ammo type to determine how many shots
	// it will take to destroy an obstruction, and call it unsafe if it takes
	// more than x shots.  Also, if the ammo is explosive, and the
	// obstruction is too close, it should never be safe...

	bSafe    = true;
	hitActor = None;

	foreach TraceActors(Class'Actor', traceActor, hitLocation, hitNormal,
	                    traceEnd, traceStart, extent)
	{
		if (hitActor == None)
			hitActor = traceActor;
		if (traceActor == Level)
		{
			if (!bIgnoreLevel)
				bSafe = false;
			break;
		}
		tracePawn = Pawn(traceActor);
		if (tracePawn != None)
		{
			if (tracePawn != self)
			{
				if (GetPawnAllianceType(tracePawn) == ALLIANCE_Friendly)
					bSafe = false;
				break;
			}
		}
		traceDecoration = DeusExDecoration(traceActor);
		if (traceDecoration != None)
		{
			if (traceDecoration.bExplosive || traceDecoration.bInvincible)
			{
				bSafe = false;
				break;
			}
			if ((traceDecoration.HitPoints > 20) || (traceDecoration.minDamageThreshold > 4))  // hack
			{
				bSafe = false;
				break;
			}
		}
		traceMover = HXMover(traceActor);
		if (traceMover != None)
		{
			if (!traceMover.bBreakable)
			{
				bSafe = false;
				break;
			}
			else if ((traceMover.doorStrength > 0.2) || (traceMover.minDamageThreshold > 8))  // hack
			{
				bSafe = false;
				break;
			}
			else  // hack
				break;
		}
		if (Inventory(traceActor) != None)
		{
			bSafe = false;
			break;
		}
	}

	return (bSafe);
}

// ----------------------------------------------------------------------------
// AISafeToThrow()
// ----------------------------------------------------------------------------

function bool AISafeToThrow(vector traceEnd, vector traceStart,
                            float throwAccuracy,
                            optional vector extent)
{
	local float                   time1, time2, tempTime;
	local vector                  pos1,  pos2,  tempPos;
	local rotator                 rot1,  rot2,  tempRot;
	local rotator                 bestAngle;
	local bool                    bSafe;
	local DeusExWeapon            dxWeapon;
	local Class<HXThrownProjectile> throwClass;

	// Someday, we should check for nearby friendlies within the blast radius
	// before throwing...

	// Sanity checks
	throwClass = None;
	dxWeapon = DeusExWeapon(Weapon);
	if (dxWeapon != None)
		throwClass = Class<HXThrownProjectile>(dxWeapon.ProjectileClass);
	if (throwClass == None)
		return false;

	if (extent == vect(0,0,0))
	{
		extent = vect(1,1,0) * throwClass.Default.CollisionRadius;
		extent.Z = throwClass.Default.CollisionHeight;
	}

	if (throwAccuracy < 0.01)
		throwAccuracy = 0.01;

	bSafe = false;
	if (ComputeThrowAngles(traceEnd, traceStart, dxWeapon.ProjectileSpeed, rot1, rot2))
	{
		time1 = ParabolicTrace(pos1, Vector(rot1)*dxWeapon.ProjectileSpeed, traceStart,
		                       true, extent, 5.0,
		                       throwClass.Default.Elasticity, throwClass.Default.bBounce,
		                       60, throwAccuracy);
		time2 = ParabolicTrace(pos2, Vector(rot2)*dxWeapon.ProjectileSpeed, traceStart,
		                       true, extent, 5.0,
		                       throwClass.Default.Elasticity, throwClass.Default.bBounce,
		                       60, throwAccuracy);
		if ((time1 > 0) || (time2 > 0))
		{
			if ((time1 > time2) && (time2 > 0))
			{
				tempTime = time1;
				time1    = time2;
				time2    = tempTime;
				tempPos  = pos1;
				pos1     = pos2;
				pos2     = tempPos;
				tempRot  = rot1;
				rot1     = rot2;
				rot2     = tempRot;
			}
			if (VSize(pos1-traceEnd) <= throwClass.Default.blastRadius)
			{
				if (FastTrace(traceEnd, pos1))
				{
					if ((VSize(pos1-Location) > throwClass.Default.blastRadius*0.5) ||
					    !FastTrace(Location, pos1))
					{
						bestAngle = rot1;
						bSafe     = true;
					}
				}
			}
		}
		if (!bSafe && (time2 > 0))
		{
			if (VSize(pos2-traceEnd) <= throwClass.Default.blastRadius)
			{
				if (FastTrace(traceEnd, pos2))
				{
					if ((VSize(pos2-Location) > throwClass.Default.blastRadius*0.5) ||
					    !FastTrace(Location, pos2))
					{
						bestAngle = rot2;
						bSafe     = true;
					}
				}
			}
		}

	}

	if (bSafe)
		ViewRotation = bestAngle;

	return (bSafe);

}

// ----------------------------------------------------------------------------
// AICanShoot()
// ----------------------------------------------------------------------------

function bool AICanShoot( Pawn Target, bool bLeadTarget, bool bCheckReadiness, optional float ThrowAccuracy, optional bool bDiscountMinRange )
{
	local HXWeapon HXWeapon;
	local Vector X, Y, Z;
	local Vector ProjStart, ProjEnd;
	local float  TempMinRange, TempMaxRange;
	local float  Temp;
	local float  Dist;
	local float  ExtraDist;
	local Actor  HitActor;
	local Vector HitLocation, HitNormal;
	local Vector Extent;
	local bool   bIsThrown;
	local float  Elevation;
	local bool   bSafe;

	if ( Target==None || Target.bIgnore )
		return False;

	HXWeapon = HXWeapon(Weapon);
	if ( HXWeapon==None || (bCheckReadiness && !HXWeapon.bReadyToFire) )
		return False;

	if ( HXWeapon.ReloadCount>0 && (HXWeapon.AmmoType==None || HXWeapon.AmmoType.AmmoAmount<=0) )
		return False;

	if ( FireElevation>0 )
	{
		Elevation = Max( 10, FireElevation + (CollisionHeight+Target.CollisionHeight) );
		if ( Elevation<Abs(Location.Z-target.Location.Z) )
			return false;
	}

	// oldver.
	//bIsThrown = IsThrownWeapon( HXWeapon );
	bIsThrown = HXWeapon.bIsGrenade;

	ExtraDist = Target.CollisionRadius;

	// Range Checks.
	GetPawnWeaponRanges( Self, TempMinRange, TempMaxRange, Temp );
	if ( bDiscountMinRange )
		TempMinRange = 0;
	if ( TempMinRange>=TempMaxRange )
		return False;

	// Predict targets movement.
	ViewRotation = Rotation;
	GetAxes(ViewRotation, X, Y, Z);
	ProjStart = HXWeapon.ComputeProjectileStart( X, Y, Z );
	if ( bLeadTarget && !HXWeapon.bInstantHit && HXWeapon.ProjectileSpeed>0 )
	{
		if ( bIsThrown )
		{
			// compute target's position 1.5 seconds in the future
			ProjEnd = Target.Location + Target.Velocity*1.5;
		}
		else if ( !ComputeTargetLead(Target,ProjStart,HXWeapon.ProjectileSpeed,5.0,ProjEnd) )
		{
			return False;
		}
	}
	else
	{
		ProjEnd = Target.Location;
	}

	if ( bIsThrown )
		ProjEnd += vect(0,0,-1) * (Target.CollisionHeight - 5.0);

	// VSize() is positive definite. No clue what ION thought.
	Dist = Max( 0, VSize(ProjEnd-Location) );

	// HX_HAN: Change Dist<TempMinRange to Dist<=TempMinRange.
	if ( Dist<=TempMinRange || (Dist-ExtraDist)>TempMaxRange )
		return False;

	if ( !bIsThrown )
	{
		//if ( !FastTrace(Target.Location,ProjStart) )
		if ( !FastRepTrace(Target.Location,ProjStart) )
		{
			if ( !Target.bIsPlayer )
				return False;

			//
			// Players only... hack.
			//
			// HX_HAN: No clue why they just account for players,
			//         but I'll keep it that way (for now).
			//
			ProjEnd += vect(0,0,1) * Target.BaseEyeHeight;
			//if ( !FastTrace(Target.Location+vect(0,0,1)*Target.BaseEyeHeight,ProjStart) )
			if ( !FastRepTrace(Target.Location+vect(0,0,1)*Target.BaseEyeHeight,ProjStart) )
				return False;
		}
	}

	if ( HXWeapon.bInstantHit )
	{
		// AISafeToShoot accounts for blocking decorations, etc.
		return AISafeToShoot( HitActor, ProjEnd, ProjStart, , True );
	}
	else
	{
		Extent.X = HXWeapon.ProjectileClass.Default.CollisionRadius;
		Extent.Y = HXWeapon.ProjectileClass.Default.CollisionRadius;
		Extent.Z = HXWeapon.ProjectileClass.Default.CollisionHeight;

		if ( bIsThrown && ThrowAccuracy>0 )
		{
			// Special handling for non straight path.
			return AISafeToThrow( ProjEnd, ProjStart, ThrowAccuracy, Extent );
		}
		else
		{
			// AISafeToShoot accounts for blocking decorations, etc.
			return AISafeToShoot( HitActor, ProjEnd, ProjStart, Extent*3 );
		}
	}
}

// ----------------------------------------------------------------------------
// HXTryLocation()
// ----------------------------------------------------------------------------

function bool HXTryLocation( out Vector Position, optional float MinDist, optional bool bTraceActors, optional HXNearbyProjectileList ProjList )
{
	local float   Magnitude;
	local Vector  NormalPos;
	local Rotator Rot;
	local float   Dist;
	local bool    bSuccess;

	NormalPos = Position-Location;
	Magnitude = VSize(NormalPos);
	if ( MinDist>Magnitude )
		MinDist = Magnitude;
	Rot = Rotator(position-Location);
	bSuccess = AIDirectionReachable( Location, Rot.Yaw, Rot.Pitch, MinDist, Magnitude, Position );

	if ( bSuccess )
	{
		if ( bDefendHome && !IsNearHome(Position) )
			bSuccess = False;
		else if ( bAvoidHarm && HXIsLocationDangerous(ProjList,Position) )
			bSuccess = False;
	}

	return bSuccess;
}

// ----------------------------------------------------------------------------
// ComputeBestFiringPosition()
// ----------------------------------------------------------------------------

function EDestinationType ComputeBestFiringPosition(out vector newPosition)
{
	local float            selfMinRange, selfMaxRange;
	local float            enemyMinRange, enemyMaxRange;
	local float            temp;
	local float            dist;
	local float            innerRange[2], outerRange[2];
	local Rotator          relativeRotation;
	local float            hAngle, vAngle;
	local int              acrossDist;
	local float            awayDist;
	local float            extraDist;
	local float            fudgeMargin;
	local int              angle;
	local float            maxDist;
	local float            distDelta;
	local bool             bInnerValid, bOuterValid;
	local vector           tryVector;
	local EDestinationType destType;
	local float            moveMult;
	local float            reloadMult;
	local float            minArea;
	local float            minDist;
	local float            range;
	local float            margin;

	local HXNearbyProjectileList projList;
	local vector               projVector;
	local bool                 bUseProjVector;

	local rotator              sprintRot;
	local vector               sprintVect;
	local bool                 bUseSprint;

	destType = DEST_Failure;

	extraDist   = enemy.CollisionRadius*0.5;
	fudgeMargin = 100;
	minArea     = 35;

	GetPawnWeaponRanges(self, selfMinRange, selfMaxRange, temp);
	GetPawnWeaponRanges(enemy, enemyMinRange, temp, enemyMaxRange);

	if (selfMaxRange > 1200)
		selfMaxRange = 1200;
	if (enemyMaxRange > 1200)
		enemyMaxRange = 1200;

	// hack, to prevent non-strafing NPCs from trying to back up
	if (!bCanStrafe)
		selfMinRange  = 0;

	minDist = enemy.CollisionRadius + CollisionRadius - (extraDist+1);
	if (selfMinRange < minDist)
		selfMinRange = minDist;
	if (selfMinRange < MinRange)
		selfMinRange = MinRange;
	if (selfMaxRange > MaxRange)
		selfMaxRange = MaxRange;

	dist = VSize(enemy.Location-Location);

	innerRange[0] = selfMinRange;
	innerRange[1] = selfMaxRange;
	outerRange[0] = selfMinRange;
	outerRange[1] = selfMaxRange;

	// hack, to prevent non-strafing NPCs from trying to back up

	if (selfMaxRange > enemyMinRange)
		innerRange[1] = enemyMinRange;
	if ((selfMinRange < enemyMaxRange) && bCanStrafe)  // hack, to prevent non-strafing NPCs from trying to back up
		outerRange[0] = enemyMaxRange;

	range = outerRange[1]-outerRange[0];
	if (range < minArea)
	{
		outerRange[0] = 0;
		outerRange[1] = 0;
	}
	range = innerRange[1]-innerRange[0];
	if (range < minArea)
	{
		innerRange[0] = outerRange[0];
		innerRange[1] = outerRange[1];
		outerRange[0] = 0;
		outerRange[1] = 0;
	}

	// If the enemy can reach us through our entire weapon range, just use the range
	if ((innerRange[0] >= innerRange[1]) && (outerRange[0] >= outerRange[1]))
	{
		innerRange[0] = selfMinRange;
		innerRange[1] = selfMaxRange;
	}

	innerRange[0] += extraDist;
	innerRange[1] += extraDist;
	outerRange[0] += extraDist;
	outerRange[1] += extraDist;

	if (innerRange[0] >= innerRange[1])
		bInnerValid = false;
	else
		bInnerValid = true;
	if (outerRange[0] >= outerRange[1])
		bOuterValid = false;
	else
		bOuterValid = true;

	if (!bInnerValid)
	{
		// ugly
		newPosition = Location;
//		return DEST_SameLocation;
		return destType;
	}

	relativeRotation = Rotator(Location - enemy.Location);

	hAngle = (relativeRotation.Yaw - enemy.Rotation.Yaw) & 65535;
	if (hAngle > 32767)
		hAngle -= 65536;
	// ignore vertical angle for now

	awayDist   = dist;
	acrossDist = 0;
	maxDist    = GroundSpeed*0.6;  // distance covered in 6/10 second

	if (bInnerValid)
	{
		margin = (innerRange[1]-innerRange[0]) * 0.5;
		if (margin > fudgeMargin)
			margin = fudgeMargin;
		if (awayDist < innerRange[0])
			awayDist = innerRange[0]+margin;
		else if (awayDist > innerRange[1])
			awayDist = innerRange[1]-margin;
	}
	if (bOuterValid)
	{
		margin = (outerRange[1]-outerRange[0]) * 0.5;
		if (margin > fudgeMargin)
			margin = fudgeMargin;
		if (awayDist > outerRange[1])
			awayDist = outerRange[1]-margin;
	}

	if (awayDist > dist+maxDist)
		awayDist = dist+maxDist;
	if (awayDist < dist-maxDist)
		awayDist = dist-maxDist;

	// Used to determine whether NPCs should sprint/avoid aim
	moveMult = 1.0;
	if ((dist <= 180) && enemy.bIsPlayer && (enemy.Weapon != None) && (enemyMaxRange < 180))
		moveMult = CloseCombatMult;

	if (bAvoidAim && !enemy.bIgnore && (FRand() <= AvoidAccuracy*moveMult))
	{
		if ((awayDist < enemyMaxRange+maxDist+50) && (awayDist < 800) && (Enemy.Weapon != None))
		{
			if (dist > 0)
				angle = int(atan(CollisionRadius*2.0/dist)*32768/Pi);
			else
				angle = 16384;

			if ((hAngle >= -angle) && (hAngle <= angle))
			{
				if (hAngle < 0)
					acrossDist = (-angle-hAngle)-128;
				else
					acrossDist = (angle-hAngle)+128;
				if (Rand(20) == 0)
					acrossDist = -acrossDist;
			}
		}
	}

// projList is implicitly initialized to null...

	bUseProjVector = false;
	if (bAvoidHarm && (FRand() <= HarmAccuracy))
	{
		if (HXGetProjectileList(projList, Location) > 0)
		{
			if (HXIsLocationDangerous(projList, Location))
			{
				projVector = HXComputeAwayVector(projList);
				bUseProjVector = true;
			}
		}
	}

	reloadMult = 1.0;
	if (IsWeaponReloading() && Enemy.bIsPlayer)
		reloadMult = 0.5;

	bUseSprint = false;
	if (!bUseProjVector && bSprint && bCanStrafe && !enemy.bIgnore && (FRand() <= SprintRate*0.5*moveMult*reloadMult))
	{
		if (bOuterValid || (innerRange[1] > 100))  // sprint on long-range weapons only
		{
			sprintRot = Rotator(enemy.Location - Location);
			if (Rand(2) == 1)
				sprintRot.Yaw += 16384;
			else
				sprintRot.Yaw += 49152;
			sprintRot = RandomBiasedRotation(sprintRot.Yaw, 0.5, 0, 0);
			sprintRot.Pitch = 0;
			sprintVect = Vector(sprintRot)*GroundSpeed*(FRand()+0.5);
			bUseSprint = true;
		}
	}

	if ((acrossDist != 0) || (awayDist != dist) || bUseProjVector || bUseSprint)
	{
		if (Rand(40) != 0)
		{
			if ((destType == DEST_Failure) && bUseProjVector)
			{
				tryVector = projVector + Location;
				if (HXTryLocation(tryVector, CollisionRadius+16))
					destType = DEST_NewLocation;
			}
			if ((destType == DEST_Failure) && (acrossDist != 0) && (awayDist != dist))
			{
				tryVector = Vector(relativeRotation+(rot(0, 1, 0)*acrossDist))*awayDist + enemy.Location;
				if (HXTryLocation(tryVector, CollisionRadius+16, , projList))
					destType = DEST_NewLocation;
			}
			if ((destType == DEST_Failure) && (awayDist != dist))
			{
				tryVector = Vector(relativeRotation)*awayDist + enemy.Location;
				if (HXTryLocation(tryVector, CollisionRadius+16, , projList))
					destType = DEST_NewLocation;
			}
			if ((destType == DEST_Failure) && (acrossDist != 0))
			{
				tryVector = Vector(relativeRotation+(rot(0, 1, 0)*acrossDist))*dist + enemy.Location;
				if (HXTryLocation(tryVector, CollisionRadius+16, , projList))
					destType = DEST_NewLocation;
			}
			if ((destType == DEST_Failure) && bUseSprint)
			{
				tryVector = sprintVect + Location;
				if (HXTryLocation(tryVector, CollisionRadius+16))
					destType = DEST_NewLocation;
			}
		}
		if (destType == DEST_Failure)
		{
			if ((moveMult >= 0.5) || (FRand() <= moveMult))
			{
				if (AIPickRandomDestination(CollisionRadius+16, maxDist,
				                            relativeRotation.Yaw+32768, 0.6, -relativeRotation.Pitch, 0.6, 2,
				                            0.9, tryVector))
					if (!bDefendHome || IsNearHome(tryVector))
						if (!bAvoidHarm || !HXIsLocationDangerous(projList, tryVector))
							destType = DEST_NewLocation;
			}
			else
				destType = DEST_SameLocation;
		}
		if (destType != DEST_Failure)
			newPosition = tryVector;
	}
	else
		destType = DEST_SameLocation;

	return destType;
}

// ----------------------------------------------------------------------------
// AdjustAim()
//
// Adjust the aim at target
// ----------------------------------------------------------------------------

function rotator AdjustAim(float projSpeed, vector projStart, int aimerror, bool leadTarget, bool warnTarget)
{
	local rotator     FireRotation;
	local vector      FireSpot;
	local actor       HitActor;
	local vector      HitLocation, HitNormal;
	local vector      vectorArray[3];
	local vector      tempVector;
	local int         i;
	local int         swap;
	local Rotator     rot;
	local bool        bIsThrown;
	local HXMover dxMover;
	local actor       Target;  // evil fix -- STM

	bIsThrown = IsThrownWeapon(DeusExWeapon(Weapon));

// took this line out for evil fix...
//	if ( Target == None )

	Target = Enemy;
	if ( Target == None )
		return Rotation;
	if ( !Target.IsA('Pawn') )
		return rotator(Target.Location - Location);

	FireSpot = Target.Location;
	if (leadTarget && (projSpeed > 0))
	{
		if (bIsThrown)
		{
			// compute target's position 1.5 seconds in the future
			FireSpot = target.Location + (target.Velocity*1.5);
		}
		else
		{
			//FireSpot += (Target.Velocity * VSize(Target.Location - ProjStart)/projSpeed);
			ComputeTargetLead(Pawn(Target), ProjStart, projSpeed, 20.0, FireSpot);
		}
	}

	if (bIsThrown)
	{
		vectorArray[0] = FireSpot - vect(0,0,1)*(Target.CollisionHeight-5);  // floor
		vectorArray[1] = vectorArray[0] + Vector(rot(0,1,0)*Rand(65536))*CollisionRadius*1.2; // Rand does just go up to 0xEFFF.
		vectorArray[2] = vectorArray[0] + Vector(rot(0,1,0)*Rand(65536))*CollisionRadius*1.2;

		for (i=0; i<3; i++)
		{
			if (AISafeToThrow(vectorArray[i], ProjStart, 0.025))
				break;
		}
		if (i < 3)
		{
			FireSpot = vectorArray[i];
			FireRotation = ViewRotation;
		}
		else
			FireRotation = Rotator(FireSpot - ProjStart);
	}
	else
	{
		dxMover = HXMover(Target.Base);
		if ((dxMover != None) && dxMover.bBreakable)
		{
			tempVector = Normal((Location-Target.Location)*vect(1,1,0))*(Target.CollisionRadius*1.01) -
			             vect(0,0,1)*(Target.CollisionHeight*1.01);
			vectorArray[0] = FireSpot + tempVector;
		}
		else if (bAimForHead)
			vectorArray[0] = FireSpot + vect(0,0,1)*(Target.CollisionHeight*0.85);    // head
		else
			vectorArray[0] = FireSpot + vect(0,0,1)*((FRand()*2-1)*Target.CollisionHeight);
		vectorArray[1] = FireSpot + vect(0,0,1)*((FRand()*2-1)*Target.CollisionHeight);
		vectorArray[2] = FireSpot + vect(0,0,1)*((FRand()*2-1)*Target.CollisionHeight);

		for (i=0; i<3; i++)
		{
			if (AISafeToShoot(HitActor, vectorArray[i], ProjStart))
				break;
		}
		if (i < 3)
			FireSpot = vectorArray[i];

		FireRotation = Rotator(FireSpot - ProjStart);
	}

	if (warnTarget && Pawn(Target) != None) 
		Pawn(Target).WarnTarget(self, projSpeed, vector(FireRotation)); 

	FireRotation.Yaw = FireRotation.Yaw & 65535;
	if ( (Abs(FireRotation.Yaw - (Rotation.Yaw & 65535)) > 8192)
		&& (Abs(FireRotation.Yaw - (Rotation.Yaw & 65535)) < 57343) )
	{
		if ( (FireRotation.Yaw > Rotation.Yaw + 32768) || 
			((FireRotation.Yaw < Rotation.Yaw) && (FireRotation.Yaw > Rotation.Yaw - 32768)) )
			FireRotation.Yaw = Rotation.Yaw - 8192;
		else
			FireRotation.Yaw = Rotation.Yaw + 8192;
	}
	viewRotation = FireRotation;			
	return FireRotation;
}

// ----------------------------------------------------------------------------
// IsThrownWeapon()
//
// Just changed it for legacy code.
// For HX you can just check HXWeapon.bIsGrenade
// ----------------------------------------------------------------------------

function bool IsThrownWeapon( DeusExWeapon TestWeapon )
{
	local HXWeapon HXTestWeapon;

	if ( TestWeapon==None )
		return False;

	HXTestWeapon = HXWeapon( TestWeapon );

	// Fallback.
	if ( HXTestWeapon==None )
		return Super.IsThrownWeapon( TestWeapon );

	return HXTestWeapon.bIsGrenade;
}

// ----------------------------------------------------------------------------
// ----------------------------------------------------------------------------
// CALLBACKS AND OVERRIDDEN FUNCTIONS
// ----------------------------------------------------------------------------
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
// Tick()
// ----------------------------------------------------------------------------

simulated function Tick(float deltaTime)
{
	local HXPlayerPawn NearestPlayer;
	local float NearestPlayerDistance;
	local float        dropPeriod;
	local float        adjustedRate;
	local float        TempDist;
	local Pawn         P;
	local name         stateName;
	local vector       loc;
	local bool         bDoLowPriority;
	local bool         bCheckOther;
	local bool         bCheckPlayer;

	// Nasty update function.
	if ( Role < ROLE_Authority )
	{
		bIsSpeaking = bClientIsSpeaking;
		ClientLipSynch(deltaTime);
	}
	else
	{
		bClientHostileTowardsPlayers = GetAllianceType('Player') == ALLIANCE_Hostile;

		bClientSitting		= bSitting;
		bClientDancing		= bDancing;
		bClientIsSpeaking	= bIsSpeaking;
	}

	// Only do the other stuff on server.
	if ( Role<ROLE_Authority )
		return;

	// Ugly to update this often.
	//if ( bClientInWorld )
		//CreateShadow();
	//else
		//KillShadow();

	NearestPlayerDistance = 1000000; // some value in FindTaggedActor()
	if ( PlayerDistances[0].Player!=None )
	{
		NearestPlayerDistance = PlayerDistances[0].Distance;
		NearestPlayer         = PlayerDistances[0].Player;
	}

	bDoLowPriority = True;
	bCheckPlayer   = True;
	bCheckOther    = True;

	if ( bTickVisibleOnly )
	{
		if ( NearestPlayerDistance>1200 )
		{
			bDoLowPriority = False;
		}
		//if ( NearestPlayerDistance>(2500 + 500 * Level.Game.Difficulty) )
		if ( NearestPlayerDistance>2500 )
		{
			bCheckPlayer = False;
		}
		// HX_NOTE: LastRendered() now includes last time regarded as relevant for network.
		if ( NearestPlayerDistance>600 && LastRendered()>=5.0 )
		{
			bCheckOther = False;
		}
	}

	// [...]

	if ( bStandInterpolation )
		UpdateStanding( DeltaTime );

	// this is UGLY!
	if (bOnFire && (health > 0))
	{
		stateName = GetStateName();
		if ((stateName != 'Burning') && (stateName != 'TakingHit') && (stateName != 'RubbingEyes'))
			GotoState('Burning');
	}
	else
	{
		if ( bDoLowPriority )
		{
			// Don't allow radius-based convos to interupt other conversations!
			// Note: Maybe consider other players too.
			if ( NearestPlayer!=None && GetStateName()!='Conversation' && GetStateName()!='FirstPersonConversation' )
				HXGameInfo(Level.Game).StartConversation( NearestPlayer, Self, IM_Radius );
		}

		if ( CheckEnemyPresence(DeltaTime,bCheckPlayer,bCheckOther) )
		{
			HandleEnemy();
		}
		else
		{
			CheckBeamPresence( DeltaTime );
			if ( bDoLowPriority /*|| LastRendered()<5.0*/ )
				CheckCarcassPresence( DeltaTime );  // hacky -- may change state!
		}
	}

	// Randomly spawn an air bubble every 0.2 seconds if we're underwater
	// NOTE: This should happen clientside.
	if ( HeadRegion.Zone.bWaterZone && bSpawnBubbles && bDoLowPriority && !bIsRobot )
	{
		SwimBubbleTimer += DeltaTime;
		if ( SwimBubbleTimer>=0.2 )
		{
			SwimBubbleTimer = 0;
			if ( FRand()<0.4 )
			{
				Loc = Location + VRand() * 4;
				Loc.Z += CollisionHeight * 0.9;
				Spawn( Class'AirBubble', Self, , Loc );
			}
		}
	}

	// Handle poison damage
	UpdatePoison( DeltaTime );
}

// ----------------------------------------------------------------------------
// BaseChange()
// ----------------------------------------------------------------------------

singular function BaseChange()
{
	//Log( Self$".BaseChange() bInWorld = "$bInWorld );

	if ( bInWorld )
	{
		Super.BaseChange();

		if ( bSitting && !IsSeatValid(HXSeatActor) )
		{
			StandUp();
			if ( GetStateName()=='Sitting' )
				GotoState( 'Sitting', 'Begin' );
		}
	}
	else
	{
		SetPhysics( PHYS_None );
	}
}

// ----------------------------------------------------------------------------
// SwitchToBestWeapon()
// ----------------------------------------------------------------------------

function bool SwitchToBestWeapon()
{
	local Inventory    inv;
	local DeusExWeapon curWeapon;
	local float        score;
	local DeusExWeapon dxWeapon;
	local DeusExWeapon bestWeapon;
	local float        bestScore;
	local int          fallbackLevel;
	local int          curFallbackLevel;
	local bool         bBlockSpecial;
	local bool         bValid;
	local bool         bWinner;
	local float        minRange, accRange;
	local float        range, centerRange;
	local float        cutoffRange;
	local float        enemyRange;
	local float        minEnemy, accEnemy, maxEnemy;
	local ScriptedPawn enemyPawn;
	local HXRobot        enemyRobot;
	local DeusExPlayer enemyPlayer;
	local float        enemyRadius;
	local bool         bEnemySet;
	local int          loopCount, i;  // hack - check for infinite inventory
	local Inventory    loopInv;       // hack - check for infinite inventory

	if (ShouldDropWeapon())
	{
		DropWeapon();
		return false;
	}

	bBlockSpecial = false;
	dxWeapon = DeusExWeapon(Weapon);
	if (dxWeapon != None)
	{
		if (dxWeapon.AITimeLimit > 0)
		{
			if (SpecialTimer <= 0)
			{
				bBlockSpecial = true;
				FireTimer = dxWeapon.AIFireDelay;
			}
		}
	}

	bestWeapon      = None;
	bestScore       = 0;
	fallbackLevel   = 0;
	inv             = Inventory;

	bEnemySet   = false;
	minEnemy    = 0;
	accEnemy    = 0;
	enemyRange  = 400;  // default
	enemyRadius = 0;
	enemyPawn   = None;
	enemyRobot  = None;
	if (Enemy != None)
	{
		bEnemySet   = true;
		enemyRange  = VSize(Enemy.Location - Location);
		enemyRadius = Enemy.CollisionRadius;
		if (DeusExWeapon(Enemy.Weapon) != None)
			DeusExWeapon(Enemy.Weapon).GetWeaponRanges(minEnemy, accEnemy, maxEnemy);
		enemyPawn   = ScriptedPawn(Enemy);
		enemyRobot  = HXRobot(Enemy);
		enemyPlayer = DeusExPlayer(Enemy);
	}

	loopCount = 0;
	while (inv != None)
	{
		// THIS IS A MAJOR HACK!!!
		loopCount++;
		if (loopCount == 9999)
		{
			log("********** RUNAWAY LOOP IN SWITCHTOBESTWEAPON ("$self$") **********");
			loopInv = Inventory;
			i = 0;
			while (loopInv != None)
			{
				i++;
				if (i > 300)
					break;
				log("   Inventory "$i$" - "$loopInv);
				loopInv = loopInv.Inventory;
			}
		}

		curWeapon = DeusExWeapon(inv);
		if (curWeapon != None)
		{
			bValid = true;
			if (curWeapon.ReloadCount > 0)
			{
				if (curWeapon.AmmoType == None)
					bValid = false;
				else if (curWeapon.AmmoType.AmmoAmount < 1)
					bValid = false;
			}

			// Ensure we can actually use this weapon here
			if (bValid)
			{
				// lifted from DeusExWeapon...
				if ((curWeapon.EnviroEffective == ENVEFF_Air) || (curWeapon.EnviroEffective == ENVEFF_Vacuum) ||
				    (curWeapon.EnviroEffective == ENVEFF_AirVacuum))
					if (curWeapon.Region.Zone.bWaterZone)
						bValid = false;
			}

			if (bValid)
			{
				GetWeaponBestRange(curWeapon, minRange, accRange);
				cutoffRange = minRange+(CollisionRadius+enemyRadius);
				range = (accRange - minRange) * 0.5;
				centerRange = minRange + range;
				if (range < 50)
					range = 50;
				if (enemyRange < centerRange)
					score = (centerRange - enemyRange)/range;
				else
					score = (enemyRange - centerRange)/range;
				if ((minRange >= minEnemy) && (accRange <= accEnemy))
					score += 0.5;  // arbitrary
				if ((cutoffRange >= enemyRange-CollisionRadius) && (cutoffRange >= 256)) // do not use long-range weapons on short-range targets
					score += 10000;

				curFallbackLevel = 3;
				if (curWeapon.bFallbackWeapon && !bUseFallbackWeapons)
					curFallbackLevel = 2;
				if (!bEnemySet && !curWeapon.bUseAsDrawnWeapon)
					curFallbackLevel = 1;
				if ((curWeapon.AIFireDelay > 0) && (FireTimer > 0))
					curFallbackLevel = 0;
				if (bBlockSpecial && (curWeapon.AITimeLimit > 0) && (SpecialTimer <= 0))
					curFallbackLevel = 0;

				// Adjust score based on opponent and damage type.
				// All damage types are listed here, even the ones that aren't used by weapons... :)
				// (hacky...)

				switch (curWeapon.WeaponDamageType())
				{
					case 'Exploded':
						// Massive explosions are always good
						score -= 0.2;
						break;

					case 'Stunned':
						if (enemyPawn != None)
						{
							if (enemyPawn.bStunned)
								score += 1000;
							else
								score -= 1.5;
						}
						if (enemyPlayer != None)
							score += 10;
						break;

					case 'TearGas':
						if (enemyPawn != None)
						{
							if (enemyPawn.bStunned)
								//score += 1000;
								bValid = false;
							else
								score -= 5.0;
						}
						if (enemyRobot != None)
							//score += 10000;
							bValid = false;
						break;

					case 'HalonGas':
						if (enemyPawn != None)
						{
							if (enemyPawn.bStunned)
								//score += 1000;
								bValid = false;
							else if (enemyPawn.bOnFire)
								//score += 10000;
								bValid = false;
							else
								score -= 3.0;
						}
						if (enemyRobot != None)
							//score += 10000;
							bValid = false;
						break;

					case 'PoisonGas':
					case 'Poison':
					case 'PoisonEffect':
					case 'Radiation':
						if (enemyRobot != None)
							//score += 10000;
							bValid = false;
						break;

					case 'Burned':
					case 'Flamed':
					case 'Shot':
						if (enemyRobot != None)
							score += 0.5;
						break;

					case 'Sabot':
						if (enemyRobot != None)
							score -= 0.5;
						break;

					case 'EMP':
					case 'NanoVirus':
						if (enemyRobot != None)
							score -= 5.0;
						else if (enemyPlayer != None)
							score += 5.0;
						else
							//score += 10000;
							bValid = false;
						break;

					case 'Drowned':
					default:
						break;
				}

				// Special case for current weapon
				if ((curWeapon == Weapon) && (WeaponTimer < 10.0))
				{
					// If we last changed weapons less than five seconds ago,
					// keep this weapon
					if (WeaponTimer < 5.0)
						score = -10;

					// If between five and ten seconds, use a sliding scale
					else
						score -= (10.0 - WeaponTimer)/5.0;
				}

				// Throw a little randomness into the computation...
				else
				{
					score += FRand()*0.1 - 0.05;
					if (score < 0)
						score = 0;
				}

				if (bValid)
				{
					// ugly
					if (bestWeapon == None)
						bWinner = true;
					else if (curFallbackLevel > fallbackLevel)
						bWinner = true;
					else if (curFallbackLevel < fallbackLevel)
						bWinner = false;
					else if (bestScore > score)
						bWinner = true;
					else
						bWinner = false;
					if (bWinner)
					{
						bestScore     = score;
						bestWeapon    = curWeapon;
						fallbackLevel = curFallbackLevel;
					}
				}
			}
		}
		inv = inv.Inventory;
	}

	// If we're changing weapons, reset the weapon timers
	if (Weapon != bestWeapon)
	{
		if (!bEnemySet)
			WeaponTimer = 10;  // hack
		else
			WeaponTimer = 0;
		if (bestWeapon != None)
			if (bestWeapon.AITimeLimit > 0)
				SpecialTimer = bestWeapon.AITimeLimit;
		ReloadTimer = 0;
	}

	SetWeapon(bestWeapon);

	return false;
}

// ----------------------------------------------------------------------------
// ClientLoopBaseConvoAnim()
// ----------------------------------------------------------------------------
/*
simulated function ClientLoopBaseConvoAnim()
{
	local float rnd;

	// Special case for sitting
	if (bClientSitting)
	{
		if (!IsAnimating())
			PlaySitting();
	}

	// Special case for dancing
	else if (bClientDancing)
	{
		if (!IsAnimating())
			PlayDancing();
	}

	// Otherwise, just do the usual shit
	else
	{
		rnd = FRand();

		// move arms randomly
		if (bClientIsSpeaking)
		{
			if (animTimer[2] > 2.5)
			{
				animTimer[2] = 0;
				if (rnd < 0.1)
					PlayAnim('GestureLeft', 0.35, 0.4);
				else if (rnd < 0.2)
					PlayAnim('GestureRight', 0.35, 0.4);
				else if (rnd < 0.3)
					PlayAnim('GestureBoth', 0.35, 0.4);
			}
		}

		// if we're not playing an animation, loop the breathe
		if (!IsAnimating())
			LoopAnim('BreatheLight',, 0.4);
	}
}
*/

// ----------------------------------------------------------------------------
// Frob()
// ----------------------------------------------------------------------------

function Frob(Actor Frobber, Inventory frobWith)
{
	Super(Pawn).Frob(Frobber, frobWith);

	// Check to see if the Frobber is the player.  If so, then
	// check to see if we need to start a conversation.

	if (HXPlayerPawn(Frobber) != None)
	{
		if ( HXGameInfo(Level.Game).StartConversation( HXPlayerPawn(Frobber), Self, IM_Frob ) )
		{
			ConversationActor = Pawn(Frobber);
			return;
		}
	}
}

//
// HX_REORDER
//

// ----------------------------------------------------------------------------
// CanConverseWithPlayer2()
// ----------------------------------------------------------------------------

function bool CanConverseWithPlayer2()
{
	local name alliance1, alliance2, carcname;  // temp vars

	//if (GetPawnAllianceType(dxPlayer) == ALLIANCE_Hostile)
	if (GetAllianceType('Player') == ALLIANCE_Hostile)
		return false;
	//else if ((GetStateName() == 'Fleeing') && (Enemy != dxPlayer) && (IsValidEnemy(Enemy, false)))  // hack
	else if ((GetStateName() == 'Fleeing') && (Enemy != None) && (!Enemy.isA('HXPlayerPawn')) && (IsValidEnemy(Enemy, false)))  // hack
		return false;
	//else if (GetCarcassData(dxPlayer, alliance1, alliance2, carcname))
		//return false;
	else
		return true;
}

// ----------------------------------------------------------------------------
// state Attacking
//
// Kill!  Kill!  Kill!  Kill!
// ----------------------------------------------------------------------------

State Attacking
{
	function ReactToInjury(Pawn instigatedBy, Name damageType, EHitLocation hitPos)
	{
		local Pawn oldEnemy;
		local bool bHateThisInjury;
		local bool bFearThisInjury;

		if ((health > 0) && (bLookingForInjury || bLookingForIndirectInjury))
		{
			oldEnemy = Enemy;

			bHateThisInjury = ShouldReactToInjuryType(damageType, bHateInjury, bHateIndirectInjury);
			bFearThisInjury = ShouldReactToInjuryType(damageType, bFearInjury, bFearIndirectInjury);

			if (bHateThisInjury)
				IncreaseAgitation(instigatedBy, 1.0);
			if (bFearThisInjury)
				IncreaseFear(instigatedBy, 2.0);

			if (ReadyForNewEnemy())
				SetEnemy(instigatedBy);

			if (ShouldFlee())
			{
				SetDistressTimer();
				PlayCriticalDamageSound();
				if (RaiseAlarm == RAISEALARM_BeforeFleeing)
					SetNextState('Alerting');
				else
					SetNextState('Fleeing');
			}
			else
			{
				SetDistressTimer();
				if (oldEnemy != Enemy)
					PlayNewTargetSound();
				SetNextState('Attacking', 'ContinueAttack');
			}
			GotoDisabledState(damageType, hitPos);
		}
	}

	function SetFall()
	{
		StartFalling('Attacking', 'ContinueAttack');
	}

	function HitWall(vector HitNormal, actor Wall)
	{
		if (Physics == PHYS_Falling)
			return;
		Global.HitWall(HitNormal, Wall);
		CheckOpenDoor(HitNormal, Wall);
	}

	function Reloading(DeusExWeapon reloadWeapon, float reloadTime)
	{
		Global.Reloading(reloadWeapon, reloadTime);
		if (bReadyToReload)
			if (IsWeaponReloading())
				if (!IsHandToHand())
					TweenToShoot(0);
	}

	function EDestinationType PickDestination()
	{
		local vector               distVect;
		local vector               tempVect;
		local rotator              enemyDir;
		local float                magnitude;
		local float                calcMagnitude;
		local int                  iterations;
		local EDestinationType     destType;
		local HXNearbyProjectileList projList;

		destPoint = None;
		destLoc   = vect(0, 0, 0);
		destType  = DEST_Failure;

		if (enemy == None)
			return (destType);

		if (bCrouching && (CrouchTimer > 0))
			destType = DEST_SameLocation;

		if (destType == DEST_Failure)
		{
			if (AICanShoot(enemy, true, false, 0.025) || ActorReachable(enemy))
			{
				destType = ComputeBestFiringPosition(tempVect);
				if (destType == DEST_NewLocation)
					destLoc = tempVect;
			}
		}

		if (destType == DEST_Failure)
		{
			MoveTarget = FindPathToward(enemy);
			if (MoveTarget != None)
			{
				if (!bDefendHome || IsNearHome(MoveTarget.Location))
				{
					if (bAvoidHarm)
						HXGetProjectileList(projList, MoveTarget.Location);
					if (!bAvoidHarm || !HXIsLocationDangerous(projList, MoveTarget.Location))
					{
						destPoint = MoveTarget;
						destType  = DEST_NewLocation;
					}
				}
			}
		}

		// Default behavior, so they don't just stand there...
		if (destType == DEST_Failure)
		{
			enemyDir = Rotator(Enemy.Location - Location);
			if (AIPickRandomDestination(60, 150,
			                            enemyDir.Yaw, 0.5, enemyDir.Pitch, 0.5, 
			                            2, FRand()*0.4+0.35, tempVect))
			{
				if (!bDefendHome || IsNearHome(tempVect))
				{
					destType = DEST_NewLocation;
					destLoc  = tempVect;
				}
			}
		}

		return (destType);
	}

	function bool FireIfClearShot()
	{
		local DeusExWeapon dxWeapon;

		dxWeapon = DeusExWeapon(Weapon);
		if (dxWeapon != None)
		{
			if ((dxWeapon.AIFireDelay > 0) && (FireTimer > 0))
				return false;
			else if (AICanShoot(enemy, true, true, 0.025))
			{
				Weapon.Fire(0);
				FireTimer = dxWeapon.AIFireDelay;
				return true;
			}
			else
				return false;
		}
		else
			return false;
	}

	function CheckAttack(bool bPlaySound)
	{
		local bool bCriticalDamage;
		local bool bOutOfAmmo;
		local Pawn oldEnemy;
		local bool bAllianceSwitch;

		oldEnemy = enemy;

		bAllianceSwitch = false;
		if (!IsValidEnemy(enemy))
		{
			if (IsValidEnemy(enemy, false))
				bAllianceSwitch = true;
			SetEnemy(None, 0, true);
		}

		if (enemy == None)
		{
			if (Orders == 'Attacking')
			{
				FindOrderActor();
				SetEnemy(Pawn(OrderActor), 0, true);
			}
		}
		if (ReadyForNewEnemy())
			FindBestEnemy(false);
		if (enemy == None)
		{
			Enemy = oldEnemy;  // hack
			if (bPlaySound)
			{
				if (bAllianceSwitch)
					PlayAllianceFriendlySound();
				else
					PlayAreaSecureSound();
			}
			Enemy = None;
			if (Orders != 'Attacking')
				FollowOrders();
			else
				GotoState('Wandering');
			return;
		}

		SwitchToBestWeapon();
		if (bCrouching && (CrouchTimer <= 0) && !ShouldCrouch())
		{
			EndCrouch();
			TweenToShoot(0.15);
		}
		bCriticalDamage = False;
		bOutOfAmmo      = False;
		if (ShouldFlee())
			bCriticalDamage = True;
		else if (Weapon == None)
			bOutOfAmmo = True;
		else if (Weapon.ReloadCount > 0)
		{
			if (Weapon.AmmoType == None)
				bOutOfAmmo = True;
			else if (Weapon.AmmoType.AmmoAmount < 1)
				bOutOfAmmo = True;
		}
		if (bCriticalDamage || bOutOfAmmo)
		{
			if (bPlaySound)
			{
				if (bCriticalDamage)
					PlayCriticalDamageSound();
				else if (bOutOfAmmo)
					PlayOutOfAmmoSound();
			}
			if (RaiseAlarm == RAISEALARM_BeforeFleeing)
				GotoState('Alerting');
			else
				GotoState('Fleeing');
		}
		else if (bPlaySound && (oldEnemy != Enemy))
			PlayNewTargetSound();
	}

	function Tick(float deltaSeconds)
	{
		local bool   bCanSee;
		local float  yaw;
		local vector lastLocation;
		local Pawn   lastEnemy;
		local float  surpriseTime;

		Global.Tick(deltaSeconds);
		if (CrouchTimer > 0)
		{
			CrouchTimer -= deltaSeconds;
			if (CrouchTimer < 0)
				CrouchTimer = 0;
		}
		EnemyTimer += deltaSeconds;
		UpdateActorVisibility(Enemy, deltaSeconds, 1.0, false);
		if ((Enemy != None) && HasEnemyTimedOut())
		{
			lastLocation = Enemy.Location;
			lastEnemy    = Enemy;
			FindBestEnemy(true);
			if (Enemy == None)
			{
				SetSeekLocation(lastEnemy, lastLocation, SEEKTYPE_Guess, true);
				GotoState('Seeking');
			}
		}
		else if (bCanFire && (Enemy != None))
		{
			ViewRotation = Rotator(Enemy.Location-Location);
			if (bFacingTarget)
				FireIfClearShot();
			else if (!bMustFaceTarget)
			{
				yaw = (ViewRotation.Yaw-Rotation.Yaw) & 0xFFFF;
				if (yaw >= 32768)
					yaw -= 65536;
				yaw = Abs(yaw)*360/32768;  // 0-180 x 2
				if (yaw <= FireAngle)
					FireIfClearShot();
			}
		}
		//UpdateReactionLevel(true, deltaSeconds);
	}

	function bool IsHandToHand()
	{
		if (Weapon != None)
		{
			if (DeusExWeapon(Weapon) != None)
			{
				if (DeusExWeapon(Weapon).bHandToHand)
					return true;
				else
					return false;
			}
			else
				return false;
		}
		else
			return false;
	}

	function bool ReadyForWeapon()
	{
		local bool bReady;

		bReady = false;
		if (DeusExWeapon(weapon) != None)
		{
			if (DeusExWeapon(weapon).bReadyToFire)
				if (!IsWeaponReloading())
					bReady = true;
		}
		if (!bReady)
			if (enemy == None)
				bReady = true;
		if (!bReady)
			if (!AICanShoot(enemy, true, false, 0.025))
				bReady = true;

		return (bReady);
	}

	function bool ShouldCrouch()
	{
		if (bCanCrouch && !Region.Zone.bWaterZone && !IsHandToHand() &&
		    ((enemy != None) && (VSize(enemy.Location-Location) > 300)) &&
		    ((DeusExWeapon(Weapon) == None) || DeusExWeapon(Weapon).bUseWhileCrouched))
			return true;
		else
			return false;
	}

	function StartCrouch()
	{
		if (!bCrouching)
		{
			bCrouching = true;
			SetBasedPawnSize(CollisionRadius, GetCrouchHeight());
			CrouchTimer = 1.0+FRand()*0.5;
		}
	}

	function EndCrouch()
	{
		if (bCrouching)
		{
			bCrouching = false;
			ResetBasedPawnSize();
		}
	}

	function BeginState()
	{
		StandUp();

		// hack
		if (MaxRange < MinRange+10)
			MaxRange = MinRange+10;
		bCanFire      = false;
		bFacingTarget = false;

		SwitchToBestWeapon();

		//EnemyLastSeen = 0;
		BlockReactions();
		bCanConverse = False;
		bAttacking = True;
		bStasis = False;
		SetDistress(true);

		CrouchTimer = 0;
		EnableCheckDestLoc(false);
	}

	function EndState()
	{
		EnableCheckDestLoc(false);
		bCanFire      = false;
		bFacingTarget = false;

		ResetReactions();
		bCanConverse = True;
		bAttacking = False;
		bStasis = True;
		bReadyToReload = false;

		EndCrouch();
	}

Begin:
	if (Enemy == None)
		GotoState('Seeking');
	//EnemyLastSeen = 0;
	CheckAttack(false);

Surprise:
	if ((1.0-ReactionLevel)*SurprisePeriod < 0.25)
		Goto('BeginAttack');
	Acceleration=vect(0,0,0);
	PlaySurpriseSound();
	PlayWaiting();
	while (ReactionLevel < 1.0)
	{
		TurnToward(Enemy);
		Sleep(0);
	}

BeginAttack:
	EnemyReadiness = 1.0;
	ReactionLevel  = 1.0;
	if (PlayerAgitationTimer > 0)
		PlayAllianceHostileSound();
	else
		PlayTargetAcquiredSound();
	if (PlayBeginAttack())
	{
		Acceleration = vect(0,0,0);
		TurnToward(enemy);
		FinishAnim();
	}

RunToRange:
	bCanFire       = false;
	bFacingTarget  = false;
	bReadyToReload = false;
	EndCrouch();
	if (Physics == PHYS_Falling)
		TweenToRunning(0.05);
	WaitForLanding();
	if (!IsWeaponReloading() || bCrouching)
	{
		if (ShouldPlayTurn(Enemy.Location))
			PlayTurning();
		TurnToward(enemy);
	}
	else
		Sleep(0);
	bCanFire = true;
	while (PickDestination() == DEST_NewLocation)
	{
		if (bCanStrafe && ShouldStrafe())
		{
			PlayRunningAndFiring();
			if (destPoint != None)
				StrafeFacing(destPoint.Location, enemy);
			else
				StrafeFacing(destLoc, enemy);
			bFacingTarget = true;
		}
		else
		{
			bFacingTarget = false;
			PlayRunning();
			if (destPoint != None)
				MoveToward(destPoint, MaxDesiredSpeed);
			else
				MoveTo(destLoc, MaxDesiredSpeed);
		}
		CheckAttack(true);
	}

Fire:
	bCanFire      = false;
	bFacingTarget = false;
	Acceleration = vect(0, 0, 0);

	SwitchToBestWeapon();
	if (FRand() > 0.5)
		bUseSecondaryAttack = true;
	else
		bUseSecondaryAttack = false;
	if (IsHandToHand())
		TweenToAttack(0.15);
	else if (ShouldCrouch() && (FRand() < CrouchRate))
	{
		TweenToCrouchShoot(0.15);
		FinishAnim();
		StartCrouch();
	}
	else
		TweenToShoot(0.15);
	if (!IsWeaponReloading() || bCrouching)
		TurnToward(enemy);
	FinishAnim();
	bReadyToReload = true;

ContinueFire:
	while (!ReadyForWeapon())
	{
		if (PickDestination() != DEST_SameLocation)
			Goto('RunToRange');
		CheckAttack(true);
		if (!IsWeaponReloading() || bCrouching)
			TurnToward(enemy);
		else
			Sleep(0);
	}
	CheckAttack(true);
	if (!FireIfClearShot())
		Goto('ContinueAttack');
	bReadyToReload = false;
	if (bCrouching)
		PlayCrouchShoot();
	else if (IsHandToHand())
		PlayAttack();
	else
		PlayShoot();
	FinishAnim();
	if (FRand() > 0.5)
		bUseSecondaryAttack = true;
	else
		bUseSecondaryAttack = false;
	bReadyToReload = true;
	if (!IsHandToHand())
	{
		if (bCrouching)
			TweenToCrouchShoot(0);
		else
			TweenToShoot(0);
	}
	CheckAttack(true);
	if (PickDestination() != DEST_NewLocation)
	{
		if (!IsWeaponReloading() || bCrouching)
			TurnToward(enemy);
		else
			Sleep(0);
		Goto('ContinueFire');
	}
	Goto('RunToRange');

ContinueAttack:
ContinueFromDoor:
	CheckAttack(true);
	if (PickDestination() != DEST_NewLocation)
		Goto('Fire');
	else
		Goto('RunToRange');

}

// ----------------------------------------------------------------------------
// state Alerting
//
// Warn other NPCs that an enemy is about
// ----------------------------------------------------------------------------

State Alerting
{
	function SetFall()
	{
		StartFalling('Alerting', 'ContinueAlert');
	}

	function Tick(float deltaSeconds)
	{
		Global.Tick(deltaSeconds);
	}

	function HitWall(vector HitNormal, actor Wall)
	{
		if (Physics == PHYS_Falling)
			return;
		Global.HitWall(HitNormal, Wall);
		CheckOpenDoor(HitNormal, Wall);
	}

	function Bump(actor bumper)
	{
		if (bAcceptBump)
		{
			if (bumper == HXAlarmActor)
			{
				bAcceptBump = False;
				GotoState('Alerting', 'SoundAlarm');
			}
		}

		// Handle conversations, if need be
		Global.Bump(bumper);
	}

	function bool IsAlarmReady(Actor actorAlarm)
	{
		local bool      bReady;
		local HXAlarmUnit alarm;

		bReady = false;
		alarm = HXAlarmUnit(actorAlarm);
		if ((alarm != None) && !alarm.bDeleteMe)
			if (!alarm.bActive)
				if ((alarm.associatedPawn == None) ||
				    (alarm.associatedPawn == self))
					bReady = true;

		return bReady;
	}

	function TriggerAlarm()
	{
		if ((HXAlarmActor != None) && !HXAlarmActor.bDeleteMe)
		{
			if (HXAlarmActor.hackStrength > 0)  // make sure the alarm hasn't been hacked
				HXAlarmActor.Trigger(self, Enemy);
		}
	}

	function bool IsHXAlarmInRange(HXAlarmUnit alarm)
	{
		local bool bInRange;

		bInRange = false;
		if ((alarm != None) && !alarm.bDeleteMe)
			if ((VSize((alarm.Location-Location)*vect(1,1,0)) <
			     (CollisionRadius+alarm.CollisionRadius+24)) &&
			    (Abs(alarm.Location.Z-Location.Z) < (CollisionHeight+alarm.CollisionHeight)))
				bInRange = true;

		return (bInRange);
	}

	function vector FindAlarmPosition(Actor alarm)
	{
		local vector alarmPos;

		alarmPos = alarm.Location;
		alarmPos += vector(alarm.Rotation.Yaw*rot(0,1,0))*(CollisionRadius+alarm.CollisionRadius);

		return (alarmPos);
	}

	function bool GetNextHXAlarmPoint(HXAlarmUnit alarm)
	{
		local vector alarmPoint;
		local bool   bValid;

		destPoint = None;
		destLoc   = vect(0,0,0);
		bValid    = false;

		if ((alarm != None) && !alarm.bDeleteMe)
		{
			alarmPoint = FindAlarmPosition(alarm);
			if (PointReachable(alarmPoint))
			{
				destLoc = alarmPoint;
				bValid = true;
			}
			else
			{
				MoveTarget = FindPathTo(alarmPoint);
				if (MoveTarget != None)
				{
					destPoint = MoveTarget;
					bValid = true;
				}
			}
		}

		return (bValid);
	}

	function HXAlarmUnit FindHXTarget()
	{
		local ScriptedPawn pawnAlly;
		local HXAlarmUnit    alarm;
		local float        dist;
		local HXAlarmUnit    bestAlarm;
		local float        bestDist;

		bestAlarm = None;

		// Do we have any allies on this level?
		foreach AllActors(Class'ScriptedPawn', pawnAlly)
			if (GetPawnAllianceType(pawnAlly) == ALLIANCE_Friendly)
				break;

		// Yes, so look for an alarm box that isn't active...
		if (pawnAlly != None)
		{
			foreach RadiusActors(Class'HXAlarmUnit', alarm, 2400)
			{
				if (GetAllianceType(alarm.Alliance) != ALLIANCE_Hostile)
				{
					dist = VSize((Location-alarm.Location)*vect(1,1,2));  // use squished sphere
					if ((bestAlarm == None) || (dist < bestDist))
					{
						bestAlarm = alarm;
						bestDist  = dist;
					}
				}
			}

			// Is the nearest alarm already going off?  And can we reach it?
			if (!IsAlarmReady(bestAlarm) || !GetNextHXAlarmPoint(bestAlarm))
				bestAlarm = None;
		}

		// Return our target alarm box
		return (bestAlarm);
	}

	function bool PickDestination()
	{
		local bool      bDest;
		local HXAlarmUnit alarm;

		// Init
		destPoint = None;
		destLoc   = vect(0, 0, 0);
		bDest     = false;

		// Find an alarm we can trigger
		alarm = FindHXTarget();
		if (alarm != None)
		{
			// Find a way to get there
			HXAlarmActor = alarm;
			alarm.associatedPawn = self;
			bDest = true;  // if alarm != none, we've already computed the route to the alarm
		}

		// Return TRUE if we were successful
		return (bDest);
	}

	function BeginState()
	{
		StandUp();
		//Disable('AnimEnd');
		bAcceptBump = False;
		bCanConverse = False;
		HXAlarmActor = None;
		bStasis = False;
		BlockReactions();
		SetupWeapon(false);
		SetDistress(false);
		EnemyReadiness = 1.0;
		ReactionLevel  = 1.0;
		EnableCheckDestLoc(false);
	}

	function EndState()
	{
		EnableCheckDestLoc(false);
		ResetReactions();
		bAcceptBump = False;
		//Enable('AnimEnd');
		bCanConverse = True;
		if (HXAlarmActor != None)
			if (HXAlarmActor.associatedPawn == self)
				HXAlarmActor.associatedPawn = None;
		HXAlarmActor = None;
		bStasis = True;
	}

Begin:
	if (Enemy == None)
		GotoState('Seeking');
	//EnemyLastSeen = 0;
	destPoint = None;
	if (RaiseAlarm == RAISEALARM_Never)
		GotoState('Fleeing');
	if (AlarmTimer > 0)
		PlayGoingForAlarmSound();

Alert:
	if (AlarmTimer > 0)
		Goto('Done');

	WaitForLanding();
	if (!PickDestination())
		Goto('Done');

Moving:
	// Can we go somewhere?
	bAcceptBump = True;
	EnableCheckDestLoc(true);
	while (true)
	{
		if (destPoint != None)
		{
			if (ShouldPlayWalk(MoveTarget.Location))
				PlayRunning();
			MoveToward(MoveTarget, MaxDesiredSpeed);
			CheckDestLoc(MoveTarget.Location, true);
		}
		else
		{
			if (ShouldPlayWalk(destLoc))
				PlayRunning();
			MoveTo(destLoc, MaxDesiredSpeed);
			CheckDestLoc(destLoc);
		}
		if (IsHXAlarmInRange(HXAlarmActor))
			break;
		else if (!GetNextHXAlarmPoint(HXAlarmActor))
			break;
	}
	EnableCheckDestLoc(false);

SoundAlarm:
	Acceleration=vect(0,0,0);
	bAcceptBump = False;
	if (IsHXAlarmInRange(HXAlarmActor))
	{
		TurnToward(HXAlarmActor);
		PlayPushing();
		FinishAnim();
		TriggerAlarm();
	}

Done:
	bAcceptBump = False;
	if (RaiseAlarm == RAISEALARM_BeforeAttacking)
		GotoState('Attacking');
	else
		GotoState('Fleeing');

ContinueAlert:
ContinueFromDoor:
	Goto('Alert');

}

// ----------------------------------------------------------------------------
// IsSeatValid()
// ----------------------------------------------------------------------------

function bool IsSeatValid( Actor CheckActor )
{
	local HXDecoration CheckDecoration;

	if ( CheckActor==None || CheckActor.bDeleteMe )
		return False;

	CheckDecoration = HXDecoration(CheckActor);
	if ( CheckDecoration==None || !CheckDecoration.bIsSeat )
		return False;

	if ( !bSitAnywhere && VSize(CheckDecoration.Location-CheckDecoration.InitialPosition)> 70 )
		return False;

	//if (GetPlayerPawn().CarriedDecoration==CheckDecoration )
		//return false;
	if ( CheckDecoration.Base!=None && CheckDecoration.Base.IsA('PlayerPawn') )
		return false;

	return true;
}

// ----------------------------------------------------------------------------
// InstigatorToPawn()
// ----------------------------------------------------------------------------

function Pawn InstigatorToPawn(Actor eventActor)
{
	local Pawn pawnActor;

	if (Inventory(eventActor) != None)
	{
		if (Inventory(eventActor).Owner != None)
			eventActor = Inventory(eventActor).Owner;
	}
	else if (DeusExDecoration(eventActor) != None)
		eventActor = GetPlayerPawn();
	else if (HXProjectile(eventActor) != None)
		eventActor = eventActor.Instigator;

	pawnActor = Pawn(eventActor);
	if (pawnActor == self)
		pawnActor = None;

	return pawnActor;

}

// ----------------------------------------------------------------------------
// FindTaggedActor()
// ----------------------------------------------------------------------------

function Actor FindTaggedActor(Name actorTag, optional bool bRandom, optional Class<Actor> tagClass)
{
	local Actor TempActor, BestActor;
	local float Dist, BestDist;
	local int i;

	BestActor = None;
	BestDist  = 1000000;

	if ( TagClass==None )
		TagClass = Class'Actor';

	// If no tag, then assume the a player is the target.
	if ( ActorTag=='' )
	{
		// Use the +/- 30% for players if bRandom.
		if ( bRandom )
		{
			// Just try first 4 players.
			for ( i=0; i<4; i++ )
			{
				if ( PlayerDistances[i].Player!=None )
				{
					Dist = PlayerDistances[i].Distance * (FRand()*0.6+0.7);
					if ( BestActor==None || Dist<BestDist )
					{
						BestActor = PlayerDistances[i].Player;
						BestDist  = Dist;
					}
				}
			}
		}
		else
		{
			BestActor = PlayerDistances[0].Player;
		}
	}
	else
	{
		foreach AllActors( TagClass, TempActor, ActorTag )
		{
			if ( TempActor!=Self )
			{
				Dist = VSize(TempActor.Location - Location);
				if ( bRandom )
					Dist *= FRand()*0.6+0.7;  // +/- 30%
				if ( BestActor==None || Dist<BestDist )
				{
					BestActor = TempActor;
					BestDist  = Dist;
				}
			}
		}
	}
	return BestActor;
}

// ----------------------------------------------------------------------------
// CheckBeamPresence
// ----------------------------------------------------------------------------

function bool CheckBeamPresence(float deltaSeconds)
{
	local DeusExPlayer player;
	local Beam         beamActor;
	local bool         bReactToBeam;

	if (bReactPresence && bLookingForEnemy && (BeamCheckTimer <= 0) && (LastRendered() < 5.0))
	{
		BeamCheckTimer = 1.0;
		player = DeusExPlayer(GetPlayerPawn());
		if (player != None)
		{
			bReactToBeam = false;
			if (IsValidEnemy(player))
			{
				foreach RadiusActors(Class'Beam', beamActor, 1200)
				{
					if ((beamActor.Owner == player) && (beamActor.LightType != LT_None) && (beamActor.LightBrightness > 32))
					{
						if (VSize(beamActor.Location - Location) < (beamActor.LightRadius+1)*25)
							bReactToBeam = true;
						else
						{
							if (AICanSee(beamActor, , false, true, false, false) > 0)
							{
								if (FastTrace(beamActor.Location, Location+vect(0,0,1)*BaseEyeHeight))
									bReactToBeam = true;
							}
						}
					}
					if (bReactToBeam)
						break;
				}
			}
			if (bReactToBeam)
				HandleSighting(player);
		}
	}
}

// ----------------------------------------------------------------------------
// CheckCarcassPresence()
// ----------------------------------------------------------------------------

function bool CheckCarcassPresence(float deltaSeconds)
{
	local Actor         carcass;
	local Name          CarcassName;
	local int           lastCycle;
	local HXCarcass body;
	local DeusExPlayer  player;
	local float         visibility;
	local Name          KillerAlliance;
	local Name          killedAlliance;
	local Pawn          killer;
	local Pawn          bestKiller;
	local float         dist;
	local float         bestDist;
	local float         maxCarcassDist;
	local int           maxCarcassCount;

	if (bFearCarcass && !bHateCarcass && !bReactCarcass)  // Major hack!
		maxCarcassCount = 1;
	else
		maxCarcassCount = ArrayCount(Carcasses);

	//if ((bHateCarcass || bReactCarcass || bFearCarcass) && bLookingForCarcass && (CarcassTimer <= 0))
	if ((bHateCarcass || bReactCarcass || bFearCarcass) && (NumCarcasses < maxCarcassCount))
	{
		maxCarcassDist = 1200;
		if (CarcassCheckTimer <= 0)
		{
			CarcassCheckTimer = 0.1;
			carcass           = None;
			lastCycle         = BodyIndex;
			foreach CycleActors(Class'HXCarcass', body, BodyIndex)
			{
				if (body.Physics != PHYS_Falling)
				{
					if (VSize(body.Location-Location) < maxCarcassDist)
					{
						if (GetCarcassData(body, KillerAlliance, killedAlliance, CarcassName, true))
						{
							visibility = AICanSee(body, ComputeActorVisibility(body), true, true, true, true);
							if (visibility > 0)
								carcass = body;
							break;
						}
					}
				}
			}
			if (lastCycle >= BodyIndex)
			{
				if (carcass == None)
				{
					player = DeusExPlayer(GetPlayerPawn());
					if (player != None)
					{
						if (VSize(player.Location-Location) < maxCarcassDist)
						{
							if (GetCarcassData(player, KillerAlliance, killedAlliance, CarcassName, true))
							{
								visibility = AICanSee(player, ComputeActorVisibility(player), true, true, true, true);
								if (visibility > 0)
									carcass = player;
							}
						}
					}
				}
			}
			if (carcass != None)
			{
				CarcassTimer = 120;
				AddCarcass(CarcassName);
				if (bLookingForCarcass)
				{
					if (KillerAlliance == 'Player')
						killer = GetPlayerPawn();
					else
					{
						bestKiller = None;
						bestDist   = 0;
						foreach AllActors(Class'Pawn', killer)  // hack
						{
							if (killer.Alliance == KillerAlliance)
							{
								dist = VSize(killer.Location - Location);
								if ((bestKiller == None) || (bestDist > dist))
								{
									bestKiller = killer;
									bestDist   = dist;
								}
							}
						}
						killer = bestKiller;
					}
					if (bHateCarcass)
					{
						PotentialEnemyAlliance = KillerAlliance;
						PotentialEnemyTimer    = 15.0;
						bNoNegativeAlliances   = false;
					}
					if (bFearCarcass)
						IncreaseFear(killer, 2.0);

					if (bFearCarcass && IsFearful() && !IsValidEnemy(killer))
					{
						SetDistressTimer();
						SetEnemy(killer, , true);
						GotoState('Fleeing');
					}
					else
					{
						SetDistressTimer();
						SetSeekLocation(killer, carcass.Location, SEEKTYPE_Carcass);
						HandleEnemy();
					}
				}
			}
		}
	}

}

// ----------------------------------------------------------------------------
// HandleHacking()
// ----------------------------------------------------------------------------

function HandleHacking(Name event, EAIEventState state, XAIParams params)
{
	// Fear, Hate

	local Pawn pawnActor;

	if (state == EAISTATE_Begin || state == EAISTATE_Pulse)
	{
		//pawnActor = GetPlayerPawn();

		if ( params.BestActor != None )
		{
			if ( params.BestActor.IsA('HXMover') )
				pawnActor = HXMover(params.BestActor).pickPlayer;
			else if ( params.BestActor.IsA('HXHackableDevices') )
				pawnActor = HXHackableDevices(params.BestActor).hackInstigator;		
		}

		if (pawnActor != None)
		{
			if (bHateHacking)
				IncreaseAgitation(pawnActor, 1.0);
			if (bFearHacking)
				IncreaseFear(pawnActor, 0.51);
			if (SetEnemy(pawnActor))
			{
				SetDistressTimer();
				HandleEnemy();
			}
			else if (bFearHacking && IsFearful())
			{
				SetDistressTimer();
				SetEnemy(pawnActor, , true);
				GotoState('Fleeing');
			}
			else  // only players can hack
				ReactToFutz();
		}
	}
}

// ----------------------------------------------------------------------------
// DecreaseAgitation()
// ----------------------------------------------------------------------------

function DecreaseAgitation(Actor actorInstigator, float AgitationLevel)
{
	local float        newLevel;
	local DeusExPlayer player;
	local Pawn         instigator;

	player = DeusExPlayer(GetPlayerPawn());

	if (Inventory(actorInstigator) != None)
	{
		if (Inventory(actorInstigator).Owner != None)
			actorInstigator = Inventory(actorInstigator).Owner;
	}
	else if (DeusExDecoration(actorInstigator) != None)
		actorInstigator = player;

	instigator = Pawn(actorInstigator);
	if ((instigator == None) || (instigator == self))
		return;

	AgitationTimer  = AgitationSustainTime;
	if (AgitationLevel > 0)
	{
		bAlliancesChanged    = True;
		bNoNegativeAlliances = False;
		AgitateAlliance(instigator.Alliance, -AgitationLevel);
	}

}



// ----------------------------------------------------------------------------
// state Conversation
//
// Just sit here until the conversation is over
// ----------------------------------------------------------------------------

state Conversation
{
	function Tick(float deltaTime)
	{
		Global.Tick(deltaTime);

		LipSynch(deltaTime);

		// Keep turning towards the person we're speaking to
		if (ConversationActor != None)
		{
			if (bSitting)
			{
				if (HXSeatActor != None)
					LookAtActor(ConversationActor, true, true, true, 0, 0.5, HXSeatActor.Rotation.Yaw+49152, 5461);
				else
					LookAtActor(ConversationActor, false, true, true, 0, 0.5);
			}
			else
				LookAtActor(ConversationActor, true, true, true, 0, 0.5);
		}
	}

	function LoopHeadConvoAnim()
	{
	}

	function SetOrders( Name orderName, optional Name newOrderTag, optional bool bImmediate )
	{
		local DeusExPlayer dxPlayer;

		if ( HXGameInfo(Level.Game)!=None )
			if ( HXGameInfo(Level.Game).ConPlay!=None )
				if ( HXGameInfo(Level.Game).ConPlay.GetForcePlay() )
				{
					Global.SetOrders( OrderName, newOrderTag, bImmediate );
					return;
				}

		ConvOrders   = OrderName;
		ConvOrderTag = NewOrderTag;
	}

	function FollowOrders(optional bool bDefer)
	{
		local name tempConvOrders, tempConvOrderTag;

		// hack
		tempConvOrders   = ConvOrders;
		tempConvOrderTag = ConvOrderTag;
		ResetConvOrders();  // must do this before calling SetOrders(), or recursion will result

		if (tempConvOrders != '')
			Global.SetOrders(tempConvOrders, tempConvOrderTag, true);
		else
			Global.FollowOrders(bDefer);
	}

	function BeginState()
	{
		local DeusExPlayer dxPlayer;
		local bool         bBlock;

		ResetConvOrders();
		EnableCheckDestLoc(false);

		bBlock = True;
		if ( HXGameInfo(Level.Game)!=None )
			if ( HXGameInfo(Level.Game).ConPlay!=None )
				if ( HXGameInfo(Level.Game).ConPlay.CanInterrupt() )
					bBlock = False;

		bInterruptState = True;
		if (bBlock)
		{
			bCanConverse = False;
			MakePawnIgnored(true);
			BlockReactions(true);
		}
		else
		{
			bCanConverse = True;
			MakePawnIgnored(true);
			BlockReactions(false);
		}

		// Check if the current state is "WaitingFor", "RunningTo" or "GoingTo", in which case
		// we want the orders to be 'Standing' after the conversation is over.  UNLESS the 
		// ScriptedPawn was going somewhere else (OrderTag != '')

		if (((Orders == 'WaitingFor') || (Orders == 'RunningTo') || (Orders == 'GoingTo')) && (OrderTag == ''))
			SetOrders('Standing');

		bConversationEndedNormally = False;
		bInConversation = True;
		bStasis = False;
		SetDistress(false);
		SeekPawn = None;
	}

	function EndState()
	{
		local DeusExPlayer player;
		local bool         bForcePlay;

		bForcePlay = false;
		if ( HXGameInfo(Level.Game)!=None )
			if ( HXGameInfo(Level.Game).ConPlay!=None )
				bForcePlay = HXGameInfo(Level.Game).conPlay.GetForcePlay();

		bConvEndState = true;
		if ( !bForcePlay && !bConversationEndedNormally )
			HXGameInfo(Level.Game).AbortConversation();
		bConvEndState = false;
		ResetConvOrders();

		StopBlendAnims();
		bInterruptState = true;
		bCanConverse    = True;
		MakePawnIgnored(false);
		ResetReactions();
		bStasis = True;
		ConversationActor = None;
	}

Begin:

	Acceleration = vect(0,0,0);

	DesiredRotation.Pitch = 0;
	if (!bSitting && !bDancing)
		PlayWaiting();

	// we are now idle
}

// ----------------------------------------------------------------------------
// state FirstPersonConversation
//
// Just sit here until the conversation is over
// ----------------------------------------------------------------------------

state FirstPersonConversation
{
	function Tick(float deltaTime)
	{
		Global.Tick(deltaTime);

		LipSynch(deltaTime);

		// Keep turning towards the person we're speaking to
		if (ConversationActor != None)
		{
			if (bSitting)
			{
				if (HXSeatActor != None)
					LookAtActor(ConversationActor, true, true, true, 0, 1.0, HXSeatActor.Rotation.Yaw+49152, 5461);
				else
					LookAtActor(ConversationActor, false, true, true, 0, 1.0);
			}
			else
				LookAtActor(ConversationActor, true, true, true, 0, 1.0);
		}
	}

	function LoopHeadConvoAnim()
	{
	}

	function SetOrders(Name orderName, optional Name newOrderTag, optional bool bImmediate)
	{
		ConvOrders   = orderName;
		ConvOrderTag = newOrderTag;
	}

	function FollowOrders(optional bool bDefer)
	{
		local name tempConvOrders, tempConvOrderTag;

		// hack
		tempConvOrders   = ConvOrders;
		tempConvOrderTag = ConvOrderTag;
		ResetConvOrders();  // must do this before calling SetOrders(), or recursion will result

		if (tempConvOrders != '')
			Global.SetOrders(tempConvOrders, tempConvOrderTag, true);
		else
			Global.FollowOrders(bDefer);
	}

	function BeginState()
	{
		local DeusExPlayer dxPlayer;
		local bool         bBlock;

		ResetConvOrders();
		EnableCheckDestLoc(false);

/*
		if ( HXGameInfo(Level.Game)!=None )
			if ( HXGameInfo(Level.Game).ConPlay!=None )
				if ( HXGameInfo(Level.Game).ConPlay.CanInterrupt() )
					bBlock = False;
*/

		// 1st-person conversations will no longer block;
		// left old code in here in case people change their minds :)

		bBlock = false;

		bInterruptState = True;
		if (bBlock)
		{
			bCanConverse = False;
			MakePawnIgnored(true);
			BlockReactions(true);
		}
		else
		{
			bCanConverse = True;
			MakePawnIgnored(false);
			if ( HXGameInfo(Level.Game)!=None && HXGameInfo(Level.Game).ConPlay!=None && HXGameInfo(Level.Game).IsSpeakingActorByBindName(HXGameInfo(Level.Game).ConPlay.Con,"JCDenton") )
				SetReactions(false, false, false, false, true, false, false, true, false, false, true, true);
			else
				ResetReactions();
		}

		bConversationEndedNormally = False;
		bInConversation = True;
		bStasis = False;
		SetDistress(false);
		SeekPawn = None;

		//bClientDoLipSynch = True;
	}

	function EndState()
	{
		local HXGameInfo Game;

		bConvEndState = true;
		if ( !bConversationEndedNormally )
		{
			Game = HXGameInfo(Level.Game);
			if ( Game!=None )
				Game.AbortConversation();
		}
		bConvEndState = false;
		ResetConvOrders();

		StopBlendAnims();
		bInterruptState = true;
		bCanConverse    = True;
		MakePawnIgnored(false);
		ResetReactions();
		bStasis = True;
		ConversationActor = None;

		//bClientDoLipSynch = False;
	}

Begin:

	Acceleration = vect(0,0,0);

	DesiredRotation.Pitch = 0;
	if (!bSitting && !bDancing)
		PlayWaiting();

	// we are now idle
}

// ----------------------------------------------------------------------------
// state RunningTo
//
// Move to an actor really fast.
// ----------------------------------------------------------------------------

state RunningTo
{
	function SetFall()
	{
		StartFalling('RunningTo', 'ContinueRun');
	}

	function HitWall(vector HitNormal, actor Wall)
	{
		if (Physics == PHYS_Falling)
			return;
		Global.HitWall(HitNormal, Wall);
		CheckOpenDoor(HitNormal, Wall);
	}

	function Bump(actor bumper)
	{
		// If we hit the guy we're going to, end the state
		if (bumper == OrderActor)
			GotoState('RunningTo', 'Done');

		// Handle conversations, if need be
		Global.Bump(bumper);
	}

	function Touch(actor toucher)
	{
		// If we hit the guy we're going to, end the state
		if (toucher == OrderActor)
			GotoState('RunningTo', 'Done');

		// Handle conversations, if need be
		Global.Touch(toucher);
	}

	function BeginState()
	{
		StandUp();
		//BlockReactions();
		SetupWeapon(false);
		SetDistress(false);
		bStasis = False;
		SeekPawn = None;
		EnableCheckDestLoc(true);
	}
	function EndState()
	{
		EnableCheckDestLoc(false);
		//ResetReactions();
		bStasis = True;
	}

	function AnimEnd()
	{
		PlayRunning2();
	}

Begin:
	Acceleration = vect(0, 0, 0);
	if (orderActor == None)
		Goto('Done');

	PlayRunning2();

Follow:
	if (IsOverlapping(orderActor))
		Goto('Done');
	MoveTarget = GetNextWaypoint(orderActor);
	if ((MoveTarget != None) && (!MoveTarget.Region.Zone.bWaterZone) &&
	    (MoveTarget.Physics != PHYS_Falling))
	{
		if ((MoveTarget == orderActor) && MoveTarget.IsA('Pawn'))
		{
			if (GetNextVector(orderActor, useLoc))
			{
				if (ShouldPlayWalk(useLoc))
				{
					//PlayRunning();
				}
				MoveTo(useLoc, MaxDesiredSpeed);
				CheckDestLoc(useLoc);
			}
			else
				Goto('Pause');
		}
		else
		{
			if (ShouldPlayWalk(MoveTarget.Location))
			{
				//PlayRunning();
			}
			MoveToward(MoveTarget, MaxDesiredSpeed);
			CheckDestLoc(MoveTarget.Location, true);
		}
		if (IsOverlapping(orderActor))
			Goto('Done');
		else
			Goto('Follow');
	}

Pause:
	Acceleration = vect(0, 0, 0);
	TurnToward(orderActor);
	PlayWaiting();
	Sleep(1.0);
	Goto('Follow');

Done:
	if (orderActor.IsA('PatrolPoint'))
		TurnTo(Location + PatrolPoint(orderActor).lookdir);
	GotoState('Standing');

ContinueRun:
ContinueFromDoor:
	PlayRunning();
	Goto('Follow');
}

// ----------------------------------------------------------------------------
// state Patrolling
//
// Move from point to point in a predescribed pattern.
// ----------------------------------------------------------------------------

State Patrolling
{
	function SetFall()
	{
		StartFalling('Patrolling', 'ContinuePatrol');
	}

	function HitWall(vector HitNormal, actor Wall)
	{
		if (Physics == PHYS_Falling)
			return;
		Global.HitWall(HitNormal, Wall);
		CheckOpenDoor(HitNormal, Wall);
	}
	
	function AnimEnd()
	{
		PlayWaiting();
	}

	function PatrolPoint PickStartPoint()
	{
		local NavigationPoint nav;
		local PatrolPoint     curNav;
		local float           curDist;
		local PatrolPoint     closestNav;
		local float           closestDist;

		//Log( Self $ ".Patrolling.PickStartPoint()" );

		nav = Level.NavigationPointList;
		while (nav != None)
		{
			nav.visitedWeight = 0;
			nav = nav.nextNavigationPoint;
		}

		closestNav  = None;
		closestDist = 100000;
		nav = Level.NavigationPointList;
		while (nav != None)
		{
			curNav = PatrolPoint(nav);
			if ((curNav != None) && (curNav.Tag == OrderTag))
			{
				while (curNav != None)
				{
					if (curNav.visitedWeight != 0)  // been here before
						break;
					curDist = VSize(Location - curNav.Location);
					if ((closestNav == None) || (closestDist > curDist))
					{
						closestNav  = curNav;
						closestDist = curDist;
					}
					curNav.visitedWeight = 1;
					curNav = curNav.NextPatrolPoint;
				}
			}
			nav = nav.nextNavigationPoint;
		}

		return (closestNav);
	}

	function PickDestination()
	{
		//Log( Self $ ".Patrolling.PickDestination()" );

		if (PatrolPoint(destPoint) != None)
			destPoint = PatrolPoint(destPoint).NextPatrolPoint;
		else
			destPoint = PickStartPoint();
		if (destPoint == None)  // can't go anywhere...
			GotoState('Standing');

		//Log( "(destPoint=" $ destPoint $ ")" );
	}

	function BeginState()
	{
		//Log( Self $ ".Patrolling.BeginState()" );

		StandUp();
		SetEnemy(None, EnemyLastSeen, true);
		Disable('AnimEnd');
		SetupWeapon(false);
		SetDistress(false);
		bStasis = false;
		SeekPawn = None;
		EnableCheckDestLoc(false);
	}

	function EndState()
	{
		//Log( Self $ ".Patrolling.EndState()" );

		EnableCheckDestLoc(false);
		Enable('AnimEnd');
		bStasis = true;
	}

Begin:
	//Log( Self $ ".Patrolling.Begin" );
	destPoint = None;

Patrol:
	//Log( Self $ ".Patrolling.Patrol" );
	//Disable('Bump');
	WaitForLanding();
	PickDestination();

Moving:
	//Log( Self $ ".Patrolling.Moving" );
	// Move from pathnode to pathnode until we get where we're going
	if (destPoint != None)
	{
		if (!IsPointInCylinder(self, destPoint.Location, 16-CollisionRadius))
		{
			//Log( "(InCylinder)" );

			EnableCheckDestLoc(true);
			//native(517) final function Actor FindPathToward(actor anActor, optional bool bSinglePath=False, optional bool bClearPaths=True);
			MoveTarget = FindPathToward(destPoint);

			// HX_HAN: Experimental workaround for runaway loop with Teleportes and PatrolPoints.
			if ( MoveTarget==None )
			{
				if (ShouldPlayWalk(destPoint.Location))
					PlayWalking();
				MoveToward(destPoint, GetWalkingSpeed());	
			}
			else
			{
				while (MoveTarget != None)
				{
					//Log( "(MoveTarget=" $ MoveTarget $ ")" );

					if (ShouldPlayWalk(MoveTarget.Location))
						PlayWalking();
					MoveToward(MoveTarget, GetWalkingSpeed());
					CheckDestLoc(MoveTarget.Location, true);
					if (MoveTarget == destPoint)
						break;
					MoveTarget = FindPathToward(destPoint);
				}
			}
			EnableCheckDestLoc(false);
		}
		//else
			//Log( "(NotInCylinder)" );
	}
	else
		Goto('Patrol');

Pausing:
	//Log( Self $ ".Patrolling.Pausing" );
	if (!bAlwaysPatrol)
		bStasis = true;
	Acceleration = vect(0, 0, 0);

	// Turn in the direction dictated by the WanderPoint, or a random direction
	if (PatrolPoint(destPoint) != None)
	{
		if ((PatrolPoint(destPoint).pausetime > 0) || (PatrolPoint(destPoint).NextPatrolPoint == None))
		{
			if (ShouldPlayTurn(Location + PatrolPoint(destPoint).lookdir))
				PlayTurning();
			TurnTo(Location + PatrolPoint(destPoint).lookdir);
			Enable('AnimEnd');
			TweenToWaiting(0.2);
			PlayScanningSound();
			//Enable('Bump');
			sleepTime = PatrolPoint(destPoint).pausetime * ((-0.9*restlessness) + 1);
			Sleep(sleepTime);
			Disable('AnimEnd');
			//Disable('Bump');
			FinishAnim();
		}
	}
	Goto('Patrol');

ContinuePatrol:
	//Log( Self $ ".Patrolling.ContinuePatrol" );
ContinueFromDoor:
	//Log( Self $ ".Patrolling.ContinueFromDoor" );
	FinishAnim();
	PlayWalking();
	Goto('Moving');
}

//
// lip synching support - DEUS_EX CNN
//
simulated function ClientLipSynch(float deltaTime)
{
	local name animseq;
	local float rnd;
	local float tweentime;

	//Log( "ClientLipSynch()" );

	// update the animation timers that we are using
	animTimer[0] += deltaTime;
	animTimer[1] += deltaTime;
	animTimer[2] += deltaTime;

	if ( bIsSpeaking )
	{
		// Make sure we close our mouth.
		bWasSpeaking = True;

		// if our framerate is high enough (>20fps), tween the lips smoothly
		if (Level.TimeSeconds - animTimer[3]  < 0.05)
			tweentime = 0.1;
		else
			tweentime = 0.0;

		// the last animTimer slot is used to check framerate
		animTimer[3] = Level.TimeSeconds;

		//Log( Self $ ".nextPhoneme = " $ nextPhoneme );
		if ( GetPlayerPawn() != None && GetPlayerPawn().Player != None && GetPlayerPawn().Player.Console != None && GetPlayerPawn().Player.Console.IsA('HXConsole') )
			HXConsole(GetPlayerPawn().Player.Console).PhonemeNotify( Self, nextPhoneme );

		if (nextPhoneme == "A")
			animseq = 'MouthA';
		else if (nextPhoneme == "E")
			animseq = 'MouthE';
		else if (nextPhoneme == "F")
			animseq = 'MouthF';
		else if (nextPhoneme == "M")
			animseq = 'MouthM';
		else if (nextPhoneme == "O")
			animseq = 'MouthO';
		else if (nextPhoneme == "T")
			animseq = 'MouthT';
		else if (nextPhoneme == "U")
			animseq = 'MouthU';
		else if (nextPhoneme == "X")
			animseq = 'MouthClosed';

		if (animseq != '')
		{
			if (lastPhoneme != nextPhoneme)
			{
				lastPhoneme = nextPhoneme;
				TweenBlendAnim(animseq, tweentime);
			}
		}
	}
	else if ( bWasSpeaking )
	{
		bWasSpeaking = False;
		TweenBlendAnim('MouthClosed', tweentime);
	}

	// blink randomly
	if (animTimer[0] > 2.0)
	{
		animTimer[0] = 0;
		if (FRand() < 0.4)
			PlayBlendAnim('Blink', 1.0, 0.1, 1);
	}
}

//
// lip synching support - DEUS_EX CNN
//
function LipSynch(float deltaTime)
{
	local name animseq;
	local float rnd;
	local float tweentime;

	// update the animation timers that we are using
	animTimer[0] += deltaTime;
	animTimer[1] += deltaTime;
	animTimer[2] += deltaTime;

	if (bIsSpeaking)
	{
		// if our framerate is high enough (>20fps), tween the lips smoothly
		if (Level.TimeSeconds - animTimer[3]  < 0.05)
			tweentime = 0.1;
		else
			tweentime = 0.0;

		// the last animTimer slot is used to check framerate
		animTimer[3] = Level.TimeSeconds;

		if ( GetPlayerPawn() != None && GetPlayerPawn().Player != None && GetPlayerPawn().Player.Console != None && GetPlayerPawn().Player.Console.IsA('HXConsole') )
			HXConsole(GetPlayerPawn().Player.Console).PhonemeNotify( Self, nextPhoneme );

		if (nextPhoneme == "A")
			animseq = 'MouthA';
		else if (nextPhoneme == "E")
			animseq = 'MouthE';
		else if (nextPhoneme == "F")
			animseq = 'MouthF';
		else if (nextPhoneme == "M")
			animseq = 'MouthM';
		else if (nextPhoneme == "O")
			animseq = 'MouthO';
		else if (nextPhoneme == "T")
			animseq = 'MouthT';
		else if (nextPhoneme == "U")
			animseq = 'MouthU';
		else if (nextPhoneme == "X")
			animseq = 'MouthClosed';

		if (animseq != '')
		{
			if (lastPhoneme != nextPhoneme)
			{
				lastPhoneme = nextPhoneme;
				TweenBlendAnim(animseq, tweentime);
			}
		}
	}
	else if (bWasSpeaking)
	{
		bWasSpeaking = False;
		TweenBlendAnim('MouthClosed', tweentime);
	}

	// blink randomly
	if (animTimer[0] > 2.0)
	{
		animTimer[0] = 0;
		if (FRand() < 0.4)
			PlayBlendAnim('Blink', 1.0, 0.1, 1);
	}

	LoopHeadConvoAnim();
	LoopBaseConvoAnim();
}


// ----------------------------------------------------------------------------
// state StartUp
//
// Initial state
// ----------------------------------------------------------------------------

auto simulated state StartUp
{
	function BeginState()
	{
		if ( Role == ROLE_Authority )
		{
			bInterruptState = true;
			bCanConverse = false;

			SetMovementPhysics(); 
			if (Physics == PHYS_Walking)
				SetPhysics(PHYS_Falling);

			bStasis = False;
			SetDistress(false);
			BlockReactions();
			ResetDestLoc();
		}
	}

	function EndState()
	{
		if ( Role == ROLE_Authority )
		{
			bCanConverse = true;
			bStasis = True;
			ResetReactions();
		}
	}

	simulated function Tick(float deltaSeconds)
	{
		Global.Tick(deltaSeconds);

		if ( Role == ROLE_Authority && LastRendered() <= 1.0)
		{
			PlayWaiting();
			//InitializePawn();

			// HX_Note: fixes bInWorld = false && patrolling pawns
			if ( bInWorld )
				FollowOrders();
		}
	}

Begin:

	if ( Role == ROLE_Authority )
	{
		//InitializePawn();

		Sleep(FRand()+0.2);
		WaitForLanding();
	}

Start:
	// HX_Note: fixes bInWorld = false && patrolling pawns
	//if ( Role == ROLE_Authority )
	if ( Role == ROLE_Authority && bInWorld )
		FollowOrders();

	if ( Role == Role_Authority )
		Goto('End');
/*
ClientSimAction:
	LastClientSimulatedAction = ClientSimulatedAction;

	Log( LastClientSimulatedAction );

	// check if we have an simulated action, else sleep
	if ( ClientSimulatedAction == 'Stand' )
	{
		Goto('ClientSimStand');
	}
	// unsupported or empty
	else
	{
		Sleep( FRand() + 1.0 );

		Goto('ClientSimAction');
	}

ClientSimStand:
	PlayWaiting();
	//Sleep(FRand()*14+8);
	Sleep(FRand()+2);

	if ( LastClientSimulatedAction != ClientSimulatedAction )
		Goto('ClientSimAction');

	if (FRand() < 0.5)
	{
		PlayIdle();
		FinishAnim();
	}
	else
	{
		if (FRand() > 0.5)
		{
			PlayTurnHead(LOOK_Up, 1.0, 1.0);
			Sleep(2.0);

			if ( LastClientSimulatedAction != ClientSimulatedAction )
				Goto('ClientSimAction');

			PlayTurnHead(LOOK_Forward, 1.0, 1.0);
			Sleep(0.5);
		}
		else if (FRand() > 0.5)
		{
			PlayTurnHead(LOOK_Left, 1.0, 1.0);
			Sleep(1.5);

			if ( LastClientSimulatedAction != ClientSimulatedAction )
				Goto('ClientSimAction');

			PlayTurnHead(LOOK_Forward, 1.0, 1.0);
			Sleep(0.9);

			if ( LastClientSimulatedAction != ClientSimulatedAction )
				Goto('ClientSimAction');

			PlayTurnHead(LOOK_Right, 1.0, 1.0);
			Sleep(1.2);

			if ( LastClientSimulatedAction != ClientSimulatedAction )
				Goto('ClientSimAction');

			PlayTurnHead(LOOK_Forward, 1.0, 1.0);
			Sleep(0.5);
		}
		else
		{
			PlayTurnHead(LOOK_Right, 1.0, 1.0);
			Sleep(1.5);

			if ( LastClientSimulatedAction != ClientSimulatedAction )
				Goto('ClientSimAction');

			PlayTurnHead(LOOK_Forward, 1.0, 1.0);
			Sleep(0.9);

			if ( LastClientSimulatedAction != ClientSimulatedAction )
				Goto('ClientSimAction');

			PlayTurnHead(LOOK_Left, 1.0, 1.0);
			Sleep(1.2);

			if ( LastClientSimulatedAction != ClientSimulatedAction )
				Goto('ClientSimAction');

			PlayTurnHead(LOOK_Forward, 1.0, 1.0);
			Sleep(0.5);
		}
	}

	Goto('ClientSimAction');
*/
DoNothing:
End:
	// nil
}


// ----------------------------------------------------------------------------
// StandUp()
// ----------------------------------------------------------------------------

function StandUp(optional bool bInstant)
{
	local vector placeToStand;

	if (bSitting)
	{
		bSitInterpolation = false;
		bSitting          = false;

		EnableCollision(true);
		SetBase(None);
		SetPhysics(PHYS_Falling);
		ResetBasedPawnSize();

		if (!bInstant && (HXSeatActor != None) && IsOverlapping(HXSeatActor))
		{
			bStandInterpolation = true;
			remainingStandTime  = 0.3;
			StandRate = (Vector(HXSeatActor.Rotation+Rot(0, -16384, 0))*CollisionRadius) /
			            remainingStandTime;
		}
		else
			StopStanding();
	}

	if (HXSeatActor != None)
	{
		if (HXSeatActor.sittingActor[seatSlot] == self)
			HXSeatActor.sittingActor[seatSlot] = None;
		HXSeatActor = None;
	}

	if (bDancing)
		bDancing = false;
}


// ----------------------------------------------------------------------------
// state Standing
//
// Just kinda stand there and do nothing.
// (similar to Wandering, except the pawn doesn't actually move)
// ----------------------------------------------------------------------------

state Standing
{
	ignores EnemyNotVisible;

	function SetFall()
	{
		StartFalling('Standing', 'ContinueStand');
	}

	function AnimEnd()
	{
		PlayWaiting();
	}

	function HitWall(vector HitNormal, actor Wall)
	{
		if (Physics == PHYS_Falling)
			return;
		Global.HitWall(HitNormal, Wall);
		CheckOpenDoor(HitNormal, Wall);
	}

	function Tick(float deltaSeconds)
	{
		animTimer[1] += deltaSeconds;
		Global.Tick(deltaSeconds);
	}

	function BeginState()
	{
		StandUp();
		SetEnemy(None, EnemyLastSeen, true);
		Disable('AnimEnd');
		bCanJump = false;

		bStasis = False;

		SetupWeapon(false);
		SetDistress(false);
		SeekPawn = None;
		EnableCheckDestLoc(false);
	}

	function EndState()
	{
		EnableCheckDestLoc(false);
		bAcceptBump = True;

		if (JumpZ > 0)
			bCanJump = true;
		bStasis = True;

		StopBlendAnims();

		ClientSimulatedAction = '';
	}

Begin:
	WaitForLanding();
	if (!bUseHome)
		Goto('StartStand');

MoveToBase:
	if (!IsPointInCylinder(self, HomeLoc, 16-CollisionRadius))
	{
		EnableCheckDestLoc(true);
		while (true)
		{
			if (PointReachable(HomeLoc))
			{
				if (ShouldPlayWalk(HomeLoc))
					PlayWalking();
				MoveTo(HomeLoc, GetWalkingSpeed());
				CheckDestLoc(HomeLoc);
				break;
			}
			else
			{
				MoveTarget = FindPathTo(HomeLoc);
				if (MoveTarget != None)
				{
					if (ShouldPlayWalk(MoveTarget.Location))
						PlayWalking();
					MoveToward(MoveTarget, GetWalkingSpeed());
					CheckDestLoc(MoveTarget.Location, true);
				}
				else
					break;
			}
		}
		EnableCheckDestLoc(false);
	}
	TurnTo(Location+HomeRot);

StartStand:
	Acceleration=vect(0,0,0);
	Goto('Stand');

ContinueFromDoor:
	Goto('MoveToBase');

Stand:
ContinueStand:
	if (bPlayIdle)
		ClientSimulatedAction = 'Stand';

	// nil
	bStasis = True;

	PlayWaiting();
	if (!bPlayIdle)
		Goto('DoNothing');
	Sleep(FRand()*14+8);

Fidget:
	if (FRand() < 0.5)
	{
		PlayIdle();
		FinishAnim();
	}
	else
	{
		if (FRand() > 0.5)
		{
			PlayTurnHead(LOOK_Up, 1.0, 1.0);
			Sleep(2.0);
			PlayTurnHead(LOOK_Forward, 1.0, 1.0);
			Sleep(0.5);
		}
		else if (FRand() > 0.5)
		{
			PlayTurnHead(LOOK_Left, 1.0, 1.0);
			Sleep(1.5);
			PlayTurnHead(LOOK_Forward, 1.0, 1.0);
			Sleep(0.9);
			PlayTurnHead(LOOK_Right, 1.0, 1.0);
			Sleep(1.2);
			PlayTurnHead(LOOK_Forward, 1.0, 1.0);
			Sleep(0.5);
		}
		else
		{
			PlayTurnHead(LOOK_Right, 1.0, 1.0);
			Sleep(1.5);
			PlayTurnHead(LOOK_Forward, 1.0, 1.0);
			Sleep(0.9);
			PlayTurnHead(LOOK_Left, 1.0, 1.0);
			Sleep(1.2);
			PlayTurnHead(LOOK_Forward, 1.0, 1.0);
			Sleep(0.5);
		}
	}
	if (FRand() < 0.3)
		PlayIdleSound();
	Goto('Stand');

DoNothing:
	ClientSimulatedAction = '';
	// nil
}


// ----------------------------------------------------------------------------
// state Dancing
//
// Dance in place.
// (Most of this code was ripped from Standing)
// ----------------------------------------------------------------------------

state Dancing
{
	ignores EnemyNotVisible;

	function SetFall()
	{
		StartFalling('Dancing', 'ContinueDance');
	}

	function AnimEnd()
	{
		PlayDancing();
	}

	function HitWall(vector HitNormal, actor Wall)
	{
		if (Physics == PHYS_Falling)
			return;
		Global.HitWall(HitNormal, Wall);
		CheckOpenDoor(HitNormal, Wall);
	}

	function BeginState()
	{
		if (bSitting && !bDancing)
			StandUp();
		SetEnemy(None, EnemyLastSeen, true);
		Disable('AnimEnd');
		bCanJump = false;

		bStasis = False;

		SetupWeapon(false);
		SetDistress(false);
		SeekPawn = None;
		EnableCheckDestLoc(false);
	}

	function EndState()
	{
		EnableCheckDestLoc(false);
		bAcceptBump = True;

		if (JumpZ > 0)
			bCanJump = true;
		bStasis = True;

		StopBlendAnims();
	}

Begin:
	WaitForLanding();
	if (bDancing)
	{
		if (bUseHome)
			TurnTo(Location+HomeRot);
		Goto('StartDance');
	}
	if (!bUseHome)
		Goto('StartDance');

MoveToBase:
	if (!IsPointInCylinder(self, HomeLoc, 16-CollisionRadius))
	{
		EnableCheckDestLoc(true);
		while (true)
		{
			if (PointReachable(HomeLoc))
			{
				if (ShouldPlayWalk(HomeLoc))
					PlayWalking();
				MoveTo(HomeLoc, GetWalkingSpeed());
				CheckDestLoc(HomeLoc);
				break;
			}
			else
			{
				MoveTarget = FindPathTo(HomeLoc);
				if (MoveTarget != None)
				{
					if (ShouldPlayWalk(MoveTarget.Location))
						PlayWalking();
					MoveToward(MoveTarget, GetWalkingSpeed());
					CheckDestLoc(MoveTarget.Location, true);
				}
				else
					break;
			}
		}
		EnableCheckDestLoc(false);
	}
	TurnTo(Location+HomeRot);

StartDance:
	Acceleration=vect(0,0,0);
	Goto('Dance');

ContinueFromDoor:
	Goto('MoveToBase');

Dance:
ContinueDance:
	// nil
	bDancing = True;
	PlayDancing();
	bStasis = True;
	if (!bHokeyPokey)
		Goto('DoNothing');

Spin:
	Sleep(FRand()*5+5);
	useRot = DesiredRotation;
	if (FRand() > 0.5)
	{
		TurnTo(Location+1000*vector(useRot+rot(0,16384,0)));
		TurnTo(Location+1000*vector(useRot+rot(0,32768,0)));
		TurnTo(Location+1000*vector(useRot+rot(0,49152,0)));
	}
	else
	{
		TurnTo(Location+1000*vector(useRot+rot(0,49152,0)));
		TurnTo(Location+1000*vector(useRot+rot(0,32768,0)));
		TurnTo(Location+1000*vector(useRot+rot(0,16384,0)));
	}
	TurnTo(Location+1000*vector(useRot));
	Goto('Spin');

DoNothing:
	// nil
}

// ----------------------------------------------------------------------------
// state Sitting
//
// Just kinda sit there and do nothing.
// (doubles as a shitting state)
// ----------------------------------------------------------------------------

state Sitting
{
	ignores EnemyNotVisible;

	function SetFall()
	{
		StartFalling('Sitting', 'ContinueSit');
	}

	function AnimEnd()
	{
		PlayWaiting();
	}

	function HitWall(vector HitNormal, actor Wall)
	{
		if (Physics == PHYS_Falling)
			return;
		if (!bAcceptBump)
			NextDirection = TURNING_None;
		Global.HitWall(HitNormal, Wall);
		CheckOpenDoor(HitNormal, Wall);
	}

	function bool HandleTurn(Actor Other)
	{
		if (Other == HXSeatActor)
			return true;
		else
			return Global.HandleTurn(Other);
	}

	function Bump(actor bumper)
	{
		// If we hit our chair, move to the right place
		if ((bumper == HXSeatActor) && bAcceptBump)
		{
			bAcceptBump = false;
			GotoState('Sitting', 'CircleToFront');
		}

		// Handle conversations, if need be
		else
			Global.Bump(bumper);
	}

	function Tick(float deltaSeconds)
	{
		local vector endPos;
		local vector newPos;
		local float  delta;

		Global.Tick(deltaSeconds);

		if (bSitInterpolation && (HXSeatActor != None))
		{
			endPos = SeatPosition(HXSeatActor, SeatSlot);
			if ((deltaSeconds < remainingSitTime) && (remainingSitTime > 0))
			{
				delta = deltaSeconds/remainingSitTime;
				newPos = (endPos-Location)*delta + Location;
				remainingSitTime -= deltaSeconds;
			}
			else
			{
				remainingSitTime = 0;
				bSitInterpolation = false;
				newPos = endPos;
				Acceleration = vect(0,0,0);
				Velocity = vect(0,0,0);
				SetBase(HXSeatActor);
				bSitting = true;
			}
			SetLocation(newPos);
			DesiredRotation = HXSeatActor.Rotation+Rot(0, -16384, 0);
		}
	}

	// Prior SitPosition.
	function Vector SeatPosition( HXDecoration Seat, int slot)
	{
		local float newAssHeight;

		newAssHeight = GetDefaultCollisionHeight() + BaseAssHeight;
		newAssHeight = -(CollisionHeight - newAssHeight);

		return ((Seat.sitPoint[slot]>>Seat.Rotation)+Seat.Location+(vect(0,0,-1)*newAssHeight));
	}

	// Prior GetDestinationPosition.
	function vector GetSitDestinationPosition( HXDecoration Seat, optional float ExtraDist )
	{
		local Rotator SeatRot;
		local Vector  DestPos;

		if ( Seat==None )
			return Location;

		SeatRot = Seat.Rotation + Rot(0,-16384,0);
		SeatRot.Pitch = 0;
		DestPos = Seat.Location;
		DestPos += Vect(0,0,1)*(CollisionHeight-Seat.CollisionHeight);
		DestPos += Vector(SeatRot)*(Seat.CollisionRadius+CollisionRadius+ExtraDist);
		return DestPos;
	}

	function bool IsIntersectingSeat()
	{
		local bool   bIntersect;
		local vector testVector;

		bIntersect = false;
		if (HXSeatActor != None)
			bIntersect = IsOverlapping(HXSeatActor);

		return (bIntersect);
	}

	// Prior FindBestSlot.
	function int FindBestSeatSlot( HXDecoration Seat, out float SlotDist )
	{
		local float Dist, BestDist;
		local int   BestSlot, i;

		BestSlot = -1;
		BestDist = 100;

		if ( !Seat.Region.Zone.bWaterZone )
		{
			for ( i=0; i<Seat.NumSitPoints; i++ )
			{
				if ( Seat.SittingActor[i]==None )
				{
					Dist = VSize( SeatPosition(Seat,i) - Location );

					if ( BestSlot<0 || BestDist>Dist )
					{
						BestDist = Dist;
						BestSlot = i;
					}
				}
			}
		}

		SlotDist = BestDist;
		return BestSlot;
	}

	function FindBestSeat()
	{
		local HXDecoration Decoration, BestSeat;
		local float CurrentDist, BestDist;
		local int CurrentSlot, BestSlot;
		local bool bTry;

		if ( bUseFirstSeatOnly && bSeatHackUsed )
		{
			BestSeat = HXSeatHack;  // Use the seat hack.
			BestSlot = -1;

			if ( !IsSeatValid(BestSeat) )
			{
				BestSeat = None;
			}
			else
			{
				if ( GetNextWaypoint(BestSeat)==None )
				{
					BestSeat = None;
				}
				else
				{
					BestSlot = FindBestSeatSlot( BestSeat, CurrentDist );
					if ( BestSlot<0 )
						BestSeat = None;
				}
			}
		}
		else
		{
			// Try the ordered seat first.
			BestSeat = HXDecoration(OrderActor);
			if ( BestSeat!=None && !BestSeat.bIsSeat )
				BestSeat = None;

			if ( BestSeat!=None )
			{
				//if ( !IsSeatValid(OrderActor) )
				if ( !IsSeatValid(BestSeat) )
				{
					BestSeat = None;
				}
				else
				{
					if ( GetNextWaypoint(BestSeat)==None )
					{
						BestSeat = None;
					}
					else
					{
						BestSlot = FindBestSeatSlot( BestSeat, CurrentDist );
						if ( BestSlot<0 )
							BestSeat = None;
					}
				}
			}
			if ( BestSeat==None )
			{
				BestDist = 10001;
				BestSlot = -1;

				foreach RadiusActors( Class'HXDecoration', Decoration, 10000 )
				{
					if ( Decoration.bIsSeat && IsSeatValid(Decoration))
					{
						CurrentSlot = FindBestSeatSlot( Decoration, CurrentDist );
						if ( CurrentSlot>=0 )
						{
							if ( BestDist>CurrentDist )
							{
								if ( GetNextWaypoint(Decoration)!=None )
								{
									BestDist = CurrentDist;
									BestSeat = Decoration;
									BestSlot = CurrentSlot;
								}
							}
						}
					}
				}
			}
		}

		if ( BestSeat!=None )
		{
			BestSeat.SittingActor[BestSlot] = Self;
			SeatLocation       = BestSeat.Location;
			bSeatLocationValid = True;
		}
		else
		{
			bSeatLocationValid = False;
		}

		if ( bUseFirstSeatOnly && !bSeatHackUsed )
		{
			HXSeatHack    = BestSeat;
			bSeatHackUsed = True;
		}

		HXSeatActor = BestSeat;
		SeatSlot    = BestSlot;
	}

	function FollowSeatFallbackOrders()
	{
		FindBestSeat();
		if (IsSeatValid(HXSeatActor))
			GotoState('Sitting', 'Begin');
		else
			GotoState('Wandering');
	}

	function BeginState()
	{
		SetEnemy(None, EnemyLastSeen, true);
		Disable('AnimEnd');
		bCanJump = false;

		bAcceptBump = True;

		if (HXSeatActor == None)
			FindBestSeat();

		bSitInterpolation = false;

		bStasis = False;

		SetupWeapon(false);
		SetDistress(false);
		SeekPawn = None;
		EnableCheckDestLoc(true);
	}

	function EndState()
	{
		EnableCheckDestLoc(false);
		if (!bSitting)
			StandUp();

		bAcceptBump = True;

		if (JumpZ > 0)
			bCanJump = true;

		bSitInterpolation = false;

		bStasis = True;
	}

Begin:
	WaitForLanding();
	if (!IsSeatValid(HXSeatActor))
		FollowSeatFallbackOrders();
	if (!bSitting)
		WaitForLanding();
	else
	{
		TurnTo(Vector(HXSeatActor.Rotation+Rot(0, -16384, 0))*100+Location);
		Goto('ContinueSitting');
	}

MoveToSeat:
	if (IsIntersectingSeat())
		Goto('MoveToPosition');
	bAcceptBump = true;
	while (true)
	{
		if (!IsSeatValid(HXSeatActor))
			FollowSeatFallbackOrders();
		destLoc = GetSitDestinationPosition(HXSeatActor);
		if (PointReachable(destLoc))
		{
			if (ShouldPlayWalk(destLoc))
				PlayWalking();
			MoveTo(destLoc, GetWalkingSpeed());
			CheckDestLoc(destLoc);

			if (IsPointInCylinder(self, GetSitDestinationPosition(HXSeatActor), 16, 16))
			{
				bAcceptBump = false;
				Goto('MoveToPosition');
				break;
			}
		}
		else
		{
			MoveTarget = GetNextWaypoint(HXSeatActor);
			if (MoveTarget != None)
			{
				if (ShouldPlayWalk(MoveTarget.Location))
					PlayWalking();
				MoveToward(MoveTarget, GetWalkingSpeed());
				CheckDestLoc(MoveTarget.Location, true);
			}
			else
				break;
		}
	}

CircleToFront:
	bAcceptBump = false;
	if (!IsSeatValid(HXSeatActor))
		FollowSeatFallbackOrders();
	if (ShouldPlayWalk(GetSitDestinationPosition(HXSeatActor, 16)))
		PlayWalking();
	MoveTo(GetSitDestinationPosition(HXSeatActor, 16), GetWalkingSpeed());

MoveToPosition:
	if (!IsSeatValid(HXSeatActor))
		FollowSeatFallbackOrders();
	bSitting = true;
	EnableCollision(false);
	Acceleration=vect(0,0,0);

Sit:
	Acceleration=vect(0,0,0);
	Velocity=vect(0,0,0);
	if (!IsSeatValid(HXSeatActor))
		FollowSeatFallbackOrders();
	remainingSitTime = 0.8;
	PlaySittingDown();
	SetBasedPawnSize(CollisionRadius, GetSitHeight());
	SetPhysics(PHYS_Flying);
	StopStanding();
	bSitInterpolation = true;
	while (bSitInterpolation)
		Sleep(0);
	FinishAnim();
	Goto('ContinueSitting');

ContinueFromDoor:
	Goto('MoveToSeat');

ContinueSitting:
	if (!IsSeatValid(HXSeatActor))
		FollowSeatFallbackOrders();
	SetBasedPawnSize(CollisionRadius, GetSitHeight());
	SetCollision(Default.bCollideActors, Default.bBlockActors, Default.bBlockPlayers);
	PlaySitting();
	bStasis  = True;
	// nil

}

// ----------------------------------------------------------------------------
// state Fleeing
//
// Run like a bat outta hell away from an actor.
// ----------------------------------------------------------------------------

State Fleeing
{
	function ReactToInjury(Pawn instigatedBy, Name damageType, EHitLocation hitPos)
	{
		local Name currentState;
		local Pawn oldEnemy;
		local name newLabel;
		local bool bHateThisInjury;
		local bool bFearThisInjury;
		local bool bAttack;

		if ((health > 0) && (bLookingForInjury || bLookingForIndirectInjury))
		{
			currentState = GetStateName();

			bHateThisInjury = ShouldReactToInjuryType(damageType, bHateInjury, bHateIndirectInjury);
			bFearThisInjury = ShouldReactToInjuryType(damageType, bFearInjury, bFearIndirectInjury);

			if (bHateThisInjury)
				IncreaseAgitation(instigatedBy);
			if (bFearThisInjury)
				IncreaseFear(instigatedBy, 2.0);

			oldEnemy = Enemy;

			bAttack = false;
			if (SetEnemy(instigatedBy))
			{
				if (!ShouldFlee())
				{
					SwitchToBestWeapon();
					if (Weapon != None)
						bAttack = true;
				}
			}
			else
				SetEnemy(instigatedBy, , true);

			if (bAttack)
			{
				SetDistressTimer();
				SetNextState('HandlingEnemy');
			}
			else
			{
				SetDistressTimer();
				if (oldEnemy != Enemy)
					newLabel = 'Begin';
				else
					newLabel = 'ContinueFlee';
				SetNextState('Fleeing', newLabel);
			}
			GotoDisabledState(damageType, hitPos);
		}
	}

	function SetFall()
	{
		StartFalling('Fleeing', 'ContinueFlee');
	}

	function FinishFleeing()
	{
		if (bLeaveAfterFleeing)
			GotoState('Wandering');
		else
			FollowOrders();
	}

	function bool InSeat( out vector NewLocation )  // hack
	{
		local HXDecoration Decoration;

		foreach RadiusActors( Class'HXDecoration', Decoration, 200 )
		{
			if ( Decoration.bIsSeat && IsOverlapping(Decoration) )
			{
				NewLocation = Decoration.Location + Vector(Decoration.Rotation+Rot(0, -16384, 0))*(CollisionRadius+Decoration.CollisionRadius+20);
				return True;
			}
		}
		return False;
	}

	function Tick(float deltaSeconds)
	{
		UpdateActorVisibility(Enemy, deltaSeconds, 1.0, false);
		if (IsValidEnemy(Enemy))
		{
			if (EnemyLastSeen > FearSustainTime)
				FinishFleeing();
		}
		else if (!IsValidEnemy(Enemy, false))
			FinishFleeing();
		else if (!IsFearful())
			FinishFleeing();
		Global.Tick(deltaSeconds);
	}

	function HitWall(vector HitNormal, actor Wall)
	{
		if (Physics == PHYS_Falling)
			return;
		Global.HitWall(HitNormal, Wall);
		CheckOpenDoor(HitNormal, Wall);
	}
	
	function AnimEnd()
	{
		PlayWaiting();
	}

	function PickDestination()
	{
		local HidePoint      hidePoint;
		local Actor          waypoint;
		local float          dist;
		local float          score;
		local Vector         vector1, vector2;
		local Rotator        rotator1;
		local float          tmpDist;

		local float          bestDist;
		local float          bestScore;

		local FleeCandidates candidates[5];
		local int            candidateCount;
		local int            maxCandidates;

		local float          maxDist;
		local int            openSlot;
		local float          maxScore;
		local int            i;
		local bool           bReplace;

		local float          angle;
		local float          magnitude;
		local int            iterations;

		local HXNearbyProjectileList projList;
		local bool                 bSuccess;

		maxCandidates  = 3;  // must be <= size of candidates[] arrays
		maxDist        = 10000;

		// Initialize the list of candidates
		for (i=0; i<maxCandidates; i++)
		{
			candidates[i].score = -1;
			candidates[i].dist  = maxDist+1;
		}
		candidateCount = 0;

		MoveTarget = None;
		destPoint  = None;

		if (bAvoidHarm)
		{
			HXGetProjectileList(projList, Location);
			if (HXIsLocationDangerous(projList, Location))
			{
				vector1 = HXComputeAwayVector(projList);
				rotator1 = Rotator(vector1);
				if (AIDirectionReachable(Location, rotator1.Yaw, rotator1.Pitch, CollisionRadius+24, VSize(vector1), destLoc))
					return;   // eck -- hack!!!
			}
		}

		if (Enemy != None)
		{
			foreach RadiusActors(Class'HidePoint', hidePoint, maxDist)
			{
				// Can the boogeyman see our hiding spot?
				if (!enemy.LineOfSightTo(hidePoint))
				{
					// More importantly, can we REACH our hiding spot?
					waypoint = GetNextWaypoint(hidePoint);
					if (waypoint != None)
					{
						// How far is it to the hiding place?
						dist = VSize(hidePoint.Location - Location);

						// Determine vectors to the waypoint and our enemy
						vector1 = enemy.Location - Location;
						vector2 = waypoint.Location - Location;

						// Strip out magnitudes from the vectors
						tmpDist = VSize(vector1);
						if (tmpDist > 0)
							vector1 /= tmpDist;
						tmpDist = VSize(vector2);
						if (tmpDist > 0)
							vector2 /= tmpDist;

						// Add them
						vector1 += vector2;

						// Compute a score (a function of angle)
						score = VSize(vector1);
						score = 4-(score*score);

						// Find an empty slot for this candidate
						openSlot  = -1;
						bestScore = score;
						bestDist  = dist;

						for (i=0; i<maxCandidates; i++)
						{
							// Can we replace the candidate in this slot?
							if (bestScore > candidates[i].score)
								bReplace = TRUE;
							else if ((bestScore == candidates[i].score) &&
							         (bestDist < candidates[i].dist))
								bReplace = TRUE;
							else
								bReplace = FALSE;
							if (bReplace)
							{
								bestScore = candidates[i].score;
								bestDist  = candidates[i].dist;
								openSlot = i;
							}
						}

						// We found an open slot -- put our candidate here
						if (openSlot >= 0)
						{
							candidates[openSlot].point    = hidePoint;
							candidates[openSlot].waypoint = waypoint;
							candidates[openSlot].location = waypoint.Location;
							candidates[openSlot].score    = score;
							candidates[openSlot].dist     = dist;
							if (candidateCount < maxCandidates)
								candidateCount++;
						}
					}
				}
			}

			// Any candidates?
			if (candidateCount > 0)
			{
				// Find a random candidate
				// (candidates moving AWAY from the enemy have a higher
				// probability of being chosen than candidates moving
				// TOWARDS the enemy)

				maxScore = 0;
				for (i=0; i<candidateCount; i++)
					maxScore += candidates[i].score;
				score = FRand() * maxScore;
				for (i=0; i<candidateCount; i++)
				{
					score -= candidates[i].score;
					if (score <= 0)
						break;
				}
				destPoint  = candidates[i].point;
				MoveTarget = candidates[i].waypoint;
				destLoc    = candidates[i].location;
			}
			else
			{
				iterations = 4;
				magnitude = 400*(FRand()*0.4+0.8);  // 400, +/-20%
				rotator1 = Rotator(Location-Enemy.Location);
				if (!AIPickRandomDestination(100, magnitude, rotator1.Yaw, 0.6, rotator1.Pitch, 0.6, iterations,
				                             FRand()*0.4+0.35, destLoc))
					destLoc = Location+(VRand()*1200);  // we give up
			}
		}
		else
			destLoc = Location+(VRand()*1200);  // we give up
	}

	function BeginState()
	{
		StandUp();
		Disable('AnimEnd');
		//Disable('Bump');
		BlockReactions();
		if (!bCower)
			bCanConverse = False;
		bStasis = False;
		SetupWeapon(false, true);
		SetDistress(true);
		EnemyReadiness = 1.0;
		//ReactionLevel  = 1.0;
		SeekPawn = None;
		EnableCheckDestLoc(false);
	}

	function EndState()
	{
		EnableCheckDestLoc(false);
		Enable('AnimEnd');
		//Enable('Bump');
		ResetReactions();
		if (!bCower)
			bCanConverse = True;
		bStasis = True;
	}

Begin:
	//EnemyLastSeen = 0;
	destPoint = None;

Surprise:
	if ((1.0-ReactionLevel)*SurprisePeriod < 0.25)
		Goto('Flee');
	Acceleration=vect(0,0,0);
	PlaySurpriseSound();
	PlayWaiting();
	Sleep(FRand()*0.5);
	if (Enemy != None)
		TurnToward(Enemy);
	if (bCower)
		Goto('Flee');
	Sleep(FRand()*0.5+0.5);

Flee:
	if (bLeaveAfterFleeing)
	{
		bTransient = true;
		bDisappear = true;
	}
	if (bCower)
		Goto('Cower');
	WaitForLanding();
	PickDestination();

Moving:
	Sleep(0.0);

	if (enemy == None)
	{
		Acceleration = vect(0,0,0);
		PlayWaiting();
		Sleep(2.0);
		FinishFleeing();
	}

	// Move from pathnode to pathnode until we get where we're going
	if (destPoint != None)
	{
		EnableCheckDestLoc(true);
		while (MoveTarget != None)
		{
			if (ShouldPlayWalk(MoveTarget.Location))
				PlayRunning();
			MoveToward(MoveTarget, MaxDesiredSpeed);
			CheckDestLoc(MoveTarget.Location, true);
			if (enemy.bDetectable && enemy.AICanSee(destPoint, 1.0, false, false, false, true) > 0)
			{
				PickDestination();
				EnableCheckDestLoc(false);
				Goto('Moving');
			}
			if (MoveTarget == destPoint)
				break;
			MoveTarget = FindPathToward(destPoint);
		}
		EnableCheckDestLoc(false);
	}
	else if (PointReachable(destLoc))
	{
		if (ShouldPlayWalk(destLoc))
			PlayRunning();
		MoveTo(destLoc, MaxDesiredSpeed);
		if (enemy.bDetectable && enemy.AICanSee(Self, 1.0, false, false, true, true) > 0)
		{
			PickDestination();
			Goto('Moving');
		}
	}
	else
	{
		PickDestination();
		Goto('Moving');
	}

Pausing:
	Acceleration = vect(0,0,0);

	if (enemy != None)
	{
		if (HidePoint(destPoint) != None)
		{
			if (ShouldPlayTurn(Location + HidePoint(destPoint).faceDirection))
				PlayTurning();
			TurnTo(Location + HidePoint(destPoint).faceDirection);
		}
		Enable('AnimEnd');
		TweenToWaiting(0.2);
		while (AICanSee(enemy, 1.0, false, false, true, true) <= 0)
			Sleep(0.25);
		Disable('AnimEnd');
		FinishAnim();
	}

	Goto('Flee');

Cower:
	if (!InSeat(useLoc))
		Goto('CowerContinue');

	PlayRunning();
	MoveTo(useLoc, MaxDesiredSpeed);

CowerContinue:
	Acceleration = vect(0,0,0);
	PlayCowerBegin();
	FinishAnim();
	PlayCowering();

	// behavior 3 - cower and occasionally make short runs
	while (true)
	{
		Sleep(FRand()*3+6);

		PlayCowerEnd();
		FinishAnim();
		if (AIPickRandomDestination(60, 150, 0, 0, 0, 0,
		                            2, FRand()*0.3+0.6, useLoc))
		{
			if (ShouldPlayWalk(useLoc))
				PlayRunning();
			MoveTo(useLoc, MaxDesiredSpeed);
		}
		PlayCowerBegin();
		FinishAnim();
		PlayCowering();
	}

	/* behavior 2 - cower forever
	// don't stop cowering
	while (true)
		Sleep(1.0);
	*/

	/* behavior 1 - cower only when enemy watching
	if (enemy != None)
	{
		while (AICanSee(enemy, 1.0, false, false, true, true) > 0)
			Sleep(0.25);
	}
	PlayCowerEnd();
	FinishAnim();
	Goto('Pausing');
	*/

ContinueFlee:
ContinueFromDoor:
	FinishAnim();
	PlayRunning();
	if (bCower)
		Goto('Cower');
	else
		Goto('Moving');

}


// ----------------------------------------------------------------------------
// state WaitingFor
//
// Wait for a pawn to become visible, then move to it
// ----------------------------------------------------------------------------

state WaitingFor
{
	function SetFall()
	{
		StartFalling('WaitingFor', 'ContinueFollow');
	}

	function HitWall(vector HitNormal, actor Wall)
	{
		if (Physics == PHYS_Falling)
			return;
		Global.HitWall(HitNormal, Wall);
		CheckOpenDoor(HitNormal, Wall);
	}

	function Bump(actor bumper)
	{
		// If we hit the guy we're going to, end the state
		if (bumper == OrderActor)
			GotoState('WaitingFor', 'Done');

		// Handle conversations, if need be
		Global.Bump(bumper);
	}

	function Touch(actor toucher)
	{
		// If we hit the guy we're going to, end the state
		if (toucher == OrderActor)
			GotoState('WaitingFor', 'Done');

		// Handle conversations, if need be
		Global.Touch(toucher);
	}

	function BeginState()
	{
		StandUp();
		//BlockReactions();
		SetupWeapon(false);
		SetDistress(false);
		bStasis = True;
		SeekPawn = None;
		EnableCheckDestLoc(true);
	}
	function EndState()
	{
		EnableCheckDestLoc(false);
		//ResetReactions();
		bStasis = True;
	}

	function AnimEnd()
	{
		PlayRunning2();
	}

Begin:
	Acceleration = vect(0, 0, 0);

	if (orderActor == None && orderTag != '')
		GotoState('Idle');

	PlayWaiting();

Wait:
	Sleep(1.0);

	FindOrderActor();
	
	if (orderActor == None)
		Goto('Wait');

	if (AICanSee(orderActor, 1.0, false, true, false, true) <= 0)
		Goto('Wait');

	bStasis = False;

	PlayRunning2();

Follow:
	if (IsOverlapping(orderActor))
		Goto('Done');
	MoveTarget = GetNextWaypoint(orderActor);
	if ((MoveTarget != None) && (!MoveTarget.Region.Zone.bWaterZone) &&
	    (MoveTarget.Physics != PHYS_Falling))
	{
		if ((MoveTarget == orderActor) && MoveTarget.IsA('Pawn'))
		{
			if (GetNextVector(orderActor, useLoc))
			{
				if (ShouldPlayWalk(useLoc))
				{
					//if ( !IsAnimating() )
						//PlayRunning();
				}
				MoveTo(useLoc, MaxDesiredSpeed);
				CheckDestLoc(useLoc);
			}
			else
				Goto('Pause');
		}
		else
		{
			if (ShouldPlayWalk(MoveTarget.Location))
			{
				//if ( !IsAnimating() )
					//PlayRunning();
			}
			MoveToward(MoveTarget, MaxDesiredSpeed);
			CheckDestLoc(MoveTarget.Location, true);
		}
		if (IsOverlapping(orderActor))
			Goto('Done');
		else
			Goto('Follow');
	}

Pause:
	Acceleration = vect(0, 0, 0);
	TurnToward(orderActor);
	//PlayWaiting();
	Sleep(1.0);
	Goto('Follow');

Done:
	GotoState('Standing');

ContinueFollow:
ContinueFromDoor:
	PlayRunning();
	Goto('Follow');
}

// ----------------------------------------------------------------------------
// state AvoidingProjectiles
//
// Run away from a projectile.
// ----------------------------------------------------------------------------

state AvoidingProjectiles
{
	ignores EnemyNotVisible;

	function SetFall()
	{
		StartFalling('RunningTo', 'ContinueRun');
	}

	function HitWall(vector HitNormal, actor Wall)
	{
		if (Physics == PHYS_Falling)
			return;
		Global.HitWall(HitNormal, Wall);
		CheckOpenDoor(HitNormal, Wall);
	}

	function AnimEnd()
	{
		PlayWaiting();
	}

	function PickDestination(bool bGotoWatch)
	{
		local HXNearbyProjectileList projList;
		local bool                 bMove;
		local vector               projVector;
		local rotator              projRot;
		local int                  i;
		local int                  bestSlot;
		local float                bestDist;

		destLoc   = vect(0,0,0);
		destPoint = None;
		bMove = false;

		if (HXGetProjectileList(projList, Location) > 0)
		{
			if (HXIsLocationDangerous(projList, Location))
			{
				projVector = HXComputeAwayVector(projList);
				projRot    = Rotator(projVector);
				if (AIDirectionReachable(Location, projRot.Yaw, projRot.Pitch, CollisionRadius+24, VSize(projVector), destLoc))
				{
					useLoc = Location + vect(0,0,1)*BaseEyeHeight;  // hack
					bMove = true;
				}
			}
		}

		if (bMove)
			GotoState('AvoidingProjectiles', 'RunAway');
		else if (bGotoWatch)
			GotoState('AvoidingProjectiles', 'Watch');
	}

	function BeginState()
	{
		StandUp();
		Disable('AnimEnd');
		bCanJump = false;
		SetReactions(true, true, true, true, false, true, true, true, true, true, true, true);
		bStasis = False;
		useLoc = Location + vect(0,0,1)*BaseEyeHeight + Vector(Rotation);
		bCanConverse = False;
		EnableCheckDestLoc(false);
	}

	function EndState()
	{
		EnableCheckDestLoc(false);
		if (JumpZ > 0)
			bCanJump = true;
		ResetReactions();
		bStasis = True;
		bCanConverse = True;
	}

Begin:
	Acceleration = vect(0,0,0);
	PickDestination(true);

RunAway:
	PlayTurnHead(LOOK_Forward, 1.0, 0.0001);
	if (ShouldPlayWalk(destLoc))
		PlayRunning();
	MoveTo(destLoc, MaxDesiredSpeed);
	PickDestination(true);

Watch:
	Acceleration = vect(0,0,0);
	PlayWaiting();
	LookAtVector(useLoc, true, false, true);
	TurnTo(Vector(DesiredRotation)*1000+Location);
	sleepTime = 3.0;
	while (sleepTime > 0)
	{
		sleepTime -= 0.5;
		Sleep(0.5);
		PickDestination(false);
	}

Done:
	if (Orders != 'AvoidingProjectiles')
		FollowOrders();
	else
		GotoState('Wandering');

ContinueRun:
ContinueFromDoor:
	PickDestination(false);
	Goto('Done');

}

// ----------------------------------------------------------------------------
// PlayRunning2()
//
// Experimental
//
// function LoopAnimPivot(name Sequence, optional float Rate, optional float TweenTime, optional float MinRate,optional vector NewPrePivot )
// ----------------------------------------------------------------------------

function PlayRunning2()
{
//	ClientMessage("PlayRunning()");
	bIsWalking = False;
	if (Region.Zone.bWaterZone)
		LoopAnimPivot('Tread',,,,GetSwimPivot());
	else
	{
		if (HasTwoHandedWeapon())
			LoopAnimPivot('RunShoot2H', runAnimMult, 0.05);
		else
			LoopAnimPivot('Run', runAnimMult, 0.05);
	}
}



// ----------------------------------------------------------------------------
// PlayWaiting()
// ----------------------------------------------------------------------------

//simulated function PlayWaiting()
function PlayWaiting()
{
//	ClientMessage("PlayWaiting()");
	if (Region.Zone.bWaterZone)
		LoopAnimPivot('Tread', , 0.3, , GetSwimPivot());
	else
	{
		if (HasTwoHandedWeapon())
			LoopAnimPivot('BreatheLight2H', , 0.3);
		else
			LoopAnimPivot('BreatheLight', , 0.3);
	}
}


// ----------------------------------------------------------------------------
// PlayIdle()
// ----------------------------------------------------------------------------

//simulated function PlayIdle()
function PlayIdle()
{
	ClientMessage("PlayIdle()");

	if (Region.Zone.bWaterZone)
		LoopAnimPivot('Tread', , 0.3, , GetSwimPivot());
	else
	{
		if (HasTwoHandedWeapon())
			PlayAnimPivot('Idle12H', , 0.3);
		else
			PlayAnimPivot('Idle1', , 0.3);
	}
}


// ----------------------------------------------------------------------------
// PlayTurnHead()
// ----------------------------------------------------------------------------
/*
simulated function bool PlayTurnHead(ELookDirection Dir, float rate, float tweentime)
{
	local name lookName;
	local bool bSuccess;

	if ( Role == ROLE_Authority )
	{
		return Super(ScriptedPawn).PlayTurnHead( Dir, rate,  tweentime );
	}
	else
	{
		//Log("PlayTurnHead()");

		if (dir == LOOK_Left)
			lookName = 'HeadLeft';
		else if (dir == LOOK_Right)
			lookName = 'HeadRight';
		else if (dir == LOOK_Up)
			lookName = 'HeadUp';
		else if (dir == LOOK_Down)
			lookName = 'HeadDown';
		else
			lookName = 'Still';

		bSuccess = false;

		if (BlendAnimSequence[3] != lookName)
		{
			if (animTimer[1] > 0.00)
			{
				animTimer[1] = 0;
				if (BlendAnimSequence[3] == '')
					BlendAnimSequence[3] = 'Still';
				PlayBlendAnim(lookName, rate, tweentime, 3);
				bSuccess = true;
			}
		}

		return (bSuccess);
	}
}



//
// LookAtActor - DEUS_EX STM
//

simulated function LookAtActor(Actor targ, bool bRotate,
                     bool bLookHorizontal, bool bLookVertical,
                     optional float DelayTime, optional float rate,
                     optional float LockAngle, optional float AngleOffset)
{
	local vector lookTo;

	if ( targ != None )
	{
		//Log( Self @ "LookAtActor()" );
		// If we're looking at a pawn, look at the head;
		// otherwise, look at the center point

		lookTo = targ.Location;
		if (Pawn(targ) != None)
			lookTo += (vect(0,0,1)*Pawn(targ).BaseEyeHeight);
		else if (Decoration(targ) != None)
			lookTo += (vect(0,0,1)*Decoration(targ).BaseEyeHeight);
		else
			lookTo += (vect(0,0,1)*targ.CollisionHeight*0.75);

		LookAtVector(lookTo, bRotate, bLookHorizontal, bLookVertical,
								 DelayTime, rate, LockAngle, AngleOffset);
	}
}

//
// LookAtVector - DEUS_EX STM
//

simulated function LookAtVector(vector lookTo, bool bRotate,
                      bool bLookHorizontal, bool bLookVertical,
                      optional float DelayTime, optional float rate,
                      optional float LockAngle, optional float AngleOffset)
{
	local vector         lookFrom;
	local rotator        lookAngle;
	local int            hPos, vPos;
	local int            hAngle, vAngle;
	local int            hAbs, vAbs;
	local int            hRot;
	local ELookDirection lookDir;

	//Log( Self @ "LookAtVector()" );

	if (rate <= 0)
		rate = 1.0;

	// Head movement angles
	hAngle = 5461;  // 30 degrees horizontally
	vAngle = 2731;  // 15 degrees vertically

	// Determine our angle to the target
	lookFrom  = Location + (vect(0,0,1)*CollisionHeight*0.9);
	lookAngle = Rotator(lookTo-lookFrom);
	lookAngle.Yaw = (lookAngle.Yaw - Rotation.Yaw) & 65535;
	if (lookAngle.Yaw > 32767)
		lookAngle.Yaw -= 65536;
	if (lookAngle.Pitch > 32767)
		lookAngle.Pitch -= 65536;

	// hPos and vPos determine which way the pawn needs to look
	// horizontally and vertically

	hPos = 0;
	vPos = 0;

	// Do we need to look up or down?
	if (bLookVertical)
	{
		if (lookAngle.Pitch > vAngle*0.9)
			vPos = 1;
		else if (lookAngle.Pitch < -vAngle*0.75)
			vPos = -1;
	}

	// Do we need to look left or right?
	if (bLookHorizontal)
	{
		if (lookAngle.Yaw > hAngle*0.5)
			hPos = 1;
		else if (lookAngle.Yaw < -hAngle*0.5)
			hPos = -1;
	}

	hAbs = Abs(lookAngle.Yaw);
	vAbs = Abs(lookAngle.Pitch);

	if (bRotate)
	{
		hRot = lookAngle.Yaw;

		// Hack -- NPCs that look horizontally or vertically, AND rotate, will use inexact rotations
		if (bLookHorizontal && (vPos == 0))
		{
			if (hRot > hAngle*1.2)
				hRot -= hAngle*1.2;
			else if (hRot < -hAngle*1.2)
				hRot += hAngle*1.2;
			else
				hRot = 0;
		}
		else if (bLookVertical && (hPos == 0))
		{
			if (hRot > hAngle*0.35)
				hRot -= hAngle*0.35;
			else if (hRot < -hAngle*0.35)
				hRot += hAngle*0.35;
			else
				hRot = 0;
		}

		// Clamp the rotation angle, based on the angles passed in
		if (AngleOffset > 0)
		{
			hRot = (hRot + (Rotation.Yaw-LockAngle) + 65536*4) & 65535;
			if (hRot > 32767)
				hRot -= 65536;
			if      (hRot < -AngleOffset)
				hRot = -AngleOffset;
			else if (hRot > AngleOffset)
				hRot = AngleOffset;
			hRot = (hRot + (LockAngle-Rotation.Yaw) + 65536*4) & 65535;
			if (hRot > 32767)
				hRot -= 65536;
		}

		// Compute actual rotation, based on new angle
		hAbs = (65536 + lookAngle.Yaw - hRot) & 65535;
		if (hAbs > 32767)
			hAbs = 65536-hAbs;
	}

	// No rotation
	else
		hRot = 0;

	// We can't look vertically AND horizontally at the same time
	// (we need a skeletal animation system!!!)

	if ((hPos != 0) && (vPos != 0))
	{
		if (hAbs > vAbs)
			vPos = 0;
		else
			hPos = 0;
	}

	// Play head turning animation
	if (hPos > 0)
		lookDir = LOOK_Right;
	else if (hPos < 0)
		lookDir = LOOK_Left;
	else if (vPos > 0)
		lookDir = LOOK_Up;
	else if (vPos < 0)
		lookDir = LOOK_Down;
	else
		lookDir = LOOK_Forward;
	if ((bLookHorizontal || bLookVertical) && (animTimer[1] >= DelayTime))
		PlayTurnHead(lookDir, 1.0, rate);

	// Turn as necessary
	if (bRotate && ROLE == ROLE_Authority )
		DesiredRotation = Rotation + rot(0,1,0)*hRot;
}

//
// LoopHeadConvoAnim - DEUS_EX STM
//

function ClientLoopHeadConvoAnim()
{
	local float rnd;

	rnd = FRand();

	// move head randomly (only while not speaking)
	if (!bClientIsSpeaking && (animTimer[1] > 0.5))
	{
		if (rnd < 0.01)
			PlayTurnHead(LOOK_Left, 1.0, 2.0);
		else if (rnd < 0.02)
			PlayTurnHead(LOOK_Right, 1.0, 2.0);
		else
			PlayTurnHead(LOOK_Forward, 1.0, 1.0);
	}
}
*/

// Add Item to this pawn's inventory. 
// Returns true if successfully added, false if not.
function bool AddInventory( inventory NewItem )
{
	// Skip if already in the inventory.
	local inventory Inv;
	
	// The item should not have been destroyed if we get here.
	if (NewItem ==None )
		log("tried to add none inventory to "$self);

	for( Inv=Inventory; Inv!=None; Inv=Inv.Inventory )
		if( Inv == NewItem )
			return false;

	// DEUS_EX AJY
	// Update the previous owner's inventory chain
	// HX_INV do not remove my own proposed inventory!
	if (NewItem.Owner != None && NewItem.Owner != Self)
		Pawn(NewItem.Owner).DeleteInventory(NewItem);

	// Add to front of inventory chain.
	NewItem.SetOwner(Self);
	NewItem.Inventory = Inventory;
	Inventory = NewItem;

	return true;
}

// ----------------------------------------------------------------------------
// DeleteInventory()
// ----------------------------------------------------------------------------

function bool DeleteInventory(inventory item)
{
	local bool retval;

	retval = Super(Pawn).DeleteInventory(item);

	// HX_INV: also remove link to next item
	//Item.Inventory = None;

	return retval;
}


// ----------------------------------------------------------------------------
// SpawnCarcass()
// ----------------------------------------------------------------------------

function Carcass SpawnCarcass()
{
	local HXCarcass carc;
	local vector loc;
	local Inventory item, nextItem;
	local FleshFragment chunk;
	local int i;
	local float size;
	local HXWeapon WeaponItem;

	// If we really got blown up good, gib us and don't display a carcass.
	if ( Health<-100 && !bIsRobot )
	{
		size = (CollisionRadius + CollisionHeight) / 2;
		if (size > 10.0)
		{
			for (i=0; i<size/4.0; i++)
			{
				loc.X = (1-2*FRand()) * CollisionRadius;
				loc.Y = (1-2*FRand()) * CollisionRadius;
				loc.Z = (1-2*FRand()) * CollisionHeight;
				loc += Location;
				chunk = spawn(class'FleshFragment', None,, loc);
				if (chunk != None)
				{
					chunk.DrawScale = size / 25;
					chunk.SetCollisionSize(chunk.CollisionRadius / chunk.DrawScale, chunk.CollisionHeight / chunk.DrawScale);
					chunk.bFixedRotationDir = True;
					chunk.RotationRate = RotRand(False);
				}
			}
		}

		return None;
	}

	// spawn the carcass
	carc = HXCarcass(Spawn(CarcassType));

	if ( carc != None )
	{
		if (bStunned)
			carc.bNotDead = True;

		carc.Initfor(self);

		// move it down to the floor
		loc = Location;
		loc.z -= Default.CollisionHeight;
		loc.z += carc.Default.CollisionHeight;
		carc.SetLocation(loc);
		carc.Velocity = Velocity;
		carc.Acceleration = Acceleration;

		// HX_INV:
		// actually just put the inventory into the carcass the player is allowed
		// to pick up. Look at Carcass code.

		// give the carcass the pawn's inventory if we aren't an animal or robot
		if ( !bIsAnimal && !bIsRobot )
		{
			if ( Inventory != None )
			{
				do
				{
					item = Inventory;
					nextItem = item.Inventory;

					DeleteInventory(item);

					// Only let the player pick up ammo that's randomized into a weapon
					if ( Item.IsA('Ammo') )
					{
						Item.Destroy();
					}
					else // No Ammo.
					{	
						WeaponItem = HXWeapon(Item);

						if ( WeaponItem!=None )
						{
							// No WeaponNPCMelee or WeaponNPCRange
							if ( WeaponItem.bNativeAttack )
							{
								WeaponItem.Destroy();
							}
							// Grenades and LAMs always pickup 1
							else if ( WeaponItem.bIsGrenade )
							{
								WeaponItem.PickupAmmoCount = 1;

								Carc.AddInventory( WeaponItem );
							}
							// Any weapons have their ammo set to a random number of rounds (1-4)
							else
							{
								WeaponItem.PickupAmmoCount = Rand(4) + 1;

								Carc.AddInventory(item);
							}
						}
						// Just add DeusExPickups
						else
						{
							Carc.AddInventory(item);
						}
					}

					item = nextItem;
				}
				until (item == None);
			}
		}
	}

	return carc;
}


// ----------------------------------------------------------------------------
// GetNearestPlayer()
//
// Workaround for: Error, Context expression: Variable is too large (256 bytes, 255 max)
// ----------------------------------------------------------------------------

function HXPlayerPawn GetNearestPlayer()
{
	return PlayerDistances[0].Player;
}

// ----------------------------------------------------------------------------
// LipSynchTimer()
// ----------------------------------------------------------------------------

simulated event LipSynchTimer();

// ----------------------------------------------------------------------------
// HandleShot()
// ----------------------------------------------------------------------------

/*function HandleShot( Name Event, EAIEventState State, XAIParams Params )
{
	Log( Self $ ".HandleShot()" );
	Log( "  Event = " $ Event );
	Log( "  State = " $ State );
	Log( "  Params.BestActor = " $ Params.BestActor );
	Log( "  Params.Score = " $ Params.Score );
	Log( "  Params.Visibility = " $ Params.Visibility );
	Log( "  Params.Volume = " $ Params.Volume );
	Log( "  Params.Smell = " $ Params.Smell );

	Super.HandleShot( Event, State, Params );
}*/

// ----------------------------------------------------------------------------
// DropDecoration()
// ----------------------------------------------------------------------------

function DropDecoration()
{
	if (CarriedDecoration != None)
	{
		CarriedDecoration.bWasCarried = true;
		CarriedDecoration.SetBase(None);
		CarriedDecoration.SetPhysics(PHYS_Falling);
		CarriedDecoration.Velocity = Velocity + 10 * VRand();
		CarriedDecoration.Instigator = self;

		// turn off translucency - DEUS_EX CNN
		/*CarriedDecoration.Style = CarriedDecoration.Default.Style;
		CarriedDecoration.ScaleGlow = CarriedDecoration.Default.ScaleGlow;
		CarriedDecoration.bUnlit = CarriedDecoration.Default.bUnlit;*/

		CarriedDecoration = None;
	}
}

// ----------------------------------------------------------------------------
// GrabDecoration()
// ----------------------------------------------------------------------------

function GrabDecoration()
{
	local vector lookDir, HitLocation, HitNormal, T1, T2, extent;
	local actor HitActor;

	if ( carriedDecoration == None )
	{
		//first trace to find it
		lookDir = vector(Rotation);
		lookDir.Z = 0;
		T1 = Location + BaseEyeHeight * vect(0,0,1) + lookDir * 0.8 * CollisionRadius;
		T2 = T1 + lookDir * 1.2 * CollisionRadius;
		HitActor = Trace(HitLocation, HitNormal, T2, T1, true);
		if ( HitActor == None )
		{
			T1 = T2 - (BaseEyeHeight + CollisionHeight - 2) * vect(0,0,1);
			HitActor = Trace(HitLocation, HitNormal, T1, T2, true);
		}
		else if ( HitActor == Level )
		{
			T2 = HitLocation - lookDir;
			T1 = T2 - (BaseEyeHeight + CollisionHeight - 2) * vect(0,0,1);
			HitActor = Trace(HitLocation, HitNormal, T1, T2, true);
		}	
		if ( (HitActor == None) || (HitActor == Level) )
		{
			extent.X = CollisionRadius;
			extent.Y = CollisionRadius;
			extent.Z = CollisionHeight;
			HitActor = Trace(HitLocation, HitNormal, Location + lookDir * 1.2 * CollisionRadius, Location, true, extent);
		}

		if ( Mover(HitActor) != None )
		{
			if ( Mover(HitActor).bUseTriggered )
				HitActor.Trigger( self, self );
		}		
		else if ( (Decoration(HitActor) != None)  && ((weapon == None) || (weapon.Mass < 20)) )
		{
			CarriedDecoration = Decoration(HitActor);
			if ( !CarriedDecoration.bPushable || (CarriedDecoration.Mass > 40) 
				|| (CarriedDecoration.StandingCount > 0) )
			{
				CarriedDecoration = None;
				return;
			}
			lookDir.Z = 0;				
			if ( CarriedDecoration.SetLocation(Location + (0.5 * CollisionRadius + CarriedDecoration.CollisionRadius) * lookDir) )
			{
				CarriedDecoration.SetPhysics(PHYS_None);
				CarriedDecoration.SetBase(self);

				// make it translucent - DEUS_EX CNN
				/*CarriedDecoration.Style = STY_Translucent;
				CarriedDecoration.ScaleGlow = 1.0;
				CarriedDecoration.bUnlit = True;*/
			}
			else
				CarriedDecoration = None;
		}
	}
}

// ----------------------------------------------------------------------------
// GetPlayerDistance()
//
// Workaround for
// 'Error, Context expression: Variable is too large (256 bytes, 255 max)'
// compiler bug, when called from outside
// ----------------------------------------------------------------------------

simulated function GetPlayerDistance( int i, out HXPlayerPawn OutPlayer, out float OutDistance )
{
	OutPlayer   = PlayerDistances[i].Player;
	OutDistance = PlayerDistances[i].Distance;
}

// ----------------------------------------------------------------------------
// ----------------------------------------------------------------------------
// STIMULI AND AGITATION ROUTINES
// ----------------------------------------------------------------------------
// ----------------------------------------------------------------------------

// ----------------------------------------------------------------------------
// WeaponDrawnScore()
// ----------------------------------------------------------------------------

function float WeaponDrawnScore( Actor Receiver, Actor Sender, float Score )
{
	local Pawn PawnSender;

	// Cull events received from enemies
	PawnSender = Pawn(Sender);
	if ( PawnSender==None )
		PawnSender = Pawn(Sender.Owner);
	if ( PawnSender==None )
		PawnSender = Sender.Instigator;
	if ( PawnSender==None )
		Score = 0;

	// Changed to align with LoudNoiseScore and in particular making more sense. --han
	//else if ( IsValidEnemy(PawnSender) )
	else if ( !IsValidEnemy(PawnSender) )
		Score = 0;

	return Score;
}

// ----------------------------------------------------------------------------
// GetStyleTexture()
// ----------------------------------------------------------------------------

function Texture GetStyleTexture( ERenderStyle InStyle, texture OriginalTexture, optional Texture ReplacementTexture )
{
	local Texture MaskTexture;

	switch ( InStyle )
	{
		case STY_Translucent:
			MaskTexture = Texture'BlackMaskTex';
			break;
		case STY_Modulated:
			MaskTexture = Texture'GrayMaskTex';
			break;
		case STY_Masked:
			MaskTexture = Texture'PinkMaskTex';
			break;
		default:
			MaskTexture = Texture'BlackMaskTex';
			break;
	}

	if ( OriginalTexture==None )
		return MaskTexture;

	switch ( OriginalTexture.Name )
	{
		case 'BlackMaskTex':
			return Texture'BlackMaskTex'; // Hack. Unsure why --han
			break;
		case 'GrayMaskTex':
		case 'PinkMaskTex':
			return MaskTexture;
			break;
	}

	if ( ReplacementTexture!=None )
		return ReplacementTexture;

	return OriginalTexture;
}

// ----------------------------------------------------------------------------
// SetSkinStyle()
// ----------------------------------------------------------------------------

function SetSkinStyle( ERenderStyle InStyle, optional Texture InTexture, optional float InScaleGlow )
{
	local int i;

	if ( InScaleGlow==0.0 )
		InScaleGlow = ScaleGlow;

	for ( i=0; i<8; i++ )
		MultiSkins[i] = GetStyleTexture( InStyle, GetMeshTexture(i), InTexture );

	Skin      = GetStyleTexture( InStyle, Skin, InTexture );
	Texture   = GetStyleTexture( InStyle, Skin, InTexture );
	ScaleGlow = InScaleGlow;
	Style     = InStyle;
}

// ----------------------------------------------------------------------------
// ResetSkinStyle()
// ----------------------------------------------------------------------------

function ResetSkinStyle()
{
	local int i;

	for ( i=0; i<8; i++ )
		MultiSkins[i] = default.MultiSkins[i];

	Skin      = default.Skin;
	Texture   = default.Texture;
	ScaleGlow = default.ScaleGlow;
	Style     = default.Style;
}

// ----------------------------------------------------------------------------
// ----------------------------------------------------------------------------

defaultproperties
{
	bSpawning=True
	NameArticle=""
	PawnInUseMsg="This Pawn is already in use by %s."
	bIsRobot=False
	bIsHuman=False
	bIsAnimal=False
	bIsChild=False
	bOn=False
	NetPriority=2.000000
}
