//=============================================================================
// HXScriptedRenderIterator.
//
// Basic idea is to have an RenderIterator which exposes it's virtual C++
// interface to UnrealScript (No RenderIterator just does that for Init()).
//
// Use is like: Set the RenderIteratorClass property of the Actor you want
// to use it to your RenderIterator class. The Render adds a new
// FDynamicSprite for each time you return an Actor in CurrentItem().
// The ScaleGlow, DrawScale, Location and Rotation property gets copied over
// to the FDynamicSprite, so you can modify these each time CurretItem() is
// called and if needed restore them afterwards (I save the Location/Rotation
// in my DecorationRenderIterator in Init() and restore it is IsDone()).
// If you need to adjust more of these properties (like I need to do
// for getting rid of the translucent draw style of carried decorations
// on other pawns) use another Actor as a render proxy.
// 
// Event flow is basically: 
//  Init();
//  First();
//  while( !IsDone() )
//  {
//    CurrentItem();
//    Next();
//  }
//
// A few more words about RenderIterators in general: If there is no
// RenderIteratorClass set, then just the Actor as it is currently is used
// for the creating of the FDynamicSprite. If the RenderIteratorClass is not
// empty and RenderInterface is None a new RenderIterator is constructed.
// If the RenderIterator is not None but RenderIteratorClass gets set to None
// the current RenderIterator gets destroyed. The Outer property contains 
// the Actor for which this RenderIterface was constructed.
// RenderIterators are not used inside the Editor.
//
// "I didn't understand a word you wrote, so whats the purpose of all these?"
// Ion used RenderIterators for all kinds of particle systems, like smoke,
// dripping water, electric arcs and even the laserbeams. For HX RenderIterators
// proved quite invaluable to achive smooth display of carried decorations 
// in network, so they are quite versatile. If you use them for particle 
// system this will be times faster then using an Actor based approach, but
// you get the penalty of giving up collision, etc. However, you for something
// like a rain system you could use FastTrace() to determine the volume you
// want to operate in or, where a rain drop would hit the floor and update
// occasionally distributed over a couple of frames the current volume to use.
//
// If your ScriptedRenderIterator does work as you like, you should consider
// porting it over to C++, which offers exactly the same interface, so this
// is a straight forward process, and at least for larger particle systems
// this should give a performance increase.
//
// When you are just returning a proxy in CurrentItem() and not the Owner, 
// you should update Owner.LastRenderTime (e.g. setting it to Level.TimeSeconds).
// Without you might experience problems like actor falling into statis although 
// the actor should not be in stasis.
//=============================================================================
class HXScriptedRenderIterator extends HXRenderIterator
	native;

//
// Copies a whole bunch of rendering related variables.
//
native(3100) final function CopyRenderProperties( Actor Dest, Actor Source );

// ----------------------------------------------------------------------------
// Init()
// ----------------------------------------------------------------------------

function Init( PlayerPawn Camera )
{
	Observer = Camera;
}

// ----------------------------------------------------------------------------
// First()
// ----------------------------------------------------------------------------

function First()
{
	Index = 0;
}

// ----------------------------------------------------------------------------
// Next()
// ----------------------------------------------------------------------------

function Next()
{
	Index++;
}

// ----------------------------------------------------------------------------
// IsDone()
// ----------------------------------------------------------------------------

function bool IsDone()
{
	return Index>=MaxItems;
}

// ----------------------------------------------------------------------------
// ActorCurrentItem()
// ----------------------------------------------------------------------------

function Actor CurrentItem()
{
	return Actor(Outer);
}

// ----------------------------------------------------------------------------
// ----------------------------------------------------------------------------

defaultproperties
{
	MaxItems=1
}
