//=============================================================================
// HXTAD.
//=============================================================================
class HXTAD extends HXElectronicDevices;

var float BlinkTime; // Used for BlinkTimer()

var() float BeepInterval;
var() Sound BeepSound;
var() Sound NoMessagesSound;
var() Texture OnTexture, OffTexture;
var HXTADLight Light;
var bool bOn;

// ----------------------------------------------------------------------------
// PostPostBeginPlay()
// ----------------------------------------------------------------------------

function PostPostBeginPlay()
{
	Super.PostPostBeginPlay();

	BlinkTime = BeepInterval * 0.5;
}

// ----------------------------------------------------------------------------
// Succeeds()
//
// Intended to be called by Mutator code if Self was spawned as a successor
// to Other. Super call recommended.
// ----------------------------------------------------------------------------

function Succeeds( Actor Other )
{
	local TAD OtherTAD;

	// Call Super.
	Super.Succeeds( Other );

	// Special TAD Init.
	OtherTAD = TAD(Other);
	if ( OtherTAD!=None )
	{
		BeepInterval = OtherTAD.BeepInterval;
		BeepSound    = OtherTAD.BeepSound;
	}
}

// ----------------------------------------------------------------------------
// Frob()
// ----------------------------------------------------------------------------

function Frob( Actor Frobber, Inventory FrobWith )
{
	Super.Frob( Frobber, FrobWith );

	// Play a Sound if we have are not inside a conversation
	// and have no new messages to play.
	if ( !IsInState('Conversation') && !IsInState('FirstPersonConversation') && !CheckForMessages() )
		PlaySound( NoMessagesSound, SLOT_Misc,,, 512 );
}

// ----------------------------------------------------------------------------
// Tick()
// ----------------------------------------------------------------------------

function Tick( float DeltaTime )
{
	Super.Tick( DeltaTime );

	// Additional BlinkTimer().
	if ( BlinkTime>0.0 )
	{
		BlinkTime -= DeltaTime;

		if ( BlinkTime<0.001 )
		{
			BlinkTime = 0.0;
			BlinkTimer();
		}
	}
}

// ----------------------------------------------------------------------------
// Timer()
// ----------------------------------------------------------------------------

function BlinkTimer()
{
	if ( CheckForMessages(True) )
	{
		if ( Light==None )
			Light = Spawn( Class'HXTADLight', Self,, Location+vect(0,0,32) );

		// Beep periodically.
		if ( !IsInState('Conversation') && !IsInState('FirstPersonConversation') )
		{
			bOn = !bOn;

			if ( bOn )
			{
				// On - Beep.
				PlaySound( BeepSound, SLOT_Misc,,, 512 );

				if ( Light!=None )
					Light.LightType = LT_Steady;
				Skin = OnTexture;
			}
			else
			{
				// Off.
				if ( Light!=None )
					Light.LightType = LT_None;
				Skin = OffTexture;
			}

			BlinkTime = BeepInterval * 0.5;
		}
		else
		{
			// Turn light off during conversation.
			bOn = False;
			if ( Light!=None )
				Light.LightType = LT_None;
			Skin = OffTexture;

			// Just set to some time in the future until
			// the conversation is over.
			BlinkTime = 1.0;
		}
	}
	else
	{
		// We currently have no conversation, so remove all effects,
		// kill the Light, but do set the timer in case another conversation
		// for us gets valid after some flags got set.
		bOn = False;
		if ( Light!=None )
		{
			Light.Destroy();
			Light = None;
		}
		Skin = OffTexture;

		// Check every 5s for new conversations.
		BlinkTime = 5.0;
	}
}

// ----------------------------------------------------------------------
// CheckForMessages()
//
// Based on DeusExPlayer.GetActiveConversation(), but does no range
// check and has an optional handling to check for unplayed messages
// in case the messages are not bDisplayOnce (which perfectly makes
// sense for a TAD).
// ----------------------------------------------------------------------

function bool CheckForMessages( optional bool bOnlyNew )
{
	local Conversation Conversation;
	local ConListItem ConListItem;
	local HXGameInfo Game;

	// Sanity check.
	Game = HXGameInfo(Level.Game);
	if ( Game==None || Game.Steve==None || Game.Steve.FlagBase==None )
		return false;

	// In a loop, go through the conversations, checking each.
	for ( ConListItem=ConListItem(ConListItems); ConListItem!=None; ConListItem=ConListItem.Next )
	{
		Conversation = ConListItem.Con;

		// Skip if the conversation is not frobable (or legacy bumpable).
		if ( !Conversation.bInvokeFrob && !Conversation.bInvokeBump )
			continue;

		// Check if this conversation is only to be played once or we only want new messages.
		if ( (Conversation.bDisplayOnce || bOnlyNew) && Game.Steve.FlagBase.GetBool(StringToName(Conversation.ConName$"_Played")) )
			continue;

		// Now Check FlagRefs.
		if ( !Game.CheckFlagRefs(Conversation.FlagRefList) )
			continue;

		// Passed all checks.
		return true;
	}

	// Nothing found.
	return false;
}

// ----------------------------------------------------------------------------
// ----------------------------------------------------------------------------

defaultproperties
{
	NetPriority=2.1
	BeepInterval=2.0
	BeepSound=Beep5
	NoMessagesSound=Beep5
	ItemName="Telephone Answering Machine"
	Mesh=TAD
	Texture=TADTex1
	OffTexture=TADTex1
	OnTexture=TADTex2
	CollisionRadius=7.40
	//CollisionHeight=2.13
	CollisionHeight=1.38
	Mass=10.0
	Buoyancy=5.0
}
