//=============================================================================
// HXTailMutator.
//
// Batch replaces Inventory/Actors using the ModifyInventoryClass and
// ModifyActorClass Mutator interface.
// The TailMutator is supposed to sit at the end of the Mutator chain.
//=============================================================================
class HXTailMutator extends HXMutator
	//native
	;

var bool bTailMutatorPreBeginPlayDone;

// ----------------------------------------------------------------------
// PreBeginPlay()
// ----------------------------------------------------------------------

event PreBeginPlay()
{
	local HXSpawnNotify SpawnNotify;

	Super.PreBeginPlay();

	// PreBeginPlay() is called twice because Mutator is Spawned during
	// GameInfo.InitGame(). Prevent it from running twice.
	if ( !bTailMutatorPreBeginPlayDone )
	{
		// Remove the guns AutoTurrets do spawn, even if when Owner itsself is bDeleteMe.
		SpawnNotify = Spawn( Class'HXAutoTurretGunSpawnNotify' );
		SpawnNotify.Link();

		// Prevent running it twice.
		bTailMutatorPreBeginPlayDone = true;
	}
}

// ----------------------------------------------------------------------------
// CheckReplacement()
// ----------------------------------------------------------------------------

function bool CheckReplacement( Actor Other, out byte bSuperRelevant )
{
	// Inventory
	if ( Other.IsA('Inventory') )
	{
		return ReplaceInventory( Inventory(Other) );
	}
	// Keypoint
	else if ( Other.isA('Keypoint') )
	{
		// InterpolationPoint
		if ( Other.IsA('InterpolationPoint') )
		{
			//return ReplaceActor( Other );
			return true;
		}
		// CameraPoint
		else if ( Other.IsA('CameraPoint') )
		{
			return ReplaceActor( Other, true );
		}
	}
	// Mover
	//else if ( Other.bIsMover ) )
	//{
		//return true;
	//}
	// NavigationPoint
	else if ( Other.IsA('NavigationPoint') )
	{
		// MapExit
		if ( Other.IsA('MapExit') )
		{
			return ReplaceActor( Other, true );
		}
		// Teleporter
		else if ( Other.IsA('Teleporter') )
		{
			return ReplaceActor( Other, true );
		}
	}
	// ZoneInfos.
	else if ( Other.IsA('ZoneInfo') )
	{
		return ModifyZoneInfo( ZoneInfo(Other) );
	}

	// Fallback.
	return ReplaceActor( Other );
}

// ----------------------------------------------------------------------------
// CheckInventoryClassReplacement()
//
// Conveniance interface for ModifyInventoryClass similiar to CheckReplacement.
// ----------------------------------------------------------------------------

function bool CheckInventoryClassReplacement( out Class<Inventory> InventoryClass )
{
	local class<Inventory> NewInventoryClass;

	NewInventoryClass = class<Inventory>(DynamicLoadObject("HX.HX"$InventoryClass.Name,Class'Class',true));
	if ( NewInventoryClass!=None )
		InventoryClass = NewInventoryClass;

	// End of chain anyhow.
	return true;
}

// ----------------------------------------------------------------------------
// CheckActorClassReplacement()
//
// Conveniance interface for ModifyActorClass similiar to CheckReplacement.
// ----------------------------------------------------------------------------

function bool CheckActorClassReplacement( out Class<Actor> ActorClass )
{
	local class<Actor> NewActorClass;

	// Hacks for a few base/special classes. Used for Triggers and such.
	switch ( ActorClass )
	{
		case class'Carcass':          ActorClass = class'Carcass';       return true; break;
		case class'Decoration':       ActorClass = class'Decoration';    return true; break;
		case class'DeusExCarcass':    ActorClass = class'HXCarcass';     return true; break;
		case class'DeusExDecoration': ActorClass = class'HXDecoration';  return true; break;
		case class'DeusExWeapon':     ActorClass = class'HXWeapon';      return true; break;
		case class'MoverCollider':    ActorClass = class'MoverCollider'; return true; break; // Was used for Trigger.ClassProximityType. I doubt it's really needed. --han

		// Nope, don't ask.
		case class'CameraPoint':      ActorClass = class'HXDummyCameraPoint'; return true; break;
	}

	NewActorClass = class<Actor>(DynamicLoadObject("HX.HX"$ActorClass.Name,Class'Class',true));
	if ( NewActorClass!=None && !ClassIsChildOf(NewActorClass,class'Inventory') ) // Filter out Inventory classes here as well.
		ActorClass = NewActorClass;

	// End of chain anyhow.
	return true;
}

// ----------------------------------------------------------------------------
// ReplaceActor()
//
// Conveniance function for quickly spawning the tail successor for an Actor.
// ----------------------------------------------------------------------------

function bool ReplaceActor( Actor Other, optional bool bEvenSpawnWhenStaticNoDelete, optional bool bEvenSpawnForSameClass, optional bool bEvenSpawnWhenOwned )
{
	local class<Actor> SpawnClass;

	SpawnClass = Other.Class;
	Game.ModifyActorClass( SpawnClass );

	return SpawnSuccessor( Other, SpawnClass, bEvenSpawnWhenStaticNoDelete, bEvenSpawnForSameClass, bEvenSpawnWhenOwned );
}

// ----------------------------------------------------------------------------
// ReplaceInventory()
//
// Conveniance function for quickly spawning the tail successor for an Item.
// ----------------------------------------------------------------------------

function bool ReplaceInventory( Inventory Other, optional bool bEvenSpawnWhenStaticNoDelete, optional bool bEvenSpawnForSameClass, optional bool bEvenSpawnWhenOwned )
{
	local class<Inventory> SpawnClass;

	SpawnClass = Other.Class;
	Game.ModifyInventoryClass( SpawnClass );

	return SpawnSuccessor( Other, SpawnClass, bEvenSpawnWhenStaticNoDelete, bEvenSpawnForSameClass, bEvenSpawnWhenOwned );
}

// ----------------------------------------------------------------------------
// ModifyZoneInfo()
// ----------------------------------------------------------------------------

function bool ModifyZoneInfo( ZoneInfo ZoneInfo )
{
	local Class<Actor> Result;

	// Assume EntryActor and ExitActor are not Inventory items.
	if ( ZoneInfo.EntryActor!=None )
		Game.ModifyActorClass( ZoneInfo.EntryActor );
	if ( ZoneInfo.ExitActor!=None )
		Game.ModifyActorClass( ZoneInfo.ExitActor );

	return true;
}

// ----------------------------------------------------------------------------
// ----------------------------------------------------------------------------

defaultproperties
{
}
