//=============================================================================
// HXValve.
//=============================================================================
class HXValve extends HXDecoration;

var() Sound OpenSound, MoveSound, CloseSound;
var() Sound RushingSound;
var() bool bOpen;

// Needed for Triggering.
var transient Actor Actor;

// ----------------------------------------------------------------------------
// PostBeginPlay()
// ----------------------------------------------------------------------------

simulated function PostBeginPlay()
{
	Super.PostBeginPlay();

	if ( Role==ROLE_Authority )
	{
		// Actually not so nice if the player starts next to valve in single player
		// or on a listen server.
		if ( bOpen )
		{
			PlayAnim( 'Open', 10.0 );
			InitialState = 'ValveOpen';
		}
		else
		{
			PlayAnim( 'Close', 10.0 );
			InitialState = 'ValveClosed';
		}
	}
}

// ----------------------------------------------------------------------------
// Succeeds()
//
// Intended to be called by Mutator code if Self was spawned as a successor
// to Other. Super call recommended.
// ----------------------------------------------------------------------------

function Succeeds( Actor Other )
{
	local Valve OtherValve;

	// Call Super.
	Super.Succeeds( Other );

	// Special Valve Init.
	OtherValve = Valve(Other);
	if ( OtherValve!=None )
	{
		bOpen = OtherValve.bOpen;

		// Some special handling, because Ambient sound should just be played in HX when the valve is open.
		RushingSound = OtherValve.AmbientSound;
		OtherValve.AmbientSound = None;
	}
}

// ----------------------------------------------------------------------------
// state ValveOpen
// ----------------------------------------------------------------------------

state ValveOpen
{
	function BeginState()
	{
		bOpen = True;

		// Add flowing water ambient sound.
		AmbientSound = RushingSound;

		// End AI Event.
		AIEndEvent( 'LoudNoise', EAITYPE_Audio );
	}

	function Frob( Actor Frobber, Inventory FrobWith )
	{
		// No Super call to avoid triggering the events before valve is fully closed.

		// Save Frobber (could have used Instigator,  but opted against for minor reasons).
		if ( Instigator==None )
			Instigator = Pawn(Frobber);

		// Start closing the valve.
		GotoState( 'ValveClosing' );
	}
}

// ----------------------------------------------------------------------------
// state ValveClosed
// ----------------------------------------------------------------------------

state ValveClosed
{
	function BeginState()
	{
		bOpen = False;

		// Remove flowing water ambient sound.
		AmbientSound = None;

		// End AI Event.
		AIEndEvent( 'LoudNoise', EAITYPE_Audio );
	}

	function Frob( Actor Frobber, Inventory FrobWith )
	{
		// No Super call to avoid triggering the events before valve is fully open.

		// Save Frobber (could have used Instigator,  but opted against for minor reasons).
		if ( Instigator==None )
			Instigator = Pawn(Frobber);

		// Start opening the valve.
		GotoState( 'ValveOpening' );
	}
}

// ----------------------------------------------------------------------------
// state ValveOpening.
// ----------------------------------------------------------------------------

state ValveOpening
{
	function Frob( Actor Frobber, Inventory FrobWith )
	{
		// Ignore.
	}

Begin:
	// Start Open anim.
	PlayAnim( 'Open' );

	// Start to play OpenSound.
	PlaySound( OpenSound, SLOT_Misc,,, 512 );

	// Notify AI.
	AIStartEvent( 'LoudNoise', EAITYPE_Audio,, 512 );

	// Sleep until the OpenSound is done.
	Sleep( GetSoundDuration(OpenSound) );

	// Start (looped) MoveSound.
	PlaySound( MoveSound, SLOT_Misc,,, 512 );

	// Start ambient sound here.
	AmbientSound = RushingSound;

	// Wait until Open animation is done.
	FinishAnim();

	// Stop MoveSound.
	StopSoundSlot( SLOT_Misc );

	// Trigger Event *after* valve is fully open.
	if ( Event!='' )
		foreach AllActors( Class 'Actor', Actor, Event )
			Actor.Trigger( Self, Instigator );
	Instigator = None;
	Actor = None;

	GotoState( 'ValveOpen' );
}

// ----------------------------------------------------------------------------
// state ValveOpening.
// ----------------------------------------------------------------------------

state ValveClosing
{
	function Frob( Actor Frobber, Inventory FrobWith )
	{
		// Ignore.
	}

Begin:
	// Start Close anim.
	PlayAnim( 'Close' );

	// Start to play CloseSound.
	PlaySound( CloseSound, SLOT_Misc,,, 512 );

	// Notify AI.
	AIStartEvent( 'LoudNoise', EAITYPE_Audio,, 512 );

	// Sleep until the CloseSound is done.
	Sleep( GetSoundDuration(CloseSound) );

	// Start (looped) MoveSound.
	PlaySound( MoveSound, SLOT_Misc,,, 512 );

	// Remove flowing water ambient sound.
	//AmbientSound = None; // Feels to early here..

	// Wait until Close animation is done.
	FinishAnim();

	// Stop MoveSound.
	StopSoundSlot( SLOT_Misc );

	// Trigger Event *after* valve is fully open.
	if ( Event!='' )
		foreach AllActors( Class 'Actor', Actor, Event )
			Actor.Trigger( Self, Instigator );
	Instigator = None;
	Actor = None;

	GotoState( 'ValveClosed' );
}

// ----------------------------------------------------------------------------
// ----------------------------------------------------------------------------

defaultproperties
{
	OpenSound=ValveOpen
	MoveSound=ValveMove
	CloseSound=ValveClose
	RushingSound=WaterRushing
	bInvincible=True
	ItemName="Valve"
	bPushable=False
	Physics=PHYS_None
	Mesh=Valve
	SoundRadius=6
	SoundVolume=48
	SoundPitch=96
	CollisionRadius=7.20
	//CollisionHeight=1.92
	CollisionHeight=1.17
	Mass=20.0
	Buoyancy=10.0
	NetPriority=2.1
}
