//=============================================================================
// HXRobot
//=============================================================================
class HXRobot extends HXScriptedPawn
	abstract;

var(Sounds) sound SearchingSound;
var(Sounds) sound SpeechTargetAcquired;
var(Sounds) sound SpeechTargetLost;
var(Sounds) sound SpeechOutOfAmmo;
var(Sounds) sound SpeechCriticalDamage;
var(Sounds) sound SpeechScanning;
var(Sounds) sound SpeechAreaSecure;

var() int EMPHitPoints;
var ParticleGenerator sparkGen;
var float crazedTimer;

var(Sounds) sound explosionSound;


// ------------------------------------------------------------
// InitGenerator()
// ------------------------------------------------------------

function InitGenerator()
{
	local Vector loc;

	if ((sparkGen == None) || (sparkGen.bDeleteMe))
	{
		loc = Location;
		loc.z += CollisionHeight/2;
		sparkGen = Spawn(class'ParticleGenerator', Self,, loc, rot(16384,0,0));
		if (sparkGen != None)
			sparkGen.SetBase(Self);
	}
}


// ------------------------------------------------------------
// DestroyGenerator()
// ------------------------------------------------------------

function DestroyGenerator()
{
	if (sparkGen != None)
	{
		sparkGen.DelayedDestroy();
		sparkGen = None;
	}
}


// ------------------------------------------------------------
// Tick()
// ------------------------------------------------------------

function Tick(float deltaTime)
{
	local float pct, mod;

	Super.Tick(deltaTime);

   // DEUS_EX AMSD All the MP robots have massive numbers of EMP hitpoints, not equal to the default.  In multiplayer, at least, only do this if
   // they are DAMAGED.
	if ((Default.EMPHitPoints != EMPHitPoints) && (EMPHitPoints != 0) && ((Level.Netmode == NM_Standalone) || (EMPHitPoints < Default.EMPHitPoints)))
	{
		pct = (Default.EMPHitPoints - EMPHitPoints) / Default.EMPHitPoints;
		mod = pct * (1.0 - (2.0 * FRand()));
		DesiredSpeed = MaxDesiredSpeed + (mod * MaxDesiredSpeed * 0.5);
		SoundPitch = Default.SoundPitch + (mod * 8.0);
	}

	if (CrazedTimer > 0)
	{
		CrazedTimer -= deltaTime;
		if (CrazedTimer < 0)
			CrazedTimer = 0;
	}

	if (CrazedTimer > 0)
		bReverseAlliances = true;
	else
		bReverseAlliances = false;
}


// ------------------------------------------------------------
// ShouldFlee()
// ------------------------------------------------------------

function bool ShouldFlee()
{
	return (Health <= MinHealth);
}


// ------------------------------------------------------------
// ShouldDropWeapon()
// ------------------------------------------------------------

function bool ShouldDropWeapon()
{
	return false;
}


// ------------------------------------------------------------
// Destroyed()
// ------------------------------------------------------------

simulated event Destroyed()
{
	Super.Destroyed();

	DestroyGenerator();
}


// ------------------------------------------------------------
// SpawnCarcass()
// ------------------------------------------------------------

function Carcass SpawnCarcass()
{
	Explode(Location);

	return None;
}


// ------------------------------------------------------------
// IgnoreDamageType()
// ------------------------------------------------------------

function bool IgnoreDamageType(Name damageType)
{
	if ((damageType == 'TearGas') || (damageType == 'HalonGas') || (damageType == 'PoisonGas') || (damageType == 'Radiation'))
		return True;
	else if ((damageType == 'Poison') || (damageType == 'PoisonEffect'))
		return True;
	else if (damageType == 'KnockedOut')
		return True;
	else
		return False;
}


// ------------------------------------------------------------
// SetOrders()
// ------------------------------------------------------------

function SetOrders(Name orderName, optional Name newOrderTag, optional bool bImmediate)
{
	if (EMPHitPoints > 0)  // ignore orders if disabled
		Super.SetOrders(orderName, newOrderTag, bImmediate);
}


// ----------------------------------------------------------------------
// TakeDamageBase()
// ----------------------------------------------------------------------

function TakeDamageBase(int Damage, Pawn instigatedBy, Vector hitlocation, Vector momentum, name damageType, bool bPlayAnim)
{
	local float actualDamage;
	local int oldEMPHitPoints;

	// Robots are invincible to EMP in multiplayer as well
	if (( Level.NetMode != NM_Standalone ) && (damageType == 'EMP') && (Self.IsA('MedicalBot') || Self.IsA('RepairBot')) )
		return;

	if ( bInvincible )
		return;

	// robots aren't affected by gas or radiation
	if (IgnoreDamageType(damageType))
		return;

	// enough EMP damage shuts down the robot
	if (damageType == 'EMP')
	{
		oldEMPHitPoints = EMPHitPoints;
		EMPHitPoints   -= Damage;

		// make smoke!
		if (EMPHitPoints <= 0)
		{
			EMPHitPoints = 0;
			if (oldEMPHitPoints > 0)
			{
				PlaySound(sound'EMPZap', SLOT_None,,, (CollisionRadius+CollisionHeight)*8, 2.0);
				InitGenerator();
				if (sparkGen != None)
				{
					sparkGen.LifeSpan = 6;
					sparkGen.particleTexture = Texture'Effects.Smoke.SmokePuff1';
					sparkGen.particleDrawScale = 0.3;
					sparkGen.bRandomEject = False;
					sparkGen.ejectSpeed = 10.0;
					sparkGen.bGravity = False;
					sparkGen.bParticlesUnlit = True;
					sparkGen.frequency = 0.3;
					sparkGen.riseRate = 3;
					sparkGen.spawnSound = Sound'Spark2';
				}
			}
			AmbientSound = None;
			if (GetStateName() != 'Disabled')
				GotoState('Disabled');
		}

		// make sparks!
		else if (sparkGen == None)
		{
			InitGenerator();
			if (sparkGen != None)
			{
				sparkGen.particleTexture = Texture'Effects.Fire.SparkFX1';
				sparkGen.particleDrawScale = 0.2;
				sparkGen.bRandomEject = True;
				sparkGen.ejectSpeed = 100.0;
				sparkGen.bGravity = True;
				sparkGen.bParticlesUnlit = True;
				sparkGen.frequency = 0.2;
				sparkGen.riseRate = 10;
				sparkGen.spawnSound = Sound'Spark2';
			}
		}

		return;
	}
	else if (damageType == 'NanoVirus')
	{
		CrazedTimer += 0.5*Damage;
		return;
	}

	// play a hit sound
	PlayTakeHitSound(Damage, damageType, 1);

	// increase the pitch of the ambient sound when damaged
	if (SoundPitch == Default.SoundPitch)
		SoundPitch += 16;

	actualDamage = Level.Game.ReduceDamage(Damage, DamageType, self, instigatedBy);

	// robots don't have soft, squishy bodies like humans do, so they're less
	// susceptible to gunshots...
	if (damageType == 'Shot')
		actualDamage *= 0.25;  // quarter strength

	// hitting robots with a prod won't stun them, and will only do a limited
	// amount of damage...
	else if ((damageType == 'Stunned') || (damageType == 'KnockedOut'))
		actualDamage *= 0.5;  // half strength

	// flame attacks don't really hurt robots much, either
	else if ((damageType == 'Flamed') || (damageType == 'Burned'))
		actualDamage *= 0.25;  // quarter strength

	if ((actualDamage > 0.01) && (actualDamage < 1))
		actualDamage = 1;
	actualDamage = int(actualDamage+0.5);

	if (ReducedDamageType == 'All') //God mode
		actualDamage = 0;
	else if (Inventory != None) //then check if carrying armor
		actualDamage = Inventory.ReduceDamage(int(actualDamage), DamageType, HitLocation);

	if (!bInvincible)
		Health -= int(actualDamage);

	if (Health <= 0)
	{
		ClearNextState();
		//PlayDeathHit(actualDamage, hitLocation, damageType);
		if ( actualDamage > mass )
			Health = -1 * actualDamage;
		Enemy = instigatedBy;
		Died(instigatedBy, damageType, HitLocation);
	}
	MakeNoise(1.0);

	ReactToInjury(instigatedBy, damageType, HITLOC_None);
}


// ------------------------------------------------------------
// ReactToInjury()
// ------------------------------------------------------------

function ReactToInjury(Pawn instigatedBy, Name damageType, EHitLocation hitPos)
{
	local Pawn oldEnemy;

	if (IgnoreDamageType(damageType))
		return;

	if (EMPHitPoints > 0)
	{
		if (damageType == 'NanoVirus')
		{
			oldEnemy = Enemy;
			FindBestEnemy(false);
			if (oldEnemy != Enemy)
				PlayNewTargetSound();
			instigatedBy = Enemy;
		}
		Super.ReactToInjury(instigatedBy, damageType, hitPos);
	}
}


// ------------------------------------------------------------
// GotoDisabledState()
// ------------------------------------------------------------

function GotoDisabledState(name damageType, EHitLocation hitPos)
{
	if (!bCollideActors && !bBlockActors && !bBlockPlayers)
		return;
	else if (!IgnoreDamageType(damageType) && CanShowPain())
		TakeHit(hitPos);
	else
		GotoNextState();
}


// ------------------------------------------------------------
// Explode()
// ------------------------------------------------------------

function Explode(vector HitLocation)
{
	local int i, num;
	local float explosionRadius;
	local Vector loc;
	local DeusExFragment s;
	local ExplosionLight light;

	explosionRadius = (CollisionRadius + CollisionHeight) / 2;
	PlaySound(explosionSound, SLOT_None, 2.0,, explosionRadius*32);

	if (explosionRadius < 48.0)
		PlaySound(sound'LargeExplosion1', SLOT_None,,, explosionRadius*32);
	else
		PlaySound(sound'LargeExplosion2', SLOT_None,,, explosionRadius*32);

	AISendEvent('LoudNoise', EAITYPE_Audio, , explosionRadius*16);

	// draw a pretty explosion
	light = Spawn(class'ExplosionLight',,, HitLocation);
	for (i=0; i<explosionRadius/20+1; i++)
	{
		loc = Location + VRand() * CollisionRadius;
		if (explosionRadius < 16)
		{
			Spawn(class'ExplosionSmall',,, loc);
			light.size = 2;
		}
		else if (explosionRadius < 32)
		{
			Spawn(class'ExplosionMedium',,, loc);
			light.size = 4;
		}
		else
		{
			Spawn(class'ExplosionLarge',,, loc);
			light.size = 8;
		}
	}

	// spawn some metal fragments
	num = FMax(3, explosionRadius/6);
	for (i=0; i<num; i++)
	{
		s = Spawn(class'MetalFragment', Owner);
		if (s != None)
		{
			s.Instigator = Instigator;
			s.CalcVelocity(Velocity, explosionRadius);
			s.DrawScale = explosionRadius*0.075*FRand();
			s.Skin = GetMeshTexture();
			if (FRand() < 0.75)
				s.bSmoking = True;
		}
	}

	// cause the damage
	HurtRadius(0.5*explosionRadius, 8*explosionRadius, 'Exploded', 100*explosionRadius, Location);
}


// ------------------------------------------------------------
// TweenToRunningAndFiring()
// ------------------------------------------------------------

function TweenToRunningAndFiring(float tweentime)
{
	bIsWalking = FALSE;
	TweenAnimPivot('Run', tweentime);
}


// ------------------------------------------------------------
// PlayRunningAndFiring()
// ------------------------------------------------------------

function PlayRunningAndFiring()
{
	bIsWalking = FALSE;
	LoopAnimPivot('Run');
}


// ------------------------------------------------------------
// TweenToShoot()
// ------------------------------------------------------------

function TweenToShoot(float tweentime)
{
	TweenAnimPivot('Still', tweentime);
}


// ------------------------------------------------------------
// PlayShoot()
// ------------------------------------------------------------

function PlayShoot()
{
	PlayAnimPivot('Still');
}


// ------------------------------------------------------------
// TweenToAttack()
// ------------------------------------------------------------

function TweenToAttack(float tweentime)
{
	TweenAnimPivot('Still', tweentime);
}


// ------------------------------------------------------------
// PlayAttack()
// ------------------------------------------------------------

function PlayAttack()
{
	PlayAnimPivot('Still');
}


// ------------------------------------------------------------
// PlayTurning()
// ------------------------------------------------------------

function PlayTurning()
{
	LoopAnimPivot('Walk');
}


// ------------------------------------------------------------
// TweenToWalking()
// ------------------------------------------------------------

function TweenToWalking(float tweentime)
{
	bIsWalking = True;
	TweenAnimPivot('Walk', tweentime);
}


// ------------------------------------------------------------
// PlayWalking()
// ------------------------------------------------------------

function PlayWalking()
{
	bIsWalking = True;
	LoopAnimPivot('Walk');
}


// ------------------------------------------------------------
// TweenToRunning()
// ------------------------------------------------------------

function TweenToRunning(float tweentime)
{
	bIsWalking = False;
	PlayAnimPivot('Run',, tweentime);
}


// ------------------------------------------------------------
// PlayRunning()
// ------------------------------------------------------------

function PlayRunning()
{
	bIsWalking = False;
	LoopAnimPivot('Run');
}


// ------------------------------------------------------------
// TweenToWaiting()
// ------------------------------------------------------------

function TweenToWaiting(float tweentime)
{
	TweenAnimPivot('Idle', tweentime);
}


// ------------------------------------------------------------
// PlayWaiting()
// ------------------------------------------------------------

function PlayWaiting()
{
	PlayAnimPivot('Idle');
}


// ------------------------------------------------------------
// PlaySwimming()
// ------------------------------------------------------------

function PlaySwimming()
{
	LoopAnimPivot('Still');
}


// ------------------------------------------------------------
// TweenToSwimming()
// ------------------------------------------------------------

function TweenToSwimming(float tweentime)
{
	TweenAnimPivot('Still', tweentime);
}


// ------------------------------------------------------------
// PlayLanded()
// ------------------------------------------------------------

function PlayLanded(float impactVel)
{
	bIsWalking = True;
}


// ------------------------------------------------------------
// PlayDuck()
// ------------------------------------------------------------

function PlayDuck()
{
	TweenAnimPivot('Still', 0.25);
}


// ------------------------------------------------------------
// PlayRising()
// ------------------------------------------------------------

function PlayRising()
{
	PlayAnimPivot('Still');
}


// ------------------------------------------------------------
// PlayCrawling()
// ------------------------------------------------------------

function PlayCrawling()
{
	LoopAnimPivot('Still');
}


// ------------------------------------------------------------
// PlayFiring()
// ------------------------------------------------------------

function PlayFiring()
{
	LoopAnimPivot('Still',,0.1);
}


// ------------------------------------------------------------
// PlayReloadBegin()
// ------------------------------------------------------------

function PlayReloadBegin()
{
	PlayAnimPivot('Still',, 0.1);
}


// ------------------------------------------------------------
// PlayReload()
// ------------------------------------------------------------

function PlayReload()
{
	PlayAnimPivot('Still');
}


// ------------------------------------------------------------
// PlayReloadEnd()
// ------------------------------------------------------------

function PlayReloadEnd()
{
	PlayAnimPivot('Still',, 0.1);
}


// ------------------------------------------------------------
// PlayDisabled()
// ------------------------------------------------------------

function PlayDisabled()
{
	TweenAnimPivot('Still', 0.2);
}


// ------------------------------------------------------------
// PlayScanningSound()
// ------------------------------------------------------------

function PlayScanningSound()
{
	PlaySound(SearchingSound, SLOT_None,,, 2048);
	PlaySound(SpeechScanning, SLOT_None,,, 2048);
}


// ------------------------------------------------------------
// PlaySearchingSound()
// ------------------------------------------------------------

function PlaySearchingSound()
{
	PlaySound(SearchingSound, SLOT_None,,, 2048);
	PlaySound(SpeechScanning, SLOT_None,,, 2048);
}


// ------------------------------------------------------------
// PlayTargetAcquiredSound()
// ------------------------------------------------------------

function PlayTargetAcquiredSound()
{
	PlaySound(SpeechTargetAcquired, SLOT_None,,, 2048);
}


// ------------------------------------------------------------
// PlayTargetLostSound()
// ------------------------------------------------------------

function PlayTargetLostSound()
{
	PlaySound(SpeechTargetLost, SLOT_None,,, 2048);
}


// ------------------------------------------------------------
// PlayOutOfAmmoSound()
// ------------------------------------------------------------

function PlayOutOfAmmoSound()
{
	PlaySound(SpeechOutOfAmmo, SLOT_None,,, 2048);
}


// ------------------------------------------------------------
// PlayCriticalDamageSound()
// ------------------------------------------------------------

function PlayCriticalDamageSound()
{
	PlaySound(SpeechCriticalDamage, SLOT_None,,, 2048);
}


// ------------------------------------------------------------
// PlayAreaSecureSound()
// ------------------------------------------------------------

function PlayAreaSecureSound()
{
	PlaySound(SpeechAreaSecure, SLOT_None,,, 2048);
}


// ------------------------------------------------------------
// state Disabled
// ------------------------------------------------------------

state Disabled
{
	ignores bump, frob, reacttoinjury;
	function BeginState()
	{
		StandUp();
		BlockReactions(true);
		bCanConverse = False;
		SeekPawn = None;
	}
	function EndState()
	{
		ResetReactions();
		bCanConverse = True;
	}

Begin:
	Acceleration=vect(0,0,0);
	DesiredRotation=Rotation;
	PlayDisabled();

Disabled:
}


// ------------------------------------------------------------
// state Fleeing
// ------------------------------------------------------------

state Fleeing
{
	function PickDestination()
	{
		local int     iterations;
		local float   magnitude;
		local rotator rot1;

		iterations = 4;
		magnitude  = 400*(FRand()*0.4+0.8);  // 400, +/-20%
		rot1       = Rotator(Location-Enemy.Location);
		if (!AIPickRandomDestination(40, magnitude, rot1.Yaw, 0.6, rot1.Pitch, 0.6, iterations,
		                             FRand()*0.4+0.35, destLoc))
			destLoc = Location;  // we give up
	}
}


// ------------------------------------------------------------
// IsImmobile()
// If the bots are immobile, then we can make them always relevant
// ------------------------------------------------------------

function bool IsImmobile()
{
   local bool bHasReactions;
   local bool bHasFears;
   local bool bHasHates;

   if (Orders != 'Standing')
      return false;

   bHasReactions = bReactFutz || bReactPresence || bReactLoudNoise || bReactAlarm || bReactShot || bReactCarcass || bReactDistress || bReactProjectiles;

   bHasFears = bFearHacking || bFearWeapon || bFearShot || bFearInjury || bFearIndirectInjury || bFearCarcass || bFearDistress || bFearAlarm || bFearProjectiles;

   bHasHates = bHateHacking || bHateWeapon || bHateShot || bHateInjury || bHateIndirectInjury || bHateCarcass || bHateDistress;

   return (!bHasReactions && !bHasFears && !bHasHates);
}


// ------------------------------------------------------------
// stubbed out functions...
// ------------------------------------------------------------

function ImpartMomentum(Vector momentum, Pawn instigatedBy) {}
function ComputeFallDirection(float totalTime, int numFrames, out vector moveDir, out float stopTime) {}
function PlayFalling() {}
function PlayCowerBegin() {}
function PlayCowering() {}
function PlayCowerEnd() {}
function PlayWeaponSwitch(Weapon newWeapon) {}
function PlayIdleSound() {}
function PlayGoingForAlarmSound() {}


// ------------------------------------------------------------
// ------------------------------------------------------------

defaultproperties
{
	EMPHitPoints=50
	explosionSound=Sound'DeusExSounds.Robot.RobotExplode'
	maxRange=512.000000
	MinHealth=0.000000
	RandomWandering=0.150000
	bCanBleed=False
	bShowPain=False
	bCanSit=False
	bAvoidAim=False
	bAvoidHarm=False
	bHateShot=False
	bReactAlarm=True
	bReactProjectiles=False
	bEmitDistress=False
	RaiseAlarm=RAISEALARM_Never
	bMustFaceTarget=False
	FireAngle=60.000000
	MaxProvocations=0
	SurprisePeriod=0.000000
	EnemyTimeout=7.000000
	walkAnimMult=1.000000
	bCanStrafe=False
	bCanSwim=False
	bIsHuman=False
	JumpZ=0.000000
	MaxStepHeight=4.000000
	Health=50
	HitSound1=Sound'DeusExSounds.Generic.Spark1'
	HitSound2=Sound'DeusExSounds.Generic.Spark1'
	Die=Sound'DeusExSounds.Generic.Spark1'
	VisibilityThreshold=0.006000
	BindName="Robot"
}
