//=============================================================================
// HXCoopWeapon
//=============================================================================
class HXCoopWeapon extends DeusExWeapon
	abstract;


// ----------------------------------------------------------------------
// PreBeginPlay()
// ----------------------------------------------------------------------

function PreBeginPlay()
{
	Super(Weapon).PreBeginPlay();
}


// ----------------------------------------------------------------------
// PostBeginPlay()
// ----------------------------------------------------------------------

function PostBeginPlay()
{
	Super(Weapon).PostBeginPlay();
}


// ----------------------------------------------------------------------
// SetWeaponStay()
// ----------------------------------------------------------------------

function SetWeaponStay()
{
	//bWeaponStay = bWeaponStay || Level.Game.bCoopWeaponMode;
	bWeaponStay = false;
}


// ----------------------------------------------------------------------
// CalculateAccuracy()
// ----------------------------------------------------------------------

simulated function float CalculateAccuracy()
{
	local float accuracy;	// 0 is dead on, 1 is pretty far off
	local float tempacc, div;
	local float weapskill; // so we don't keep looking it up (slower).
	local int HealthArmRight, HealthArmLeft, HealthHead;
	local int BestArmRight, BestArmLeft, BestHead;
	local bool checkit;
	local DeusExPlayer player;

	accuracy = BaseAccuracy;		// start with the weapon's base accuracy
	weapskill = GetWeaponSkill();

	player = DeusExPlayer(Owner);

	if (player != None)
	{
		// check the player's skill
		// 0.0 = dead on, 1.0 = way off
		accuracy += weapskill;

		// get the health values for the player
		HealthArmRight = player.HealthArmRight;
		HealthArmLeft  = player.HealthArmLeft;
		HealthHead     = player.HealthHead;
		BestArmRight   = player.Default.HealthArmRight;
		BestArmLeft    = player.Default.HealthArmLeft;
		BestHead       = player.Default.HealthHead;
		checkit = True;
	}
	else if (ScriptedPawn(Owner) != None)
	{
		// update the weapon's accuracy with the ScriptedPawn's BaseAccuracy
		// (BaseAccuracy uses higher values for less accuracy, hence we add)
		accuracy += ScriptedPawn(Owner).BaseAccuracy;

		// get the health values for the NPC
		HealthArmRight = ScriptedPawn(Owner).HealthArmRight;
		HealthArmLeft  = ScriptedPawn(Owner).HealthArmLeft;
		HealthHead     = ScriptedPawn(Owner).HealthHead;
		BestArmRight   = ScriptedPawn(Owner).Default.HealthArmRight;
		BestArmLeft    = ScriptedPawn(Owner).Default.HealthArmLeft;
		BestHead       = ScriptedPawn(Owner).Default.HealthHead;
		checkit = True;
	}
	else
		checkit = False;

	if (checkit)
	{
		if (HealthArmRight < 1)
			accuracy += 0.5;
		else if (HealthArmRight < BestArmRight * 0.34)
			accuracy += 0.2;
		else if (HealthArmRight < BestArmRight * 0.67)
			accuracy += 0.1;

		if (HealthArmLeft < 1)
			accuracy += 0.5;
		else if (HealthArmLeft < BestArmLeft * 0.34)
			accuracy += 0.2;
		else if (HealthArmLeft < BestArmLeft * 0.67)
			accuracy += 0.1;

		if (HealthHead < BestHead * 0.67)
			accuracy += 0.1;
	}

	// increase accuracy (decrease value) if we haven't been moving for awhile
	// this only works for the player, because NPCs don't need any more aiming help!
	if (player != None)
	{
		tempacc = accuracy;
		if (standingTimer > 0)
		{
			// higher skill makes standing bonus greater
			div = Max(15.0 + 29.0 * weapskill, 0.0);
			accuracy -= FClamp(standingTimer/div, 0.0, 0.6);
	
			// don't go too low
			if ((accuracy < 0.1) && (tempacc > 0.1))
				accuracy = 0.1;
		}
	}

	// make sure we don't go negative
	if (accuracy < 0.0)
		accuracy = 0.0;

	return accuracy;
}


// ----------------------------------------------------------------------
// LoadAmmo()
// ----------------------------------------------------------------------

function bool LoadAmmo(int ammoNum)
{
	local class<Ammo> newAmmoClass;
	local Ammo newAmmo;
	local Pawn P;

	if ((ammoNum < 0) || (ammoNum > 2))
		return False;

	P = Pawn(Owner);

	// sorry, only pawns can have weapons
	if (P == None)
		return False;

	newAmmoClass = AmmoNames[ammoNum];

	if (newAmmoClass != None)
	{
		if (newAmmoClass != AmmoName)
		{
			newAmmo = Ammo(P.FindInventoryType(newAmmoClass));
			if (newAmmo == None)
			{
				P.ClientMessage(Sprintf(msgOutOf, newAmmoClass.Default.ItemName));
				return False;
			}
			
			// if we don't have a projectile for this ammo type, then set instant hit
			if (ProjectileNames[ammoNum] == None)
			{
				bInstantHit = True;
				bAutomatic = Default.bAutomatic;
				ShotTime = Default.ShotTime;

				if (HasReloadMod())
					ReloadTime = Default.ReloadTime * (1.0+ModReloadTime);
				else
					ReloadTime = Default.ReloadTime;

				FireSound = Default.FireSound;
				ProjectileClass = None;
			}
			else
			{
				// otherwise, set us to fire projectiles
				bInstantHit = False;
				bAutomatic = False;
				ShotTime = 1.0;
				if (HasReloadMod())
					ReloadTime = 2.0 * (1.0+ModReloadTime);
				else
					ReloadTime = 2.0;
				FireSound = None;		// handled by the projectile
				ProjectileClass = ProjectileNames[ammoNum];
				ProjectileSpeed = ProjectileClass.Default.Speed;
			}

			AmmoName = newAmmoClass;
			AmmoType = newAmmo;

			// AlexB had a new sound for 20mm but there's no mechanism for playing alternate sounds per ammo type
			// Same for WP rocket
			if (newAmmo.isA('HXAmmo20mm'))
				FireSound=Sound'AssaultGunFire20mm';
			else if (newAmmo.isA('HXAmmoRocketWP'))
				FireSound=Sound'GEPGunFireWP';
			else if (newAmmo.isA('HXAmmoRocket'))
				FireSound=Sound'GEPGunFire';

			if (Level.NetMode != NM_Standalone)
				SetClientAmmoParams(bInstantHit, bAutomatic, ShotTime, FireSound, ProjectileClass, ProjectileSpeed);

			// Notify the object belt of the new ammo
			if (P.isA('DeusExPlayer'))
				DeusExPlayer(P).UpdateBeltText(Self);

			ReloadAmmo();

			P.ClientMessage(Sprintf(msgNowHas, ItemName, newAmmoClass.Default.ItemName));

			return True;
		}
		else
		{
			P.ClientMessage(Sprintf(MsgAlreadyHas, ItemName, newAmmoClass.Default.ItemName));
		}
	}

	return False;
}


// ----------------------------------------------------------------------
// NearWallCheck()
// ----------------------------------------------------------------------

simulated function bool NearWallCheck()
{
	local Vector StartTrace, EndTrace, HitLocation, HitNormal;
	local Actor HitActor;

	// Scripted pawns can't place LAMs
	if (Owner.isA('ScriptedPawn'))
		return false;

	// trace out one foot in front of the pawn
	StartTrace = Owner.Location;
	EndTrace = StartTrace + Vector(Pawn(Owner).ViewRotation) * 32;

	StartTrace.Z += Pawn(Owner).BaseEyeHeight;
	EndTrace.Z += Pawn(Owner).BaseEyeHeight;

	HitActor = Trace(HitLocation, HitNormal, EndTrace, StartTrace);

	if ((HitActor == Level) || ((HitActor != None) && HitActor.IsA('Mover')))
	{
		placeLocation = HitLocation;
		placeNormal = HitNormal;
		placeMover = Mover(HitActor);
		return True;
	}

	return False;
}


// ----------------------------------------------------------------------
// Tick()
// ----------------------------------------------------------------------

simulated function Tick(float deltaTime)
{
	local vector loc;
	local rotator rot;
	local float beepspeed, recoil;
	local DeusExPlayer player;
  local Actor RealTarget;
	local Pawn pawn;

	player = DeusExPlayer(Owner);
	pawn = Pawn(Owner);

	Super.Tick(deltaTime);

	// don't do any of this if this weapon isn't currently in use
	if (pawn == None)
	{
		LockMode = LOCK_None;
		MaintainLockTimer = 0;
		LockTarget = None;
		LockTimer = 0;
		return;
	}

	if (pawn.Weapon != self)
	{
		LockMode = LOCK_None;
		MaintainLockTimer = 0;
		LockTarget = None;
		LockTimer = 0;
		return;
	}

	// all this should only happen IF you have ammo loaded
	if (ClipCount < ReloadCount)
	{
		// check for LAM or other placed mine placement
		if (bHandToHand && (ProjectileClass != None) && (!Self.IsA('HXWeaponShuriken')))
		{
			if (NearWallCheck())
			{
				if (!bNearWall || (AnimSequence == 'Select'))
				{
					PlayAnim('PlaceBegin',, 0.1);
					bNearWall = True;
				}
			}
			else
			{
				if (bNearWall)
				{
					PlayAnim('PlaceEnd',, 0.1);
					bNearWall = False;
				}
			}
		}


		SoundTimer += deltaTime;

		if ((Level.Netmode == NM_Standalone) || ( (Player != None) && (Player.PlayerIsClient())))
		{
			if (bCanTrack)
			{
				Target = AcquireTarget();
				RealTarget = Target;

				// calculate the range
				if (Target != None)
					 TargetRange = Abs(VSize(Target.Location - Location));

				// update our timers
				//SoundTimer += deltaTime;
				MaintainLockTimer -= deltaTime;

				// check target and range info to see what our mode is
				if ((Target == None) || IsInState('Reload'))
				{
					if (MaintainLockTimer <= 0)
					{				
						SetLockMode(LOCK_None);
						MaintainLockTimer = 0;
						LockTarget = None;
					}
					else if (LockMode == LOCK_Locked)
					{
						Target = LockTarget;
					}
				}
				else if ((Target != LockTarget) && (Target.IsA('Pawn')) && (LockMode == LOCK_Locked))
				{
					SetLockMode(LOCK_None);
					LockTarget = None;
				}
				else if (!Target.IsA('Pawn'))
				{
					if (MaintainLockTimer <=0 )
					{
						SetLockMode(LOCK_Invalid);
					}
				}
				else if ( (Target.IsA('DeusExPlayer')) && (Target.Style == STY_Translucent) )
				{
					//DEUS_EX AMSD Don't allow locks on cloaked targets.
					SetLockMode(LOCK_Invalid);
				}
				//else if ( (Target.IsA('DeusExPlayer')) && (Player.DXGame.IsA('TeamDMGame')) && (TeamDMGame(Player.DXGame).ArePlayersAllied(Player,DeusExPlayer(Target))) )
				//{
					//DEUS_EX AMSD Don't allow locks on allies.
					//SetLockMode(LOCK_Invalid);
				//}
				else
				{
					if (TargetRange > MaxRange)
					{
						SetLockMode(LOCK_Range);
					}
					else
					{
						// change LockTime based on skill
						// -0.7 = max skill
						// DEUS_EX AMSD Only do weaponskill check here when first checking.
						if (LockTimer == 0)
						{
							LockTime = FMax(Default.LockTime + 3.0 * GetWeaponSkill(), 0.0);
							//if ((Level.Netmode != NM_Standalone) && (LockTime < 0.25))
								//LockTime = 0.25;
						}

						LockTimer += deltaTime;
						if (LockTimer >= LockTime)
						{
							SetLockMode(LOCK_Locked);
						}
						else
						{
							SetLockMode(LOCK_Acquire);
						}
					}
				}
            
				// act on the lock mode
				switch (LockMode)
				{
					case LOCK_None:
						TargetMessage = msgNone;
						LockTimer -= deltaTime;
						break;

					case LOCK_Invalid:
						TargetMessage = msgLockInvalid;
						LockTimer -= deltaTime;
						break;

					case LOCK_Range:
						TargetMessage = msgLockRange @ Int(TargetRange/16) @ msgRangeUnit;
						LockTimer -= deltaTime;
						break;

					case LOCK_Acquire:
						TargetMessage = msgLockAcquire @ Left(String(LockTime-LockTimer), 4) @ msgTimeUnit;
						beepspeed = FClamp((LockTime - LockTimer) / Default.LockTime, 0.2, 1.0);
						if (SoundTimer > beepspeed)
						{
							Owner.PlaySound(TrackingSound, SLOT_None);
							SoundTimer = 0;
						}
						break;

					case LOCK_Locked:
						// If maintaining a lock, or getting a new one, increment maintainlocktimer
						if ((RealTarget != None) && ((RealTarget == LockTarget) || (LockTarget == None)))
						{
							//if (Level.NetMode != NM_Standalone)
								//MaintainLockTimer = default.MaintainLockTimer;
							//else
							MaintainLockTimer = 0;

							LockTarget = Target;
						}
						TargetMessage = msgLockLocked @ Int(TargetRange/16) @ msgRangeUnit;
						// DEUS_EX AMSD Moved out so server can play it so that client knows for sure when locked.
						/*if (SoundTimer > 0.1)
						{
						Owner.PlaySound(LockedSound, SLOT_None);
						SoundTimer = 0;
						}*/
						break;

				}
			}
			else
			{
				LockMode = LOCK_None;
				TargetMessage = msgNone;
				LockTimer = 0;
				MaintainLockTimer = 0;
				LockTarget = None;
			}

			if (LockTimer < 0)
				LockTimer = 0;
    }
	}
	else
	{
		LockMode = LOCK_None;
		TargetMessage=msgNone;
		MaintainLockTimer = 0;
		LockTarget = None;
		LockTimer = 0;
	}

	if ((LockMode == LOCK_Locked) && (SoundTimer > 0.1) && (Role == ROLE_Authority))
	{
		PlayLockSound();
		SoundTimer = 0;
	}

	currentAccuracy = CalculateAccuracy();

	if (player != None)
	{
		// reduce the recoil based on skill
		recoil = recoilStrength + GetWeaponSkill() * 2.0;
		if (recoil < 0.0)
			recoil = 0.0;

		// simulate recoil while firing
		if (bFiring && IsAnimating() && (AnimSequence == 'Shoot') && (recoil > 0.0))
		{
			player.ViewRotation.Yaw += deltaTime * (Rand(4096) - 2048) * recoil;
			player.ViewRotation.Pitch += deltaTime * (Rand(4096) + 4096) * recoil;

			if ((player.ViewRotation.Pitch > 16384) && (player.ViewRotation.Pitch < 32768))
				player.ViewRotation.Pitch = 16384;
		}
	}

	// if were standing still, increase the timer
	if (VSize(Owner.Velocity) < 10)
		standingTimer += deltaTime;
	else	// otherwise, decrease it slowly based on velocity
		standingTimer = FMax(0, standingTimer - 0.03*deltaTime*VSize(Owner.Velocity));

	if (bLasing || bZoomed)
	{
		// shake our view to simulate poor aiming
		if (ShakeTimer > 0.25)
		{
			ShakeYaw = currentAccuracy * (Rand(4096) - 2048);
			ShakePitch = currentAccuracy * (Rand(4096) - 2048);
			ShakeTimer -= 0.25;
		}

		ShakeTimer += deltaTime;

		if (bLasing && (Emitter != None))
		{
			loc = Owner.Location;
			loc.Z += Pawn(Owner).BaseEyeHeight;

			// add a little random jitter - looks cool!
			rot = Pawn(Owner).ViewRotation;
			rot.Yaw += Rand(5) - 2;
			rot.Pitch += Rand(5) - 2;

			Emitter.SetLocation(loc);
			Emitter.SetRotation(rot);
		}

		if ((player != None) && bZoomed)
		{
			player.ViewRotation.Yaw += deltaTime * ShakeYaw;
			player.ViewRotation.Pitch += deltaTime * ShakePitch;
		}
	}
}


// ----------------------------------------------------------------------
// ClientFire()
// ----------------------------------------------------------------------

simulated function bool ClientFire(float value)
{
	local bool bWaitOnAnim;
	local vector shake;

	// check for surrounding environment
	if ((EnviroEffective == ENVEFF_Air) || (EnviroEffective == ENVEFF_Vacuum) || (EnviroEffective == ENVEFF_AirVacuum))
	{
		if (Region.Zone.bWaterZone)
		{
			if (Pawn(Owner) != None)
			{
				Pawn(Owner).ClientMessage(msgNotWorking);
				if (!bHandToHand)
					PlaySimSound( Misc1Sound, SLOT_None, TransientSoundVolume * 2.0, 1024 );
			}
			return false;
		}
	}

	if ( !bLooping ) // Wait on animations when not looping
	{
		bWaitOnAnim = ( IsAnimating() && ((AnimSequence == 'Select') || (AnimSequence == 'Shoot') || (AnimSequence == 'ReloadBegin') || (AnimSequence == 'Reload') || (AnimSequence == 'ReloadEnd') || (AnimSequence == 'Down')));
	}
	else
	{
		bWaitOnAnim = False;
		bLooping = False;
	}

	if ( (Owner.IsA('DeusExPlayer') && (DeusExPlayer(Owner).NintendoImmunityTimeLeft > 0.01)) ||
		  (!bClientReadyToFire) || bInProcess || bWaitOnAnim )
	{
		DeusExPlayer(Owner).bJustFired = False;
		bPointing = False;
		bFiring = False;
		return false;
	}

	if (!Self.IsA('HXWeaponFlamethrower'))
		ServerForceFire();

	if (bHandToHand)
	{
		SimAmmoAmount = AmmoType.AmmoAmount - 1;

		bClientReadyToFire = False;
		bInProcess = True;
		GotoState('ClientFiring');
		bPointing = True;

		if (Owner.isA('PlayerPawn'))
			PlayerPawn(Owner).PlayFiring();

		PlaySelectiveFiring();
		PlayFiringSound();
	}
	else if ((ClipCount < ReloadCount) || (ReloadCount == 0))
	{
		if ((ReloadCount == 0) || (AmmoType.AmmoAmount > 0))
		{
			SimClipCount = ClipCount + 1;

			if ( AmmoType != None )
				AmmoType.SimUseAmmo();

			bFiring = True;
			bPointing = True;
			bClientReadyToFire = False;
			bInProcess = True;
			GotoState('ClientFiring');

			if (Owner.isA('PlayerPawn'))
			{
				shake.X = 0.0;
				shake.Y = 100.0 * (ShakeTime*0.5);
				shake.Z = 100.0 * -(currentAccuracy * ShakeVert);
				PlayerPawn(Owner).ClientShake( shake );
				PlayerPawn(Owner).PlayFiring();
			}

			// Don't play firing anim for 20mm
			if (!AmmoType.isA('HXAmmo20mm'))
				PlaySelectiveFiring();

			PlayFiringSound();

			if (bInstantHit &&  !AmmoType.isA('HXAmmo20mm'))
				TraceFire(currentAccuracy);

			else
			{
				if ( !bFlameOn && Self.IsA('HXWeaponFlamethrower'))
				{
					bFlameOn = True;
					StartFlame();
				}
				ProjectileFire(ProjectileClass, ProjectileSpeed, bWarnTarget);
			}
		}
		else
		{
			if ( Owner.IsA('DeusExPlayer') && DeusExPlayer(Owner).bAutoReload )
			{
				if ( MustReload() && CanReload() )
				{
					bClientReadyToFire = False;
					bInProcess = False;
					if ((AmmoType.AmmoAmount == 0) && (AmmoName != AmmoNames[0]))
						CycleAmmo();

					ReloadAmmo();
				}
			}
			PlaySimSound( Misc1Sound, SLOT_None, TransientSoundVolume * 2.0, 1024 );		// play dry fire sound
		}
	}
	else
	{
		if ( Owner.IsA('DeusExPlayer') && DeusExPlayer(Owner).bAutoReload )
		{
			if ( MustReload() && CanReload() )
			{
				bClientReadyToFire = False;
				bInProcess = False;
				if ((AmmoType.AmmoAmount == 0) && (AmmoName != AmmoNames[0]))
					CycleAmmo();
				ReloadAmmo();
			}
		}
		PlaySimSound( Misc1Sound, SLOT_None, TransientSoundVolume * 2.0, 1024 );		// play dry fire sound
	}
	return true;
}


// ----------------------------------------------------------------------
// Fire()
// ----------------------------------------------------------------------

function Fire(float Value)
{
	local float sndVolume;
	local bool bListenClient;

	bListenClient = (Owner.IsA('DeusExPlayer') && DeusExPlayer(Owner).PlayerIsListenClient());

	sndVolume = TransientSoundVolume;

	if (Level.NetMode != NM_Standalone)
	{
		// Turn up the sounds a bit in mulitplayer
		//sndVolume = TransientSoundVolume * 2.0; // HX_NOTE: disabled for now
		if (Owner.IsA('DeusExPlayer'))
		{
			if (DeusExPlayer(Owner).NintendoImmunityTimeLeft > 0.01 || (!bClientReady && !bListenClient))
			{
				DeusExPlayer(Owner).bJustFired = False;
				bReadyToFire = True;
				bPointing = False;
				bFiring = False;
				return;
			}
		}
	}

	// check for surrounding environment
	if ((EnviroEffective == ENVEFF_Air) || (EnviroEffective == ENVEFF_Vacuum) || (EnviroEffective == ENVEFF_AirVacuum))
	{
		if (Region.Zone.bWaterZone)
		{
			if (Owner.isA('Pawn'))
			{
				Pawn(Owner).ClientMessage(msgNotWorking);
				if (!bHandToHand)
					PlaySimSound( Misc1Sound, SLOT_None, sndVolume, 1024 );		// play dry fire sound
			}
			GotoState('Idle');
			return;
		}
	}

	if (bHandToHand)
	{
		if (ReloadCount > 0)
			AmmoType.UseAmmo(1);

		if ((Level.NetMode != NM_Standalone) && !bListenClient )
			bClientReady = False;

		bReadyToFire = False;
		GotoState('NormalFire');
		bPointing=True;

		if (Owner.IsA('PlayerPawn'))
			PlayerPawn(Owner).PlayFiring();

		PlaySelectiveFiring();
		PlayFiringSound();
	}

	// if we are a single-use weapon, then our ReloadCount is 0 and we don't use ammo
	else if ((ClipCount < ReloadCount) || (ReloadCount == 0))
	{
		if ((ReloadCount == 0) || AmmoType.UseAmmo(1))
		{
			if ((Level.NetMode != NM_Standalone ) && !bListenClient)
				bClientReady = False;

			ClipCount++;
			bFiring = True;
			bReadyToFire = False;

			GotoState('NormalFire');

			if ((Level.NetMode == NM_Standalone) || ( Owner.IsA('DeusExPlayer') && DeusExPlayer(Owner).PlayerIsListenClient()))
			{
				if (Owner.isA('PlayerPawn'))		// shake us based on accuracy
					PlayerPawn(Owner).ShakeView(ShakeTime, currentAccuracy * ShakeMag + ShakeMag, currentAccuracy * ShakeVert);
			}

			bPointing = True;

			if (bInstantHit)
				TraceFire(currentAccuracy);
			else
				ProjectileFire(ProjectileClass, ProjectileSpeed, bWarnTarget);

			if (Owner.IsA('PlayerPawn'))
				PlayerPawn(Owner).PlayFiring();

			// Don't play firing anim for 20mm
			if (!AmmoType.isA('HXAmmo20mm'))
				PlaySelectiveFiring();

			PlayFiringSound();

			if (Owner.bHidden)
				CheckVisibility();

		}
		else
			PlaySimSound(Misc1Sound, SLOT_None, sndVolume, 1024); // play dry fire sound
	}
	else
		PlaySimSound(Misc1Sound, SLOT_None, sndVolume, 1024); // play dry fire sound

	// Update ammo count on object belt
	if (Owner.isA('DeusExPlayer'))
		DeusExPlayer(Owner).UpdateBeltText(Self);
}


// ----------------------------------------------------------------------
// PlayFiringSound()
// ----------------------------------------------------------------------

simulated function PlayFiringSound()
{
	if (bHasSilencer)
		PlaySimSound(Sound'StealthPistolFire', SLOT_None, TransientSoundVolume, 2048);
	else
		PlaySimSound(FireSound, SLOT_None, TransientSoundVolume, 2048);
}


// ----------------------------------------------------------------------
// ProjectileFire()
// ----------------------------------------------------------------------

simulated function Projectile ProjectileFire(class<projectile> ProjClass, float ProjSpeed, bool bWarn)
{
	local Vector Start, X, Y, Z;
	local DeusExProjectile proj;
	local float mult;
	local float volume, radius;
	local int i, numProj;
	local Pawn aPawn;

	// AugCombat increases our speed (distance) if hand to hand
	mult = 1.0;
	if (bHandToHand && (DeusExPlayer(Owner) != None))
	{
		mult = DeusExPlayer(Owner).AugmentationSystem.GetAugLevelValue(class'AugCombat');
		if (mult == -1.0)
			mult = 1.0;
		ProjSpeed *= mult;
	}

	// skill also affects our damage
	// GetWeaponSkill returns 0.0 to -0.7 (max skill/aug)
	mult += -2.0 * GetWeaponSkill();

	// make noise if we are not silenced
	if (!bHasSilencer && !bHandToHand)
	{
		GetAIVolume(volume, radius);
		Owner.AISendEvent('WeaponFire', EAITYPE_Audio, volume, radius);
		Owner.AISendEvent('LoudNoise', EAITYPE_Audio, volume, radius);
		if (!Owner.IsA('PlayerPawn'))
			Owner.AISendEvent('Distress', EAITYPE_Audio, volume, radius);
	}

	// should we shoot multiple projectiles in a spread?
	if (AreaOfEffect == AOE_Cone)
		numProj = 3;
	else
		numProj = 1;

	GetAxes(Pawn(owner).ViewRotation,X,Y,Z);
	Start = ComputeProjectileStart(X, Y, Z);

	for (i=0; i<numProj; i++)
	{
    // If we have multiple slugs, then lower our accuracy a bit after the first slug so the slugs DON'T all go to the same place
    //if ((i > 0) && (Level.NetMode != NM_Standalone))
       //if (currentAccuracy < MinProjSpreadAcc)
          //currentAccuracy = MinProjSpreadAcc;
         
		AdjustedAim = pawn(owner).AdjustAim(ProjSpeed, Start, AimError, True, bWarn);
		AdjustedAim.Yaw += currentAccuracy * (Rand(1024) - 512);
		AdjustedAim.Pitch += currentAccuracy * (Rand(1024) - 512);


		if (( Level.NetMode == NM_Standalone ) || ( Owner.IsA('DeusExPlayer') && DeusExPlayer(Owner).PlayerIsListenClient()) )
		{
			proj = DeusExProjectile(Spawn(ProjClass, Owner,, Start, AdjustedAim));
			if (proj != None)
			{
				// AugCombat increases our damage as well
				proj.Damage *= mult;

				// send the targetting information to the projectile
				if (bCanTrack && (LockTarget != None) && (LockMode == LOCK_Locked))
				{
					proj.Target = LockTarget;
					proj.bTracking = True;
				}
			}
		}
		else
		{
			if (( Role == ROLE_Authority ) || (DeusExPlayer(Owner) == DeusExPlayer(GetPlayerPawn())) )
			{
				// Do it the old fashioned way if it can track, or if we are a projectile that we could pick up again
				if ( bCanTrack || Self.IsA('HXWeaponShuriken') || Self.IsA('HXWeaponMiniCrossbow') || Self.IsA('HXWeaponLAM') || Self.IsA('HXWeaponEMPGrenade') || Self.IsA('HXWeaponGasGrenade'))
				{
					if ( Role == ROLE_Authority )
					{
						proj = DeusExProjectile(Spawn(ProjClass, Owner,, Start, AdjustedAim));
						if (proj != None)
						{
							// AugCombat increases our damage as well
								proj.Damage *= mult;
							// send the targetting information to the projectile
							if (bCanTrack && (LockTarget != None) && (LockMode == LOCK_Locked))
							{
								proj.Target = LockTarget;
								proj.bTracking = True;
							}
						}
					}
				}
				else
				{
					proj = DeusExProjectile(Spawn(ProjClass, Owner,, Start, AdjustedAim));
					if (proj != None)
					{
						proj.RemoteRole = ROLE_None;
						// AugCombat increases our damage as well
						if ( Role == ROLE_Authority )
							proj.Damage *= mult;
						else
							proj.Damage = 0;
					}
					if ( Role == ROLE_Authority )
					{
						for ( aPawn = Level.PawnList; aPawn != None; aPawn = aPawn.nextPawn )
						{
							if ( aPawn.IsA('DeusExPlayer') && ( DeusExPlayer(aPawn) != DeusExPlayer(Owner) ))
								DeusExPlayer(aPawn).ClientSpawnProjectile( ProjClass, Owner, Start, AdjustedAim );
						}
					}
				}
			}
		}

	}
	return proj;
}


// ----------------------------------------------------------------------
// TraceFire()
// ----------------------------------------------------------------------

simulated function TraceFire(float Accuracy)
{
	local vector HitLocation, HitNormal, StartTrace, EndTrace, X, Y, Z;
	local Rotator rot;
	local actor Other;
	local float dist, alpha, degrade;
	local int i, numSlugs;
	local float volume, radius;

	// make noise if we are not silenced
	if (!bHasSilencer && !bHandToHand)
	{
		GetAIVolume(volume, radius);
		Owner.AISendEvent('WeaponFire', EAITYPE_Audio, volume, radius);
		Owner.AISendEvent('LoudNoise', EAITYPE_Audio, volume, radius);
		if (!Owner.IsA('PlayerPawn'))
			Owner.AISendEvent('Distress', EAITYPE_Audio, volume, radius);
	}

	GetAxes(Pawn(owner).ViewRotation,X,Y,Z);
	StartTrace = ComputeProjectileStart(X, Y, Z);
	AdjustedAim = pawn(owner).AdjustAim(1000000, StartTrace, 2.75*AimError, False, False);

	// check to see if we are a shotgun-type weapon
	if (AreaOfEffect == AOE_Cone)
		numSlugs = 5;
	else
		numSlugs = 1;

	// if there is a scope, but the player isn't using it, decrease the accuracy
	// so there is an advantage to using the scope
	if (bHasScope && !bZoomed)
		Accuracy += 0.2;
	// if the laser sight is on, make this shot dead on
	// also, if the scope is on, zero the accuracy so the shake makes the shot inaccurate
	else if (bLasing || bZoomed)
		Accuracy = 0.0;


	for (i=0; i<numSlugs; i++)
	{
    // If we have multiple slugs, then lower our accuracy a bit after the first slug so the slugs DON'T all go to the same place
    //if ((i > 0) && (Level.NetMode != NM_Standalone) && !(bHandToHand))
       //if (Accuracy < MinSpreadAcc)
          //Accuracy = MinSpreadAcc;

    // Let handtohand weapons have a better swing
    if ((bHandToHand) && (NumSlugs > 1) && (Level.NetMode != NM_Standalone))
    {
       StartTrace = ComputeProjectileStart(X,Y,Z);
       StartTrace = StartTrace + (numSlugs/2 - i) * SwingOffset;
    }

    EndTrace = StartTrace + Accuracy * (FRand()-0.5)*Y*1000 + Accuracy * (FRand()-0.5)*Z*1000 ;
    EndTrace += (FMax(1024.0, MaxRange) * vector(AdjustedAim));
    
    Other = Pawn(Owner).TraceShot(HitLocation,HitNormal,EndTrace,StartTrace);

		// randomly draw a tracer for relevant ammo types
		// don't draw tracers if we're zoomed in with a scope - looks stupid
      // DEUS_EX AMSD In multiplayer, draw tracers all the time.
		if ( ((Level.NetMode == NM_Standalone) && (!bZoomed && (numSlugs == 1) && (FRand() < 0.5))) ||
           ((Level.NetMode != NM_Standalone) && (Role == ROLE_Authority) && (numSlugs == 1)) )
		{
			if ((AmmoName == Class'HXAmmo10mm') || (AmmoName == Class'HXAmmo3006') ||
				(AmmoName == Class'HXAmmo762mm'))
			{
				if (VSize(HitLocation - StartTrace) > 250)
				{
					rot = Rotator(EndTrace - StartTrace);
               if ((Level.NetMode != NM_Standalone) && (Self.IsA('WeaponRifle')))
                  Spawn(class'SniperTracer',,, StartTrace + 96 * Vector(rot), rot);
               else
                  Spawn(class'Tracer',,, StartTrace + 96 * Vector(rot), rot);
				}
			}
		}

		// check our range
		dist = Abs(VSize(HitLocation - Owner.Location));

		if (dist <= AccurateRange)		// we hit just fine
			ProcessTraceHit(Other, HitLocation, HitNormal, vector(AdjustedAim),Y,Z);
		else if (dist <= MaxRange)
		{
			// simulate gravity by lowering the bullet's hit point
			// based on the owner's distance from the ground
			alpha = (dist - AccurateRange) / (MaxRange - AccurateRange);
			degrade = 0.5 * Square(alpha);
			HitLocation.Z += degrade * (Owner.Location.Z - Owner.CollisionHeight);
			ProcessTraceHit(Other, HitLocation, HitNormal, vector(AdjustedAim),Y,Z);
		}
	}

	// otherwise we don't hit the target at all
}


// ----------------------------------------------------------------------
// ProcessTraceHit()
// ----------------------------------------------------------------------

simulated function ProcessTraceHit(Actor Other, Vector HitLocation, Vector HitNormal, Vector X, Vector Y, Vector Z)
{
	local float        mult;
	local name         damageType;
	local DeusExPlayer dxPlayer;

	if (Other != None)
	{
		// AugCombat increases our damage if hand to hand
		mult = 1.0;
		if (bHandToHand && (DeusExPlayer(Owner) != None))
		{
			mult = DeusExPlayer(Owner).AugmentationSystem.GetAugLevelValue(class'AugCombat');
			if (mult == -1.0)
				mult = 1.0;
		}

		// skill also affects our damage
		// GetWeaponSkill returns 0.0 to -0.7 (max skill/aug)
		mult += -2.0 * GetWeaponSkill();

		// Determine damage type
		damageType = WeaponDamageType();

		if (Other != None)
		{
			if (Other.bOwned)
			{
				dxPlayer = DeusExPlayer(Owner);
				if (dxPlayer != None)
					dxPlayer.AISendEvent('Futz', EAITYPE_Visual);
			}
		}
		if ((Other == Level) || (Other.IsA('Mover')))
		{
			if ( Role == ROLE_Authority )
				Other.TakeDamage(HitDamage * mult, Pawn(Owner), HitLocation, 1000.0*X, damageType);

			SelectiveSpawnEffects( HitLocation, HitNormal, Other, HitDamage * mult);
		}
		else if ((Other != self) && (Other != Owner))
		{
			if ( Role == ROLE_Authority )
				Other.TakeDamage(HitDamage * mult, Pawn(Owner), HitLocation, 1000.0*X, damageType);
			if (bHandToHand)
				SelectiveSpawnEffects( HitLocation, HitNormal, Other, HitDamage * mult);

			if (bPenetrating && Other.IsA('Pawn') && !Other.IsA('HXRobot'))
				SpawnBlood(HitLocation, HitNormal);
		}
	}
   if (DeusExMPGame(Level.Game) != None)
   {
      if (DeusExPlayer(Other) != None)
         DeusExMPGame(Level.Game).TrackWeapon(self,HitDamage * mult);
      else
         DeusExMPGame(Level.Game).TrackWeapon(self,0);
   }
}


// ----------------------------------------------------------------------
// UpdateInfo()
// ----------------------------------------------------------------------

simulated function bool UpdateInfo(Object winObject)
{
	local PersonaInventoryInfoWindow winInfo;
	local string str;
	local int i, dmg;
	local float mod;
	local bool bHasAmmo;
	local bool bAmmoAvailable;
	local class<DeusExAmmo> ammoClass;
	local Pawn P;
	local Ammo weaponAmmo;
	local int  ammoAmount;

	P = Pawn(Owner);
	winInfo = PersonaInventoryInfoWindow(winObject);

	if (P == none || winInfo == none)
		return false;

	winInfo.SetTitle(itemName);
	winInfo.SetText(msgInfoWeaponStats);
	winInfo.AddLine();

	// Create the ammo buttons.  Start with the AmmoNames[] array,
	// which is used for weapons that can use more than one 
	// type of ammo.

	if (AmmoNames[0] != None)
	{
		for (i=0; i<ArrayCount(AmmoNames); i++)
		{
			if (AmmoNames[i] != None) 
			{
				// Check to make sure the player has this ammo type
				// *and* that the ammo isn't empty
				weaponAmmo = Ammo(P.FindInventoryType(AmmoNames[i]));

				if (weaponAmmo != None)
				{
					ammoAmount = weaponAmmo.AmmoAmount;
					bHasAmmo = (weaponAmmo.AmmoAmount > 0);
				}
				else
				{
					ammoAmount = 0;
					bHasAmmo = False;
				}

				winInfo.AddAmmo(AmmoNames[i], bHasAmmo, ammoAmount);
				bAmmoAvailable = True;

				if (AmmoNames[i] == AmmoName)
				{
					winInfo.SetLoaded(AmmoName);
					ammoClass = class<DeusExAmmo>(AmmoName);
				}
			}
		}
	}
	else
	{
		// Now peer at the AmmoName variable, but only if the AmmoNames[] 
		// array is empty
		if ((AmmoName != class'AmmoNone') && (!bHandToHand) && (ReloadCount != 0))
		{	
			weaponAmmo = Ammo(P.FindInventoryType(AmmoName));

			if (weaponAmmo != None)
			{
				ammoAmount = weaponAmmo.AmmoAmount;
				bHasAmmo = (weaponAmmo.AmmoAmount > 0);
			}
			else
			{
				ammoAmount = 0;
				bHasAmmo = False;
			}

			winInfo.AddAmmo(AmmoName, bHasAmmo, ammoAmount);
			winInfo.SetLoaded(AmmoName);
			ammoClass = class<DeusExAmmo>(AmmoName);
			bAmmoAvailable = True;
		}
	}

	// Only draw another line if we actually displayed ammo.
	if (bAmmoAvailable)
		winInfo.AddLine();	

	// Ammo loaded
	if ((AmmoName != class'AmmoNone') && (!bHandToHand) && (ReloadCount != 0))
		winInfo.AddAmmoLoadedItem(msgInfoAmmoLoaded, AmmoType.itemName);

	// ammo info
	if ((AmmoName == class'AmmoNone') || bHandToHand || (ReloadCount == 0))
		str = msgInfoNA;
	else
		str = AmmoName.Default.ItemName;
	for (i=0; i<ArrayCount(AmmoNames); i++)
		if ((AmmoNames[i] != None) && (AmmoNames[i] != AmmoName))
			str = str $ "|n" $ AmmoNames[i].Default.ItemName;

	winInfo.AddAmmoTypesItem(msgInfoAmmo, str);

	// base damage
	if (AreaOfEffect == AOE_Cone)
	{
		if (bInstantHit)
			dmg = Default.HitDamage * 5;
		else
			dmg = Default.HitDamage * 3;
	}
	else
		dmg = Default.HitDamage;


	str = String(dmg);
	mod = 1.0 - GetWeaponSkill();

	if (mod != 1.0)
		str = str @ BuildPercentString(mod - 1.0) @ "=" @ FormatFloatString(dmg * mod, 1.0);

	winInfo.AddInfoItem(msgInfoDamage, str, (mod != 1.0));

	// clip size
	if ((Default.ReloadCount == 0) || bHandToHand)
		str = msgInfoNA;
	else
		str = Default.ReloadCount @ msgInfoRounds;

	if (HasClipMod())
		str = str @ BuildPercentString(ModReloadCount) @ "=" @ ReloadCount @ msgInfoRounds;

	winInfo.AddInfoItem(msgInfoClip, str, HasClipMod());

	// rate of fire
	if ((Default.ReloadCount == 0) || bHandToHand)
	{
		str = msgInfoNA;
	}
	else
	{
		if (bAutomatic)
			str = msgInfoAuto;
		else
			str = msgInfoSingle;

		str = str $ "," @ FormatFloatString(1.0/Default.ShotTime, 0.1) @ msgInfoRoundsPerSec;
	}
	winInfo.AddInfoItem(msgInfoROF, str);

	// reload time
	if ((Default.ReloadCount == 0) || bHandToHand)
		str = msgInfoNA;
	else
		str = FormatFloatString(Default.ReloadTime, 0.1) @ msgTimeUnit;

	if (HasReloadMod())
		str = str @ BuildPercentString(ModReloadTime) @ "=" @ FormatFloatString(ReloadTime, 0.1) @ msgTimeUnit;

	winInfo.AddInfoItem(msgInfoReload, str, HasReloadMod());

	// recoil
	str = FormatFloatString(Default.recoilStrength, 0.01);
	if (HasRecoilMod())
		str = str @ BuildPercentString(ModRecoilStrength) @ "=" @ FormatFloatString(recoilStrength, 0.01);

	winInfo.AddInfoItem(msgInfoRecoil, str, HasRecoilMod());

	str = Int((2.0 - Default.BaseAccuracy)*50.0) $ "%";
	mod = (Default.BaseAccuracy - (BaseAccuracy + GetWeaponSkill())) * 0.5;
	if (mod != 0.0)
	{
		str = str @ BuildPercentString(mod);
		str = str @ "=" @ Min(100, Int(100.0*mod+(2.0 - Default.BaseAccuracy)*50.0)) $ "%";
	}

	winInfo.AddInfoItem(msgInfoAccuracy, str, (mod != 0.0));

	// accurate range
	if (bHandToHand)
		str = msgInfoNA;
	else
		str = FormatFloatString(Default.AccurateRange/16.0, 1.0) @ msgRangeUnit;

	if (HasRangeMod())
		str = str @ BuildPercentString(ModAccurateRange) @ "=" @ FormatFloatString(AccurateRange/16.0, 1.0) @ msgRangeUnit;

	winInfo.AddInfoItem(msgInfoAccRange, str, HasRangeMod());

	// max range
	if (bHandToHand)
		str = msgInfoNA;
	else
		str = FormatFloatString(Default.MaxRange/16.0, 1.0) @ msgRangeUnit;

	winInfo.AddInfoItem(msgInfoMaxRange, str);

	// mass
	winInfo.AddInfoItem(msgInfoMass, FormatFloatString(Default.Mass, 1.0) @ msgMassUnit);

	// laser mod
	if (bCanHaveLaser)
	{
		if (bHasLaser)
			str = msgInfoYes;
		else
			str = msgInfoNo;
	}
	else
	{
		str = msgInfoNA;
	}
	winInfo.AddInfoItem(msgInfoLaser, str, bCanHaveLaser && bHasLaser && (Default.bHasLaser != bHasLaser));

	// scope mod
	if (bCanHaveScope)
	{
		if (bHasScope)
			str = msgInfoYes;
		else
			str = msgInfoNo;
	}
	else
	{
		str = msgInfoNA;
	}
	winInfo.AddInfoItem(msgInfoScope, str, bCanHaveScope && bHasScope && (Default.bHasScope != bHasScope));

	// silencer mod
	if (bCanHaveSilencer)
	{
		if (bHasSilencer)
			str = msgInfoYes;
		else
			str = msgInfoNo;
	}
	else
	{
		str = msgInfoNA;
	}
	winInfo.AddInfoItem(msgInfoSilencer, str, bCanHaveSilencer && bHasSilencer && (Default.bHasSilencer != bHasSilencer));

	// Governing Skill
	winInfo.AddInfoItem(msgInfoSkill, GoverningSkill.default.SkillName);

	winInfo.AddLine();
	winInfo.SetText(Description);

	// If this weapon has ammo info, display it here
	if (ammoClass != None)
	{
		winInfo.AddLine();
		winInfo.AddAmmoDescription(ammoClass.Default.ItemName $ "|n" $ ammoClass.Default.description);
	}

	return True;
}


// ----------------------------------------------------------------------
// state DownWeapon
// ----------------------------------------------------------------------

state DownWeapon
{
ignores Fire, AltFire;

	function bool PutDown()
	{
		// alert NPCs that I'm putting away my gun
		AIEndEvent('WeaponDrawn', EAITYPE_Visual);
		return Super.PutDown();
	}

Begin:
  ScopeOff();
	LaserOff();

	if (( Level.NetMode == NM_DedicatedServer ) || ((Level.NetMode == NM_ListenServer) && Owner.IsA('DeusExPlayer') && !DeusExPlayer(Owner).PlayerIsListenClient()))
		ClientDownWeapon();

	TweenDown();
	FinishAnim();

	//if (Level.NetMode != NM_Standalone)
		//ClipCount = 0;	// Auto-reload in multiplayer (when putting away)

	bOnlyOwnerSee = false;
	if (Pawn(Owner) != None)
		Pawn(Owner).ChangedWeapon();
}


// ----------------------------------------------------------------------
// BecomePickup()
// ----------------------------------------------------------------------

function BecomePickup()
{
	Super(Inventory).BecomePickup();
	SetDisplayProperties(Default.Style, Default.Texture, Default.bUnlit, Default.bMeshEnviromap );
/*  if (Level.NetMode != NM_Standalone)
      if (bTossedOut)
         Lifespan = 5.0;
*/
}


// ----------------------------------------------------------------------
// HandlePickupQuery()
// ----------------------------------------------------------------------

function bool HandlePickupQuery(Inventory Item)
{
	local DeusExWeapon W;
	local DeusExPlayer player;
	local bool bResult;
	local class<Ammo> defAmmoClass;
	local Ammo defAmmo;
	
	// make sure that if you pick up a modded weapon that you
	// already have, you get the mods
	W = DeusExWeapon(Item);
	if ((W != None) && (W.Class == Class))
	{
		if (W.ModBaseAccuracy > ModBaseAccuracy)
			ModBaseAccuracy = W.ModBaseAccuracy;
		if (W.ModReloadCount > ModReloadCount)
			ModReloadCount = W.ModReloadCount;
		if (W.ModAccurateRange > ModAccurateRange)
			ModAccurateRange = W.ModAccurateRange;

		// these are negative
		if (W.ModReloadTime < ModReloadTime)
			ModReloadTime = W.ModReloadTime;
		if (W.ModRecoilStrength < ModRecoilStrength)
			ModRecoilStrength = W.ModRecoilStrength;

		if (W.bHasLaser)
			bHasLaser = True;
		if (W.bHasSilencer)
			bHasSilencer = True;
		if (W.bHasScope)
			bHasScope = True;

		// copy the actual stats as well
		if (W.ReloadCount > ReloadCount)
			ReloadCount = W.ReloadCount;
		if (W.AccurateRange > AccurateRange)
			AccurateRange = W.AccurateRange;

		// these are negative
		if (W.BaseAccuracy < BaseAccuracy)
			BaseAccuracy = W.BaseAccuracy;
		if (W.ReloadTime < ReloadTime)
			ReloadTime = W.ReloadTime;
		if (W.RecoilStrength < RecoilStrength)
			RecoilStrength = W.RecoilStrength;
	}
	player = DeusExPlayer(Owner);

	if (Item.Class == Class)
	{
      if (!( (Weapon(item).bWeaponStay && (Level.NetMode == NM_Standalone)) && (!Weapon(item).bHeldItem || Weapon(item).bTossedOut)))
		{
			// Only add ammo of the default type
			// There was an easy way to get 32 20mm shells, buy picking up another assault rifle with 20mm ammo selected
			if ( AmmoType != None )
			{
				// Add to default ammo only
				if ( AmmoNames[0] == None )
					defAmmoClass = AmmoName;
				else
					defAmmoClass = AmmoNames[0];

				defAmmo = Ammo(player.FindInventoryType(defAmmoClass));
				defAmmo.AddAmmo( Weapon(Item).PickupAmmoCount );

				if ( Level.NetMode != NM_Standalone )
				{
					if (( player != None ) && ( player.InHand != None ))
					{
						if ( DeusExWeapon(item).class == DeusExWeapon(player.InHand).class )
							ReadyToFire();
					}
				}
			}
		}
	}

	bResult = Super.HandlePickupQuery(Item);

	// Notify the object belt of the new ammo
	if (player != None)
		player.UpdateBeltText(Self);

	return bResult;
}


// ----------------------------------------------------------------------
// WeaponDamageType()
// ----------------------------------------------------------------------

function name WeaponDamageType()
{
	local name                    damageType;
	local Class<DeusExProjectile> projClass;

	projClass = Class<DeusExProjectile>(ProjectileClass);
	if (bInstantHit)
	{
		if (StunDuration > 0)
			damageType = 'Stunned';
		else
			damageType = 'Shot';

		if (AmmoType != None)
			if (AmmoType.IsA('HXAmmoSabot'))
				damageType = 'Sabot';
	}
	else if (projClass != None)
		damageType = projClass.Default.damageType;
	else
		damageType = 'None';

	return (damageType);
}


// ----------------------------------------------------------------------
// SelectiveSpawnEffects()
// Continues the simulated chain for the owner, and spawns the effects 
// for other players that can see them
// No actually spawning occurs on the server itself
// ----------------------------------------------------------------------

simulated function SelectiveSpawnEffects( Vector HitLocation, Vector HitNormal, Actor Other, float Damage)
{
	local DeusExPlayer fxOwner;
	local Pawn aPawn;

	// The normal path before there was multiplayer
	if ( Level.NetMode == NM_Standalone )
	{
		SpawnEffects(HitLocation, HitNormal, Other, Damage);
		return;
	}

	fxOwner = DeusExPlayer(Owner);

	if ( Role == ROLE_Authority )
	{
		SpawnEffectSounds(HitLocation, HitNormal, Other, Damage );

		for ( aPawn = Level.PawnList; aPawn != None; aPawn = aPawn.nextPawn )
		{
			if ( aPawn.IsA('DeusExPlayer') && ( DeusExPlayer(aPawn) != fxOwner ) )
			{
				if ( DeusExPlayer(aPawn).FastTrace( DeusExPlayer(aPawn).Location, HitLocation ))
					DeusExPlayer(aPawn).ClientSpawnHits( bPenetrating, bHandToHand, HitLocation, HitNormal, Other, Damage );
			}
		}
	}
	if ( fxOwner == DeusExPlayer(GetPlayerPawn()) )
	{
			fxOwner.ClientSpawnHits( bPenetrating, bHandToHand, HitLocation, HitNormal, Other, Damage );
			SpawnEffectSounds( HitLocation, HitNormal, Other, Damage );
	}
}


// ----------------------------------------------------------------------
// ----------------------------------------------------------------------

defaultproperties
{
}
