//=============================================================================
// HXGamePlayState
//=============================================================================
class HXGamePlayState extends Info;

// List of Keys
var NanoKeyInfo KeyList;

// Goal Tracking
var DeusExGoal FirstGoal;	
var DeusExGoal LastGoal;

// string for nanokey
var localized String AddedNanoKeyBy;
var localized String NanoKeyAlreadyFound;

// flags
var FlagBase flags;

// dry runner for datalinks
var HXDataLinkEventDryRunner dryrunner;
var bool bDryRunning;
var HXDataLinkTrack tracklist;

// for bound datalinks
var HXDataLinkBindDecoration bindDeco;

// ----------------------------------------------------------------------
// PreBeginPlay()
// ----------------------------------------------------------------------

function PreBeginPlay()
{
	// create flagbase
	flags = new class'FlagBase';

	// spawn deco for datalinks
	bindDeco = spawn(class'HXDataLinkBindDecoration');
	if (bindDeco == none)
		Log("(WW) Spawning BindDeco failed");

	// spawn dry runner
	dryrunner = spawn(class'HXDataLinkEventDryRunner');
	bDryRunning = false;

	Super.PreBeginPlay();
}


// ----------------------------------------------------------------------
// BroadcastDataLink()
// ----------------------------------------------------------------------

function bool BroadcastDataLink
(
	string datalinkName,
	string invokerName,
	optional HXDataLinkTrigger datalinkTrigger
)
{
	local ConListItem conListItem;
	local Conversation con;
	local HXDataLinkTrack track;
	local Pawn P;

	// find conversation
	conListItem = ConListItem(bindDeco.conListItems);

	while (conListItem != none)
	{
		if (Caps(datalinkName) == Caps(conListItem.con.conName))
		{
			// ok this is the conversation we are searching for
			con = conListItem.con;
			break;
		}
		conListItem = conListItem.next;	
	}

	if (con == none || IsConQueued(con))
		return false;

	// check for played flag
	if (con.bDisplayOnce)
		if (flags.GetBool(flags.StringToName(con.conName $ "_Played")))
			return false;

	track = spawn(class'HXDataLinkTrack');
	track.con = con;
	track.invokerName = invokerName;
	track.trigger = datalinkTrigger;
	
	QueueTrack(track);

	// broadcast datalink
	for(P = Level.PawnList; P != none; P = P.nextPawn)
	{
		if(P.bIsPlayer && P.isA('HXHuman'))
		{
			HXHuman(P).QueueDataLink(datalinkName, invokerName, datalinkTrigger);
		}
	}

	return true;
}


// ----------------------------------------------------------------------
// StartDryRunner()
// ----------------------------------------------------------------------

function StartDryRunner()
{
	local HXDataLinkTrack track;

	if (bDryRunning || tracklist == none)
		return;

	bDryRunning = true;

	track = tracklist;

	dryrunner.con					= track.con;
	dryrunner.current			= track.con.eventList;
	dryrunner.invokerName	= track.invokerName;

	dryrunner.GotoState('Starting');
}


// ----------------------------------------------------------------------
// QueueTrack()
// ----------------------------------------------------------------------

function QueueTrack(HXDataLinkTrack track)
{
	if (tracklist != none)
		tracklist.QueueTrack(track);
	else
		tracklist = track;

	if (!bDryRunning)
		StartDryRunner(); // play track instantly if idle
}


// ----------------------------------------------------------------------
// IsConQueued()
// ----------------------------------------------------------------------

function bool IsConQueued(Conversation con)
{
	local HXDataLinkTrack track;
	
	if (con == none)
		return false;

	track = tracklist;

	while (track != none)
	{
		if (track.con == con)
			return true;

		track = track.next;
	}

	return false;
}


// ----------------------------------------------------------------------
// DryRunnerFinished()
// ----------------------------------------------------------------------

function DryRunnerFinished()
{
	local HXDataLinkTrack track;

	track = tracklist;

	// notice trigger that datalink has finished
 	if (track.trigger != none)
		track.trigger.DatalinkFinished();

	// remove first track from list and destroy it
	tracklist = track.next;
	track.Destroy();

	bDryRunning = false;

	if (tracklist != none)
		StartDryRunner();
}


// ----------------------------------------------------------------------
// AddNanoKey()
// ----------------------------------------------------------------------

function AddNanoKey(name ID, string desc, HXHuman by)
{
	local Pawn P;
	local NanoKeyInfo aKey;

	// check if key was already found
	aKey = KeyList;

	while(aKey != None)
	{
		if (aKey.KeyID == ID)
		{
			by.ClientMessage(Sprintf(NanoKeyAlreadyFound, desc));
			return;
		}
		aKey = aKey.NextKey;
	}

	// save in keylist 
	aKey = new(Self) Class'NanoKeyInfo';
	aKey.KeyID				= ID;
	aKey.Description	= desc;
	aKey.NextKey			= KeyList;
	KeyList						= aKey;


	// give key to all players
	for(P = Level.PawnList; P != none; P = P.nextPawn)
	{
		if(P.bIsPlayer)
		{
			// give the player the key
			HXHuman(P).KeyRing.GiveKey(ID, desc);

			// notifiy that key was added and who found it
			P.ClientMessage(Sprintf(AddedNanoKeyBy, desc, by.PlayerReplicationInfo.PlayerName));

			// propagate to client
			if (Role == ROLE_Authority)
				HXHuman(P).KeyRing.GiveClientKey(ID, desc);
		}
	}
}


// ----------------------------------------------------------------------
// GetAllKeys()
// ----------------------------------------------------------------------

function GetAllKeys(HXHuman Other)
{
	local NanoKeyInfo Key;

	if (Other == none)
		return;

	Key = KeyList;

	while (Key != None)
	{
		Other.KeyRing.GiveKey(Key.KeyID, Key.Description);

		if (Role == ROLE_Authority)
			Other.KeyRing.GiveClientKey(Key.KeyID, Key.Description);

		Key = Key.NextKey;
	}
}


// ----------------------------------------------------------------------
// AddGoal()
//
// Adds a new goal to the list of goals the player is carrying around.
// ----------------------------------------------------------------------

function DeusExGoal AddGoal(Name goalName, bool bPrimaryGoal, string goalText, string by)
{	
	local DeusExGoal newGoal;
	local Pawn P;

	// First check to see if this goal already exists.  If so, we'll just
	// return it.  Otherwise create a new goal

	newGoal = FindGoal(goalName);

	if (newGoal == None)
	{
		newGoal = new(Self) Class'DeusExGoal';
		newGoal.SetName(goalName);
		newGoal.SetText(goalText);
		newGoal.SetPrimaryGoal(bPrimaryGoal);

		// Insert goal at the Top so goals are displayed in 
		// Newest order first.
		if (FirstGoal == None)
			LastGoal  = newGoal;
		else
			newGoal.next = FirstGoal;

		FirstGoal    = newGoal;

		for(P = Level.PawnList; P != none; P = P.nextPawn)
		{
			if(P.bIsPlayer)
			{
				HXHuman(P).GoalCacheAddGoal(goalName, bPrimaryGoal, goalText);

				if (by == "")
					P.ClientMessage(class'DeusExPlayer'.Default.GoalAdded);
				else
					P.ClientMessage(Sprintf((class'DeusExPlayer'.Default.GoalAdded $ " (%s)"), by));
			}
		}

		// TODO
		//DeusExRootWindow(rootWindow).hud.msgLog.PlayLogSound(Sound'LogGoalAdded');
	}

	return newGoal;	
}

// ----------------------------------------------------------------------
// FindGoal()
// ----------------------------------------------------------------------

function DeusExGoal FindGoal(Name goalName)
{
	local DeusExGoal goal;

	goal = FirstGoal;

	while(goal != none)
	{
		if (goalName == goal.goalName)
			break;

		goal = goal.next;
	}

	return goal;
}


// ----------------------------------------------------------------------
// GoalCompleted()
//
// Looks up the goal and marks it as completed.
// ----------------------------------------------------------------------

function GoalCompleted(name goalName, string by)
{
	local DeusExGoal goal;
	local Pawn P;

	// Loop through all the goals until we hit the one we're 
	// looking for.
	goal = FindGoal(goalName);

	if (goal != none)
	{
		// Only mark a goal as completed once!
		if (!goal.IsCompleted())
		{
			goal.SetCompleted();

			// TODO
			//DeusExRootWindow(rootWindow).hud.msgLog.PlayLogSound(Sound'LogGoalCompleted');

			for(P = Level.PawnList; P != none; P = P.nextPawn)
			{
				if(P.bIsPlayer)
				{
					HXHuman(P).GoalCompleted(goalName);

					// Let the player know
					if (by == "")
					{
						if (goal.bPrimaryGoal)
							P.ClientMessage(class'DeusExPlayer'.Default.PrimaryGoalCompleted);
						else
							P.ClientMessage(class'DeusExPlayer'.Default.SecondaryGoalCompleted);					
					}
					else
					{
						if (goal.bPrimaryGoal)
							P.ClientMessage(Sprintf((class'DeusExPlayer'.Default.PrimaryGoalCompleted $ " (%s)"), by));
						else
							P.ClientMessage(Sprintf((class'DeusExPlayer'.Default.SecondaryGoalCompleted $ " (%s)"), by));
					}
				}
			}
		}
	}
}

// ----------------------------------------------------------------------
// DeleteGoal()
//
// Deletes the specified note
// Returns True if the note successfully deleted
// ----------------------------------------------------------------------

function Bool DeleteGoal(DeusExGoal goalToDelete)
{
	local DeusExGoal goal;
	local DeusExGoal previousGoal;
	local Bool bGoalDeleted;

	bGoalDeleted = False;
	goal = FirstGoal;
	previousGoal = None;

	while( goal != None )
	{
		if ( goal == goalToDelete )
		{
			if ( goal == FirstGoal )
				FirstGoal = goal.next;

			if ( goal == LastGoal )
				LastGoal = previousGoal;

			if ( previousGoal != None )
				previousGoal.next = goal.next;

			goal = None;
						
			bGoalDeleted = True;	
			break;
		}
		previousGoal = goal;
		goal = goal.next;
	}

	return bGoalDeleted;
}

// ----------------------------------------------------------------------
// DeleteAllGoals()
//
// Deletes *ALL* Goals
// ----------------------------------------------------------------------

function DeleteAllGoals()
{
	local DeusExGoal goal;
	local DeusExGoal goalNext;
	local Pawn P;

	goal = FirstGoal;

	while(goal != None)
	{
		goalNext = goal.next;
		DeleteGoal(goal);
		goal = goalNext;
	}

	FirstGoal = None;
	LastGoal = None;


	for(P=Level.PawnList; P!=None; P=P.nextPawn)
		if(P.bIsPlayer)
			HXHuman(P).DeleteAllGoals();

}

// ----------------------------------------------------------------------
// ResetGoals()
// 
// Called when progressing to the next mission.  Deletes all 
// completed Primary Goals as well as *ALL* Secondary Goals 
// (regardless of status)
// ----------------------------------------------------------------------

function ResetGoals()
{
	local DeusExGoal goal;
	local DeusExGoal goalNext;
	local Pawn P;

	goal = FirstGoal;

	while( goal != None )
	{
		goalNext = goal.next;

		// Delete:
		// 1) Completed Primary Goals
		// 2) ALL Secondary Goals

		if ((!goal.IsPrimaryGoal()) || (goal.IsPrimaryGoal() && goal.IsCompleted()))
			DeleteGoal(goal);

		goal = goalNext;
	}

	for(P=Level.PawnList; P!=None; P=P.nextPawn)
		if(P.bIsPlayer)
			HXHuman(P).ResetGoals();
}

// ----------------------------------------------------------------------
// ----------------------------------------------------------------------

defaultproperties
{
	AddedNanoKeyBy="%s added to Nano Key Ring (%s)"
	NanoKeyAlreadyFound="%s was already found"
}
